
package org.ifarchive.glk;

/** A file reference contains platform-specific information about the name
 * and location of the file, and possibly its type, if the platform has
 * a notion of file type. It also includes a flag indication whether the
 * file is a text file or binary file.
 *
 * @see java.io.File
 */

public class FileRef extends OpaqueObject
{
	FileRef(int id)
	{
		super(clFileRef, id);
	}

	static FileRef findFileRef(int id)
	{
		return (FileRef)(find(clFileRef, id));
	}

        /** Iterate over file references. 
         * @see OpaqueObject#iterate
         */
        public static FileRef iterate(FileRef f, Object r[])
        {
                return (FileRef)iterate(clFileRef,f,r);
        }


	/** This file stores arbitrary data (preferences, statistics...) */
	public static final int fileusage_Data        = 0x00;
	/** This file stores a saved game. */
	public static final int fileusage_SavedGame   = 0x01;
	/** This file stores a stream of text from the game. */
	public static final int fileusage_Transcript  = 0x02;
	/** This file stores a recording of player input. */
	public static final int fileusage_InputRecord = 0x03;
	/** Mask for file type bits. */
	public static final int fileusage_TypeMask    = 0x0F;

	/** Open the file this refers to in text mode. */	
	public static final int fileusage_TextMode    = 0x100;
	/** Open the file this refers to in binary mode. */	
	public static final int fileusage_BinaryMode  = 0x000;

	/** Open the file this refers to in write mode. */	
	public static final int filemode_Write        = 0x01;
	/** Open the file this refers to in read mode. */	
	public static final int filemode_Read         = 0x02;
	/** Open the file this refers to in read/write mode. */	
	public static final int filemode_ReadWrite    = 0x03;
	/** Open the file this refers to in append mode. */	
	public static final int filemode_WriteAppend  = 0x05;

	/** Create a file reference by asking the player to type a name.	
	 * @param usage The file mode (text/binary) and type. 
	 * @param mode The mode in which to open the file (filemode_*)
	 * @return A new Fileref, or null if one was not created (eg: user cancel).
	 */
	public static FileRef createByPrompt(int usage, int mode)
	{
		return createByPrompt(usage, mode, null);
	}

	/** Create a file reference by asking the player to type a name.	
	 * @param usage The file mode (text/binary) and type. 
	 * @param mode The mode in which to open the file (filemode_*)
	 * @param jrock The rock value to store.
	 * @return A new Fileref, or null if one was not created (eg: user cancel).
	 */
	public static FileRef createByPrompt(int usage, int fmode, Object jrock)
	{
		int irock = castRock(jrock);
		int id = Glk.jniglk_fileref_create_by_prompt(usage, fmode, irock);	
		FileRef f = findFileRef(id);
		if (f != null) f.m_jrock = jrock;
		return f;
	}

	/** This creates a reference to a temporary file. It is always a new file
	 * (one which does not yet exist). The file (once created) will be somewhere
	 * out of the player's way. 
	 * 
	 * <p>A temporary file should not be used for long-term storage. It may be
	 * deleted automatically when the program exits, or at some later time,
	 * say when the machine is turned off or rebooted. You do not have to worry
	 * about deleting it yourself.</p>
	 * @param usage The file mode (text/binary) and type. 
	 * @param jrock The rock value to store.
	 * @return A new Fileref, or null if one was not created.
	 */
        public static FileRef createTemp(int usage, Object jrock)
	{
		int irock = castRock(jrock);
		int id = Glk.jniglk_fileref_create_temp(usage, irock);
		FileRef f = findFileRef(id);
		if (f != null) f.m_jrock = jrock;
		return f;
	}
	/** Create a reference to a file with a specific name.
	 * @see #createByName(usage,java.lang.String,java.lang.Object)
	 * @param usage The file mode (text/binary) and type. 
	 * @param name The filename to open.
	 * @return A new Fileref, or null if one was not created.
	 */
        public static FileRef createByName(int usage, String name)
	{
		return createByName(usage, name, null);
	}
	/** Create a reference to a file with a specific name.
	 * The file will be in a fixed location relevant to your program, 
	 * and visible to the player.
	 * 
	 * <p>Since filenames are highly platform-specific, you should use
	 * createByName() with care. It is legal to pass any string
	 * in the name argument. However, the library may have to mangle, transform,
	 * or truncate the string to make it a legal native filename.</p>
	 * <p>The most conservative approach is to pass a string of no more than
	 * 8 characters, consisting entirely of upper-case letters and numbers,
	 * starting with a letter. You can then be reasonably sure that the resulting
	 * filename will display all the characters you specify -- in some form.</p>
	 * @param usage The file mode (text/binary) and type. 
	 * @param name The filename to open.
	 * @param jrock The rock value to store.
	 * @return A new Fileref, or null if one was not created.
	 */
        public static FileRef createByName(int usage, String name, Object jrock)
	{
		int irock = castRock(jrock);
		int id = Glk.jniglk_fileref_create_by_name(usage, name, irock);
		FileRef f = findFileRef(id);
		if (f != null) f.m_jrock = jrock;
		return f;
	}

	/** This copies an existing file reference, but changes the usage. 
	 * @param usage The new usage.
	 * @return A copy of this file reference.
	 * @see #createFromFileRef(int,java.lang.Object)
	 */
	public FileRef createFromFileRef(int usage) 
	{
		return createFromFileRef(usage, null);
	}	

	/** This copies an existing file reference, but changes the usage. 
	 * (The original fileref is not modified.)
	 * 
	 * <p>The use of this function can be tricky. If you change the type of the
	 * fileref (fileusage_Data, fileusage_SavedGame, etc), the new reference may
	 * or may not point to the same actual disk file.</p>
	 * 
	 * <p>If you do this, and open both file references for writing, the results are
	 * unpredictable. It is safest to change the type of a fileref only if it
	 * refers to a nonexistent file.</p>
	 * 
	 * <p>If you change the mode of a fileref (fileusage_TextMode,
	 * fileusage_BinaryMode), but leave the rest of the type unchanged, the
	 * new fileref will definitely point to the same disk file as the old one.</p>
	 * @param usage The new usage.
	 * @param jrock The rock value to store.
	 * @return A copy of this file reference.
	 */ 
	public FileRef createFromFileRef(int usage, Object jrock)
	{
		int irock = castRock(jrock);
		int id = Glk.jniglk_fileref_create_from_fileref(usage, m_dispid, irock);
		FileRef f = findFileRef(id);
		if (f != null) f.m_jrock = jrock;
		return f;
	}

	protected void finalize() { if (m_dispid != 0) destroy(); }

	/** Destroys a fileref which you have created. This does <em>not</em> affect
	 * the disk file; it just reclaims the resources allocated by the
	 * FileRef.create... function.
	 * 
	 * <p>It is legal to destroy a fileref after opening a file with it (while the
	 * file is still open.) The fileref is only used for the opening operation,
	 * not for accessing the file stream.</p>
	 */
	public void destroy()
	{
                Glk.jniglk_fileref_destroy(m_dispid);
                super.destroy();
	}

	/** Open a stream on the file this refers to. 
	 * @param fmode The mode to open the file in (filemode_*)
	 * @return The opened stream, or null if opening failed. */
	public Stream openFile(int fmode) { return openFile(fmode, null); }
	/** Open a stream on the file this refers to. 
	 * @param fmode The mode to open the file in (filemode_*)
	 * @param jrock The rock for the stream.	
	 * @return The opened stream, or null if opening failed. */
	public Stream openFile(int fmode, Object jrock)
	{
		int irock = castRock(jrock);
		int id = Glk.jniglk_stream_open_file(m_dispid, fmode, irock);
		Stream s = Stream.findStream(id);
		if (s != null) s.m_jrock = jrock;
		return s;
	}

	/** This deletes the file this FileRef refers to.
	 * It does not destroy the FileRef itself. */
	public void deleteFile()
	{
		Glk.jniglk_fileref_delete_file(m_dispid);
	}
	/** See if this fileref refers to an existing file.
	 * @return true if the file exists, else false.
	 */
	public boolean doesFileExist()
	{
		return Glk.jniglk_fileref_does_file_exist(m_dispid);
	}


        long getGlkRock()       
        {
                return Glk.jniglk_fileref_get_rock(m_dispid);
        }
}


