/* things.c: Code for various Floo structures.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/floo/index.html
*/

#include "glk.h"
#include "floo.h"
#include <string.h>
#include <stdlib.h>

/* Some counters for debugging. */
static int objcount = 0;
static int refcount = 0;

/* Create a stack. */
stack_t *new_stack()
{
    stack_t *st = (stack_t *)malloc(sizeof(stack_t));
    if (!st)
        return NULL;
    st->stack_size = 128;
    st->stack = (obj_t **)malloc(st->stack_size * sizeof(obj_t *));
    if (!st->stack)
        return NULL;
    st->stackpos = 0;
    return st;
}

/* Takes a reference from you, puts it on stack. */
void stack_push(stack_t *st, obj_t *ob)
{
    if (st->stackpos >= st->stack_size) {
        st->stack_size *= 2;
        st->stack = (obj_t **)realloc(st->stack, st->stack_size * sizeof(obj_t *));
        if (!st->stack)
            floo_err("Stack overflow.");
    }
    st->stack[st->stackpos] = ob;
    st->stackpos++;
}

/* Takes reference from stack, hands it to you. */
obj_t *stack_pop(stack_t *st)
{
    if (st->stackpos <= 0)
        return NULL;
    st->stackpos--;
    return st->stack[st->stackpos];
}

/* Makes a new reference to an item of the stack, and hands it over. */
obj_t *stack_peek(stack_t *st, int depth)
{
    if (depth < 0 || st->stackpos <= depth)
        return NULL;
    return obj_newref(st->stack[st->stackpos-(depth+1)]);
}

/* How many objects are there above the closest matching Mark?
    -1 means there are no such marks. */
int stack_searchmark(stack_t *st, int key)
{
    int ix, jx;
    for (ix = st->stackpos-1, jx=0; ix >= 0; ix--, jx++) {
        if (st->stack[ix]->type == otyp_Mark
            && st->stack[ix]->u.num == key)
            return jx;
    }
    return -1;
}

/* Creates an uninitialized object. Hands you a reference. */
obj_t *new_obj(int type)
{
    obj_t *ob = (obj_t *)malloc(sizeof(obj_t));
    ob->type = type;
    ob->u.num = 0;
    ob->u.arr.len = 0;
    ob->u.objframe.pos = 0;
    ob->refs = 1;
    objcount++;
    refcount++;
    return ob;
}

/* Creates a new array object. The elements of the array are all NULL
    pointers (not null objects!) and must be filled in before the array
    is made available to the user. */
obj_t *new_obj_array(int len, int executable)
{
    int ix;
    obj_t **ls;
    obj_t *ob = new_obj(executable ? otyp_Proc : otyp_Array);
    if (!ob)
        return NULL;
    
    if (len == 0) {
        ob->u.arr.len = 0;
        ob->u.arr.o = NULL;
        return ob;
    }
    
    ls = (obj_t **)malloc(len * sizeof(obj_t *));
    if (!ls)
        return NULL;
        
    for (ix=0; ix<len; ix++) {
        ls[ix] = NULL;
    }
    ob->u.arr.o = ls;
    ob->u.arr.len = len;
    return ob;
}

/* Throws away a reference. Deletes the object if there are no more references. */
void delete_obj(obj_t *ob)
{
    int ix;
    obj_t *ob2;
    
    if (!ob)
        floo_err("Deleting ???null.");
    
    refcount--;
    if (ob->refs > 1) {
        ob->refs--;
        return;
    }
    switch (ob->type) {
        case otyp_String:
            if (ob->u.str.s) {
                ob->u.str.s[0] = '\0';
                free(ob->u.str.s);
                ob->u.str.s = NULL;
                ob->u.str.len = 0;
            }
            break;
        case otyp_ID:
        case otyp_XID:
            if (ob->u.name.atom) {
                ob->u.name.atom = NULL;
            }
            break;
        case otyp_Array:
        case otyp_Proc:
            if (ob->u.arr.o) {
                for (ix=0; ix<ob->u.arr.len; ix++) {
                    ob2 = ob->u.arr.o[ix];
                    if (ob2) {
                        delete_obj(ob2);
                        ob->u.arr.o[ix] = NULL;
                    }
                }
                free(ob->u.arr.o);
                ob->u.arr.o = NULL;
            }
            ob->u.arr.len = 0;
            break;
        case otyp_ObjFrame:
            if (ob->u.objframe.proc) {
                delete_obj(ob->u.objframe.proc);
                ob->u.objframe.proc = NULL;
            }
            break;
        case otyp_FileFrame:
            /* don't close the file */
            break;
        case otyp_LoopFrame:
            if (ob->u.loopframe.obj) {
                delete_obj(ob->u.loopframe.obj);
                ob->u.loopframe.obj = NULL;
            }
            break;
        case otyp_Integer:
        case otyp_Boolean:
        case otyp_Mark:
        case otyp_Null:
        case otyp_FFunc:
            break;
        default:
            floo_err("Deleting unknown type.");
            break;
    }
    ob->type = 0;
    ob->refs = 0;
    free(ob);
    objcount--;
}

/* Leaves you the original ref, and also returns a newly-created one. */
obj_t *obj_newref(obj_t *ob)
{
    refcount++;
    ob->refs++;
    return ob;
}

int obj_debugobjcounter()
{
    return objcount;
}

int obj_debugrefcounter()
{
    return refcount;
}

/* Dump a visual representation of the object to the given stream. If str
    is zero, use the current output stream. (If there is none, forget it.)
    No reffing. */
void obj_stream(obj_t *ob, strid_t str)
{
    int ix;
    char buf[32];
    
    if (!str) {
        str = glk_stream_get_current();
        if (!str)
            return;
    }
    
    if (!ob) {
        glk_put_string_stream(str, "<???null>");
        return;
    }
    
    switch (ob->type) {
        case otyp_String:
            glk_put_char_stream(str, '"');
            glk_put_string_stream(str, ob->u.str.s);
            glk_put_char_stream(str, '"');
            break;
        case otyp_ID:
            glk_put_char_stream(str, '/');
            glk_put_buffer_stream(str, ob->u.name.atom->str, ob->u.name.atom->len);
            break;
        case otyp_XID:
            glk_put_buffer_stream(str, ob->u.name.atom->str, ob->u.name.atom->len);
            break;
        case otyp_Boolean:
            if (ob->u.num)
                glk_put_string_stream(str, "true");
            else
                glk_put_string_stream(str, "false");
            break;
        case otyp_Integer: 
            num_to_str(buf, ob->u.num);
            glk_put_string_stream(str, buf);
            break;
        case otyp_Null:
            glk_put_string_stream(str, "<null>");
            break;
        case otyp_Mark:
            glk_put_string_stream(str, "<mark>");
            break;
        case otyp_FFunc: 
            glk_put_string_stream(str, "<operator:");
            glk_put_buffer_stream(str, ob->u.ffunc.defatom->str, ob->u.ffunc.defatom->len);
            glk_put_string_stream(str, ">");
            break;
        case otyp_Array:
            glk_put_string_stream(str, "[ ");
            for (ix=0; ix<ob->u.arr.len; ix++) {
                obj_stream(ob->u.arr.o[ix], str);
                glk_put_char_stream(str, ' ');
            }
            glk_put_string_stream(str, "]");
            break;
        case otyp_Proc:
            glk_put_string_stream(str, "{ ");
            for (ix=0; ix<ob->u.arr.len; ix++) {
                obj_stream(ob->u.arr.o[ix], str);
                glk_put_char_stream(str, ' ');
            }
            glk_put_string_stream(str, "}");
            break;
        default:
            glk_put_string_stream(str, "<???>");
            break;
    }
}

/* The hashtable... another simple object, although it ought to be more
    complicated. */
    
#define HASHSIZE (127)

typedef struct hashnode_struct {
    atom_t *id;
    obj_t *val;
    struct hashnode_struct *next;
} hashnode_t;

struct floodict_struct {
    hashnode_t *bucket[HASHSIZE];
};

/* Create a new dictionary. */
floodict_t *new_floodict()
{
    int ix;
    floodict_t *tab = (floodict_t *)malloc(sizeof(floodict_t));
    if (!tab)
        return NULL;
    for (ix=0; ix<HASHSIZE; ix++)
        tab->bucket[ix] = NULL;
    return tab;
}

/* Find the hash bucket for the given string. */
int floodict_string_to_key(char *id, int len)
{
    int res = 0;
    int ix = 0;
    unsigned char ch;
    
    for (ix=0; ix<len; ix++) {
        ch = (*id);
        res += (ch << (ix & 7));
        id++;
    }
    return (res % HASHSIZE);
}

/* Look up an atom in a dictionary. Creates a reference and hands it to 
    you. */
obj_t *floodict_get(floodict_t *tab, atom_t *atom)
{
    int buck = atom->bucket;
    hashnode_t *nod;
    for (nod = tab->bucket[buck]; nod; nod = nod->next) {
        if (atom == nod->id) {
            return obj_newref(nod->val);
        }
    }
    return NULL;
}

/* Takes a reference from you, puts it in the dict (throwing away its old
    reference, if it had one.) */
void floodict_put(floodict_t *tab, atom_t *atom, obj_t *val)
{
    int buck = atom->bucket;
    hashnode_t *nod;
    for (nod = tab->bucket[buck]; nod; nod = nod->next) {
        if (atom == nod->id) {
            delete_obj(nod->val);
            nod->val = val;
            return;
        }
    }
    nod = (hashnode_t *)malloc(sizeof(hashnode_t));
    if (!nod)
        return;
    nod->id = atom;
    nod->val = val;
    nod->next = tab->bucket[buck];
    tab->bucket[buck] = nod;
}

/* Utility function, which might as well go here. Turn a number to a string. */
void num_to_str(char *buf, glsi32 num)
{
    int ix;
    int size = 0;
    char tmpc;
    
    if (num == 0) {
        strcpy(buf, "0");
        return;
    }
    
    if (num < 0) {
        buf[0] = '-';
        buf++;
        num = -num;
    }
    
    while (num) {
        buf[size] = '0' + (num % 10);
        size++;
        num /= 10;
    }
    for (ix=0; ix<size/2; ix++) {
        tmpc = buf[ix];
        buf[ix] = buf[size-ix-1];
        buf[size-ix-1] = tmpc;
    }
    buf[size] = '\0';
}
