/* floo.h: Header file for Floo.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/floo/index.html
*/

#ifndef _FLOO_H
#define _FLOO_H

/* We define our own TRUE and FALSE and NULL, because ANSI
    is a strange world. */
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef NULL
#define NULL 0
#endif

/* The atom structure. Atoms are stored in a red-black binary tree; see
    atom.c. */
typedef struct atom_struct atom_t;
struct atom_struct {
    int len; /* length of atom name */
    char *str; /* name, not null-terminated */
    int bucket; /* for floodict only */
    atom_t *left, *right;
    int isred;
};

/* Execution statuses. */
#define stat_Ok (0)
#define stat_Continue (1)
#define stat_Exit (2)
#define stat_Stop (3)
#define stat_Err (4)

/* Types of Floo objects. */
#define otyp_Boolean (1)
#define otyp_Integer (2)
#define otyp_String (3)
#define otyp_ID (4)
#define otyp_XID (5)
#define otyp_Array (6)
#define otyp_Proc (7)
#define otyp_FFunc (8)
#define otyp_Mark (9)
#define otyp_Null (10)

/* Some Floo objects which never appear to the user; they're used only in
    the execution stack. */
#define otyp_FileFrame (-1)
#define otyp_ObjFrame (-2)
#define otyp_LoopFrame (-3)

/* And the Floo object structure itself. */
typedef struct obj_struct obj_t;
struct obj_struct {
    int type; /* otyp_* */
    int refs; /* Count of references to this object */
    union {
        glsi32 num;
        struct {
            char *s;
            int len;
        } str;
        struct {
            atom_t *atom;
        } name;
        struct {
            int len;
            obj_t **o;
        } arr;
        struct {
            int (*funcptr)(void);
            atom_t *defatom;
        } ffunc;
        struct {
            obj_t *proc;
            int pos;
        } objframe;
        struct {
            strid_t file;
        } fileframe;
        struct {
            obj_t *obj;
            glsi32 cur, inc, end;
            int pushflag;
        } loopframe;
    } u;
};

/* Forward type declarations. */
typedef struct floodict_struct floodict_t;
typedef struct floostack_struct stack_t;

/* A stack. Very simple object. */
struct floostack_struct {
    obj_t **stack; /* An array of object pointers. */
    int stack_size; /* Total size of the array. */
    int stackpos; /* How much of the array is used. */
};

/* Global variable declarations. These are defined in main.c. */
extern strid_t gamefile;
extern floodict_t *sysdict;
extern floodict_t *errdict;
extern floodict_t *errinfodict;
extern stack_t *valst, *exst;

/* These are defined in ffuncs.c/ */
extern obj_t *err_undefined, *err_stackunderflow, *err_rangecheck, *err_typecheck,
    *err_undefinedresult, *err_unmatchedmark;

/* Function declarations. */

extern void floo_err(char *str);

extern obj_t *new_obj(int type);
extern obj_t *new_obj_array(int len, int executable);
extern void delete_obj(obj_t *ob);
extern void obj_stream(obj_t *ob, strid_t str);
extern obj_t *obj_newref(obj_t *ob);
extern int obj_debugrefcounter(void);
extern int obj_debugobjcounter(void);

extern int init_atomdict(void);
extern atom_t *atomdict_find(char *str, int len, int copystr);
extern void atomdict_debugdata(int *numatoms, int *mindepth, int *maxdepth);

extern floodict_t *new_floodict(void);
extern void delete_floodict(floodict_t *dict);
extern void floodict_put(floodict_t *dict, atom_t *atom, obj_t *val);
extern obj_t *floodict_get(floodict_t *dict, atom_t *atom);
extern int floodict_string_to_key(char *id, int len);

extern void num_to_str(char *buf, glsi32 num);

extern stack_t *new_stack(void);
extern void stack_push(stack_t *st, obj_t *ob);
extern obj_t *stack_pop(stack_t *st);
extern obj_t *stack_peek(stack_t *st, int depth);
extern int stack_searchmark(stack_t *st, int key);
/* extern int stack_height(stack_t *st); */
#define stack_height(st)  ((st)->stackpos)

extern int init_sysdict(void);
extern obj_t *new_nullref(void);

extern int init_parser(void);
extern obj_t *parse_object(strid_t file);
extern int is_floo_file(strid_t file);

extern void execute_loop(void);
extern int exec_error(obj_t *err);

extern int init_dispatch(void);
extern void floo_dispatch(void);

#endif /* _FLOO_H */
