/* exec.c: Floo object execution.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/floo/index.html
*/

#include <string.h>

#include "glk.h"
#include "floo.h"

static obj_t *err_object = NULL;
static obj_t *err_name = NULL;
static int execute_object(obj_t *ob, int direct);

/* This is the top-level execution loop. It executes objects on the exec
    stack until they're all gone. */
void execute_loop()
{
    obj_t *ob;
    obj_t *frame;
    int direct;
    int res;
    
    while (1) {
        glk_tick(); /* Let the library yield time, if necessary */
        
        /* Find the top item on the exec stack. */
        frame = stack_peek(exst, 0);
        if (!frame) {
            ob = NULL;
        }
        else if (frame->type == otyp_FileFrame) {
            /* Execute the next object in an open file. */
            strid_t file = frame->u.fileframe.file;
            delete_obj(frame);
            ob = parse_object(file);
            if (!ob) {
                delete_obj(stack_pop(exst));
                continue;
            }
            direct = TRUE;
        }
        else if (frame->type == otyp_ObjFrame) {
            /* Execute the next element of an array. */
            obj_t *proc = frame->u.objframe.proc; /* nonref */
            int pos = frame->u.objframe.pos;
            frame->u.objframe.pos++;
            delete_obj(frame);
            
            if (pos >= proc->u.arr.len) {
                delete_obj(stack_pop(exst));
                continue;
            }
            else {
                ob = obj_newref(proc->u.arr.o[pos]);
                direct = TRUE;
            }
        }
        else if (frame->type == otyp_LoopFrame) {
            /* Execute the next iteration of a loop. */
            obj_t *proc = obj_newref(frame->u.loopframe.obj); 
            int doneflag = FALSE;
            int pushflag = frame->u.loopframe.pushflag;
            if (frame->u.loopframe.inc > 0) {
                if (frame->u.loopframe.cur > frame->u.loopframe.end)
                    doneflag = TRUE;
            }
            else if (frame->u.loopframe.inc < 0) {
                if (frame->u.loopframe.cur < frame->u.loopframe.end)
                    doneflag = TRUE;
            }
            else {
                /* leave it FALSE */
            }
            delete_obj(frame);
            if (doneflag) {
                delete_obj(proc);
                delete_obj(stack_pop(exst));
                continue;
            }
            if (pushflag) {
                obj_t *obv = new_obj(otyp_Integer);
                obv->u.num = frame->u.loopframe.cur;
                stack_push(valst, obv);
            }
            frame->u.loopframe.cur += frame->u.loopframe.inc;
            ob = proc; 
            direct = FALSE;
        }
        else {
            /* Execute the object directly. */
            delete_obj(frame);
            ob = stack_pop(exst);
            direct = FALSE;
        }
        
        if (ob == NULL) {
            /* No more objects to execute. Goodbye. */
            return;
        }
        
        /* Now we have an object to execute. Go for it. */
        res = execute_object(ob, direct);
        
        /* Handle errors and loop-exits. */
        
        if (res) {
            if (res == stat_Err) {
                /* An error has occured. Store the error data in errinfodict,
                    and push the handler on the execution stack (to be executed
                    next.) */ 
                obj_t *errproc = NULL;
                /* errproc is looked up by finding what err_name is bound to in
                    errdict.  err_name is set by exec_error(). */
                if (err_name) {
                    errproc = floodict_get(errdict, err_name->u.name.atom);
                }
                if (!errproc)
                    floo_err("Unknown error.");
                stack_push(exst, errproc);
                /* err_object is set by execute_object(). */
                if (!err_object) {
                    err_object = new_nullref();
                }
                floodict_put(errinfodict, atomdict_find("command", -1, FALSE), 
                    err_object);
                err_object = NULL;
                /* err_name is set by exec_error(). */
                if (!err_name) {
                    err_name = new_nullref();
                }
                floodict_put(errinfodict, atomdict_find("errorname", -1, FALSE), 
                    err_name);
                err_name = NULL;
            }
            else {
                /* Exit or continue the innermost loop. */
                int cont = FALSE;
                while (!cont && (frame = stack_pop(exst)) != NULL) {
                    if (res == stat_Exit) {
                        if (frame->type == otyp_LoopFrame)
                            cont = TRUE;
                    }
                    else if (res == stat_Continue) {
                        if (frame->type == otyp_LoopFrame) {
                            cont = TRUE;
                            stack_push(exst, obj_newref(frame));
                        }
                    }
                    delete_obj(frame);
                }
            }
        }
    }
}

/* Execute one object.
    Takes possession of the reference. */
static int execute_object(obj_t *ob, int direct)
{
    obj_t *ob2;
    int (*funcptr)(void);
    int res = stat_Ok;
    
    switch (ob->type) {
        case otyp_Proc:
            if (direct) {
                stack_push(valst, ob);
            }
            else {
                ob2 = new_obj(otyp_ObjFrame);
                ob2->u.objframe.proc = ob;
                ob2->u.objframe.pos = 0;
                stack_push(exst, ob2);
            }
            break;
        case otyp_XID:
            ob2 = floodict_get(sysdict, ob->u.name.atom);
            if (!ob2) {
                res = exec_error(err_undefined);
                err_object = ob;
                break;
            }
            delete_obj(ob);
            if (!(ob2->type == otyp_Proc 
                || ob2->type == otyp_XID 
                || ob2->type == otyp_FFunc))
                stack_push(valst, ob2);
            else
                stack_push(exst, ob2);
            break;
        case otyp_FFunc:
            funcptr = ob->u.ffunc.funcptr;
            res = (*funcptr)();
            if (res == stat_Err) {
                err_object = ob;
                break;
            }
            delete_obj(ob);
            break;
        case otyp_Null:
            delete_obj(ob);
            break;
        default:
            stack_push(valst, ob);
            break;
    }
    
    return res;
}

/* Execute an error. Lots of code in Floo calls this. This doesn't actually
    do much; it stores the error name in a global variable, for later reference.
    The execute_loop() assumes that if execute_object() returns stat_Err, the
    object being executed is the one that actually threw the error. */
int exec_error(obj_t *err)
{
    err_name = obj_newref(err);
    return stat_Err;
}
