
/*! 
*
* Rogue is a runner for Interactive fiction games. It supports games
* written in many different toolkits. These include Adrift, AGT/AGX
* and TADS.
*
* \file rg_basic.c
* 	General operations required by all interpreters such as
*	prompting for filename, outputting strings, etc.
* 	
* 	
* \author Pallav Nawani
* Copyright Pallav Nawani (c) 2005
* 
*/


#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>

#include "support.h"
#include "rogue.h"
#include "interface.h"




//==///////////////////////////////////////////////////////////////////
//
///
/// This function creates tags for use in scaling the fonts, giving
/// bold, italic, or underlined output.
///
/// \param
///	sdata:	All the data used by Rogue.
///	buffer:	GtkTextBuffer to which the tags are added.
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void make_tags(Gtk_sdata *sdata, GtkTextBuffer *buffer)
{

   /* create tags for text output formatting */
   gtk_text_buffer_create_tag (buffer, "italic",
	 "style", PANGO_STYLE_ITALIC, NULL);

   gtk_text_buffer_create_tag (buffer, "bold",
	 "weight", PANGO_WEIGHT_BOLD, NULL);

   gtk_text_buffer_create_tag (buffer, "underline",
	 "underline", PANGO_UNDERLINE_SINGLE, NULL);

   gtk_text_buffer_create_tag (buffer, "center",
	 "justification", GTK_JUSTIFY_CENTER, NULL);

   gtk_text_buffer_create_tag (buffer, "right_justify",
	 "justification", GTK_JUSTIFY_RIGHT, NULL);

   gtk_text_buffer_create_tag (buffer, "userinput",
	 "style", PANGO_STYLE_ITALIC,
	 "weight", PANGO_WEIGHT_BOLD,
	 NULL);
   
   gtk_text_buffer_create_tag (buffer, "importantinfo",
	 "foreground", "DarkGreen", NULL);

   gtk_text_buffer_create_tag (buffer, "debuginfo",
	 "foreground", "DarkRed", NULL);

   gtk_text_buffer_create_tag (buffer, "secondary_color",
	 "foreground", "Brown", NULL);

   gtk_text_buffer_create_tag (buffer, "question",
	 "foreground", "DarkRed",
	 "weight", PANGO_WEIGHT_BOLD, NULL);

   gtk_text_buffer_create_tag (buffer, "bluetag",
	 "foreground", "Blue", NULL);
   
   gtk_text_buffer_create_tag (buffer, "cyantag",
	 "foreground", "Cyan", NULL);

   gtk_text_buffer_create_tag (buffer, "redtag",
	 "foreground", "Red", NULL);

   gtk_text_buffer_create_tag (buffer, "greentag",
	 "foreground", "Green", NULL);

   gtk_text_buffer_create_tag (buffer, "normal",
	 "foreground", "black",
	 "weight", PANGO_WEIGHT_NORMAL,
	 NULL);
   
   gtk_text_buffer_create_tag (buffer, "fixedwidth",
	 "family", "monospace", NULL);

   /* initialize tag stack */
   strcpy(sdata->tags[0], "normal");
   sdata->tsize = 1;
   sdata->ftop  = 0;



}







//==///////////////////////////////////////////////////////////////////
//
/// Outputs a string to the text view widget.
///
/// \param
///	idata:	All the data used by Rogue.
///	string:	A Null terminated string to be outputted to the widget.
/// \return
///	Nothing
//==///////////////////////////////////////////////////////////////////
void rg_output_string(Gtk_sdata *idata, const char *string)
{

   GtkTextBuffer *tbuf;
   GtkTextIter giter;
   GtkTextIter siter;
   GtkTextView *tview;
   GtkTextMark *tmark;

   const char *cptr;
   char warn[] = "Rogue: NULL STRING!\n";
   int i;
  
   cptr = string;
   if(NULL == string)
      cptr = warn;

   tview = GTK_TEXT_VIEW(idata->textw);
   tbuf = gtk_text_view_get_buffer(tview);

   gtk_text_buffer_get_end_iter(tbuf, &giter);
   tmark = gtk_text_buffer_create_mark(tbuf,
	 "datamark",
	 &giter,
	 TRUE);
   
   gtk_text_buffer_place_cursor(tbuf, &giter);
   gtk_text_buffer_insert(tbuf, &giter, cptr, -1);
   gtk_text_view_scroll_to_mark(tview,
	 gtk_text_buffer_get_insert(tbuf),
	 0.2, 1, 0, 0.5);
   
   gtk_text_buffer_get_end_iter(tbuf, &giter);
   gtk_text_buffer_get_iter_at_mark(tbuf, &siter, tmark);
   gtk_text_buffer_delete_mark(tbuf, tmark);

   for(i = 1; i < idata->tsize; i++)
      gtk_text_buffer_apply_tag_by_name(tbuf,
	    idata->tags[i], &siter, &giter);

   if(idata->ftag != NULL)
      gtk_text_buffer_apply_tag(tbuf,
	    idata->ftag,
	    &siter, &giter);

}







//==///////////////////////////////////////////////////////////////////
//
/// Get a line of input from the user.
///
/// \param
///	idata:	All the data used by Rogue.
/// \return
///	None
//
//==///////////////////////////////////////////////////////////////////
void rg_read_line(Gtk_sdata *idata)
{
   /* Wait for input */
   idata->got_input = 0;
   idata->command  = WAIT_FOR_INPUT;

   while(!idata->got_input) {
      gtk_main_iteration_do(1);
   }

}





//==///////////////////////////////////////////////////////////////////
//
/// Update the status bar with the given string. First, the previous
/// string is popped from the status bar, then the new one is pushed.
///
/// \param
///	idata:	The data required by rogue to function.
///	string:	The string to be written to the status bar.
///
/// \return
///	None
//
//==///////////////////////////////////////////////////////////////////
void rg_update_status_bar(Gtk_sdata *idata, const char *string)
{
   int cid;

   /* Update statusbar */
   cid = gtk_statusbar_get_context_id(GTK_STATUSBAR(idata->sbar),
	 "simple_status");

   if(idata->status)
      gtk_statusbar_pop(GTK_STATUSBAR(idata->sbar),
	    idata->status);
   idata->status = cid;

   gtk_statusbar_push(GTK_STATUSBAR(idata->sbar),
	 cid,
	 string);

}




//==///////////////////////////////////////////////////////////////////
//
/// Get a filename as input from the user.
///
/// \param
///	idata:	 The data required by rogue to function.
///	is_save: If 1, the Title of the fileselector is set to
///		 'Save as', else it is set to 'Select file to open'.
/// \return
///	1 if we get a filename, 0 if the user cancels the fileselector
///	dialog box.
//
//==///////////////////////////////////////////////////////////////////
int rg_get_filename(Gtk_sdata *idata, int is_save)
{
   GtkWidget *fsel;

   idata->yesno   = DNORESPONSE;
   idata->menu    = 0;
   idata->command = SAVE_GAME;

   if(is_save)
      fsel = create_fileselection (idata, 1);
   else
      fsel = create_fileselection (idata, 0);

   gtk_file_selection_set_filename(GTK_FILE_SELECTION(fsel),
	 idata->name);

   gtk_widget_show(fsel);
   while(idata->yesno == DNORESPONSE)
   {
      gtk_main_iteration_do(1);
   }

   if(idata->yesno == DYES)
      return 1;

   return 0;

}



//==///////////////////////////////////////////////////////////////////
//
/// Clears the text buffer.
///
/// \param
///	idata: The data required by rogue to function.
/// \return
///	Nothing
//==///////////////////////////////////////////////////////////////////
void rg_clear_screen(Gtk_sdata *idata)
{
   GtkTextBuffer *tbuf;

   tbuf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(idata->textw));
   gtk_text_buffer_set_text(tbuf, " ", -1);

}





//==///////////////////////////////////////////////////////////////////
//
/// Opens a text file and shows it in a TextView widget.
///
/// \param
///	fname:	Full path for the text file.
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void rg_show_info(char *fname)
{

   char *buf;
   int done;

   FILE *fptr;

   GtkWidget *info;
   GtkTextView *tview;
   GtkTextBuffer *tbuf;
   GtkTextIter giter;
 
   buf = (char *)malloc(strlen(fname) + 256);
   fptr = fopen(fname, "ra");

   if(NULL == fptr) {
      sprintf(buf, "Could not open the file %s\n", fname);
      info = create_about_dlg("Oops!", buf);
      gtk_widget_show(info);
      free(buf);
      return;
   }

   strcpy(buf, fname);
   info  = create_game_text_window(buf);
   tview = (GtkTextView *)lookup_widget(info, "game_info_textview");

   tbuf = gtk_text_view_get_buffer(tview);
   gtk_text_buffer_get_end_iter(tbuf, &giter);

   while(!feof(fptr)) {
      done = fread(buf, 1, 255, fptr);
      gtk_text_buffer_insert(tbuf, &giter, buf, done);
      if(done < 255)
	 break;
   }

   free(buf);
   fclose(fptr);
   gtk_widget_show(info);


}

//==///////////////////////////////////////////////////////////////////
//
/// Send input to the game. This function is useful to send commands
/// to the game from menu.
///
/// \param
///	idata: The interpreter data
///	input: command to send to game interpreter.
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void rg_send_command(Gtk_sdata *idata, char *input)
{
   strcpy(idata->userin, input);
   idata->got_input = 1;

}


//==///////////////////////////////////////////////////////////////////
//
///
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////


/*  End of file rg_basic.c  */

