
/*! 
*
* Rogue is a runner for Interactive fiction games. It supports games
* written in many different toolkits. These include Adrift, AGT/AGX
* and TADS.
*
* \file rg_agx.c
* 	Functions for interfacing with Agility
* 	
* 	
* \author Pallav Nawani
* Copyright Pallav Nawani (c) 2005
* 
*/


#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

#include "support.h"
#include "rogue.h"
#include "interface.h"
#include "rg_basic.h"


#ifdef __cplusplus
extern "C" {
#endif

extern Gtk_sdata *idata;
extern char fstring[256];




/* Wait for n seconds */
void agt_delay(int n)
{
   if(!fast_replay)
      sleep(n);
   print_statline();
}



/* Produce a hz-Hertz sound for ms milliseconds */
void agt_tone(int hz,int ms)
{
   return;
}



/* Return random number from a to b inclusive */
int agt_rand(int a,int b)
{
    return a+(rand()>>2)%(b-a+1);
}



//==///////////////////////////////////////////////////////////////////
//
/// Read a line from the keyboard, allocating space for it using malloc.
/// in_type: 0= command, 1=number, 2=question, 3=userstr, 4=filename,
///          5= RESTART, RESTORE, UNDO, QUIT
///
/// \param
///	in_type: The type of input desired.
///
/// \return
///	Pointer to a string with the input.
//
//==///////////////////////////////////////////////////////////////////
char *agt_input(int in_type)
{

   print_statline();
   switch(in_type)
   {
      case 0:
      case 1:
      case 2:
      case 3:
      case 5:
	 rg_read_line(idata);
	 return strdup(idata->userin);
	 break;

      case 4:
	 rg_get_filename(idata, 0);
	 return strdup(idata->name);
	 break;

      default: 
	 break;
   }

   return NULL;
}




//==///////////////////////////////////////////////////////////////////
//
/// Reads a character and returns it, possibly reading in a full line.
/// If echo_char=1, echo character. If 0, then the character is not 
/// required to be echoed (and ideally shouldn't be)
///
/// \param
///	echo_char: Flag telling us whether the character should be
///		   echoed.
/// \return
///	The character read in.
//==///////////////////////////////////////////////////////////////////
char agt_getkey(unsigned char echo_char)
{
   char c;
   int i, len;

   print_statline();
   rg_read_line(idata);

   c = idata->userin[0];
   len = strlen(idata->userin);
   for(i=0; i < len; i++) {
      if(idata->userin[i] != ' ') {
	 c = idata->userin[i];
	 break;
      }
   }
   return c;
}





//==///////////////////////////////////////////////////////////////////
//
///
/// Set text color to color c, where the colors are as follows:
/// 0 = Black, 1=Blue,    2=Green, 3=Cyan,
/// 4 = Red,   5=Magenta, 6=Brown,
/// 7 = Normal This should restore the text mode to its default appearance.
/// 8 = Turn on blinking.
/// 9 = White
/// 10= Turn on fixed pitch font.
/// 11= Turn off fixed pitch font
/// -1= Emphasized text, used (e.g.) for room titles
/// -2= End emphasized text
///
/// \param
///	c:	The color.
/// \return
///	Nothing
//==///////////////////////////////////////////////////////////////////
void agt_textcolor(int c)
{
   char *tag_names[] = {
      "normal",
      "bold",
      "normal",
      "bluetag",
      "greentag",
      "cyantag",
      "redtag",
      "debuginfo",
      "secondary_color",
      "normal",
      "underline",
      "normal",
      "fixedwidth",
      "normal"
   };

   switch(c+2) {
      case 1:
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 10:
      case 12:
	 if(idata->tsize < CMDBUF) {
	    strcpy(idata->tags[idata->tsize], tag_names[c+2]);
	    idata->tsize++;
	 }
	 break;

      default:
	 if(idata->tsize > 1)
	    idata->tsize--;
	 break;
   }

   return;
}




//==///////////////////////////////////////////////////////////////////
//
/// Output a string on the status line.
///
/// \param
///	s:	String to be written to the status bar.
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void agt_statline(const char *s)
{
   rg_update_status_bar(idata, s);
}



//==///////////////////////////////////////////////////////////////////
//
/// Clears the screen
/// \param
///	None
/// \return
///	None
//==///////////////////////////////////////////////////////////////////
void agt_clrscr(void)
{
   rg_clear_screen(idata);
   print_statline();
}




//==///////////////////////////////////////////////////////////////////
//
/// Output a string to the screen/text widget.
///
/// \param
///	s: String to be written to the text widget.
/// \return
///	None
//==///////////////////////////////////////////////////////////////////
void agt_puts(const char *s)
{
   rg_output_string(idata, s);
   curr_x += strlen(s);
}




//==///////////////////////////////////////////////////////////////////
//
/// Output a newline to the screen/text widget.
///
/// \param
///	None
/// \return
///	None
//==///////////////////////////////////////////////////////////////////
void agt_newline( )
{
   rg_output_string(idata, "\n");
   curr_x = 0;
}




//==///////////////////////////////////////////////////////////////////
//
/// Used to make boxes using ascii symbols. We won't be using this.
/// We will just center all the text inside box instead.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void agt_makebox(int width, int height, unsigned long flags)
{

   if(idata->tsize < CMDBUF) {
      strcpy(idata->tags[idata->tsize], "center");
      idata->tsize++;
   }

   if(idata->tsize < CMDBUF) {
      strcpy(idata->tags[idata->tsize], "debuginfo");
      idata->tsize++;
   }
}



//==///////////////////////////////////////////////////////////////////
//
/// It is supposed to output a quoted newline. However, we just
/// output a newline, and a text justification tag.
///
/// \param
///	None
/// \return
///	None.
//==///////////////////////////////////////////////////////////////////
void agt_qnewline(void)
{

   rg_output_string(idata, "\n");
   curr_x = 1;

}



//==///////////////////////////////////////////////////////////////////
//
/// It is supposed to end the box. However, we just
/// output a newline, and a text justification tag.
///
/// \param
///	None
/// \return
///	None.
//==///////////////////////////////////////////////////////////////////
void agt_endbox(void)
{
   rg_output_string(idata, "\n");
   curr_x = 0;

   if(idata->tsize > 1)
      idata->tsize--;

   if(idata->tsize > 1)
      idata->tsize--;
}



//==///////////////////////////////////////////////////////////////////
//
/// Open agility configuration file. If fid is non zero, we return
/// NULL.
///
/// \param
///	fid: The type of the file to be opened.
/// \return
///	
//==///////////////////////////////////////////////////////////////////
FILE *agt_globalfile(int fid)
{
   char *homedir;
   FILE *cfgfile;
   char *fname;

   if (fid!=0)
      return NULL;
   
   homedir = getenv("HOME");
   fname   = (char *)malloc(strlen(homedir) + 10);
   sprintf(fname, "%s/.agilrc", homedir);
   cfgfile = fopen(fname, "r");
   
   return cfgfile;
}





//==///////////////////////////////////////////////////////////////////
//
/// Parse and Turn on/off various options. If an option was
/// recognized, return 1. Else return 0.
/// If setflag is 0, then the option was prefixed with NO_
///
/// \param
///	optnum: Number of options
///	optstr: Option string
///	setflag: If zero, then it means the opposite.
/// \return
///	1 if an option was recognized, 0 otherwise.
//
//==///////////////////////////////////////////////////////////////////
unsigned char agt_option(int optnum,
      char *optstr[],
      unsigned char setflag)
{

   if (strcasecmp("COLORTERM", optstr[0]) == 0
	 && setflag) {
      return 1;
   }

   if (strcasecmp("HISTORY", optstr[0]) == 0
	 && setflag) {
      return 1;
   }

   return 0;
}


void init_interface(int argc, char *argv[])
{
   char *cptr;

   script_on	= 0;
   scriptfile	= badfile(fSCR);
   center_on	= 0;
   par_fill_on	= 0;
   curr_x	= 0;
   sound_on	= 0;
   debugfile	= stderr;
   
   /* If set, we echo all output to debugfile */
   /* This should be 1 if stderr points to a different */
   /* device than stdin. (E.g. to a file instead of to the */
   /* console) */
   DEBUG_OUT	= 0;
   DEBUG_AGT_CMD= 0;
   
   screen_height= 40;  /* Assume PC dimensions */
   status_width	= 100;
   screen_width	= 100;

   if(gamepath == NULL)
   {
      gamepath    = (char **)malloc(sizeof(gamepath)*3);
      gamepath[0] = (char *)malloc(4);
      gamepath[0][0] = '\0';
      gamepath[1] = strdup(idata->name);

      cptr = strrchr(gamepath[1], '/');
      if(NULL != cptr)
	 *cptr = '\0';
      gamepath[2] = NULL;
   }

   agt_clrscr();
}

void start_interface(fc_type fc)
{
  if (stable_random)  
    srand(6);
  else 
    srand(time(0));
}


//==///////////////////////////////////////////////////////////////////
//
///
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void close_interface(void)
{
  if (filevalid(scriptfile,fSCR))
    close_pfile(scriptfile,0);
}


//==///////////////////////////////////////////////////////////////////
//
///
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////

#ifdef __cplusplus
}
#endif


/*  End of file rg_agx.c  */

