/* helioinp.c - os_* functions for user input (uses mygetlin.c)
 *
 * This file is part of HelioFrotz.
 *
 * HelioFrotz is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * HelioFrotz is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <system.h>
#include <string.h>
#include "frotz.h"
#include "heliofro.h"
#include "..\stdlib\fnt.h"

int more_prompt = 0;

/* Translate in place all the escape characters in s.  */
static void translate_special_chars(char *s)
{
  char *src = s, *dest = s;
  while (*src)
    switch(*src++) {
    default: *dest++ = src[-1]; break;
    case '\n': *dest++ = ZC_RETURN; break;
    case '\\':
      switch (*src++) {
      case '\n': *dest++ = ZC_RETURN; break;
      case '\\': *dest++ = '\\'; break;
      case '?': *dest++ = ZC_BACKSPACE; break;
      case '[': *dest++ = ZC_ESCAPE; break;
      case '_': *dest++ = ZC_RETURN; break;
      case '^': *dest++ = ZC_ARROW_UP; break;
      case '.': *dest++ = ZC_ARROW_DOWN; break;
      case '<': *dest++ = ZC_ARROW_LEFT; break;
      case '>': *dest++ = ZC_ARROW_RIGHT; break;
      case 'R': *dest++ = ZC_HKEY_RECORD; break;
      case 'P': *dest++ = ZC_HKEY_PLAYBACK; break;
      case 'S': *dest++ = ZC_HKEY_SEED; break;
      case 'U': *dest++ = ZC_HKEY_UNDO; break;
      case 'N': *dest++ = ZC_HKEY_RESTART; break;
      case 'X': *dest++ = ZC_HKEY_QUIT; break;
      case 'D': *dest++ = ZC_HKEY_DEBUG; break;
      case 'H': *dest++ = ZC_HKEY_HELP; break;
      case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
	*dest++ = ZC_FKEY_MIN + src[-1] - '0' - 1; break;
      case '0': *dest++ = ZC_FKEY_MIN + 9; break;
      default:
	FntDrawString("Unknown Escape Character", 0, 0, TRUE);
      }
    }
  *dest = '\0';
}


/* Read a line, processing commands (lines that start with a backslash
 * (that isn't the start of a special character)), and write the
 * first non-command to s.
 * Return true if timed-out.  */
static bool timed_read_line(char *s, char *prompt, bool show_cursor,
			   int timeout,
			   zchar *continued_line_chars)
{
  BitmapTemplate b;
  extern int mygetline(char *str, char *pl, int timeout);
  int did_timeout;

  b.xcoord = 0;
  b.ycoord = 0;
  b.height = h_screen_height;
  b.width = h_screen_width;

  for (;;) {
    LcdGetBitmap(&b);
    if (prompt)
      FntDrawString(prompt, 0, h_screen_height-FntLineHeight(), TRUE);
    did_timeout = mygetline(s, continued_line_chars, timeout);
    LcdDrawBitmap(&b, FALSE);
    qfree(b.bitmap_data);

    translate_special_chars(s);
    if (continued_line_chars) continued_line_chars[0] = '\0';
    return did_timeout;
  }
}

/* For allowing the user to input in a single line keys to be returned
 * for several consecutive calls to read_char, with no screen update
 * in between.  Useful for traversing menus.  */
static char read_key_buffer[INPUT_BUFFER_SIZE];

/* Similar.  Useful for using function key abbreviations.  */
static char read_line_buffer[INPUT_BUFFER_SIZE];

zchar os_read_key (int timeout, bool show_cursor)
{
  char c;
  int timed_out;

  /* Discard any keys read for line input.  */
  read_line_buffer[0] = '\0';

  if (read_key_buffer[0] == '\0') {
    timed_out = timed_read_line(read_key_buffer, NULL, show_cursor, timeout, NULL);
    /* An empty input line is reported as a single CR.
     * If there's anything else in the line, we report only the line's
     * contents and not the terminating CR.  */
    if (strlen(read_key_buffer) > 1)
      read_key_buffer[strlen(read_key_buffer) - 1] = '\0';
  } else
    timed_out = FALSE;
    
  if (timed_out)
    return ZC_TIME_OUT;

  c = read_key_buffer[0];
  memmove(read_key_buffer, read_key_buffer + 1, strlen(read_key_buffer));

  return c;
}

void display_user_input(char *s)
{
  os_display_string(s);
}


zchar os_read_line (int max, zchar *buf, int timeout, int width, int continued)
{
  char *p;
  int terminator;
  static bool timed_out_last_time;
  int timed_out;

  /* Discard any keys read for single key input.  */
  read_key_buffer[0] = '\0';

  /* After timing out, discard any further input unless we're continuing.  */
  if (timed_out_last_time && !continued)
    read_line_buffer[0] = '\0';

  if (read_line_buffer[0] == '\0')
    timed_out = timed_read_line(read_line_buffer, NULL, TRUE, timeout, buf);
  else
    timed_out = FALSE;
  
  if (timed_out) {
    timed_out_last_time = TRUE;
    return ZC_TIME_OUT;
  }
    
  /* find the terminating character.  */
  for (p = read_line_buffer;; p++) {
    if (is_terminator(*p)) {
      terminator = *p;
      *p++ = '\0';
      break;
    }
  }

  /* copy to screen */
  display_user_input(read_line_buffer);

  /* copy to the buffer and save the rest for next time.  */
  strcat(buf, read_line_buffer);
  p = read_line_buffer + strlen(read_line_buffer) + 1;
  memmove(read_line_buffer, p, strlen(p) + 1);
    
  /* If there was just a newline after the terminating character,
   * don't save it.  */
  if ((read_line_buffer[0] == '\r') && (read_line_buffer[1] == '\0'))
    read_line_buffer[0] = '\0';

  timed_out_last_time = FALSE;
  return terminator;
}

extern int GetFileName(char *def, char *type, char *prompt, char *outfile, int checkover);
int os_read_file_name (char *file_name, const char *default_name, int flag)
{
  char buf[INPUT_BUFFER_SIZE], defname[INPUT_BUFFER_SIZE];
  extern char storyname[90];
  char ftype[64];
  int checkover;
  char *prompt;
  BitmapTemplate b;
  int ok;

  b.xcoord = 0;
  b.ycoord = 0;
  b.height = h_screen_height;
  b.width = h_screen_width;

  strcpy(defname, default_name);
  checkover = 0;
  switch(flag) {
  case FILE_SCRIPT:
    prompt = "Select Script To Write";
    strcpy(ftype, "Text");
    checkover = 1;
    break;
  case FILE_RESTORE:
    prompt = "Select Game To Restore";
    sprintf(ftype, "Frotz-%s", storyname);
    break;
  case FILE_SAVE:
    prompt = "Select Game To Save";
    sprintf(ftype, "Frotz-%s", storyname);
    checkover = 1;
    break;
  case FILE_LOAD_AUX:
    prompt = "Select Aux File To Load";
    sprintf(ftype, "FrotzAux-%s", storyname);
    break;
  case FILE_SAVE_AUX:
    prompt = "Select Aux File To Save";
    sprintf(ftype, "FrotzAux-%s", storyname);
    checkover = 1;
    break;
  default:
    prompt = "Select File";
    strcpy(ftype, "Binary");
    break;
  }
  buf[0]=0;
  
  LcdGetBitmap(&b);
  ok = GetFileName(defname, ftype, prompt, buf, checkover);
  LcdDrawBitmap(&b, FALSE);
  qfree(b.bitmap_data);
  
  if (ok==0)
   return FALSE;
  
  if (strlen(buf) > MAX_FILE_NAME) {
    return FALSE;
  }
  strcpy (file_name, buf[0] ? buf : default_name);
  
  if (flag==FILE_SAVE || flag==FILE_SAVE_AUX) {
    filetype(ftype);
  } else
    filetype(NULL);

  return TRUE;
}

void os_more_prompt (void)
{
  char buf[INPUT_BUFFER_SIZE];
  more_prompt = 1;
  timed_read_line(buf, "***MORE***", 0, 0, 0);
  more_prompt = 0;
}

void os_exit_prompt (void)
{
  char buf[INPUT_BUFFER_SIZE];
  more_prompt = 1;
  timed_read_line(buf, "Hit any key to exit.", 0, 0, 0);
  more_prompt = 0;
}
