/* pix2gif.c V7/3
 *
 * MG1 to GIF format converter.
 *
 * Converts Infocom MG1/EG1 picture files to separate GIF files.
 *
 * usage: pix2gif picture-file
 *
 * Mark Howell 13 September 1992 howell_ma@movies.enet.dec.com
 *
 *	Reduced and adapted for Frostz by Fabio Concas, 4 February 2005
 *
 * History:
 *    Make stack allocated arrays static
 *    Fix lint warnings
 *    Fix 64KB MS-DOS problems
 *    Handle transparent colours properly
 *    Put transparency information into GIF (now version 89a)
 */
#ifdef _WIN32_WCE
#include <windows.h>
#endif


#include "pix2gif.h"


#ifdef __MSDOS__
//#include <alloc.h>
#define malloc(n) farmalloc(n)
#define calloc(n,s) farcalloc(n,s)
#define free(p) farfree(p)
#endif

static short mask[16] = {
    0x0000, 0x0001, 0x0003, 0x0007,
    0x000f, 0x001f, 0x003f, 0x007f,
    0x00ff, 0x01ff, 0x03ff, 0x07ff,
    0x0fff, 0x1fff, 0x3fff, 0x7fff
};

static unsigned char ega_colourmap[16][3] = {
	{   0,  0,  0 },
	{   0,  0,170 },
	{   0,170,  0 },
	{   0,170,170 },
	{ 170,  0,  0 },
	{ 170,  0,170 },
	{ 170,170,  0 },
	{ 170,170,170 },
	{  85, 85, 85 },
	{  85, 85,255 },
	{  85,255, 85 },
	{  85,255,255 },
	{ 255, 85, 85 },
	{ 255, 85,255 },
	{ 255,255, 85 },
	{ 255,255,255 }
};

#if defined(AMIGA) && defined(_DCC)
__far
#endif
unsigned char colourmap[16][3];
unsigned char code_buffer[CODE_TABLE_SIZE];
char file_name[FILENAME_MAX + 1];
short code_table[CODE_TABLE_SIZE][2];
unsigned char buffer[CODE_TABLE_SIZE];

void process_image (FILE *, pdirectory_t *);
void decompress_image (FILE *, image_t *);
short read_code (FILE *, compress_t *);
unsigned char read_byte (FILE *);
void write_byte (FILE *, int);
unsigned short read_word (FILE *);
void write_word (FILE *, unsigned short);
void read_bytes (FILE *, int, void *);
void write_bytes (FILE *, int, const void *);




//*** This is _our_ write_bmp, which writes on the device screen ***
void write_bmp (image_t *Img, int n);


#ifndef _WIN32_WCE
int main (int argc, char *argv[])
{
    int i;
    FILE *fp;
    header_t header;
    pdirectory_t *directory;

    if (argc != 2) {
	(void) fprintf (stderr, "usage: %s picture-file\n\n", argv[0]);
	(void) fprintf (stderr, "PIX2GIF version 7/2 - convert Infocom MG1/EG1 files to GIF. By Mark Howell\n");
	(void) fprintf (stderr, "Works with V6 Infocom games.\n");
	exit (EXIT_FAILURE);
    }

    if ((fp = fopen (argv[1], "rb")) == NULL) {
	perror ("fopen");
	exit (EXIT_FAILURE);
    }

    header.part = read_byte (fp);
    header.flags = read_byte (fp);
    header.unknown1 = read_word (fp);
    header.images = read_word (fp);
    header.unknown2 = read_word (fp);
    header.dir_size = read_byte (fp);
    header.unknown3 = read_byte (fp);
    header.checksum = read_word (fp);
    header.unknown4 = read_word (fp);
    header.version = read_word (fp);

    (void) printf ("Total number of images is %d.\n", (int) header.images);

    if ((directory = (pdirectory_t *) calloc ((size_t) header.images, sizeof (pdirectory_t))) == NULL) {
	(void) fprintf (stderr, "Insufficient memory\n");
	exit (EXIT_FAILURE);
    }

    for (i = 0; (unsigned int) i < (unsigned int) header.images; i++) {
	directory[i].image_number = read_word (fp);
	directory[i].image_width = read_word (fp);
	directory[i].image_height = read_word (fp);
	directory[i].image_flags = read_word (fp);
	directory[i].image_data_addr = (unsigned long) read_byte (fp) << 16;
	directory[i].image_data_addr += (unsigned long) read_byte (fp) << 8;
	directory[i].image_data_addr += (unsigned long) read_byte (fp);
	if ((unsigned int) header.dir_size == 14) {
	    directory[i].image_cm_addr = (unsigned long) read_byte (fp) << 16;
	    directory[i].image_cm_addr += (unsigned long) read_byte (fp) << 8;
	    directory[i].image_cm_addr += (unsigned long) read_byte (fp);
	} else {
	    directory[i].image_cm_addr = 0;
	    (void) read_byte (fp);
	}
    }
    for (i = 0; (unsigned int) i < (unsigned int) header.images; i++)
	process_image (fp, &directory[i]);
    free (directory);
    (void) fclose (fp);

    exit (EXIT_SUCCESS);

    return (0);

}/* main */
#endif // _WIN32_WCE



void process_image (FILE *fp, pdirectory_t *directory)
{
    int colours = 16, i;
    image_t image;

    for (i = 0; i < 16; i++) {
	colourmap[i][RED] = ega_colourmap[i][RED];
	colourmap[i][GREEN] = ega_colourmap[i][GREEN];
	colourmap[i][BLUE] = ega_colourmap[i][BLUE];
    }

    if (directory->image_cm_addr)
	{
		if (fseek (fp, directory->image_cm_addr, SEEK_SET) != 0) {
			//perror ("fseek");
			MessageBox (GetActiveWindow (), L"Invalid palette address", L"Error",
			     MB_APPLMODAL | MB_ICONSTOP);
			return;//exit (EXIT_FAILURE);
		}
		colours = read_byte (fp);

		/* Fix for some buggy _Arthur_ pictures. */
		if (colours > 14)
		    colours = 14;
		read_bytes (fp, colours * 3, (void *) &colourmap[2][RED]);
		colours += 2;
    }

    if (directory->image_flags & 1) {
	colourmap[directory->image_flags >> 12][0] = 0xFF; //*** Transparent now white
	colourmap[directory->image_flags >> 12][1] = 0xFF;
	colourmap[directory->image_flags >> 12][2] = 0xFF;
    }

/*    (void) printf ("pic %03d   size %3d x %3d   %2d colours   colour map ",
	    (int) directory->image_number,
	    (int) directory->image_width, (int) directory->image_height,
	    (int) colours);

    if (directory->image_cm_addr != 0)
	(void) printf ("$%05lx", (long) directory->image_cm_addr);
    else
	(void) printf ("------");
*/
    if (directory->image_flags & 1) {
	image.transflag = 1;
	image.transpixel = (unsigned short) directory->image_flags >> 12;
//	(void) printf ("   transparent is %u\n", image.transpixel);
    }
    else {
	image.transpixel = 0;
	image.transflag = 0;
//	(void) printf ("\n");
    }

    if (directory->image_data_addr == 0)
	return;

    image.width = directory->image_width;
    image.height = directory->image_height;
    image.colours = colours;
    image.pixels = 0;
    if ((image.image = (unsigned char *) calloc ((size_t) directory->image_width, (size_t) directory->image_height)) == NULL) {
	//(void) fprintf (stderr, "Insufficient memory\n");
	MessageBox (GetActiveWindow (), L"Could not allocate image buffer.", L"Error",
	   MB_APPLMODAL | MB_ICONSTOP);
	return;//exit (EXIT_FAILURE);
    }
    image.colourmap = colourmap;

    if (fseek (fp, directory->image_data_addr, SEEK_SET) != 0) {
	MessageBox (GetActiveWindow (), L"Invalid image data address.", L"Error",
	   MB_APPLMODAL | MB_ICONSTOP);
	//perror ("fseek");
	exit (EXIT_FAILURE);
    }
    decompress_image (fp, &image);

    //write_file ((int) directory->image_number, &image);
    write_bmp (&image, directory->image_number);

    free (image.image);

}/* process image */


void decompress_image (FILE *fp, image_t *image)
{
    int i;
    short code, old = 0, first, clear_code;
    compress_t comp;

    clear_code = 1 << CODE_SIZE;
    comp.next_code = clear_code + 2;
    comp.slen = 0;
    comp.sptr = 0;
    comp.tlen = CODE_SIZE + 1;
    comp.tptr = 0;

    for (i = 0; i < CODE_TABLE_SIZE; i++) {
	code_table[i][PREFIX] = CODE_TABLE_SIZE;
	code_table[i][PIXEL] = i;
    }

    for (;;) {
	if ((code = read_code (fp, &comp)) == (clear_code + 1))
	    return;
	if (code == clear_code) {
	    comp.tlen = CODE_SIZE + 1;
	    comp.next_code = clear_code + 2;
	    code = read_code (fp, &comp);
	} else {
	    first = (code == comp.next_code) ? old : code;
	    while (code_table[first][PREFIX] != CODE_TABLE_SIZE)
                first = code_table[first][PREFIX];
            code_table[comp.next_code][PREFIX] = old;
            code_table[comp.next_code++][PIXEL] = code_table[first][PIXEL];
        }
        old = code;
        i = 0;
        do
            buffer[i++] = (unsigned char) code_table[code][PIXEL];
        while ((code = code_table[code][PREFIX]) != CODE_TABLE_SIZE);
        do 
            image->image[image->pixels++] = buffer[--i]; 
        while (i > 0);
    }

}/* decompress_image */


short read_code (FILE *fp, compress_t *comp)
{
    short code, bsize, tlen, tptr;

    code = 0;
    tlen = comp->tlen;
    tptr = 0;

    while (tlen) {
        if (comp->slen == 0) {
            if ((comp->slen = fread (code_buffer, 1, MAX_BIT, fp)) == 0) {
                //perror ("fread");
                exit (EXIT_FAILURE);
            }
            comp->slen *= 8;
            comp->sptr = 0;
        }
	bsize = ((comp->sptr + 8) & ~7) - comp->sptr;
        bsize = (tlen > bsize) ? bsize : tlen;
        code |= (((unsigned int) code_buffer[comp->sptr >> 3] >> (comp->sptr & 7)) & mask[bsize]) << tptr;

        tlen -= bsize;
        tptr += bsize;
        comp->slen -= bsize;
        comp->sptr += bsize;
    }
    if ((comp->next_code == mask[comp->tlen]) && (comp->tlen < 12))
        comp->tlen++;

    return (code);

}/* read_code */


unsigned char read_byte (FILE *fp)
{
    int c;

    if ((c = fgetc (fp)) == EOF) {
        //perror ("fgetc");
        exit (EXIT_FAILURE);
    }

    return ((unsigned char) c);

}/* read_byte */


unsigned short read_word (FILE *fp)
{
    unsigned int w;

    w = (unsigned int) read_byte (fp);
    w += (unsigned int) read_byte (fp) << 8;

    return (w);

}/* read_word */


void read_bytes (FILE *fp, int size, void *s)
{

    if (fread (s, (size_t) size, 1, fp) != 1) {
        //perror ("fread");
        //exit (EXIT_FAILURE);
    }

}/* read_bytes */
