/***************************************************************************/
/* zcdiff.c version 2.3                                                    */
/*                                                                         */
/* Creates a diff (patch) between two infocom (zcode) gamefiles.           */
/* This is done by XORing the two files; thus this can be seen as          */
/* "encryption".                                                           */
/* The patchfile is in the following format:                               */
/*                                                                         */
/*   DB 'PFG'  Patchfile identifer string                                  */
/*   Game name (32 bytes) -- unused bytes are filled with 0's.             */
/*   Resulting (target) gamefile version                                   */
/*     (gamefile version 1 byte / release 2 bytes / version 6 bytes)       */
/*     -- taken from gamefile header                                       */
/*   Required (source) gamefile version 9 bytes (same format as above)     */
/*   XOR Data to translate gamefile                                        */
/*                                                                         */
/* If the end of the source file as specified by the length in the header  */
/* is reached, the program will return to the start of the source file to  */
/* continue differencing.                                                  */
/*                                                                         */
/*                                                                         */
/* Copyright (c) 1997/98 by Paul Gilbert (paulfgilbert AT gmail.com)       */
/* Modifications Copyright (c) 2000/2001 Rodney Hester and Nils Barth      */
/* and 2009 by Mike Ciul and Nils Barth                                    */
/*                                                                         */
/*                                                                         */
/* This program is free software: you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation, either version 3 of the License, or       */
/* (at your option) any later version.                                     */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program.  If not, see <http://www.gnu.org/licenses/>.   */
/*                                                                         */
/***************************************************************************/

#define VERSION "2.3"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

/* Compiler check that unsigned long ints are at least 4 bytes */
#if (ULONG_MAX < 4294967295)
#error "Code requires that unsigned long ints be at least 32 bits"
#endif

typedef struct {
  unsigned int zcode_version;
  unsigned int release;
  unsigned char version[6];
  unsigned long int filesize;
} FILE_DATA;

char *inname, *cryptname, *outname;
FILE *infile, *cryptfile, *outfile;
FILE_DATA infile_data, cryptfile_data;
char game[33];

void open_files(void);
void verify_gamefiles(void);
void retrieve_details(FILE_DATA *data, FILE *f, char *FileDesc, char *name);
void get_version_details(void);
void extract_gamename(int argv, char *argc[]);
void print_hex_digit(int d);
void print_version(FILE_DATA *d);
void encrypt_file(void);
void close_files(void);


/* open_files
 * Opens up the input file, output file, and encryption file for access.
 * Exits the program if any of the openings fail.
 */

void open_files()
{
 /* Open up the input file for access */
  if ((infile = fopen(inname, "rb")) == NULL)
  {
    /* Input file could not be opened, exit with error */
    fprintf(stderr, "Could not open target game file '%s'\n", inname);
    exit(1);
  }

  /* Open up the encryption file for access */
  if ((cryptfile = fopen(cryptname, "rb")) == NULL)
  {
    /* Encryption file opening failed, exit with error */
    fclose(infile);
    fprintf(stderr, "Could not open source gamefile '%s'.\n", cryptname);
    exit(1);
  }

  /* Open up the output file for writing */
  if ((outfile = fopen(outname, "wb")) == NULL)
  {
    /* Output file opening failed, exit with error */
    fclose(infile); fclose(cryptfile);
    fprintf(stderr, "Could not open output patch file '%s'.\n", outname);
    exit(1);
  }
}

/* verify_gamefiles
 * Verifies that the input and encryption files that have already been
 * opened are valid gamefiles (At the moment this is done only by reading
 * the first byte to ensure it's in the range of 1 to 8).
 */

void verify_gamefiles()
{
  int c1, c2;

  /* Get the first byte from both the input and encryption file */
  c1 = fgetc(infile);
  c2 = fgetc(cryptfile);

  /* If either c1 or c2 is not in the range 1 to 8, then not an Infocom game */
  if ((c1 < 1) || (c1 > 8))
    fprintf(stderr, "Target file '%s' is not a valid gamefile.\n", inname);
  if ((c2 < 1) || (c2 > 8))
    fprintf(stderr, "Source file '%s' is not a valid gamefile.\n", cryptname);
  if ((c1 < 1) || (c1 > 8) || (c2 < 1) || (c2 > 8))
  {
    close_files();
    exit(1);
  }
}

/* get_version_details
 * Gets in the version details for both the input file and encryption file.
 * Uses retrieve_details to retrieve the data.
 */

void retrieve_details(FILE_DATA *data, FILE *f, char *FileDesc, char *fname)
{
  unsigned int c, ctr;

  /* Get in the zcode version number */
  fseek(f, 0, SEEK_SET);
  data->zcode_version = fgetc(f);

  /* Get release number */
  fseek(f, 2, SEEK_SET);           /* move to release number    */
  c = fgetc(f);                    /* Get in HI byte of release */
  data->release = c*256;           /* Move into release number  */
  c = fgetc(f);                    /* Get in LO byte of release */
  data->release += c;              /* Add into release value    */

  /* Get in version */
  fseek(f, 0x12, SEEK_SET);
  for (ctr=0; ctr<6; ++ctr)
    data->version[ctr] = (char) fgetc(f);

  /* Get in the filesize */
  fseek(f, 0x1A, SEEK_SET);
  c = fgetc(f);                    /* Get HI byte of compressed filesize */
  data->filesize = ((unsigned int) c) << 8;  /* Store in filesize        */
  c = fgetc(f);                    /* Get LO byte of compressed filesize */
  data->filesize += ((unsigned int) c);      /* Store in filesize        */
  switch(data->zcode_version)      /* Apply filesize multiplier          */
  {
    case 1: case 2: case 3:
      data->filesize *= 2; break;
    case 4: case 5:
      data->filesize *= 4; break;
    case 6: case 7: case 8:
      data->filesize *= 8; break;
  }
  /* Check if the filesize is zero. */
  if (data->filesize == 0)
  {
    /* No filesize provided, so use entire file */
    printf("Warning: %s '%s' filesize not specified, using entire file.\n", FileDesc, fname);
    printf("File should be checked with a program like InfoCut to ensure ");
    printf("there is no\n");
    printf("padding on the file, or the resulting patchfile will be ");
    printf("inaccurate.\n\n");

    /* Work out the physical file size */
    fseek(f, 0, SEEK_END);
    data->filesize = ftell(f);
  }

  /* Reset back to the start of the gamefile */
  fseek(f, 0, SEEK_SET);
}

void get_version_details()
{
  retrieve_details(&infile_data, infile, "Target File", inname);
  retrieve_details(&cryptfile_data, cryptfile, "Source File", cryptname);
}

/* extract_gamename
 * Extracts the gamefile name from the command line. This is necessary, since
 * the C startup code splits up individual words: we need to concatenate the
 * possible multiple words of the game name back together again.
 */

void extract_gamename(int argc, char *argv[])
{
  int ctr;

  /* Clear the name */
  for (ctr=0; ctr<34; ctr++) game[ctr] = '\0';

  /* Extract words from the command line */
  ctr = 4;   /* parameter index for first word of game name */
  while (ctr < argc)
  {
    /* Special check: If first word begins with a '"' then ignore it */
    /* when copying (to support a "name" format)                     */
    if ((ctr == 4) && (argv[ctr][0] == '"'))
    {
      if (strlen(argv[4]) > 32) return;
      strcpy(game, &argv[4][1]);
    }
    else
    {
      if (strlen(game) + 1 + strlen(argv[ctr]) > 32) return;
      if (ctr>4) strcat(game, " ");
      strcat(game, argv[ctr]);
    }
    ++ctr;
  }
  if ((strlen(game)>0) && (game[strlen(game)-1] == '"'))
    game[strlen(game)-1] = '\0';
}

/* print_version
 * prints out the version given a data structure
 * Format: release/version [vzcode-version]
 */

void print_hex_digit(int d)
{
  char digits[16] = "0123456789ABCDEF";
  printf("%c", digits[d]);
}

void print_version(FILE_DATA *d)
{
  int ctr;
  int printable;

  printf("%d/", d->release);

  /* Check to make sure all 6 version digits are printable. If not, */
  /* print all six in hexadecimal format                            */
  printable = 1;
  for (ctr=0; (ctr<6) && (printable); ctr++)
  {
    if ((printable) && ((d->version[ctr] < ' ') || (d->version[ctr] > '\127')))
      printable = 0;
  }
  /* Print digits depending on resulting printable flag */
  if (printable)
  {
    for (ctr=0; ctr<6; ctr++) printf("%c", d->version[ctr]);
  }
  else
  {
    printf("$");
    for (ctr=0; ctr<6; ctr++)
    {
      print_hex_digit(((unsigned char) d->version[ctr]) / 16);
      print_hex_digit(((unsigned char) d->version[ctr]) % 16);
    }
  }

  printf(" [v%d]", d->zcode_version);
}

/* encrypt_file
 * Performs the encryption of the input file and writing to the output file
 */

void encrypt_file()
{
  int ctr;
  unsigned long int infilepos, cryptfilepos;
  unsigned int c1, c2;

  /* Print out the details of what is being encrypted */
  printf("Diff'ing game \"%s\" ", game);
  print_version(&infile_data);
  printf("\nusing ");
  print_version(&cryptfile_data);
  printf(" as source.\n\n");


  /* Write header details out */
  fputs("PFG", outfile);         /* Patchfile identifier */

  for (ctr=0; ctr<32; ctr++)     /* Gamefile name */
    fputc(game[ctr], outfile);

  /* Write input file version */
  fputc(infile_data.zcode_version, outfile);
  fputc(infile_data.release / 256, outfile);
  fputc(infile_data.release % 256, outfile);
  for (ctr=0; ctr<6; ctr++) fputc(infile_data.version[ctr], outfile);

  /* Write encryption file version */
  fputc(cryptfile_data.zcode_version, outfile);
  fputc(cryptfile_data.release / 256, outfile);
  fputc(cryptfile_data.release % 256, outfile);
  for (ctr=0; ctr<6; ctr++) fputc(cryptfile_data.version[ctr], outfile);

  /* Main loop for encrypting the file */
  infilepos = 0; cryptfilepos = 0;
  for (;;)
  {

    /* Read a byte in from both the input file and encryption file */
    if ((c1 = fgetc(infile)) == (unsigned int) -1)
    {
      /* Input file EOF passed. This this not happen even if a filesize */
      /* is not embedded in the gamefile, since the initialization code */
      /* simply takes the physical filesize anyway.                     */
      fprintf(stderr, "EOF passed in Target File '%s'. Likely cause is an incorrrectly \
calculated filesize.\n\n", inname);
      return;
    }
    if ((c2 = fgetc(cryptfile)) == (unsigned int) -1)
    {
      /* Encryption file EOF passed. Again, this should never happen */
      fprintf(stderr, "EOF passed in Source File '%s'. Likely cause is an \
incorrrectly calculated filesize.\n\n", cryptname);
    }

    /* Encrypt the byte via XOR and write it out to the output file */
    fputc((char) (c1 ^ c2), outfile);
    ++infilepos; ++cryptfilepos;

    /* If reached the end of input gamefile's specified filesize */
    /* then finish the encryption                                */
    if (infilepos >= infile_data.filesize) return;

    /* If the encryption file position moves past that specified in the */
    /* header, reset back to the file start. This ensures that files    */
    /* with padding bytes can still be used for decryption              */
    if (cryptfilepos >= cryptfile_data.filesize)
    {
      cryptfilepos = 0;
      fseek(cryptfile, 0, SEEK_SET);
    }
  }
}

/* close_files
 * Closes all three of the files
 */

void close_files()
{
  fclose(infile);
  fclose(outfile);
  fclose(cryptfile);
}

/*-------------------------------------------------------------------------*/

int main(int argc, char *argv[])
{
  /* Print out the name of the program */
  printf("zcdiff version " VERSION "\n\n");

  /* First check to make sure two parameters (the input file and */
  /* encryption file) have been entered.                         */
  if (argc < 4)
  {
    printf("Produces a diff (patchfile) between two Infocome gamefiles.\n\n");
    printf("zcdiff source.dat target.dat output.pat [Name]\n\n");
    printf("  source.dat   Source gamefile to use for encryption\n");
    printf("  target.dat   Target gamefile to encrypt\n");
    printf("  output.pat   Output name for resulting patchfile\n");
    printf("  Name         An optional name for the file\n\n");
    printf("The target.dat can later be regenerated with the command:\n\n");
    printf("zcpatch output.pat source.dat target.dat\n");
    exit(1);
  }

  /* Open up the files */
  cryptname = argv[1];
  inname = argv[2];
  outname = argv[3];
  open_files();

  /* Make sure the files are valid gamefiles */
  verify_gamefiles();

  /* Extract the game's name from the command line */
  extract_gamename(argc, argv);

  /* Extract the input file and game version data */
  get_version_details();
  
  /* Encrypt the file */
  encrypt_file();

  /* Close the files */
  close_files();

  printf("Patch produced successfully.\n");
  return 0;
}

