package zpplet.system;

import java.util.concurrent.*;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;

import zpplet.ZUserConfig;
import zpplet.header.ZHeader5;
import zpplet.machine.ZMachine;
import zpplet.machine.ZMachine6;
import zpplet.misc.*;

public class ZScreen
		extends JPanel
		implements KeyListener, MouseListener
	{
	private BufferedImage backimage;
	private Graphics gback;

	private BlockingQueue inputcodes;
	private StringBuilder inputline;
	private String previnput = null;
	private ZCursor cursor;

	private ZMachine zm;

	private int lines, chars; // in character units
	private int lineheight, charwidth; // nominal pixel sizes
	private int mousewindow; // -1 = whole screen
	int unity, unitx;

	private boolean scaled; // for Z6, locks dimensions
	private int mousebuttons;
	private int curwidth, curheight; // actual JPanel dimensions

	public ZScreen()
		{
		super();
		
		inputcodes = new ArrayBlockingQueue(16);
		inputline = new StringBuilder();
		cursor = new ZCursor(this);
		scaled = false;

		setFocusable(true);
		addMouseListener(this);
		addKeyListener(this);
		}

	public void attach(ZMachine zm)
		{
		this.zm = zm;
		inputline.setLength(0);
		mousewindow = -1;
		mousebuttons = 0;

		setScaling(zm instanceof ZMachine6);
		}

	public void detach()
		{
		cursor.setVisible(false);
		zm = null;
		try
			{
			inputcodes.put(new Integer(-1));
			}
		catch (InterruptedException e)
			{}
		inputcodes.clear();

		clear();
		}

	public int getChars()
		{
		return chars;
		}

	public int getLines()
		{
		return lines;
		}

	public int getCharWidth()
		{
		return charwidth;
		}

	public int getLineHeight()
		{
		return lineheight;
		}

	public void setMouseWindow(int value)
		{
		mousewindow = value;
		}

	public int getMouseButtons()
		{
		return mousebuttons;
		}

	public int getBoundsHeight()
		{
		return backimage.getHeight();
		}

	public int getBoundsWidth()
		{
		return backimage.getWidth();
		}
	
	public boolean isScaled()
		{
		return scaled;
		}

	public void keyPressed(KeyEvent e)
		{
		if ((zm != null) && e.isActionKey())
			{
			int result = ZChars.fromActionKeyToZAscii(e.getKeyCode());
			if (result != ZChars.INVALID) inputcodes.add(new Integer(result));
			}
		}

	public void keyReleased(KeyEvent e)
		{} // ignore

	public void keyTyped(KeyEvent e)
		{
		if (zm == null) return;
		int code = zm.zc.fromUnicodeToZAscii(e.getKeyChar());
		if (code != ZChars.INVALID) inputcodes.add(new Integer(code));
		}

	private int buttonMask(int b)
		{
		switch (b)
			{
			case MouseEvent.NOBUTTON:
				return 0;
			case MouseEvent.BUTTON1:
				return 1;
			case MouseEvent.BUTTON2:
				return 2;
			case MouseEvent.BUTTON3:
				return 4;
			default:
				return 0;
			}
		}

	public void mouseClicked(MouseEvent e)
		{
		if (!this.isFocusOwner()) requestFocus();

		if (zm == null) return;

		int x = e.getX();
		int y = e.getY();
		if (mousewindow >= 0)
			{
			ZWindow mw = zm.w[mousewindow];
			if ((x < mw.left) || (y < mw.top) || (x >= (mw.left + mw.width)) || (y >= (mw.top + mw.height))) return; // click outside of mouse window bounds; ignore
			}
		if (zm.hd instanceof ZHeader5)
			{
			((ZHeader5)zm.hd).setMouseClick(x, y); // screen relative
			if ((zm.hd.getVersion() == 5) || (e.getClickCount() == 1))
				inputcodes.add(new Integer(ZChars.MOUSE_CLICK));
			else
				inputcodes.add(new Integer(ZChars.MOUSE_DBLCLICK));
			}
		}

	public void mouseEntered(MouseEvent e)
		{} // ignore

	public void mouseExited(MouseEvent e)
		{} // ignore

	public void mousePressed(MouseEvent e)
		{
		mousebuttons |= buttonMask(e.getButton());
		}

	public void mouseReleased(MouseEvent e)
		{
		mousebuttons &= ~buttonMask(e.getButton());
		}

	public int readKeyCode(int timeout, int timeoutroutine)
		{
		try
			{
			if (zm.input != null) try
				{ // read from log file
				int r = zm.input.read();
				if (r >= 0) return r;
				zm.setInputStream(ZMachine.ISTREAM_KEYBOARD);
				}
			catch (java.io.IOException e)
				{}

			// regular input
			Integer result;
			if (timeout != 0)
				do
					{
					result = (Integer)inputcodes.poll(timeout * 100, TimeUnit.MILLISECONDS);
					if (result == null) if (zm.zi.interrupt(timeoutroutine) != 0) return ZChars.TIMEOUT;
					}
				while (result == null);
			else
				result = (Integer)inputcodes.take();
			return result.intValue();
			}
		catch (InterruptedException e)
			{
			return ZChars.ABORT;
			}
		}

	public void primeInput()
		{
		try
			{
			int r = zm.input.read();
			if (r >= 0)
				inputcodes.add(new Integer(r));
			else
				zm.setInputStream(ZMachine.ISTREAM_KEYBOARD);
			}
		catch (java.io.IOException e)
			{}
		}

	public int readBufferedCode(int timeout, int timeoutroutine)
		{
		// if input in the queue, provide it forthwith
		if (inputline.length() > 0)
			{
			int x = inputline.charAt(0);
			inputline.deleteCharAt(0);
			return x;
			}

		ZWindow inputwindow = zm.curw;

		// prepare to prompt
		inputwindow.flush();
		cursor.setColor(inputwindow.font.getForeground());
		int index = 0; // position in buffer of cursor

		while (true)
			{
			int x = inputwindow.getPosX();
			int y = inputwindow.getPosY();
			
			// draw input line
			gback.setColor(inputwindow.font.getBack());
			gback.fillRect(x, y, inputwindow.width - x, lineheight);
			drawText(x, y, inputline.toString(), inputwindow);

			// position cursor
			int w = gback.getFontMetrics().stringWidth(inputline.substring(0, index));
			cursor.setBounds(inputwindow.getPosX() + w, inputwindow.getPosY(), index >= inputline.length() ? charwidth
					: charwidth / 3, lineheight);
			repaint();
			cursor.setVisible(true);
			int incode = readKeyCode(timeout, timeoutroutine);

			switch (incode)
				{
				case ZChars.ABORT:
					cursor.setVisible(false);
					return incode;

				case ZChars.BACKSPACE:
					if (index > 0)
						inputline.deleteCharAt(--index);
					else
						Toolkit.getDefaultToolkit().beep();
					break;

				case ZChars.DELETE:
					if (index < inputline.length())
						inputline.deleteCharAt(index);
					else
						Toolkit.getDefaultToolkit().beep();
					break;

				case ZChars.CURSOR_LEFT:
				case ZChars.KEYPAD_KEYS + 4:
					if (index > 0)
						index--;
					else
						Toolkit.getDefaultToolkit().beep();
					break;

				case ZChars.CURSOR_RIGHT:
				case ZChars.KEYPAD_KEYS + 6:
					if (index < inputline.length())
						index++;
					else
						Toolkit.getDefaultToolkit().beep();
					break;

				case ZChars.KEYPAD_KEYS + 1: // END
					index = inputline.length();
					break;

				case ZChars.KEYPAD_KEYS + 7: // HOME
					index = 0;
					break;

				case ZChars.MOUSE_CLICK:
				case ZChars.MOUSE_DBLCLICK:
					inputline.append((char)incode);
					inputline.append((char)ZChars.RETURN_CHAR);
					cursor.setVisible(false);
					return readBufferedCode(0, 0);

				case ZChars.TIMEOUT: // timed out
					inputline.delete(0, inputline.length());
					return ZChars.TIMEOUT;
					
				case ZChars.CURSOR_UP:
					if (previnput != null)
						{
						inputline.delete(0, inputline.length());
						inputline.append(previnput);
						index = inputline.length();
						}
					break;

				default:
					if (incode == ZChars.RETURN_CHAR)
						{
						previnput = inputline.toString();
						inputline.append((char)incode);
						inputwindow.resetLineCount();
						inputwindow.postInputLine(inputline.toString());
						inputwindow.newLine();
						cursor.setVisible(false);
						return readBufferedCode(0, 0);
						// just snatches the first char
						}

					if (zm.zc.isStorable(incode)) inputline.insert(index++, (char)incode);
				}
			}
		}

	public void redoMetrics()
		{
		if (scaled) return;

		// establish size information
		FontMetrics fm = getFontMetrics(new Font(ZUserConfig.fixedfont, Font.PLAIN, ZUserConfig.fontsize));
		unitx = charwidth = fm.charWidth('0'); // spec definition
		unity = lineheight = fm.getHeight();

		chars = curwidth / charwidth;
		lines = curheight / lineheight;
		}
	
	public void setSize(int width, int height)
		{
		if ((zm != null) && zm.running) zm.resize0(height);
		curwidth = width;
		curheight = height;
		redoMetrics();

		// create backing image
		Image oldimage = backimage;
		backimage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
		gback = backimage.getGraphics();
		gback.setColor(ZColor.def.bg);
		gback.fillRect(0, 0, width, height);
		if (oldimage != null) gback.drawImage(oldimage, 0, 0, null);
		oldimage = null;

		if ((zm != null) && zm.running) zm.resize();
		if (cursor.isVisible()) cursor.move(zm.curw.getPosX(), zm.curw.getPosY());
		}

	public synchronized void setBounds(int x, int y, int width, int height)
		{
		super.setBounds(x, y, width, height);

		if (scaled) return;

		// don't deal with useless or redundant resizings
		if ((width <= 0) || (height <= 0)) return;
		if ((backimage != null) && (width == curwidth) && (height == curheight)) return;

		setSize(width, height);
		}

	protected synchronized int drawText(int x, int y, String text, ZWindow w)
		{
		gback.setFont(w.font.getFont());
		int width = gback.getFontMetrics().stringWidth(text);
		gback.setClip(w.left + w.leftmargin, w.top, w.getTextWidth(), w.height);

		Color bg = w.font.getBack();
		if (bg != null)
			{
			gback.setColor(bg);
			gback.fillRect(x, y, width, lineheight);
			}

		gback.setColor(w.font.getForeground());
		gback.drawString(text, x, y + gback.getFontMetrics().getAscent());
		gback.setClip(0, 0, backimage.getWidth(), backimage.getHeight());
		return width;
		}

	public synchronized void drawImage(Image img, int x, int y, ZWindow w)
		{
		gback.setClip(w.left, w.top, w.width, w.height);
		gback.drawImage(img, x, y, null);
		gback.setClip(0, 0, backimage.getWidth(), backimage.getHeight());
		}

	public synchronized void scrollUp(ZWindow w, int pixels)
		{
		gback.copyArea(w.left, w.top + pixels, w.width, w.height - pixels, 0, -pixels);
		gback.setColor(w.font.getBack());
		gback.fillRect(w.left, w.top + w.height - pixels, w.width, pixels);
		}

	public synchronized void scrollDown(ZWindow w, int pixels)
		{
		gback.copyArea(w.left, w.top, w.width, w.height - pixels, 0, pixels);
		gback.setColor(w.font.getBack());
		gback.fillRect(w.left, w.top, w.width, pixels);
		}

	public synchronized void paintComponent(Graphics g)
		{
		super.paintComponent(g);
		g.drawImage(backimage, 0, 0, null);
		cursor.paint(g);
		}

	public void clear()
		{
		gback.setColor(ZColor.def.bg);
		gback.fillRect(0, 0, getBoundsWidth(), getBoundsHeight());
		}

	public void eraseRegion(int left, int top, int width, int height, Color bg)
		{
		gback.setColor(bg);
		gback.fillRect(left, top, width, height);
		}

	public Dimension getMinimumSize()
		{
		return new Dimension(100, 100);
		}

	public int charWidth(char ch)
		{
		return gback.getFontMetrics().charWidth(ch);
		}

	public int getStringWidth(String s, ZFont font)
		{
		return gback.getFontMetrics(font.getFont()).stringWidth(s);
		}

	Frame getFrame()
		{
		Component cursor;

		cursor = this;
		while (!(cursor instanceof Frame))
			cursor = cursor.getParent();

		return (Frame)cursor;
		}

	public String getFileName(String title, boolean save)
		{
		try
			{
			FileDialog fd = new FileDialog(this.getFrame(), title, save ? FileDialog.SAVE : FileDialog.LOAD);
			fd.setVisible(true);
			if (!this.isFocusOwner()) requestFocus();
			if (fd.getFile() != null) return fd.getDirectory() + fd.getFile();
			}
		catch (AWTError e)
			{}
		return null;
		}

	public Color getColorAt(int x, int y)
		{
		if ((x >= 0) && (y >= 0) && (x < backimage.getWidth()) & (y < backimage.getHeight()))
			return new Color(backimage.getRGB(x, y));
		return ZColor.def.bg;
		}

	private void setScaling(boolean scale)
		{
		scaled = scale;
		if (scaled)
			{
			backimage = new BufferedImage(ZUserConfig.STATIC_WIDTH * ZUserConfig.v6scale, ZUserConfig.STATIC_HEIGHT
					* ZUserConfig.v6scale, BufferedImage.TYPE_INT_RGB);
			gback = backimage.getGraphics();

			ZColor.def = new ZColor(ZColor.Z_BLACK, ZColor.Z_WHITE);

			FontMetrics fm = getFontMetrics(new Font(ZUserConfig.fixedfont, Font.PLAIN, ZUserConfig.fontsize));
			charwidth = fm.charWidth('0'); // spec definition
			lineheight = fm.getHeight();

			lines = backimage.getHeight() / lineheight;
			chars = backimage.getWidth() / charwidth;
			unitx = 1;
			unity = 1;

			clear();
			return;
			}

		ZColor.def = new ZColor(ZColor.INITIAL_FG_COLOR, ZColor.INITIAL_BG_COLOR);
		setSize(getWidth(), getHeight());
		}

	void debugShowWindowBounds(ZWindow w)
		{
		gback.setColor(Color.RED);
		gback.drawRect(w.left, w.top, w.width, w.height);
		gback.setFont(new Font("Sans", Font.BOLD, 24));
		gback.drawString(Integer.toString(w.id), w.left + 2, w.top + gback.getFontMetrics().getAscent());
		}
	}