// Created with Squiffy 5.1.3
// https://github.com/textadventures/squiffy

(function(){
/* jshint quotmark: single */
/* jshint evil: true */

var squiffy = {};

(function () {
    'use strict';

    squiffy.story = {};

    var initLinkHandler = function () {
        var handleLink = function (link) {
            if (link.hasClass('disabled')) return;
            var passage = link.data('passage');
            var section = link.data('section');
            var rotateAttr = link.attr('data-rotate');
            var sequenceAttr = link.attr('data-sequence');
            if (passage) {
                disableLink(link);
                squiffy.set('_turncount', squiffy.get('_turncount') + 1);
                passage = processLink(passage);
                if (passage) {
                    currentSection.append('<hr/>');
                    squiffy.story.passage(passage);
                }
                var turnPassage = '@' + squiffy.get('_turncount');
                if (turnPassage in squiffy.story.section.passages) {
                    squiffy.story.passage(turnPassage);
                }
                if ('@last' in squiffy.story.section.passages && squiffy.get('_turncount')>= squiffy.story.section.passageCount) {
                    squiffy.story.passage('@last');
                }
            }
            else if (section) {
                currentSection.append('<hr/>');
                disableLink(link);
                section = processLink(section);
                squiffy.story.go(section);
            }
            else if (rotateAttr || sequenceAttr) {
                var result = rotate(rotateAttr || sequenceAttr, rotateAttr ? link.text() : '');
                link.html(result[0].replace(/&quot;/g, '"').replace(/&#39;/g, '\''));
                var dataAttribute = rotateAttr ? 'data-rotate' : 'data-sequence';
                link.attr(dataAttribute, result[1]);
                if (!result[1]) {
                    disableLink(link);
                }
                if (link.attr('data-attribute')) {
                    squiffy.set(link.attr('data-attribute'), result[0]);
                }
                squiffy.story.save();
            }
        };

        squiffy.ui.output.on('click', 'a.squiffy-link', function () {
            handleLink(jQuery(this));
        });

        squiffy.ui.output.on('keypress', 'a.squiffy-link', function (e) {
            if (e.which !== 13) return;
            handleLink(jQuery(this));
        });

        squiffy.ui.output.on('mousedown', 'a.squiffy-link', function (event) {
            event.preventDefault();
        });
    };

    var disableLink = function (link) {
        link.addClass('disabled');
        link.attr('tabindex', -1);
    }
    
    squiffy.story.begin = function () {
        if (!squiffy.story.load()) {
            squiffy.story.go(squiffy.story.start);
        }
    };

    var processLink = function(link) {
		link = String(link);
        var sections = link.split(',');
        var first = true;
        var target = null;
        sections.forEach(function (section) {
            section = section.trim();
            if (startsWith(section, '@replace ')) {
                replaceLabel(section.substring(9));
            }
            else {
                if (first) {
                    target = section;
                }
                else {
                    setAttribute(section);
                }
            }
            first = false;
        });
        return target;
    };

    var setAttribute = function(expr) {
        var lhs, rhs, op, value;
        var setRegex = /^([\w]*)\s*=\s*(.*)$/;
        var setMatch = setRegex.exec(expr);
        if (setMatch) {
            lhs = setMatch[1];
            rhs = setMatch[2];
            if (isNaN(rhs)) {
				if(startsWith(rhs,"@")) rhs=squiffy.get(rhs.substring(1));
                squiffy.set(lhs, rhs);
            }
            else {
                squiffy.set(lhs, parseFloat(rhs));
            }
        }
        else {
			var incDecRegex = /^([\w]*)\s*([\+\-\*\/])=\s*(.*)$/;
            var incDecMatch = incDecRegex.exec(expr);
            if (incDecMatch) {
                lhs = incDecMatch[1];
                op = incDecMatch[2];
				rhs = incDecMatch[3];
				if(startsWith(rhs,"@")) rhs=squiffy.get(rhs.substring(1));
				rhs = parseFloat(rhs);
                value = squiffy.get(lhs);
                if (value === null) value = 0;
                if (op == '+') {
                    value += rhs;
                }
                if (op == '-') {
                    value -= rhs;
                }
				if (op == '*') {
					value *= rhs;
				}
				if (op == '/') {
					value /= rhs;
				}
                squiffy.set(lhs, value);
            }
            else {
                value = true;
                if (startsWith(expr, 'not ')) {
                    expr = expr.substring(4);
                    value = false;
                }
                squiffy.set(expr, value);
            }
        }
    };

    var replaceLabel = function(expr) {
        var regex = /^([\w]*)\s*=\s*(.*)$/;
        var match = regex.exec(expr);
        if (!match) return;
        var label = match[1];
        var text = match[2];
        if (text in squiffy.story.section.passages) {
            text = squiffy.story.section.passages[text].text;
        }
        else if (text in squiffy.story.sections) {
            text = squiffy.story.sections[text].text;
        }
        var stripParags = /^<p>(.*)<\/p>$/;
        var stripParagsMatch = stripParags.exec(text);
        if (stripParagsMatch) {
            text = stripParagsMatch[1];
        }
        var $labels = squiffy.ui.output.find('.squiffy-label-' + label);
        $labels.fadeOut(1000, function() {
            $labels.html(squiffy.ui.processText(text));
            $labels.fadeIn(1000, function() {
                squiffy.story.save();
            });
        });
    };

    squiffy.story.go = function(section) {
        squiffy.set('_transition', null);
        newSection();
        squiffy.story.section = squiffy.story.sections[section];
        if (!squiffy.story.section) return;
        squiffy.set('_section', section);
        setSeen(section);
        var master = squiffy.story.sections[''];
        if (master) {
            squiffy.story.run(master);
            squiffy.ui.write(master.text);
        }
        squiffy.story.run(squiffy.story.section);
        // The JS might have changed which section we're in
        if (squiffy.get('_section') == section) {
            squiffy.set('_turncount', 0);
            squiffy.ui.write(squiffy.story.section.text);
            squiffy.story.save();
        }
    };

    squiffy.story.run = function(section) {
        if (section.clear) {
            squiffy.ui.clearScreen();
        }
        if (section.attributes) {
            processAttributes(section.attributes);
        }
        if (section.js) {
            section.js();
        }
    };

    squiffy.story.passage = function(passageName) {
        var passage = squiffy.story.section.passages[passageName];
        if (!passage) return;
        setSeen(passageName);
        var masterSection = squiffy.story.sections[''];
        if (masterSection) {
            var masterPassage = masterSection.passages[''];
            if (masterPassage) {
                squiffy.story.run(masterPassage);
                squiffy.ui.write(masterPassage.text);
            }
        }
        var master = squiffy.story.section.passages[''];
        if (master) {
            squiffy.story.run(master);
            squiffy.ui.write(master.text);
        }
        squiffy.story.run(passage);
        squiffy.ui.write(passage.text);
        squiffy.story.save();
    };

    var processAttributes = function(attributes) {
        attributes.forEach(function (attribute) {
            if (startsWith(attribute, '@replace ')) {
                replaceLabel(attribute.substring(9));
            }
            else {
                setAttribute(attribute);
            }
        });
    };

    squiffy.story.restart = function() {
        if (squiffy.ui.settings.persist && window.localStorage) {
            var keys = Object.keys(localStorage);
            jQuery.each(keys, function (idx, key) {
                if (startsWith(key, squiffy.story.id)) {
                    localStorage.removeItem(key);
                }
            });
        }
        else {
            squiffy.storageFallback = {};
        }
        if (squiffy.ui.settings.scroll === 'element') {
            squiffy.ui.output.html('');
            squiffy.story.begin();
        }
        else {
            location.reload();
        }
    };

    squiffy.story.save = function() {
        squiffy.set('_output', squiffy.ui.output.html());
    };

    squiffy.story.load = function() {
        var output = squiffy.get('_output');
        if (!output) return false;
        squiffy.ui.output.html(output);
        currentSection = jQuery('#' + squiffy.get('_output-section'));
        squiffy.story.section = squiffy.story.sections[squiffy.get('_section')];
        var transition = squiffy.get('_transition');
        if (transition) {
            eval('(' + transition + ')()');
        }
        return true;
    };

    var setSeen = function(sectionName) {
        var seenSections = squiffy.get('_seen_sections');
        if (!seenSections) seenSections = [];
        if (seenSections.indexOf(sectionName) == -1) {
            seenSections.push(sectionName);
            squiffy.set('_seen_sections', seenSections);
        }
    };

    squiffy.story.seen = function(sectionName) {
        var seenSections = squiffy.get('_seen_sections');
        if (!seenSections) return false;
        return (seenSections.indexOf(sectionName) > -1);
    };
    
    squiffy.ui = {};

    var currentSection = null;
    var screenIsClear = true;
    var scrollPosition = 0;

    var newSection = function() {
        if (currentSection) {
            disableLink(jQuery('.squiffy-link', currentSection));
        }
        var sectionCount = squiffy.get('_section-count') + 1;
        squiffy.set('_section-count', sectionCount);
        var id = 'squiffy-section-' + sectionCount;
        currentSection = jQuery('<div/>', {
            id: id,
        }).appendTo(squiffy.ui.output);
        squiffy.set('_output-section', id);
    };

    squiffy.ui.write = function(text) {
        screenIsClear = false;
        scrollPosition = squiffy.ui.output.height();
        currentSection.append(jQuery('<div/>').html(squiffy.ui.processText(text)));
        squiffy.ui.scrollToEnd();
    };

    squiffy.ui.clearScreen = function() {
        squiffy.ui.output.html('');
        screenIsClear = true;
        newSection();
    };

    squiffy.ui.scrollToEnd = function() {
        var scrollTo, currentScrollTop, distance, duration;
        if (squiffy.ui.settings.scroll === 'element') {
            scrollTo = squiffy.ui.output[0].scrollHeight - squiffy.ui.output.height();
            currentScrollTop = squiffy.ui.output.scrollTop();
            if (scrollTo > currentScrollTop) {
                distance = scrollTo - currentScrollTop;
                duration = distance / 0.4;
                squiffy.ui.output.stop().animate({ scrollTop: scrollTo }, duration);
            }
        }
        else {
            scrollTo = scrollPosition;
            currentScrollTop = Math.max(jQuery('body').scrollTop(), jQuery('html').scrollTop());
            if (scrollTo > currentScrollTop) {
                var maxScrollTop = jQuery(document).height() - jQuery(window).height();
                if (scrollTo > maxScrollTop) scrollTo = maxScrollTop;
                distance = scrollTo - currentScrollTop;
                duration = distance / 0.5;
                jQuery('body,html').stop().animate({ scrollTop: scrollTo }, duration);
            }
        }
    };

    squiffy.ui.processText = function(text) {
        function process(text, data) {
            var containsUnprocessedSection = false;
            var open = text.indexOf('{');
            var close;
            
            if (open > -1) {
                var nestCount = 1;
                var searchStart = open + 1;
                var finished = false;
             
                while (!finished) {
                    var nextOpen = text.indexOf('{', searchStart);
                    var nextClose = text.indexOf('}', searchStart);
         
                    if (nextClose > -1) {
                        if (nextOpen > -1 && nextOpen < nextClose) {
                            nestCount++;
                            searchStart = nextOpen + 1;
                        }
                        else {
                            nestCount--;
                            searchStart = nextClose + 1;
                            if (nestCount === 0) {
                                close = nextClose;
                                containsUnprocessedSection = true;
                                finished = true;
                            }
                        }
                    }
                    else {
                        finished = true;
                    }
                }
            }
            
            if (containsUnprocessedSection) {
                var section = text.substring(open + 1, close);
                var value = processTextCommand(section, data);
                text = text.substring(0, open) + value + process(text.substring(close + 1), data);
            }
            
            return (text);
        }

        function processTextCommand(text, data) {
            if (startsWith(text, 'if ')) {
                return processTextCommand_If(text, data);
            }
            else if (startsWith(text, 'else:')) {
                return processTextCommand_Else(text, data);
            }
            else if (startsWith(text, 'label:')) {
                return processTextCommand_Label(text, data);
            }
            else if (/^rotate[: ]/.test(text)) {
                return processTextCommand_Rotate('rotate', text, data);
            }
            else if (/^sequence[: ]/.test(text)) {
                return processTextCommand_Rotate('sequence', text, data);   
            }
            else if (text in squiffy.story.section.passages) {
                return process(squiffy.story.section.passages[text].text, data);
            }
            else if (text in squiffy.story.sections) {
                return process(squiffy.story.sections[text].text, data);
            }
			else if (startsWith(text,'@') && !startsWith(text,'@replace')) {
				processAttributes(text.substring(1).split(","));
				return "";
			}
            return squiffy.get(text);
        }

        function processTextCommand_If(section, data) {
            var command = section.substring(3);
            var colon = command.indexOf(':');
            if (colon == -1) {
                return ('{if ' + command + '}');
            }

            var text = command.substring(colon + 1);
            var condition = command.substring(0, colon);
			condition = condition.replace("<", "&lt;");
            var operatorRegex = /([\w ]*)(=|&lt;=|&gt;=|&lt;&gt;|&lt;|&gt;)(.*)/;
            var match = operatorRegex.exec(condition);

            var result = false;

            if (match) {
                var lhs = squiffy.get(match[1]);
                var op = match[2];
                var rhs = match[3];

				if(startsWith(rhs,'@')) rhs=squiffy.get(rhs.substring(1));
				
                if (op == '=' && lhs == rhs) result = true;
                if (op == '&lt;&gt;' && lhs != rhs) result = true;
                if (op == '&gt;' && lhs > rhs) result = true;
                if (op == '&lt;' && lhs < rhs) result = true;
                if (op == '&gt;=' && lhs >= rhs) result = true;
                if (op == '&lt;=' && lhs <= rhs) result = true;
            }
            else {
                var checkValue = true;
                if (startsWith(condition, 'not ')) {
                    condition = condition.substring(4);
                    checkValue = false;
                }

                if (startsWith(condition, 'seen ')) {
                    result = (squiffy.story.seen(condition.substring(5)) == checkValue);
                }
                else {
                    var value = squiffy.get(condition);
                    if (value === null) value = false;
                    result = (value == checkValue);
                }
            }

            var textResult = result ? process(text, data) : '';

            data.lastIf = result;
            return textResult;
        }

        function processTextCommand_Else(section, data) {
            if (!('lastIf' in data) || data.lastIf) return '';
            var text = section.substring(5);
            return process(text, data);
        }

        function processTextCommand_Label(section, data) {
            var command = section.substring(6);
            var eq = command.indexOf('=');
            if (eq == -1) {
                return ('{label:' + command + '}');
            }

            var text = command.substring(eq + 1);
            var label = command.substring(0, eq);

            return '<span class="squiffy-label-' + label + '">' + process(text, data) + '</span>';
        }

        function processTextCommand_Rotate(type, section, data) {
            var options;
            var attribute = '';
            if (section.substring(type.length, type.length + 1) == ' ') {
                var colon = section.indexOf(':');
                if (colon == -1) {
                    return '{' + section + '}';
                }
                options = section.substring(colon + 1);
                attribute = section.substring(type.length + 1, colon);
            }
            else {
                options = section.substring(type.length + 1);
            }
            var rotation = rotate(options.replace(/"/g, '&quot;').replace(/'/g, '&#39;'));
            if (attribute) {
                squiffy.set(attribute, rotation[0]);
            }
            return '<a class="squiffy-link" data-' + type + '="' + rotation[1] + '" data-attribute="' + attribute + '" role="link">' + rotation[0] + '</a>';
        }

        var data = {
            fulltext: text
        };
        return process(text, data);
    };

    squiffy.ui.transition = function(f) {
        squiffy.set('_transition', f.toString());
        f();
    };

    squiffy.storageFallback = {};

    squiffy.set = function(attribute, value) {
        if (typeof value === 'undefined') value = true;
        if (squiffy.ui.settings.persist && window.localStorage) {
            localStorage[squiffy.story.id + '-' + attribute] = JSON.stringify(value);
        }
        else {
            squiffy.storageFallback[attribute] = JSON.stringify(value);
        }
        squiffy.ui.settings.onSet(attribute, value);
    };

    squiffy.get = function(attribute) {
        var result;
        if (squiffy.ui.settings.persist && window.localStorage) {
            result = localStorage[squiffy.story.id + '-' + attribute];
        }
        else {
            result = squiffy.storageFallback[attribute];
        }
        if (!result) return null;
        return JSON.parse(result);
    };

    var startsWith = function(string, prefix) {
        return string.substring(0, prefix.length) === prefix;
    };

    var rotate = function(options, current) {
        var colon = options.indexOf(':');
        if (colon == -1) {
            return [options, current];
        }
        var next = options.substring(0, colon);
        var remaining = options.substring(colon + 1);
        if (current) remaining += ':' + current;
        return [next, remaining];
    };

    var methods = {
        init: function (options) {
            var settings = jQuery.extend({
                scroll: 'body',
                persist: true,
                restartPrompt: true,
                onSet: function (attribute, value) {}
            }, options);

            squiffy.ui.output = this;
            squiffy.ui.restart = jQuery(settings.restart);
            squiffy.ui.settings = settings;

            if (settings.scroll === 'element') {
                squiffy.ui.output.css('overflow-y', 'auto');
            }

            initLinkHandler();
            squiffy.story.begin();
            
            return this;
        },
        get: function (attribute) {
            return squiffy.get(attribute);
        },
        set: function (attribute, value) {
            squiffy.set(attribute, value);
        },
        restart: function () {
            if (!squiffy.ui.settings.restartPrompt || confirm('Are you sure you want to restart?')) {
                squiffy.story.restart();
            }
        }
    };

    jQuery.fn.squiffy = function (methodOrOptions) {
        if (methods[methodOrOptions]) {
            return methods[methodOrOptions]
                .apply(this, Array.prototype.slice.call(arguments, 1));
        }
        else if (typeof methodOrOptions === 'object' || ! methodOrOptions) {
            return methods.init.apply(this, arguments);
        } else {
            jQuery.error('Method ' +  methodOrOptions + ' does not exist');
        }
    };
})();

var get = squiffy.get;
var set = squiffy.set;


squiffy.story.start = '_default';
squiffy.story.id = 'bad0a664f9';
squiffy.story.sections = {
	'_default': {
		'text': "<h1 id=\"externoon\">externoon</h1>\n<h2 id=\"an-interactive-novel-by-nune-\">An interactive novel by <a href=\"https://nune.itch.io\">nune</a>.</h2>\n<p><a class=\"squiffy-link link-section\" data-section=\"begin\" role=\"link\" tabindex=\"0\">begin</a></p>",
		'passages': {
		},
	},
	'begin': {
		'text': "<p>Dear Angie,</p>\n<p>I woke up this morning at three o&#39;clock. I know. I can hear Mom saying &quot;that&#39;s yesterday&quot; in my head, too.</p>\n<p>I couldn&#39;t really sleep.</p>\n<p>Remember when I told you I&#39;d <a class=\"squiffy-link link-section\" data-section=\"travel\" role=\"link\" tabindex=\"0\">travel</a> someday? That day is today. Please. Try not to worry. I&#39;m OK. I think.</p>\n<p>I&#39;m sorry.</p>",
		'passages': {
		},
	},
	'travel': {
		'text': "<p>I&#39;m at the entrance of the <a class=\"squiffy-link link-section\" data-section=\"bus terminal\" role=\"link\" tabindex=\"0\">bus terminal</a> before I know it. I was expecting the cab ride to feel like a million\nyears. To my great relief, the driver wasn&#39;t in a talkative mood. Sitting behind tinted windows, I checked &mdash; and\ndouble checked &mdash; my supplies. Three bags: a small leather purse, a <a class=\"squiffy-link link-passage\" data-passage=\"canvas tote bag\" role=\"link\" tabindex=\"0\">canvas tote bag</a> from my grandmother, and the trusty <a class=\"squiffy-link link-passage\" data-passage=\"nylon backpack\" role=\"link\" tabindex=\"0\">nylon backpack</a> I bought just before I started <a class=\"squiffy-link link-passage\" data-passage=\"college\" role=\"link\" tabindex=\"0\">college</a>.</p>",
		'passages': {
			'canvas tote bag': {
				'text': "<p>At least half of the bag is stacked with pads, microfiber towels, and bamboo paper towels. You can never have too many\npaper towels. The rest is of the bag is standard toiletries: a stick of deodorant, moisturizer, hand sanitizer, castile\nsoap, sports bras, underwear, and some aloe vera. Not to mention, a few printouts of a map of the settlement and a\nsolar powered, hand crank cell phone charger/flashlight combo.</p>",
			},
			'nylon backpack': {
				'text': "<p>I bought this thing twenty years ago now, and it hasn&#39;t failed me yet. It&#39;s stuffed with three notebooks, a couple novels, four bottles of mineral water, two pairs of leggings, four tank tops, two hoodies, a knitted blanket, and a pillow case. My clothes double as pillow stuffing.</p>",
			},
			'college': {
				'text': "<p>A lifetime ago. An alternate universe.</p>",
			},
		},
	},
	'bus terminal': {
		'text': "<Describe the bus terminal itself and the people waiting. Go buy a ticket at a kiosk>\n\n<p>The bus terminal smells like old piss, and oddly enough, I&#39;m comforted by this. I haven&#39;t been here <a class=\"squiffy-link link-passage\" data-passage=\"in years\" role=\"link\" tabindex=\"0\">in years</a>, and it hasn&#39;t changed much. The same dull hustle and bustle of the train station, the same awkward arrangement of slated steel benches, the same blur of tired faces and <a class=\"squiffy-link link-passage\" data-passage=\"downcast eyes\" role=\"link\" tabindex=\"0\">downcast eyes</a>.</p>\n<p><em>I&#39;ll almost miss you.</em></p>\n<p>It&#39;s busier than I was expecting. I wonder where everyone is <a class=\"squiffy-link link-section\" data-section=\"going\" role=\"link\" tabindex=\"0\">going</a>.</p>",
		'passages': {
			'in years': {
				'text': "<p>Not since I moved in with Angie.</p>",
			},
			'downcast eyes': {
				'text': "<p>It&#39;s early October. And it&#39;s actually cold. </p>",
			},
		},
	},
	'going': {
		'text': "<p>Luckily, there&#39;s no line at the ticket kiosk. <em>Everyone probably bought their tickets on the app.</em> I dig my wallet out\nof my purse bag, hunching over the fat wad of cash that springs forth when I unzip it. I have {sequence:$480:$478} in cash on hand. It&#39;s all I have left after <a class=\"squiffy-link link-passage\" data-passage=\"selling my stuff\" role=\"link\" tabindex=\"0\">selling my stuff</a> and splurging on my supplies.</p>\n<p>The next bus isn&#39;t for another four hours, and the ticket costs me $172 &mdash; not as much as I was expecting. I spot a lone seat in the corner, facing a wall. I <a class=\"squiffy-link link-section\" data-section=\"shuffle over\" role=\"link\" tabindex=\"0\">shuffle over</a> and take a look.</p>",
		'passages': {
			'selling my stuff': {
				'text': "<p>I kept it pretty light when I moved in with Angie. The single most valuable thing I owned was my laptop. But I did have a respectable book collection. I sold off what I could at rock bottom prices and recycled the rest. Going back and forth to the Post Office was a bitch too...</p>",
			},
		},
	},
	'shuffle over': {
		'text': "<p><em>I see.</em> There&#39;s a crumpled up white paper bag on the seat. <em>No wonder it&#39;s not taken.</em> A bit hesitantly, I pick it up\n&mdash; <em>nothing amiss</em> &mdash; and toss it in a nearby waste basket.</p>\n<p>I sink down into the seat, <a class=\"squiffy-link link-passage\" data-passage=\"suddenly exhausted\" role=\"link\" tabindex=\"0\">suddenly exhausted</a>. The wall of quiet hits me. No cranky baby? No disgruntled customers or staff?</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"Oh well.\" role=\"link\" tabindex=\"0\">Oh well.</a></p>",
		'passages': {
			'suddenly exhausted': {
				'text': "<p>Well, I&#39;ve been up for nearly twelve hours already. I scrubbed my bedroom and bathroom of all traces of me. Of my essence, anyway. </p>",
			},
		},
	},
	'Oh well.': {
		'text': "<p> I wake up with a start &mdash; eyes popped wide. I must have been snoring. Trying not to create a minor scene, I discretely check my <a class=\"squiffy-link link-passage\" data-passage=\"phone\" role=\"link\" tabindex=\"0\">phone</a>.</p>\n<p> <em>Ugh.</em> About an hour left until my departure. I slept for three hours, undisturbed. <em>Shit, now I won&#39;t be able to sleep on the bus.</em></p>\n<p> My backpack is still clutched to my chest. Tote bag still clenched between my feet. <em>OK. We&#39;re good.</em></p>\n<p> I take a deep breath, taking in the smell of warm bread and &mdash; what is that? &mdash; cream cheese (probably with\n chives). <em>Ah, fuck you random bagel person.</em> It&#39;s a little after five o&#39;clock in the evening now, and the hunger\n creeps down my throat and into my chest, echoing through my stomach. I&#39;ll have to <a class=\"squiffy-link link-section\" data-section=\"Chapter 2\" role=\"link\" tabindex=\"0\">distract myself</a> for another forty-five minutes or so. </p>\n<p> I probably have enough money for one meal for each day of the trip. After that, who knows.</p>",
		'passages': {
			'phone': {
				'text': "<p>Maybe it was a terrible idea, but I upgraded my phone before I left. This thing could get me killed, or it could save my life.</p>",
			},
		},
	},
	'Chapter 2': {
		'clear': true,
		'text': "<p>Dear Angie,</p>\n<p>Right about now I would be turning a knob on the stove, then filling a small pan with water, waiting for it to boil. I\nwould be pulling open two pre-prepared dinners, and placing our favorite spices on the counter to remind myself to use them. I&#39;d have my hands on my hips, humming to myself, eagerly waiting to hear your key turn in the lock.</p>\n<p>Instead, I&#39;m slowly chewing on my second granola bar, hoping it&#39;ll make me feel full. I&#39;ve got my phone&#39;s flashlight on in my right hand and a stack of <a class=\"squiffy-link link-passage\" data-passage=\"creased printouts\" role=\"link\" tabindex=\"0\">creased printouts</a> in my left.</p>\n<p>I got <a class=\"squiffy-link link-passage\" data-passage=\"your text\" role=\"link\" tabindex=\"0\">your text</a>. I know I should have responded sooner, but... What can I say? I&#39;m ashamed of myself for running away, but part of me thinks it would have been even more shameful to stay.</p>\n<p>Coming to rely on you at this stage of my life &mdash; at my age! &mdash; felt like such a shock.\nFirst stop: <a class=\"squiffy-link link-section\" data-section=\"Cleveland Station\" role=\"link\" tabindex=\"0\">Cleveland</a>.</p>",
		'passages': {
			'creased printouts': {
				'text': "<p>A map of the Oregon desert.</p>",
			},
			'Cleveland': {
				'text': "<p>Then Denver.</p>",
			},
			'your text': {
				'text': "<p>&quot;Fuck you,&quot; it says.</p>",
			},
		},
	},
	'Cleveland Station': {
		'text': "<p>It&#39;s 2:35am and I&#39;ve walked into a time warp.</p>\n<p><em>Maybe I&#39;ll be waiting at this station for eternity. Maybe I&#39;m already dead. I&#39;ll go up to Customer Service and ask when my bus is scheduled to arrive. &quot;Oh,&quot; the unassuming old man behind the desk will lift his giant glasses and stare intently at a paper bus schedule. &quot;It says here the bus to Denver just left.&quot;</p>\n<p>&quot;Just left?&quot; I&#39;ll nearly shout. &quot;But my bus just got here! How can thay be?&quot; The world&#39;s saturatioin will shift to black and white, but I won&#39;t notice.</p>\n<p>The unassuming old man will smile politely and say, &quot;I apologize for the mix-up young lady, but the next bus to Denver is only two hours away. Why don&#39;t you have a seat?&quot;</p>\n<p>I&#39;ll plop down <a class=\"squiffy-link link-section\" data-section=\"on a bench\" role=\"link\" tabindex=\"0\">on a bench</a> in a huff, occasionally looking up at the giant analog clock on a far wall. The second hand will tick slowly forward. I won&#39;t be able to leave this purgatory until I make peace with my inner demons. Or something.&quot;</em></p>",
		'passages': {
		},
	},
	'on a bench': {
		'text': "<p>But. Yeah. Cleveland.</p>\n<p>I can&#39;t see much of it at this time of night. The station is so bright it feels like we&#39;re on stage. Getting off of the bus, I was so focused on shuffling down the aisle, I didn&#39;t really take stock of my fellow passengers. Whoever they are, bless them. There was nothing but silence on the ride here. I was able to get a <a class=\"squiffy-link link-passage\" data-passage=\"bit of reading\" role=\"link\" tabindex=\"0\">bit of reading</a> done after going over my maps, and at some point I passed out. I woke up to the sound of the bus driver announcing our arrival.</p>\n<p>The Cleveland Station is cleaner than the one back east, but it&#39;s older and smells like a moldy wet mop. I see a couple people meet cars out front, but most of the passengers find a spot and settle in. <em>I guess we&#39;re all just passing through.</em></p>\n<p>I consider cracking open a <a class=\"squiffy-link link-passage\" data-passage=\"bottle of mineral water\" role=\"link\" tabindex=\"0\">bottle of mineral water</a> when I spot a <a class=\"squiffy-link link-section\" data-section=\"water fountain\" role=\"link\" tabindex=\"0\">water fountain</a>.</p>",
		'passages': {
			'bit of reading': {
				'text': "<p>The Man With Six Senses by Muriel Jaeger.</p>",
			},
			'bottle of mineral water': {
				'text': "<p>Agh! I better not. Getting these bottles off my back would be wonderful, but I brought these for the last leg of the\ntrip.</p>",
			},
		},
	},
	'water fountain': {
		'text': "<p>There&#39;s a short line forming in front of the water fountain in the back of the station. It seems to take me forever to gather my things, and I hobble on over to the <a class=\"squiffy-link link-passage\" data-passage=\"fountain\" role=\"link\" tabindex=\"0\">fountain</a>. It stands at the end of a long hallway, bathrooms on either side. I&#39;m <a class=\"squiffy-link link-section\" data-section=\"standing\" role=\"link\" tabindex=\"0\">standing</a> behind a woman and her <a class=\"squiffy-link link-passage\" data-passage=\"two kids\" role=\"link\" tabindex=\"0\">two kids</a>.</p>",
		'passages': {
			'fountain': {
				'text': "<p>A stainless steel box attached to a drab yellow wall. Business cards, ads, and missing persons are attached above.</p>",
			},
			'two kids': {
				'text': "<p>A one-year-old strapped to her front, a five-year-old holding her right hand.</p>",
			},
		},
	},
	'standing': {
		'text': "<p>A wide-bodied man in a brown leather jacket and light blue jeans <a class=\"squiffy-link link-passage\" data-passage=\"brushes by me\" role=\"link\" tabindex=\"0\">brushes by me</a> inserting himself somewhere at <a class=\"squiffy-link link-section\" data-section=\"the front of the line\" role=\"link\" tabindex=\"0\">the front of the line</a>. The woman in front of me turns around, lips pursed, and head tilted, &quot;Some people,&quot; she says.</p>\n<p>I&#39;m caught off guard. &quot;That&#39;s for sure,&quot; I say, trying to chuckle. <em>Was that weird?</em></p>",
		'passages': {
			'brushes by me': {
				'text': "<p>Why do some people seem to take up way more space than necessary?</p>",
			},
		},
	},
	'the front of the line': {
		'text': "<p>&quot;Hey, excuse me?&quot; A man&#39;s voice <a class=\"squiffy-link link-passage\" data-passage=\"silences the room\" role=\"link\" tabindex=\"0\">silences the room</a>. &quot;Excuse me!&quot; he says again after a long pause. &quot;You&#39;re <a class=\"squiffy-link link-section\" data-section=\"cutting\" role=\"link\" tabindex=\"0\">cutting</a>! Head to the back of the line like <a class=\"squiffy-link link-passage\" data-passage=\"everybody else\" role=\"link\" tabindex=\"0\">everybody else</a>!&quot;</p>",
		'passages': {
			'silences the room': {
				'text': "<p>Even the kids pipe down at the intensity and urgency of his voice.</p>",
			},
			'everybody else': {
				'text': "<p>I turn to look behind me and notice the line is much longer. <em>Is the entire station in line?</em></p>",
			},
		},
	},
	'cutting': {
		'text': "<p>The man in the brown leather jacket continues facing forward, and tosses a reply over his shoulder. &quot;Don&#39;t tell me what the fuck to do <a class=\"squiffy-link link-section\" data-section=\"Mr. Righteous\" role=\"link\" tabindex=\"0\">man</a>, I didn&#39;t cut <em>you</em>!&quot;</p>",
		'passages': {
		},
	},
	'Mr. Righteous': {
		'text': "<p>The <a class=\"squiffy-link link-passage\" data-passage=\"righteous man\" role=\"link\" tabindex=\"0\">righteous man</a> steps out of the line, and stands to the right of Leather Jacker. &quot;So what. You are cutting in line.&quot; He <a class=\"squiffy-link link-passage\" data-passage=\"gestures\" role=\"link\" tabindex=\"0\">gestures</a> to all of us standing behind him, &quot;We were waiting, and so should you.</p>\n<p>&quot;I said <a class=\"squiffy-link link-section\" data-section=\"fuck off\" role=\"link\" tabindex=\"0\">fuck off</a>, man. <em>You</em> get back in line.&quot;</p>",
		'passages': {
			'gestures': {
				'text': "<p>A sweeping and elegant, exaggerated whip of his arm as if he were a butler guiding his master.</p>",
			},
			'righteous man': {
				'text': "<p>Mr. Righteous looks like you&#39;d expect: All white hair with a thick mustache and a pair of old metal glasses. He&#39;s got on a gray sweatshirt, blue jeans, and white sneakers. He&#39;s holding a backpack in one hand and a couple plastic bags in the other.</p>",
			},
		},
	},
	'fuck off': {
		'text': "<p>&quot;Fuck off?&quot; Mr. Righteous slowly sets his bags on the floor, and rests his fists on his hips. &quot;Fuck? Off?&quot; He stretches his arms wide and scoffs, <a class=\"squiffy-link link-passage\" data-passage=\"looking incredulously\" role=\"link\" tabindex=\"0\">looking incredulously</a> to the rest of us, and then back to <a class=\"squiffy-link link-section\" data-section=\"Leather Jacket\" role=\"link\" tabindex=\"0\">Leather Jacket</a>. &quot;Everybody&#39;s gotta wait. Otherwise, why don&#39;t we <em>all</em> just cut in line?!&quot; He huffs, &quot;You drive a red car too?&quot;</p>",
		'passages': {
			'looking incredulously': {
				'text': "<p>There&#39;s nothing here yet.</p>",
			},
		},
	},
	'Leather Jacket': {
		'text': "<p>Hands still in his pants, Leather Jacket shrugs, &quot;<a class=\"squiffy-link link-section\" data-section=\"Bro\" role=\"link\" tabindex=\"0\">Bro</a>, are you really gonna make a scene right now?&quot; his voice is elevated and sneering.</p>",
		'passages': {
		},
	},
	'Bro': {
		'text': "<p>Righteous Man extends his arms again. &quot;<em>You&#39;re</em> the one <a class=\"squiffy-link link-section\" data-section=\"making a scene\" role=\"link\" tabindex=\"0\">making a scene</a>! You cut the goddamn line!&quot;</p>",
		'passages': {
		},
	},
	'making a scene': {
		'text': "<p>I take a deep breath, preparing to sigh performatively, and I inhale a gulp of foreign shit smell and the fumes of a generic\nindustrial strength cleaner. The intrusion makes me cough and wretch loudly enough to break the tension. A woman&#39;s raspy voice phones in from the back of the line. &quot;<a class=\"squiffy-link link-section\" data-section=\"Sir. Please.\" role=\"link\" tabindex=\"0\">Sir. Please.</a> Get back in line. It&#39;s three o&#39;clock in the morning.&quot;</p>",
		'passages': {
		},
	},
	'Sir. Please.': {
		'text': "<p>Mr. Righteous is momentarily stunned. &quot;Lady! I did what I was supposed to do! This motherfucker is the one in the wrong!&quot; <a class=\"squiffy-link link-section\" data-section=\"Mr. Righteous points\" role=\"link\" tabindex=\"0\">Mr. Righteous points</a> at Leather Jacket, his finger poking him slightly.</p>\n<p>The woman sighs, &quot;Lord, help me.&quot;</p>\n<p>Leather Jacket, who was standing virtually motionless during Mr. Righteous&#39;s tirade, slowly turns to face him. &quot;Bitch, if you touch me again I <em>will</em> smack you in your fucking mouth.&quot; He speaks so calmly and quietly, Mr. Righteous is stunned a second time. <em>Or maybe a third?</em></p>",
		'passages': {
		},
	},
	'Mr. Righteous points': {
		'text': "<p>Disgusted, Mr. Righteous&#39;s eyes are flitting side to side. He seems to be looking into Leather Jacket&#39;s eyes, looking for something.</p>\n<p>&quot;Un-fucking-believable,&quot; he scoffs. He snatches his wares off the floor and heads to the seating area. He&#39;s trying to\nmake a lot of noise, but everything he has is soft. &quot;Fucking bastards!&quot; he spits from <a class=\"squiffy-link link-section\" data-section=\"Chapter 3\" role=\"link\" tabindex=\"0\">far away</a>.</p>\n<p>Overhead, the speaker crackles in, &quot;If you all don&#39;t <a class=\"squiffy-link link-passage\" data-passage=\"calm down now\" role=\"link\" tabindex=\"0\">calm down now</a>, I will have to call the police.&quot;</p>",
		'passages': {
			'calm down now': {
				'text': "<p>I&#39;m thirsty.</p>",
			},
		},
	},
	'Chapter 3': {
		'clear': true,
		'text': "<p>Dear Angie,</p>\n<p>I used to think my life was going somewhere. Where was I going, anyway? To higher ground? Maybe I thought I&#39;d be safer\nthere. Turns out, higher ground is where the <a class=\"squiffy-link link-section\" data-section=\"survivors\" role=\"link\" tabindex=\"0\">survivors</a> duke it out.</p>\n<p>I&#39;m tired of life. Why has it seemed so long? I ussed to hear people say, &quot;...and then I woke up and I was sixty,&quot; and\npart of me would think, &quot;How nice.&quot; Another part of me would break out into a cold sweat, screaming internally: Maybe I\nshould have a kid, after all? Maybe I should have gotten a Master&#39;s degree? Maybe I should have worked longer hours,\ngotten into more fights, been more vulnerable, learned to <em>really</em> play chess, majored in computer science, done more drugs, filmed a documentary, spoken louder.</p>\n<p>Maybe maybe maybe maybe maybe maybe maybe.</p>\n<p>If I say it enough times, maybe &quot;maybe&quot; won&#39;t mean anything to me anymore.</p>",
		'passages': {
		},
	},
	'survivors': {
		'text': "<p>The ride to Denver is also quiet &mdash; except for the occasional phlegmy cough. It&#39;s a bright morning, and the air is really starting to feel different. I&#39;m somewhere new. Elsewhere.</p>\n<p>I plug in my phone. Against, I&#39;m seated alone. I&#39;m starting to resent it. Is there something on my face? No. Do I smell? Not that I can tell. Do I have a strange look in my eyes? Not particularly.</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"No messages.\" role=\"link\" tabindex=\"0\">No messages.</a> New phone. New number. Who&#39;s this?</p>",
		'passages': {
		},
	},
	'No messages.': {
		'text': "<p>After deleting all of my work-related documents and unfinished job applications, I only have <a class=\"squiffy-link link-passage\" data-passage=\"old pictures\" role=\"link\" tabindex=\"0\">old pictures</a> and an MP4 in my eyeCloud folder.</p>\n<p>It&#39;s the documentary, &quot;<a class=\"squiffy-link link-section\" data-section=\"Farming in the Desert\" role=\"link\" tabindex=\"0\">Farming in the Desert</a>: A Guide to Life.&quot; I put on my headphones, take a deep breath, and sink back, getting ready for my <a class=\"squiffy-link link-passage\" data-passage=\"dopamine hit\" role=\"link\" tabindex=\"0\">dopamine hit</a>.</p>\n<p>From the corner of my eye, I catch a young man in a baseball cap on the opposite side of the bus holding a family-sized bag of skittles. He is holding nothing else. No backpack, no suitcase, no briefcase, no plastic bag. No phone. No tablet. No gaming console. No book of any kind. </p>\n<p>In the seat next to him is another family-sized bag of skittles. <em>Yikes.</em></p>",
		'passages': {
			'old pictures': {
				'text': "<p>I left the photo album with Angie, but I scanned all of our baby photos. I&#39;m tempted to go through them now, but it&#39;s too early for nostalgia.</p>",
			},
			'dopamine hit': {
				'text': "<p><em>Why does dopamine get all the credit?</em></p>",
			},
		},
	},
	'Farming in the Desert': {
		'text': "<p>&quot;Farming in the Desert&quot; starts with a slow, silent fade-in from black to a sprawling desert rolling by an open car window. You can&#39;t see who&#39;s filming or who&#39;s driving. There&#39;s no narration &mdash; just the ambient hum of an old car engine and the <a class=\"squiffy-link link-section\" data-section=\"wheels on the road\" role=\"link\" tabindex=\"0\">wheels on the road</a>.</p>\n<p>It was probably recorded with a high quality phone or tablet. It&#39;s a long documentary &mdash; running close to three hours. It&#39;s likely the filmmaker&#39;s first, or one of their earliest, <a class=\"squiffy-link link-passage\" data-passage=\"film projects\" role=\"link\" tabindex=\"0\">film projects</a> and they just didn&#39;t know what to leave out. <em>I get that.</em></p>\n<p>I don&#39;t know how many times I&#39;ve seen it.</p>",
		'passages': {
			'film projects': {
				'text': "<p>A college kid, maybe.</p>",
			},
		},
	},
	'wheels on the road': {
		'text': "<p>Cut to a sign on the outskirts of an Indian Reservation reading &quot;DINOSAURS HERE.&quot; I smile. <em>I want to see dinosaurs, too.</em> Cut to a trail of dinosaur tracks set in stone across what was once flat, muddy terrain. Cut to the road again, but the bushes are more sparse.</p>\n<p>Cut to a tilted wooden signed etched with fire: &quot;The UnCooperative.&quot; I giggle out loud a little, covering my mouth. <em>That&#39;s so clever.</em></p>\n<p>The filmmakers may be from Colorado, or thereabouts, and drove in an L shape up through Utah and a bit of Idaho. Wherever they&#39;re from, I know where they&#39;re going.</p>",
		'passages': {
		},
	},
	'the bushes': {
		'text': "<p>The filmmakers looks up, centering their phone&#39;s camera on what, at first glance, looks like a gigantic service of <a class=\"squiffy-link link-passage\" data-passage=\"ice cream\" role=\"link\" tabindex=\"0\">ice cream</a>. It&#39;s surrounded by red bushes and the occasioinal patch of grass.</p>\n<p>The house, made of burnt orange clay, uses glass bottes for bricks, and it sparkles under the sun as the filmmaker approaches. <a class=\"squiffy-link link-section\" data-section=\"There's smoke\" role=\"link\" tabindex=\"0\">There&#39;s smoke</a> billowing out from behind the house.</p>",
		'passages': {
			'ice cream': {
				'text': "<p>The structure is a series of clay domes, each extending out from the center.</p>",
			},
		},
	},
	'There\'s smoke': {
		'text': "<p>&quot;OK folks, now please gather all your things and exit the vehicle in an orderly fashion.&quot; The driver&#39;s voice. The credits are rolling. <em>It&#39;s only been two hours. Why are we stopped?</em></p>\n<p>There&#39;re groans, cursing, and crying as people make their way, inch by inch off the bus. Out the window, I see people standing on the side of the road, bags in hand. We&#39;re not at the station. We&#39;re in the <a class=\"squiffy-link link-section\" data-section=\"middle of nowhere\" role=\"link\" tabindex=\"0\">middle of nowhere</a>.</p>\n<p>On the left, traffic inches quietly by. I hunch over, grab my backpack off the bus floor, and put <a class=\"squiffy-link link-passage\" data-passage=\"my phone\" role=\"link\" tabindex=\"0\">my phone</a> in my breast pocket, headphones still on.</p>\n<p>I join <a class=\"squiffy-link link-passage\" data-passage=\"the crowd\" role=\"link\" tabindex=\"0\">the crowd</a> and try to lean on the metal banister overlooking the mountains. It&#39;s too hot, so I unlatch the small bench from my backpack, expand it, and balance on the asphalt.</p>",
		'passages': {
			'my phone': {
				'text': "<p>I have about 40% left on my battery, so I close the video. I can always watch it again later.</p>",
			},
			'the crowd': {
				'text': "<p>The anger and exasperation of the other passengers makes me feel at home. <em>I&#39;m not sure what to think about that.</em></p>",
			},
		},
	},
	'middle of nowhere': {
		'text': "<p>&quot;Wow, that&#39;s smart!&quot; I can hear a voice through my headphones rise above the others. My body tenses, and I pretend not to hear. I&#39;ve been staring at my heands for the last ten minutes, so I try digging through my tote bag to busy myself.</p>\n<p>&quot;Hey, <a class=\"squiffy-link link-section\" data-section=\"that's really smart\" role=\"link\" tabindex=\"0\">that&#39;s really smart</a>!&quot; The voice is closer now. &quot;Excuse me?&quot; There&#39;s a pair of blue sneakers with orange stripes on the ground in front of me. <em>Here we go.</em></p>\n<p>I tilt my headphones behind my beck and look up. &quot;I&#39;m sorry?&quot; It&#39;s the young man dual-wielding two family-sized bags of\nskittles &mdash; one bag in each hand, one of which is half empty. He looks Indian, but he sounds Jamaican.</p>",
		'passages': {
		},
	},
	'that\'s really smart': {
		'text': "<p>&quot;I said, &#39;That&#39;s smart.&#39;&quot; He smiles widely, bright white teeth stained by food-grade dye. &quot;That little chair. I wish I had one of those.&quot; He points to my crotch.</p>\n<p>&quot;Oh.&quot; I try to brighten my voice. &quot;Yeah! You never know when you&#39;re going to need a seat!&quot;</p>\n<p>He laughs loudly, but there&#39;s <a class=\"squiffy-link link-section\" data-section=\"no stir from the crowd\" role=\"link\" tabindex=\"0\">no stir from the crowd</a>. I look around. Most of the passengers are shielding their\neyes, squinting eagerly at the bus. &quot;So, where are you coming from?&quot;</p>",
		'passages': {
		},
	},
	'no stir from the crowd': {
		'text': "<p><em>Fuck! Please don&#39;t talk to me, dude! I don&#39;t know how to talk to people!</em> &quot;Queens, and you?&quot;\n&quot;Jersey, but Trinidad originally.&quot; He stands next to me and squats down, looking out at the bus.</p>\n<p>&quot;Oh yeah?&quot; I&#39;m surprised.</p>\n<p>&quot;I&#39;ve been living in American ever since I was a baby, though. I don&#39;t even remember Trinidad. I mean, I went to visit a couple times.&quot;</p>\n<p>&quot;Hm. So, <a class=\"squiffy-link link-section\" data-section=\"where're you headed\" role=\"link\" tabindex=\"0\">where&#39;re you headed</a>?&quot;</p>\n<p>&quot;Washington,&quot; he shifts his weight.</p>\n<p>&quot;Um, would you like a blanket to sit on? I have a blanket in my bag somewhere?&quot;</p>\n<p>The young man smiles again, looking up at me. &quot;Wow, really? I would love a blanket right now.&quot; He laughs. He&#39;s got thick black eyelashes, and deep brown eyes. <em>He&#39;s watching me.</em></p>\n<p>I smile, despite myself. &quot;Why Washington?&quot; I hand him a neatly folded picnic blanket. He unfolds it twice, to half its size, and sits down.</p>\n<p>&quot;That&#39;s better. The engine overheated, y&#39;know? The bus driver said we&#39;re gonna have to wait like two hours for another bus.&quot;</p>",
		'passages': {
		},
	},
	'where\'re you headed': {
		'text': "<p>I cross my arms over my knees, leaning forward. &quot;Why Washington?&quot;</p>\n<p>His smile faded a bit. &quot;I have an uncle out there. He said he could hook me up with a job.&quot;</p>\n<p>&quot;No jobs in Jersey?&quot;</p>\n<p>&quot;Not for me. I&#39;m tired of Jersey.&quot;</p>\n<p>I scoff. &quot;I don&#39;t blame you. Where in Jersey are you from?&quot;</p>\n<p>&quot;<a class=\"squiffy-link link-section\" data-section=\"Jersey City\" role=\"link\" tabindex=\"0\">Jersey City</a>.&quot; He stuffs the half-opened bag in his pocket and pulls out the unopened bag of Skittles. He rips it open and holds the bag up toward me. &quot;Want some?&quot;</p>\n<p><em>Huh. Nice kid.</em> &quot;Really? Which part? I lived in Jersey City for a few years.&quot; <em>Small fucking world.</em> I cup my hands and he pours some in. <em>How many years has it been since I&#39;ve had candy? Like, actual candy?</em> &quot;Thank you.&quot;</p>",
		'passages': {
		},
	},
	'Jersey City': {
		'text': "<p>He laughs. &quot;That&#39;s crazy.&quot; He tosses a few Skittles in his mouth, talking and chewing at the same time. &quot;I live in Greenville with <a class=\"squiffy-link link-section\" data-section=\"my mom and sister\" role=\"link\" tabindex=\"0\">my mom and sister</a>. What about you?&quot;</p>\n<p>&quot;I lived just outside of downtown. The McGinley Square area.&quot;</p>\n<p>He laughs even louder, covering his mouth. &quot;Oh, you lived in the ghetto!&quot;</p>\n<p>I&#39;m annoyed, but I laugh along with him. &quot;A common misconception.&quot; I crunch on a couple pieces of candy.</p>\n<p>&quot;Why&#39;d you leave? You get tired of Jersey too?&quot;</p>\n<p>&quot;Well, I lost my job, and I couldn&#39;t afford the rent anymore.&quot;</p>\n<p>&quot;And you could afford the rent in Queens?&quot;</p>\n<p>&quot;Not really.&quot; The other passengers are quiet. I wonder if they&#39;re listening to our conversation. &quot;I moved in with my sister at the ass end of Queens.&quot;</p>\n<p>&quot;It&#39;s nice out there?&quot; He&#39;s staring into my eyes.</p>\n<p>I smile. &quot;Yeah, it&#39;s nice out there.&quot;</p>\n<p>&quot;Then why are you leaving?&quot;</p>\n<p>&quot;I&#39;m just... looking for a change of pace?&quot;</p>",
		'passages': {
		},
	},
	'my mom and sister': {
		'text': "<p>&quot;I&#39;m asking too many questions, right?&quot;</p>\n<p>I chuckle a bit. &quot;No, you&#39;re fine. It&#39;s just that this is my first time talking to someone about this.&quot;</p>\n<p>&quot;For real? What&#39;s about your sister?&quot;</p>\n<p>&quot;I mentioned it to her a couple times. Just as an idea. I didn&#39;t tell her when I was leaving until I was already gone.&quot;</p>\n<p>&quot;Damn. Is she mad? I know I&#39;d be mad if my sister did <a class=\"squiffy-link link-section\" data-section=\"Chapter 4\" role=\"link\" tabindex=\"0\">something like that</a> to me.&quot;</p>\n<p>&quot;Yeah. She&#39;s mad.&quot;</p>\n<p>He laughs again. &quot;Right?&quot;</p>",
		'passages': {
		},
	},
	'Chapter 4': {
		'clear': true,
		'text': "<p>Dear Angie,</p>\n<p>The world never felt very big to me. Even now. I don&#39;t know why. Maybe I&#39;m trying to prove myself wrong.</p>\n<p>There were times in my life when I knew a lot of people. An outsider looking in might think I had a lot of friends. But\nthat was <a class=\"squiffy-link link-section\" data-section=\"all work\" role=\"link\" tabindex=\"0\">all work</a>. Everything was work. Going out to dinner, giving talks, being loud, smiling virtually nonstop, getting drinks, shaking hands, running around. Always running around. Getting up early, staying up late. It was all just work.</p>",
		'passages': {
		},
	},
	'all work': {
		'text': "<p>We&#39;ve been sitting on the side of the road on the outskirts of Colorado for <a class=\"squiffy-link link-passage\" data-passage=\"two hours and twenty-two minutes\" role=\"link\" tabindex=\"0\">two hours and twenty-two minutes</a>. Our desperation has become <a class=\"squiffy-link link-section\" data-section=\"agitation\" role=\"link\" tabindex=\"0\">agitation</a>. I hear a mean joke, &quot;Shit, I&#39;m about to grab my bag and fucking <em>walk</em> to Denver &mdash; fuck this.&quot; People laugh. A woman chimes in, &quot;Wait for me, I packed my sneakers in here somewhere!&quot;</p>\n<p>A few kids are playing in dirt, and there&#39;s <a class=\"squiffy-link link-passage\" data-passage=\"a man in the distance\" role=\"link\" tabindex=\"0\">a man in the distance</a> pacing back and forth, yelling at his wife over the phone. Or his mother.</p>",
		'passages': {
			'a man in the distance': {
				'text': "<p><em>Is that Mr. Righteous?&quot;</em></p>",
			},
			'two hours and twenty-two minutes': {
				'text': "<p>A text from Angie: &quot;When are you coming back?&quot;</p>",
			},
		},
	},
	'agitation': {
		'text': "<p>I remember when I got my first contract in <a class=\"squiffy-link link-passage\" data-passage=\"the city\" role=\"link\" tabindex=\"0\">the city</a>. <a class=\"squiffy-link link-section\" data-section=\"How embarrassing.\" role=\"link\" tabindex=\"0\">How embarrassing.</a> I thought that mean I <em>was</em> somebody. Out of\nmy depth, I floundered for years before eventually giving up.</p>\n<p>I majored in <a class=\"squiffy-link link-passage\" data-passage=\"Library Science\" role=\"link\" tabindex=\"0\">Library Science</a> for fuck&#39;s sake. </p>",
		'passages': {
			'the city': {
				'text': "<p>Thirty-something street and sixth avenue. I started getting off the train at twenty-third street and walking the rest of the way about six months into the job.</p>",
			},
			'Library Science': {
				'text': "<p>I spent a lot of my childhood at the library &mdash; one of the few public places where you&#39;re not required to spend money, and you have to be quiet. My local library&#39;s selection was terrible, but I loved it anyway.</p>",
			},
		},
	},
	'How embarrassing.': {
		'text': "<p>The new bus pulls up with a screech an hour and change later. Some of the passengers clap for the bus driver. I want to join them, but I don&#39;t. <a class=\"squiffy-link link-section\" data-section=\"He smiles politely\" role=\"link\" tabindex=\"0\">He smiles politely</a>, &quot;OK, folks. Thanks for your patience.&quot;</p>\n<p>I feel a sense of relief and trepidation. I&#39;m going to miss the clean air cascading through my lungs. &quot;Well, time to go.&quot; I told my chair and look down at my bus friend. He&#39;s sleeping on my blanket. &quot;Ah, fuck.&quot;</p>\n<p><em>How the hell did he fall asleep on the asphalt?</em></p>",
		'passages': {
		},
	},
	'He smiles politely': {
		'text': "<p>I busy myself with my tiny chair and backpack, clasping them together, unsure of what to do. From my periphery, I see\nthe bus driver laugh. &quot;We&#39;ve got a live one here, folks!&quot; The other passengers laugh with him. The bus driver trots\nover, close enough to read his name tag for the first time: &lt;a class=&quot;squiffy-link link-section&quot; data-section=&quot;&quot;Edward.&quot;&quot; role=&quot;link&quot; tabindex=&quot;0&quot;&gt;&quot;Edward.&quot;</a></p>\n<p><em>That&#39;s his last name, right?</em></p>\n<p>&quot;Young man, it&#39;s time to go.&quot; My bus friend is completely still. I&#39;m struck by the sudden fear that he may be deceased.\n&quot;Young man, you don&#39;t want us to leave you now, do you?&quot; Edward, the bus driver, leans over and grabs <a class=\"squiffy-link link-passage\" data-passage=\"the kid's shoe\" role=\"link\" tabindex=\"0\">the kid&#39;s shoe</a> &mdash; wiggling it. &quot;Young man.&quot; Firmer this time.</p>\n<p>&quot;Hey,&quot; a man in the crowd scowls at me. &quot;wake him up!&quot;</p>\n<p>&quot;No, I think I&#39;ll just leave him here.&quot; <em>Fucking moron.</em> Agitated, I bark at the kid. &quot;Kid! Get up already.&quot;</p>",
		'passages': {
			'the kid\'s shoe': {
				'text': "<p><em>Now that I think about it, all of his clothes look brand new.</em></p>",
			},
		},
	},
	'"Edward."': {
		'text': "<p>The young man springs up as if summoned from the dead.</p>\n<p>&quot;I drooled on your nice blanket, I&#39;m sorry!&quot;</p>\n<p>&quot;Ah, don&#39;t worry,&quot; I say, as he carefully folds the blanket on its creases, &quot;A little drool never killed anyone. <em>Shit,\nnow I&#39;m not going to be able to use it for a while.</em> &quot;Hey,&quot; I&#39;ve just realized something, &quot;What&#39;s your name?&quot;</p>\n<p>The young man lifts his arms up, stretching and yawning loudly. &quot;Lucas,&quot; he says simply. &quot;You ready?&quot; Lucas pats his pockets to make sure his candy&#39;s still there, and walks over to <a class=\"squiffy-link link-section\" data-section=\"the new bus\" role=\"link\" tabindex=\"0\">the new bus</a>.</p>\n<p><em>Dude, we were waiting on you...</em></p>",
		'passages': {
		},
	},
	'the new bus': {
		'text': "<p>The new bus really feels new, like it&#39;s fresh off of the assembly line . This time, I choose a seat closer to the back, but not too close to the bathroom. I decide to give myself more leg room by tucking my backpack into the overhead cabin. I&#39;m ready for a nap.</p>\n<p>I see <a class=\"squiffy-link link-section\" data-section=\"Lucas\" role=\"link\" tabindex=\"0\">Lucas</a> from the corner of my eye. He&#39;s taken his hat off, and he&#39;s looking out the window. He looks a little\nolder.</p>\n<p>Other passengers shuffle by me &mdash; some of them chatting. I take a deep breath and sit down by my own window. The seats of the new bus are firm, but comfy. Suddenly, I can see the mountains &mdash; all of them covered in a layer of <a class=\"squiffy-link link-passage\" data-passage=\"deep green vegetation\" role=\"link\" tabindex=\"0\">deep green vegetation</a>. I wonder if there&#39;s anyone out there, hiding out, fighing, preparing themselves for a long winter. <em>Weird that I should be noticing this now.</em></p>",
		'passages': {
			'deep green vegetation': {
				'text': "<p><em>Are those mountains?</em></p>\n<Insert elaborate survival fantasy here.>",
			},
		},
	},
	'Lucas': {
		'text': "<p>A hard <em>plop</em> hits my left thigh. <em>I guess someone&#39;s ready to sit next to me now. Ugh.</em> I don&#39;t turn to look.</p>\n<p>&quot;Hey!&quot; A familiar voice. It&#39;s Lucas, with a big fat grin on his face. &quot;This white lady tried to sit next to me, but I want to sit next to you!&quot;</p>\n<p>&quot;Geez!&quot; I hiss, jutting my eyes to every corner of the bus to make sure no one&#39;s listening. &quot;We&#39;re in <a class=\"squiffy-link link-section\" data-section=\"mixed company\" role=\"link\" tabindex=\"0\">mixed company</a> here!&quot;</p>\n<p>Lucas laughs again from the bottom of his throat, but this time he keeps it to a low rumble. &quot;Who cares, I don&#39;t know any of these people!&quot; He giggles.</p>",
		'passages': {
		},
	},
	'mixed company': {
		'text': "<p><em>You don&#39;t know me, either.</em> &quot;I know, but we&#39;ve still gotta be on the bus with them for some &mdash;&quot; I gesture forward. &quot;&mdash; hundreds of miles.&quot; My hands hits my leg harder than I intend on the way down.</p>\n<p>&quot;True,&quot; he concedes, leaning back. He offers me more candy, and I accept. <em>Must be the stress.</em> &quot;By the way,&quot; he pops up again, &quot;Where are you going to anyway?&quot;</p>\n<p>My back stiffens. &quot;Oh! Well, I&#39;m heading to Oregon, myself.&quot; Without my backpack, <a class=\"squiffy-link link-passage\" data-passage=\"I feel exposed\" role=\"link\" tabindex=\"0\">I feel exposed</a>.</p>\n<p>&quot;Why?&quot; Lucas is looking into my eyes again.</p>\n<p>&quot;Y&#39;know, change of pace, et cetera...&quot; <em>Didn&#39;t we cover this?</em></p>\n<p>&quot;Yeah, but why Oregon? Are there jobs out there? You <a class=\"squiffy-link link-section\" data-section=\"got a boyfriend\" role=\"link\" tabindex=\"0\">got a boyfriend</a> out there?&quot;</p>",
		'passages': {
			'I feel exposed': {
				'text': "<p>What? Are we best friends now because we lived in the same city?</p>",
			},
		},
	},
	'got a boyfriend': {
		'text': "<p>I scoff, nearly choking on the Skittles. &quot;Nope. No boyfriend. No job. And I&#39;m not looking. For either. Not right now, anyway.&quot;</p>\n<p>&quot;How are you gonna <a class=\"squiffy-link link-section\" data-section=\"Chapter 5\" role=\"link\" tabindex=\"0\">survive out there</a>?&quot; Isn&#39;t that dangerous?&quot;</p>\n<p>&quot;Yeah, it may be. Your trip could be dangerous, too.&quot;</p>\n<p>&quot;That _is_ true. No doubt.&quot; His voice lowers. &quot;You really never know what could happen.&quot;</p>",
		'passages': {
		},
	},
	'Chapter 5': {
		'clear': true,
		'text': "<p>Dear Angie,</p>\n<p>Waiting in the <a class=\"squiffy-link link-passage\" data-passage=\"raw Autumn air\" role=\"link\" tabindex=\"0\">raw Autumn air</a> of Nowhere, Colorado was one part exhilarating and one part sobering. Can one be\nsoberly exhilarated? Anyway, it was weird. <a class=\"squiffy-link link-section\" data-section=\"Is life weird for everyone?\" role=\"link\" tabindex=\"0\">Is life weird for everyone?</a> I don&#39;t know. I don&#39;t even know if I feel like getting into it.</p>",
		'passages': {
			'raw Autumn air': {
				'text': "<p>The air here is certainly better than back East, but I still feel like I&#39;m suffocating.</p>",
			},
		},
	},
	'Is life weird for everyone?': {
		'text': "<p><em>This is not the account of Michael Bristowe&#39;s strange career that Hilda asked me to write.</em></p>\n<p>The dull, flatulent groan of a metal chair on indoor concrete interrupts my reading. Lucas, energized after his [nap on\nthe bus], sits across from me in an upscale cafe in downtown Denver. At the station, I changed into something a little <a class=\"squiffy-link link-section\" data-section=\"more acceptable\" role=\"link\" tabindex=\"0\">more acceptable</a>: Black dress sneakers, dark wash skinny jeans tastefully ripped, a black and white polka dot sleeveless blouse, and a long black cardigan. I keep my hair wrapped in a black satin scarf. <em>What&#39;s the name of this place again?</em></p>\n<p>&quot;Ooh, look at you! Who you trying to look cute for?&quot; Lucas laughs from the bottom of his throat again. He sits down smoothly across from me.</p>",
		'passages': {
			'nap on the bus': {
				'text': "<p>That breakdown set us back three hours. I was so agitated I couldn&#39;t sleep. I can&#39;t miss my ride.</p>",
			},
		},
	},
	'more acceptable': {
		'text': "<p>&quot;I&#39;m just doing the bare minimum,&quot; I mumble, briefly looking up at his smiling face. Luckily, we&#39;re alone in the cafe &mdash; save for a barrista disappeared in the back. It&#39;s a little after eight o&#39;clock at night, and the place is closing soon. I&#39;ve ordered a jasmine tea and two pieces of dull broccoli and cheddar quiche.</p>\n<p><em>I have not begun that account yet, though Hilda put the matter to me urgently</em>&mdash;</p>\n<p>&quot;Excuse me!&quot; Lucas turns to the barrista wiping a spotless steel counter. &quot;Excuse me! What&#39;s there to do around here? We&#39;ve been traveling and we&#39;re bored.&quot;</p>\n<p><em>Ugh! Who&#39;s we? Leave the poor man alone!</em></p>\n<p>The barrista looks up, an amused smile on his face. &quot;Oh yeah? Where&#39;re you folks from?&quot; He has a name tag, but I can&#39;t read it from here. He&#39;s a soft spoken brunette with a short, simple haircut. He&#39;s wearing a black apron over a slight build. <em>This is the first time I&#39;m looking at him directly.</em></p>\n<p>&quot;Jersey.&quot; Lucas doesn&#39;t get up. He calls to the man from across the room. &quot;Any <a class=\"squiffy-link link-section\" data-section=\"good bars\" role=\"link\" tabindex=\"0\">good bars</a> around here?&quot;</p>\n<p>&quot;Oh cool. I&#39;m from Philly originally,&quot; the barrista calls back.</p>\n<p>&quot;I&#39;m actually from <em>New York</em> originally,&quot; I cut in, book still in hand.</p>\n<p>&quot;Nice,&quot; the barrista says, nodding.</p>",
		'passages': {
		},
	},
	'good bars': {
		'text': "<p>&quot;Not a fancy bar though,&quot; Lucas says, his voice ricocheting off of every glass and metallic surface. &quot;Just a place where we can chill for a couple hours while we wait for the next bus.&quot;</p>\n<p>The barrista tucks his hands in his apron. &quot;I don&#39;t really chill around here much myself.&quot; Then he flips his index finger up as if to point to the bulb that lit up in his head. &quot;But there is a place I&#39;ve been to a couple times with work friends? <a class=\"squiffy-link link-section\" data-section=\"Pubblie\" role=\"link\" tabindex=\"0\">Pubblie</a>? I think that&#39;s what it&#39;s called...&quot; He pulls out his phone to check. &quot;They&#39;re an old school bar with a basement club.&quot;</p>\n<p>Lucas is already searching on his phone. &quot;Yeah, I see it. Thank you!&quot;</p>\n<p>&quot;You&#39;re welcome,&quot; the barrista says pleasantly. &quot;Enjoy.&quot;</p>\n<p>&quot;Hey, you wanna come with us?&quot;</p>\n<p>&quot;Ah nah man, I can&#39;t. I gotta get home. But have one for me.&quot;</p>\n<p>Lucas chuckles, &quot;Alright, we will.&quot;</p>\n<p><em>I never said I was going with you.</em> &quot;Um, I just kind of wanted to read my book...&quot;</p>\n<p>&quot;You&#39;re not reading that shit anyway, and this place is about to close.&quot;</p>\n<p>With a sigh, I close my book. &quot;You buying?&quot;</p>",
		'passages': {
		},
	},
	'Pubblie': {
		'text': "<p>Inwardly, I count the blocks as we trot north to Pubblie. I don&#39;t bother checking the street signs. <em>Fourth block,\ncorner deli.</em> My backpack and purse <a class=\"squiffy-link link-passage\" data-passage=\"jingle merrily\" role=\"link\" tabindex=\"0\">jingle merrily</a> with each step. The blocks stretch far and wide, bisected by a long, tree-lined island with black street lamps.</p>\n<p>Lucas walks to my left, street-side. He&#39;s <a class=\"squiffy-link link-passage\" data-passage=\"smoking a menthol\" role=\"link\" tabindex=\"0\">smoking a menthol</a> and tossing the smoke over his left shoulder. &quot;Let me take one of your bags,&quot; he says, extending his right arm.</p>\n<p>&quot;Don&#39;t mind if you do,&quot; I sing, stopping in the middle of the sidewalk to take off my backpack. <em>Fifth block, Italian restraurant.</em> There are faint, anonymous figures drifting some blocks ahead and behind us. Traffic is relatively light. No honking, no car alarms.</p>\n<p>&quot;I guess this is the Williamsburg section of Denver,&quot; I look up at Lucas, and drop my backpack onto his open palm. &quot;Thank you!&quot;</p>\n<p>&quot;You&#39;re welcome.&quot; He lifts his chin up toward <a class=\"squiffy-link link-section\" data-section=\"the next block\" role=\"link\" tabindex=\"0\">the next block</a>. &quot;That&#39;s us.&quot;</p>",
		'passages': {
			'jingle merrily': {
				'text': "<p>I&#39;m grateful for the walk, I&#39;ve just realized.</p>",
			},
			'smoking a menthol': {
				'text': "<p>For a guy who smokes so much, he doesn&#39;t smell bad at all. Maybe because he&#39;s still young &mdash; the stench hasn&#39;t had\na chance to settle in yet.</p>\n<p>That must be why he had all that candy, to keep his mouth busy.</p>",
			},
		},
	},
	'the next block': {
		'text': "<p>Pubblie has an all-black, industrial storefront with tinted windows. A white guy with locs and a Dominican guy with a curly undercut stand by the door mumbling to each other and passing a blunt back and forth. I <a class=\"squiffy-link link-passage\" data-passage=\"catch a whiff\" role=\"link\" tabindex=\"0\">catch a whiff</a>.</p>\n<p>The Dominican guy nods in our direction. &quot;How&#39;re you folks doing?&quot; His voice is deeper than I was expecting. He&#39;s about my age &mdash; maybe a couple years older. I feel relief.</p>\n<p>&quot;Good, how are you?&quot; I answer, accidently slipping into resum&eacute; voice.</p>\n<p>&quot;Any black people here?&quot; Lucas chimes in, with a smile and a laugh.</p>\n<p>&quot;Oh God!&quot; I say aloud, without meaning to.</p>\n<p>The white guy covers his mouth, laughing silently. The Dominican guy scoffs, releasing smoke from his nose. &quot;It&#39;s a mixed crowd, no worries.&quot;</p>\n<p>&quot;Appreciate it.&quot; Lucas grabs the vertical door handle, letting me in. He takes a long, final drag from his cigarette and flicks it into the street. As the door <a class=\"squiffy-link link-section\" data-section=\"slowly closes\" role=\"link\" tabindex=\"0\">slowly closes</a>, he leans over, &quot;That motherfucker is sexy, ain&#39;t he?&quot;</p>\n<p>&quot;Jeez-us!&quot; I turn my head to make sure the Dominican guy hasn&#39;t heard us. Just before it closes, I catch eyes with him, a shy smirk across his face.</p>",
		'passages': {
			'catch a whiff': {
				'text': "<p>Hm.</p>",
			},
		},
	},
	'slowly closes': {
		'text': "<p>I shout-whisper, &quot;Lucas, goddammit! Now he&#39;s going to think we want to fuck him!&quot; Lucas is doubled over in\nopen-mouthed, unrestrained laughter. &quot;You don&#39;t?&quot; he says wiping tears from his eyes.</p>\n<p>&quot;I don&#39;t know that man!&quot; I nod <a class=\"squiffy-link link-section\" data-section=\"self-consciously\" role=\"link\" tabindex=\"0\">self-consciously</a> to the bouncer at the end of the long, dim corridor.</p>\n<p><em>Not gonna ID us, huh?</em></p>\n<p>&quot;Is that what I asked you?&quot;</p>\n<p>&quot;For fuck&#39;s sake...&quot;</p>\n<p>Lucas points with his free hand to the back of the bar, &quot;Let&#39;s sit in a booth in the back.&quot;</p>\n<p>&quot;After you.&quot; I haven&#39;t adjusted to the dark yet, but I can feel eyes on us. <em>They must have heard us talking on the way\nin.</em></p>",
		'passages': {
		},
	},
	'self-consciously': {
		'text': "<p>I let out a sigh of relief when we sit down.</p>\n<p><em>Ah, sweet seclusion.</em></p>\n<p>&quot;This is a nice place!&quot; Lucas is smiling ear to ear. &quot;I wasn&#39;t expecting Denver to be so classy!&quot;</p>\n<p>&quot;Some people get priced out, some people get priced in.&quot; I heave a different sigh, flipping through the drink menu.</p>\n<p>Lucas nods. &quot;That is true.&quot; He remembers something. &quot;By the way, what&#39;s in your bookbag? Boulders?&quot;</p>\n<p>&quot;Oh!&quot; I chuckle a bit. &quot;Yeah, I should have warned you.&quot; With both hands, I point to each bag. &quot;Everything I own is in\nthose bags.&quot;</p>\n<p>&quot;You too?&quot; Lucas sits back slowly, folding his arms.</p>\n<p>&quot;But&mdash;&quot; I sputter. &quot;You don&#39;t even have bags!&quot;</p>\n<p>&quot;Nope. I&#39;ve just got myself and a pack of cigarettes.&quot;</p>\n<p>&quot;Why? Looking to leave it all behind?&quot;</p>\n<p>&quot;Kinda.&quot; He looks up at our server, a bald white woman with a medusa and transparent glasses. &quot;Rum and coke, please?&quot;</p>\n<p>&quot;John Collins, with Jim Beam please.&quot; Our server nods solemnly, walking off.</p>\n<p>&quot;Damn, girl! You&#39;re getting serious!&quot;</p>\n<p>I wave it away. &quot;Does you uncle have stuff for you at his place?&quot;</p>\n<p>Lucas becomes subdued again. &quot;Not as far as I know.&quot;</p>\n<p>&quot;So then...&quot;</p>\n<p>Lucas stares at the table. &quot;I just got out of immigration jail,&quot; he said flatly.</p>\n<p>&quot;<em>Immigration jail...?</em>&quot; Slack jawed, I search Lucas&#39;s eyes.</p>\n<p>He pops out a little laugh. &quot;Yeah, I was in immigration jail for six months. I got out like a month ago.&quot;</p>\n<p>&quot;<em>Six months?!</em>&quot;</p>\n<p>Lucas lets out another low laugh. &quot;Yeah.&quot;</p>\n<p>&quot;Jesus Christ...&quot;</p>\n<p>&quot;Yeah.&quot; He&#39;s looking through me.</p>\n<p>&quot;I&mdash;&quot; A stab in the chest. <em>Don&#39;t fucking apologize.</em></p>\n<p>&quot;I cried everyday in that place. They kept trying to get me to sign papers. I didn&#39;t sign anything. I told them, &#39;I\nspeak English, y&#39;know.&#39;&quot; His eyes are stainless steel.</p>\n<p>&quot;Oh my God...&quot; I cover my mouth, eyes burning for lack of blinking.</p>\n<p>&quot;I&#39;m OK now, though. I prayed to God, and He delivered me out of that place.</p>\n<p>&quot;Wow...&quot;</p>\n<p>Lucas cackles into the stale bar air. &quot;Look at you!&quot; He points to the table. Your drink&#39;s gonna melt!&quot;</p>\n<p>Magically, my John Collins sits on top of a black coaster, ice on top. Dazed, I take a swig. &quot;<a class=\"squiffy-link link-section\" data-section=\"Chapter 6\" role=\"link\" tabindex=\"0\">Fuck</a>,&quot; the\nword gushes slowly out of my mouth. I must have been holding my breath.</p>\n<p>&quot;I&#39;m gonna copy you next round.&quot;</p>",
		'passages': {
		},
	},
	'Chapter 6': {
		'clear': true,
		'text': "<p>Dear Angie,</p>\n<p>Have you ever wondered what you&#39;re <a class=\"squiffy-link link-section\" data-section=\"made of\" role=\"link\" tabindex=\"0\">made of</a>? I mean, what is all this meat, blood, bile, and bone even for?</p>",
		'passages': {
		},
	},
	'made of': {
		'text': "<p>&quot;I know why you get these!&quot; Lucas downs the last drop. &quot;They&#39;re good as hell!&quot;</p>\n<p>The lights have been lowered even more, and there&#39;s a rhythmic rumble under our feet. A heavy wooden door somewhere\nbehind me releases old metal heads and and scene kids into the bar. The flashing light illuminate Lucas&#39;s face. <em>Good\nlooking kid.</em> </p>\n<p>A little wobbly, I slap the back of my hand on the table. &quot;Dude, <a class=\"squiffy-link link-section\" data-section=\"where are you even from\" role=\"link\" tabindex=\"0\">where are you even from</a>?&quot;</p>\n<p>&quot;Jersey City. I been told you.&quot;</p>\n<p>&quot;No. Before that. You Guyanese?&quot;</p>\n<p>Lucas is instantly repulsed. &quot;Ew, what the fuck? I ain&#39;t Guyanese!&quot;</p>\n<p>I spit with laughter, covering my mouth. &quot;My bad, my bad! But what&#39;s wrong with being Guyanese?&quot;</p>\n<p>&quot;I&#39;m from Trinidad!&quot;</p>\n<p>&quot;I&#39;m sorry!&quot; I&#39;m cradling my stomach. &quot;It&#39;s just... I didn&#39;t know there were Indians in Trinidad.&quot;</p>",
		'passages': {
		},
	},
	'where are you even from': {
		'text': "<p>&quot;Do you smoke?&quot; Lucas is distracted.</p>\n<p>&quot;Nah,&quot; I start unlocking my phone. &quot;Feel free to go for a smoke, though. I&#39;ll be here.&quot;</p>\n<p>&quot;Do you smoke <em>weed</em> I mean?&quot;</p>\n<p>&quot;Sure, but I don&#39;t have that kind of dough right now.&quot;</p>\n<p>Lucas sucks his teeth, surveying the people who&#39;ve just emerged from the basement. &quot;C&#39;mon, this is Colorado, how bad\ncould the prices be?&quot;</p>\n<p><em>Hm.</em> &quot;I guess...&quot;</p>\n<p>&quot;What about the Puerto Rican guy? Wanna ask him?&quot;</p>\n<p>&quot;You mean the <a class=\"squiffy-link link-section\" data-section=\"Dominican guy\" role=\"link\" tabindex=\"0\">Dominican guy</a> you wanna fuck?&quot;</p>\n<p>&quot;The Puerto Rican guy <em>you</em> wanna fuck, you mean?&quot;</p>\n<p>We turn heads as we choke on stale bar air. &quot;Alright, let&#39;s get the check and see if he&#39;s still around. It&#39;s been like\ntwo hours since we&#39;ve seen him...&quot; I hook an arm through the straps of my giant tote bag.</p>\n<p>&quot;Yeah, it&#39;s definitely been like forty minutes.&quot;</p>",
		'passages': {
		},
	},
	'Dominican guy': {
		'text': "<p>In front of Pubblie, Lucas is smoking a cigarette and I&#39;m on my phone. The Dominican guy and the white guy are [[still\nhere]].</p>\n<p>_OK. Where was I?&quot;</p>\n<p>_There&#39;s another sign on the front door of the dome. It reads: &quot;WELCOME&quot; and then &quot;but please knock&quot; in smaller\nletters. A thin, brown arm emerges from the bottom of the shot, some fingers are adorned with a few gold rings, closed\nin a soft fist. She knocks three times and waits in silence.</p>\n<p>Another person is standing to her right. The door opens outwardly with a metallic whine, an old white man emerges. He\nhas long gray hair, square metallic glasses, and a kind grin. He&#39;s wearing an untucked flannel shirt and light blue\njeans._</p>\n<p>Of course.</p>",
		'passages': {
		},
	},
	'still here': {
		'text': "<p><em>Cut to him giving a tour of his house. &quot;There were already people here when I got here, and they&#39;re the ones who built\nthe original dome.&quot; Some text pops in at the lower left corner of the screen: &quot;DALE.&quot;</em> Hello, Dale. _So when they\npassed away, Helen and George, I basically cut away one hole after the other every time we needed more space.</p>\n<p>&quot;I&#39;ve expanded this place three or four times over the years, which doesn&#39;t seem like much, but mud works slowly, like\nit&#39;s got a mind of its own.&quot; Dale laughs like the onomatopeia for laughter, &quot;Haha!&quot;</p>\n<p>Now this is where it gets interesting. <a class=\"squiffy-link link-section\" data-section=\"The voice\" role=\"link\" tabindex=\"0\">The voice</a> behind the camera speaks, &quot;I can&#39;t begin to imagine... This is a\ntruly amazing structure!&quot;_</p>",
		'passages': {
		},
	},
	'The voice': {
		'text': "<p>&quot;Lilianne! Lilianne!&quot;</p>\n<p>I lower my headphones.</p>\n<p>Lucas has his hand cupped around his mouth. &quot;Danilo can hook us up!&quot;</p>\n<p>I feel the humidity in the air for the first time. &quot;Lucas.&quot; I project at my normal volume, but he still hears me, or he\nreads my lips.</p>\n<p>Lucas frounds, concerned, and jogs over. &quot;What&#39;s the matter?&quot;</p>\n<p>&quot;We don&#39;t know these guys, maybe we shouldn&#39;t do this.&quot;</p>\n<p>&quot;They&#39;re just random stoners! They&#39;re not gonna kill us! This is Colorado!&quot;</p>\n<p>A chuckle escapes me. &quot;Colorado has plenty of crime. This is Denver!&quot;</p>\n<p>&quot;No it don&#39;t.&quot;</p>\n<p>&quot;Yes, it does!&quot;</p>\n<p>&quot;No it don&#39;t.&quot;</p>\n<p>&quot;Oh my fucking God, Lucas, those are strange men!&quot; I&#39;m suddenly pointing, but not on purpose.</p>\n<p>Danilo and his silent friend, previously pretending not to hear us, <a class=\"squiffy-link link-section\" data-section=\"punch the air\" role=\"link\" tabindex=\"0\">punch the air</a> with guffaws.</p>\n<p>&quot;We&#39;re not strange men!&quot; Danilo calls back to us. &quot;I&#39;m Danilo! This is Carl!&quot;</p>\n<p>&quot;Are you fifty years old? Who&#39;s named Carl?&quot; For some reason, Lucas is incredulous.</p>\n<p>&quot;I am,&quot; says Carl in a soft baritone that hovers through the fog.</p>\n<p>Lucas screams, &quot;What the fuck?!&quot;</p>",
		'passages': {
		},
	},
	'punch the air': {
		'text': "<p>&quot;Alright, alright!&quot; Let&#39;s get this shit over with, fucking hell... I stuff my phone in my small purse and grab my\nwallet. &quot;I can spare $48.&quot;</p>\n<p>&quot;I&#39;ve got a 20.&quot;</p>\n<p>&quot;Cheap ass. What were you gonna buy with that?&quot;</p>\n<p>&quot;Enough for the next couple hours. What, you distributing?&quot;</p>\n<p>&quot;I&#39;ve got standards.&quot;</p>\n<p>A tiny corner of Lucas&#39;s mouth turns up. &quot;I bet you don&#39;t.&quot; He looks to Danilo and Carl. &quot;We&#39;ve got money!&quot;</p>\n<p>&quot;Do you have an <a class=\"squiffy-link link-section\" data-section=\"Chapter 7\" role=\"link\" tabindex=\"0\">inside voice</a> by any chance?&quot;</p>\n<p>&quot;We&#39;re literally outside.&quot;</p>\n<p>&quot;And it&#39;s literally midnight.&quot;</p>",
		'passages': {
		},
	},
	'Chapter 7': {
		'clear': true,
		'text': "<p>Dear Angie,</p>\n<p>I know that I have no choice but to be myself. That doesn&#39;t mean I should settle for being a burden. You&#39;ve helped me\nheal these last few years, and I&#39;m so grateful to you. Now I need to gather the <a class=\"squiffy-link link-section\" data-section=\"scraps of myself\" role=\"link\" tabindex=\"0\">scraps of myself</a> and mold them into\na person.</p>\n<p>I hope you understand. I love you.</p>",
		'passages': {
		},
	},
	'scraps of myself': {
		'text': "<p>Danilo and Carl&#39;s apartment opens into the kitchen. It&#39;s a facsimile of a Manhattan railroad apartment in those old\ntenement buildings. It&#39;s cleaner than I was expecting.</p>\n<p>&quot;You&#39;re surprised,&quot; Danilo mumbles though a smirk, flinging it onto a tufted blue sofa.</p>\n<p>&quot;Sorry. By the way, where should I&mdash;&quot; I gesture with my bags.</p>\n<p>&quot;On top of the shoe rack by the door is fine.&quot;</p>\n<p>&quot;Nice.&quot;</p>\n<p>Carl and Lucas eventually stroll in after us. I hear Lucas before I see him.</p>\n<p>&quot;Damn, how much you paying for this place?&quot; Lucas drops my backpack on the floor with a liquidy <em>thunk</em>.</p>\n<p>I nearly jump up, &quot;Careful! Please! Thank you...&quot;</p>\n<p>Carl nods at Danilo, Danilo nods back. &quot;We may $1500 a month,&quot; Danilo says.</p>\n<p>&quot;Wow.&quot;</p>\n<p>&quot;Shit.&quot;</p>\n<p>&quot;Are you stunned because it&#39;s too much, or because it&#39;s too little?&quot;</p>\n<p>&quot;No, that&#39;s a definite steal by us.&quot;</p>\n<p>&quot;You even have hardwood floors!&quot; Lucas adds.</p>\n<p>Danilo chuckles like the first signs of an earthquake. &quot;How do you two know each other?&quot;</p>\n<p>&quot;We don&#39;t,&quot; I say into my chest while I search my wallet.</p>\n<p>Lucas grins, &quot;We&#39;ve been best friends since yesterday!&quot;</p>",
		'passages': {
		},
	},
	'how much': {
		'text': "<p>From the corner of my eye I see Danilo raise his hand to stop me. &quot;We&#39;re good. You don&#39;t have to pay me anything.&quot;</p>\n<p>I freeze, taking a glance at Lucas, who looks back at me, confused. &quot;Why&#39;s that?&quot;</p>\n<p>&quot;I don&#39;t need the money, that&#39;s all. This isn&#39;t <a class=\"squiffy-link link-section\" data-section=\"business\" role=\"link\" tabindex=\"0\">business</a>.</p>\n<p>&quot;Then it&#39;s pleasure...?&quot;</p>\n<p>&quot;The pleasure of your company?&quot; Danilo&#39;s got an idea. &quot;Everyone, come check out my room.&quot;</p>\n<p>I turn to look at Lucas. &quot;Have you gotten us murdered?&quot; My hand&#39;s still stuck in my wallet, index and middle fingers\nseparating bills from cards. <em>I gotta clean this thing out.</em></p>\n<p>Squaring his shoulders, Lucas beats his chest. &quot;Just try to murder me! I got shit to do!&quot;</p>\n<p>&quot;Dude, the neighbors...&quot; Danilo holds his head, a glint of amusement piercing through the exasperation.</p>\n<p>Lucas giggles, whispering, &quot;My bad, my bad...&quot;</p>\n<p>Carl&#39;s sitting on the couch, scrolling through his phone, laughing silently.</p>",
		'passages': {
		},
	},
	'business': {
		'text': "<p>Danilo&#39;s bedroom consists of a neatly made bed on the floor surrounded by a fortress of books, and a glass top desk\nwith hardwood legs. &quot;Interesting,&quot; I say plainly, but I&#39;m worried it sounded sarcastic.</p>\n<p>&quot;You ain&#39;t seen nothing yet. Check this out.&quot; With the <a class=\"squiffy-link link-section\" data-section=\"exaggerated\" role=\"link\" tabindex=\"0\">exaggerated</a> austery of a hotel concierge, Danilo steps\naside, swinging open a closet door. The room is showered in dark magenta light.</p>\n<p>&quot;Ooh...&quot; Lucas and I are in unison. Arranged in three rows of two sit cannabis plants in black fabric pots surrounded\nby walls covered in what looks like aluminum foil. Two boxy light fixtures hang from bars affixed across the walk-in\ncloset.</p>\n<p>Excitedly, Danilo explains the strain, its supposed psychoactive effects, and what equipment and nutrients he uses. For\nthe first time all night, I find myself relaxing.</p>",
		'passages': {
		},
	},
	'exaggerated': {
		'text': "<p>Behind Danilo and Carl&#39;s bathroom mirror, I find high potency vitamins instead of medications. <em>That figures,</em> I\nchuckly to myself. I pop a vitamin D, cut the lights, and <a class=\"squiffy-link link-section\" data-section=\"settle\" role=\"link\" tabindex=\"0\">settle</a> on the toilet. Through the door, I can hear Lucas\ntalking excited to Carl. Without headphones, I lower my phone&#39;s volume.</p>",
		'passages': {
		},
	},
	'settle': {
		'text': "<p>_Franklin sits on a floor pillow behind what looks like a solid wood coffee table. &quot;I love it here,&quot; she says with a\ntiny smile. She&#39;s petting a peppered cat sleeping angelically in her lap. &quot;Before this, I lived on the streets, and\nbefore that I lived with <a class=\"squiffy-link link-section\" data-section=\"my mother\" role=\"link\" tabindex=\"0\">my mother</a>,&quot; she pauses, &quot;so things are really looking up!&quot;</p>\n<p>A voice from behind the camera, &quot;How is this different from being homeless?&quot;</p>\n<p>Franklin is genuinely confused. &quot;Do I look homeless?&quot; She gestures around herself at her puzzle and cat paintings. &quot;I&#39;m\nliterally home. I&#39;ve been here three years already and I&#39;m never leaving.</p>\n<p>&quot;How do you eat and provide for yourself.</p>\n<p>&quot;We grow what we can. The rest comes from donations and dumpster diving.&quot;</p>\n<p>&quot;Isn&#39;t that a bit too convenient? Shouldn&#39;t you have to work a job like anyone else?&quot;</p>\n<p>Franklin becomes agitated, but she&#39;s beaming now &mdash; with pride. &quot;You come live here for_ six <em>months and you tell\nme how</em> convenient <em>it is. The soil around here is no joke.&quot;</em></p>",
		'passages': {
		},
	},
	'my mother': {
		'text': "<p>&quot;There she is!&quot; J in hand, Lucas smirks. &quot;What took you so long?&quot; He and Carl are sitting on the couch\n<a class=\"squiffy-link link-section\" data-section=\"Chapter 8\" role=\"link\" tabindex=\"0\">exchanging</a> memes. I realize there&#39;s no television.</p>\n<p>A sign escapes me. &quot;I just neede solitude...&quot;</p>\n<p>&quot;Lilianne,&quot; Danilo&#39;s magenta head is peaking out of his room. &quot;You alright?&quot;</p>\n<p>&quot;Mind if I join you?&quot;</p>\n<p>&quot;Not at all. Please do.&quot; Danilo retracts his head, letting the light envelope the tiny foyer. Inside, he&#39;s standing at\nhis desk, two freshly rolled blunts sitting on top of a bamboo cutting board. &quot;M&#39;lady,&quot; he gestures.</p>\n<p>&quot;You are so corny,&quot; I laugh, relieved. &quot;Thank you.&quot;</p>\n<p>&quot;I&#39;ll take that as a compliment.&quot;</p>\n<p>&quot;You should.&quot;</p>\n<p>He sticks one blunt in his mouth and offers me the other.</p>\n<p>&quot;All to myself?&quot;</p>\n<p>&quot;We can share if you&#39;d like, but I figured you&#39;d want your own. Danilo groans slightly as he eases onto a floor pillow\nby his bed.</p>\n<p><em>How about that. Hot</em> and <em>perceptive.</em> I light my blunt with a stainless steel lighter, and notice he&#39;s blushing.\n&quot;Christ, I said that out loud?&quot;</p>\n<p>&quot;You did,&quot; he smirks again. &quot;Would you like to sit down?&quot;</p>",
		'passages': {
		},
	},
	'Chapter 8': {
		'clear': true,
		'text': "<p>Dear Angie,</p>\n<p>I admit, sometimes the effort it takes to <a class=\"squiffy-link link-section\" data-section=\"stay alive\" role=\"link\" tabindex=\"0\">stay alive</a> is worth it.</p>",
		'passages': {
		},
	},
	'stay alive': {
		'text': "<p>&quot;What&#39;s the name of this strain again?&quot;</p>\n<p>&quot;Purple Kablooey.&quot;</p>\n<p>&quot;Purple... Kablooey?&quot;</p>\n<p>&quot;Yes.&quot;</p>\n<p>&quot;Anyway, I&#39;m impressed. This is one of the smoothest smokes I&#39;ve had in my life.</p>\n<p>&quot;That makes me happy.&quot; Danilo&#39;s eyes are twinkling shyly.</p>\n<p>I have to look away. &quot;So what&#39;s your story?&quot; I&#39;m sitting on a floor pillow across from him, hugging my knees. The room\nundulates around me. &quot;Can I sit next to you?&quot;</p>\n<p>&quot;Sure,&quot; he blurts. He takes a long hit, releasing it slowly. &quot;I&#39;m from <a class=\"squiffy-link link-section\" data-section=\"Philly\" role=\"link\" tabindex=\"0\">Philly</a> originally, by way of Colombia.</p>",
		'passages': {
		},
	},
	'Philly': {
		'text': "<p><em>Huh.</em> &quot;What a coincidence!&quot;</p>\n<p>&quot;What do you mean?&quot;</p>\n<p>&quot;Lucas and I just met a dude from Philly earlier today. Barista working at a bougie cafe?&quot;</p>\n<p>&quot;<a class=\"squiffy-link link-section\" data-section=\"Jacob\" role=\"link\" tabindex=\"0\">Jacob</a>. Yeah. I know him.&quot; Danilo ashes into a Santa Claus coffee mug. <em>That thing&#39;s gotta be fourteen or fifteen\nounces.</em> His eyes are distance and his body is stiff.</p>\n<p>&quot;What&#39;d he do?&quot;</p>\n<p>&quot;Nothing bad, sort of.&quot;</p>\n<p>&quot;Now I&#39;m intrigued!&quot;</p>\n<p>&quot;It&#39;s nothing exciting, it&#39;s just &mdash; I hate that guy.&quot; There&#39;s subdued frustration in his eyesand brow. Up close,\nDanilo&#39;s nose is bigger than I thought. He appears almost stately &mdash; statuesque. &quot;I have no evidence for this, but\nI think he&#39;s following me.&quot;</p>\n<p>&quot;Following you? From Philly?&quot;</p>\n<p>He senses my skepticism and doesn&#39;t look at me. From New Jersey, too. I wen to school there.</p>",
		'passages': {
		},
	},
	'Jacob': {
		'text': "<p>&quot;Get out of town...&quot; I say flatly, but I&#39;m taken aback. &quot;Did you go to Rutgers?&quot;</p>\n<p>&quot;Yeah!&quot; You too?&quot; Danilo&#39;s perked up a bit.</p>\n<p>&quot;Yeah... What year were you?&quot;</p>\n<p>&quot;I graduates in 2010.&quot;</p>\n<p>&quot;...I left in 2007.&quot;</p>\n<p>&quot;That&#39;s unfortunate.&quot;</p>\n<p>&quot;It is.&quot; I place my blunt on a green and gold ceramic plate, stretching backward. &quot;Have you <a class=\"squiffy-link link-section\" data-section=\"confronted\" role=\"link\" tabindex=\"0\">confronted</a> Jacob the\nStalker yet?&quot;</p>\n<p>Danilo turns to face me. &quot;I did. Once. Like a year after I found out he&#39;s moved here. I knew where he worked, and I\njust showed up one day as if by chance. I greeted him like an old friend, but I never really knew him. I drank my\ncoffee, I pretended to read a book until the place started clearing out. When I went to throw out my trash, I called\nout to him. I asked him how he&#39;d been and what brough him to Denver.&quot;</p>\n<p>I watch intently as Danilo takes an eternal sip from a small glass.</p>",
		'passages': {
		},
	},
	'confronted': {
		'text': "<p>&quot;And all he said was, &#39;Just looking for a <a class=\"squiffy-link link-section\" data-section=\"change of pace\" role=\"link\" tabindex=\"0\">change of pace</a>.&#39; In that <em>customer service</em> voice, too.&quot;</p>\n<p>&quot;That&#39;s it?&quot;</p>\n<p>&quot;Well, then I said, &#39;Wow, what a coincidence&#39; &mdash; and I was smiling as I said it &mdash; I was being friendly.\n&#39;First high school, then college, then Denver! What are the odds?&#39;&quot;</p>\n<p>&quot;Oh!&#39; My heart is racing. I can hardly stand it.</p>\n<p>&quot;And... he didn&#39;t acknowledge what I said. He shrugged, continued wiping the counter, said some noncommital bullshit\nabout catchiing up, and took the next customer.</p>\n<p>&quot;Whoa...&quot;</p>\n<p>&quot;Exactly.&quot;</p>\n<p>&quot;What do you make of that?&quot;</p>\n<p>&quot;I&#39;d do anything to forget it ever happened.&quot;</p>\n<p>&quot;You mean... You got <em>vibes</em> from him? Bad vibes?&quot;</p>\n<p>&quot;I don&#39;t know.&quot;</p>\n<p>&quot;Damn...&quot;</p>\n<p>&quot;Yeah.&quot; He lifts his glass. Would you like one?</p>",
		'passages': {
		},
	},
	'change of pace': {
		'text': "<p>&quot;Absolutely!&quot; I jump to my feet. &quot;But would it be insane if I asked to take a shower first?&quot;</p>\n<p>&quot;Um, not at all. I&#39;m assuming you have <a class=\"squiffy-link link-section\" data-section=\"everything you need\" role=\"link\" tabindex=\"0\">everything you need</a>?&quot;</p>\n<p>&quot;Yeah, but could I used one of your towerls?&quot;</p>\n<p>&quot;No problem.&quot; From the last drawer of a large black dresser, he hands me a crisp white towel.</p>\n<p>&quot;Beautiful,&quot; I say.</p>\n<p>&quot;Thank you.&quot;</p>",
		'passages': {
		},
	},
	'everything you need': {
		'text': "<p>After my shower, I scrub down the tub with some cleaner and a sponge I find under the sink. <em>I&#39;ll leave it like I left\nit.</em> I untie my hair in the mirror, basking in the release of tension. I undo each twist until I&#39;ve got a cloud of loose\ncurls. I&#39;ve cleansed, I&#39;ve exfoliated, I&#39;ve toned, and I&#39;ve moisturized.</p>\n<p>&quot;Don&#39;t overthik it. This joy is real.&quot;</p>\n<p>Lucas and Carl are nowhere to be <a class=\"squiffy-link link-section\" data-section=\"found\" role=\"link\" tabindex=\"0\">found</a>.</p>",
		'passages': {
		},
	},
	'found': {
		'text': "<p>In the morning, I find Lucas in the kitchen, making coffee. &quot;Morning!&quot;</p>\n<p>&quot;Fuck!&quot; Lucas jumps nearly a foot in the air. &quot;What are you trying to do to me?!&quot;</p>\n<p>&quot;I couldn&#39;t resist. Good morning!&quot;</p>\n<p>&quot;Good morning...&quot;</p>\n<p>&quot;I take it you didn&#39;t sleep very well.&quot;</p>\n<p>&quot;At some point, I passed out on the couch. Carl was nice enough to put a blanket on me.&quot;</p>\n<p>&quot;How do you know _I_ didn&#39;t get you the blanket?&quot;</p>\n<p>&quot;I figured you were... busy.&quot;</p>\n<p>I snicker, muffling a laugh. &quot;I didn&#39;t fuck him. We&#39;ve been talking all night.&quot;</p>\n<p>Lucas looks down at the kitchen tile. &quot;You&#39;re in a good mood.&quot;</p>\n<p>&quot;Are you upset I didn&#39;t come back from Danilo&#39;s room?&quot;</p>\n<p>Lucas turns around to fiddle with the coffee machine. &quot;I thought maybe you went to get the next bus without me.&quot;</p>\n<p>&quot;I wouldn&#39;t do that, man.&quot;</p>\n<p>&quot;But last night you said you needed &#39;solitude.&#39;&quot; He rolls his eyes and waves his hands in the air mockingly.</p>\n<p>I know he&#39;s feeling rejected, but my heart is warmed. &quot;Not from you. Sometimes I just need to be alone. That&#39;s all.&quot;</p>\n<p>&quot;OK.&quot; Lucas perks up, but he looks tired. &quot;Coffee?&quot;</p>\n<p>&quot;Can you shoot it into my veins?&quot;</p>\n<p>Lucas laughs, &quot;Right?&quot; He grabs a couple mugs from a cabinet, navigating the kitchen like he lives here. &quot;Carl said\nto make myself at home.&quot;</p>\n<p>&quot;Are you and Carl best friends now?&quot;</p>\n<p>&quot;...maybe.&quot;</p>\n<p>&quot;Lucas, I&#39;m not going any farther West than this. Not right now. Not like this.&quot;</p>\n<p>Lucas nearly stops moving. &quot;Moving in?&quot;</p>\n<p>&quot;Absolutely not! Jeez.&quot; I sit on a kitchen stool, waiting for my cup. &quot;I&#39;m going back to New York.&quot;</p>\n<p>&quot;Why in the world would you want to go back there?&quot; He sets the cups down and climbs onto a stool. He has a startling\n<a class=\"squiffy-link link-section\" data-section=\"realization\" role=\"link\" tabindex=\"0\">realization</a>. &quot;Wait, did you tell me why you were going to Oregon?&quot;</p>\n<p>I can hear the birds singing outside the kitchen window. I can hear the footsteps of Danilo and Carl&#39;s neighbors heading\nto work. I can feel the cold granite countertop under my hand. For some reason, it takes me a while to answer. &quot;No, I\nnever told you. Sorry about that.&quot;</p>",
		'passages': {
		},
	},
	'realization': {
		'text': "<p>Dear Angie,</p>\n<p>I&#39;ll be back tomorrow.</p>",
		'passages': {
		},
	},
}
})();