#include <adv3.h>
#include <en_us.h>

#include <storyguide.h>


/* No automatic look-around at the beginning of the game. */
modify runGame(look) 
{
    look = nil;
    replaced(look);
}


modify Person
    /* All characters in this game are female. */
    isHer = true
    
    /* 
     *   When the characters are in darkness, counting, the player can't see 
     *   them, but she can hear them.
     */
    soundPresence = (gPlayerChar.isIn(dark))
    
    /* 
     *   The last action this character took. It starts off at counting, as 
     *   that is what Tiana and Yvonne will be doing when the player first 
     *   sees them. (This is overridden for Emma.)
     */
    lastAction = static CountAction.createActionInstance()
    
    /* Is this character hiding? */
    isHidden = nil
    
    /* Is this character cheating? Everyone except Emma may cheat sometimes. */
    isCheating = nil
    
    /* 
     *   Does this character knows where Emma is? She does if she's cheating, or
     *   if she's already searched the bush.
     */
    knowsWhereEmmaIs = (bushNearCubby.searchedBy(self) || isCheating)
    
    /* 
     *   Is this character playing Hide and Seek (instead of Sardines)? 
     *   Instead of keeping this information separately, just check whether 
     *   she's on the list of people playing Hide and Seek. 
     */
    isPlayingHnS()
    {
        if (peoplePlayingHnS.indexOf(self))
            return true;
        else
            return nil;
    }
    
    /* 
     *   If she is, what role is she taking? Possible options: roleCounting, 
     *   roleHiding, or nil (no role, either because she isn't playing or 
     *   because the roles haven't been decided yet).
     */
    roleInHnS = nil
    
    /* 
     *   A handy list of people playing Hide and Seek, to make it easier to 
     *   keep track of who is involved. To begin with, everyone is playing 
     *   Sardines, so no one is playing Hide and Seek.
     */
    peoplePlayingHnS = []
    
    /* 
     *   The character's pathway to Emma's hiding spot. This is computed and 
     *   stored when the character moves for the first time, but may change 
     *   if she begins to cheat later on.
     */
    pathToHidingSpot = nil
    
    /* 
     *   Compute (and, for convenience, return) the character's pathway. 
     *   This isn't important for Emma and the PC, while Tiana and Yvonne 
     *   both have special handling, so by default, do nothing - it's just a 
     *   hook.
     */
    calcPathToHidingSpot() {
        return pathToHidingSpot;
    }
    
    /* A set of standard pathways, just for convenience. */
    pathAcrossLawn = [lawnWest, lawnEast, cubbyArea]
    pathBehindShed = [lawnWest, lotsaBushes, behindShed, cubbyArea]
    pathFrontOfHouse = [lawnWest, underTree, frontGarden, vegeGarden, cubbyArea]
    
    /* Characters who are hiding can't be seen. */
    canBeSensed(sense, trans, ambient)
    {
        /* 
         *   If the sense is sight, and the character is hidden, she cannot 
         *   be sensed.  Otherwise, inherit the normal handling. 
         */
        if (sense == sight && isHidden)
            return nil;
        else
            return inherited(sense, trans, ambient);
    }
    
    /* Generic code for getting into a hiding spot. */
    enterHidingSpot(hs)
    {
        /* hide */
        if (self == emma)
        {
            /* 
             *   It's not always possible for Emma to get to her destination via
             *   scripted travel, so just move her without any simulation.
             */
            moveIntoForTravel(hs);
        }
        else if (self != emma && (isPlayingHnS || 
                                  (hs == bushNearCubby && 
                                   gPlayerChar.knowsWhereEmmaIs)))
        {
            if (gAction)
            {
                nestedActorAction(self, HideIn, hs);
            }
            else
            {
                newActorAction(self, HideIn, hs);
            }
        }
        else
        {
            silentlyGoTo(hs);
        }
        
        makePosture(hiding);
        
        /* 
         *   Make the character hidden - unless she is hiding in the bush 
         *   near the cubby and the PC has already searched it or is cheating.
         */
        if (!(hs == bushNearCubby && gPlayerChar.knowsWhereEmmaIs))
            isHidden = true;
        
        /* make character's last action waiting */
        lastAction = WaitAction.createActionInstance();
    }
    
    /* 
     *   Was the character's last action interesting? Traveling and waiting 
     *   are not interesting. Everything else is - mind you, the NPCs don't 
     *   do much, so that pretty much just leaves searching as an 
     *   interesting action. (Once the NPC is hidden, this won't be called, 
     *   so while hiding would hypothetically be an interesting action, it 
     *   isn't relevant.)
     */
    lastActionWasInteresting()
    {
        local pAct = lastAction;
        
        if (pAct.ofKind(TravelViaAction) || pAct.ofKind(WaitAction))
            return nil;
        return true;
    }
    
    /* 
     *   Report a character's location and her activity based on her last 
     *   action, unless it was waiting or a travel action. If it was, just 
     *   say she is standing (or whatever posture) in the location. The 
     *   output of this should be designed to fit in between something like 
     *   "She is " and "."
     */
    reportActivity(sayLoc)
    {
        local pAct = lastAction;
        
        /* 
         *   special case - if the character has just moved or is waiting 
         *   while in the same location as the player
         */
        if (!lastActionWasInteresting() && location == gPlayerChar.location)
        {
            "just <<posture.participle>> here";
            return;
        }
        
        /* report character's activity */
        if (lastActionWasInteresting())
        {
            "<<pAct.getVerbPhrase(nil, nil)>>";
        }
        else
        {
            "<<posture.participle>>";
        }
        
        /* report location, but only if necessary */
        if (sayLoc && location != gPlayerChar.location)
        {
            " <<location.actorInName>>";
        }
    }
    
    /* 
     *   The characters' postures are already described in their 
     *   descriptions so posture reporting is unnecessary.
     */
    postureReportable = nil
    
    /* Get a list of nearby unsearched HidingSpots. */
    getNearbyHidingSpots(unsearchedOnly)
    {
        local spots = new Vector(2);
        foreach (local item in location.contents)
        {
            if (item.ofKind(HidingSpot) && 
                (!unsearchedOnly || !item.searchedBy(self)))
            {
                spots.append(item);
            }
        }
        return spots.toList();
    }
    
    /* Check whether there are any unsearched HidingSpots nearby. */
    isNearUnsearchedHidingSpot()
    {
        return getNearbyHidingSpots(true).length();
    }
    
    /* 
     *   Modify scriptedTravelTo so that it only uses nestedActorAction when 
     *   there is already an existing action; otherwise, use newActorAction.
     */
    scriptedTravelTo(dest)
    {
        local conn;

        /* find a connector from the current location to the new location */
        conn = location.getConnectorTo(self, dest);

        /* if we found the connector, perform the travel */
        if (conn != nil)
        {
            if (gAction)
                nestedActorAction(self, TravelVia, conn);
            else
                newActorAction(self, TravelVia, conn);
        }
    }

    /* Don't use anyone's specialDesc while counting. */
    actorListWith()
    {
        local group;

        /* 
         *   If the counting scene is not happening and the character has a 
         *   specialDesc, don't use any grouping - I want to use the 
         *   specialDesc instead.
         *
         *   I also want to use the specialDescs on the final turn of the 
         *   counting scene when the PC is *not* cheating. At this point, 
         *   the scene is ready to end and the specialDescs should 
         *   reflect this, but the scene machinery hasn't had a chance to 
         *   end it yet.
         */
        if (!isIn(bushNearCubby) && (!counting.active_ ||  
             (!gPlayerChar.isCheating && counting.countedTo == 50)) && 
            overrides(self, Actor, &specialDesc))
        {
            return [];
        }

        /* get the group for the posture */
        group = location.listWithActorIn(posture);

        /* 
         *   if we have a group, return a list containing the group;
         *   otherwise return an empty list 
         */
        return (group == nil ? [] : [group]);
    }
    
    
    /* 
     *   Find out which direction the character is travelling in. 
     *   scriptedTravelTo doesn't use directions, but I need to know to 
     *   write more detailed NPC travel messages. "loc" is the NPC's 
     *   starting point, and "dest" is her destination. 
     */
    getDirection(loc, dest)
    {
        foreach (local dir in Direction.allDirections)
        {
            /* get the connector that leads in this direction */
            local conn = loc.(dir.dirProp);
            
            /* does the connector lead to the destination? */
            if (conn && conn.getDestination(loc, self) == dest)
                return dir;
        }
        
        /* couldn't find a direction - give up */
        return nil;
    }
    
    
    /* 
     *   TADS 3's default implementation of following doesn't account for the
     *   possibility that the PC might see a character move through more than
     *   one room before trying to follow. This is my hacky override of that
     *   behaviour.
     *
     *   I suggest not copying this code if your NPCs are particularly active;
     *   it will result in a lot of FollowInfo objects being saved, which may in
     *   turn result in the wrong one being extracted and the PC sent off in the
     *   wrong direction. I think I can get away with it here because "It" is so
     *   short and the NPCs usually only travel through a few rooms before
     *   stopping, but I'm not sure if it might have unexpected consequences
     *   when playing hide and seek with Yvonne (she starts moving randomly
     *   while looking for a hiding spot).
     */
    trackFollowInfo(obj, conn, from)
    {
        local info;
        
        /* 
         *   If we're not tracking the given object, or we can't see the
         *   given object, ignore the notification.  In addition, we
         *   obviously have no need to track ourselves.
         */
        if (obj == self || !wantsFollowInfo(obj) || !canSee(obj))
            return;

        /* 
         *   The adv3 implementation only keeps one FollowInfo object for each
         *   NPC. I need to keep multiple copies for when the NPC moves several
         *   rooms before the player attempts to follow. For simplicity, I'll
         *   just keep all the FollowInfo objects.
         */
        
        /* create a new FollowInfo */
        info = new FollowInfo();
        info.obj = obj;

        /* 
         *   Add it to the list. Put it at the start of the list, so when
         *   getFollowInfo is searching for a FollowInfo, it will return the
         *   most recent match instead of the oldest one.
         */
        followables_ = [info] + followables_;

        /* remember information about the travel */
        info.connector = conn;
        info.sourceLocation = from;
    }

    /* 
     *   Now that I'm keeping all the FollowInfos (all... ALL the FollowInfos?),
     *   I need a more sophisticated way of extracting the appropriate one.
     *   Check the FollowInfo's source location against the PC's location.
     */
    getFollowInfo(obj)
    {
        local info = followables_.valWhich({x: x.obj == obj && 
            x.sourceLocation == gActor.getOutermostRoom});
        
        /* if it found a match, return it, otherwise return the default value */
        if (info)
            return info;
        else
            return inherited(obj);
    }
    
    
    /* 
     *   Get the NPC to travel somewhere without printing any reports, but
     *   otherwise fully simulating the action.
     */
    silentlyGoTo(loc)
    {
        /* 
         *   note any reports that have already been added to the transcript by
         *   making a copy of the existing vector
         */
        local oldReports = new Vector(gTranscript.reports_);
        
        /* 
         *   depending on the destination and game context, trigger the
         *   appropriate kind of action
         */
        if (loc.ofKind(Room))
        {
            scriptedTravelTo(loc);
        }
        else if (gAction)
        {
            nestedActorAction(self, HideIn, loc);
        }
        else
        {
            newActorAction(self, HideIn, loc);
        }
        
        /* get rid of any new reports added to the transcript */
        gTranscript.reports_ = oldReports;
    }
    
    
    /* Better NPC travel messages. */

    /*
     *   The NPC is arriving locally (staying within view throughout the 
     *   travel, and coming closer to the PC). This message is the same for 
     *   both Tiana and Yvonne, because I can't think of anything special to 
     *   put in it.
     */
    sayArrivingLocally(dest, conn)
    {
        /* get her last location */
        local last = lastTravelBack.getDestination(location, self);
        gMessageParams(dest);
        
        if (isCheating)
            "\^<<travelerName(true)>> leaves <<last.destName>> and
            joins you {in dest}. ";
        else
            "\^<<travelerName(true)>> gives up on <<last.destName>>
            and joins you {in dest}. ";
    }
    

    /* The only posture allowed at the top of the tree is sitting. */
    makePosture(newPosture)
    {
        if (newPosture != sitting && isIn(upTree))
        {
            "It wouldn't be safe to do that at the top of the tree. ";
            exit;
        }
        else
        {
            inherited(newPosture);
        }
    }
    
    
    /* 
     *   When attacking someone, move the PC into the same location as the
     *   victim and make sure the PC is standing up. This makes sure the
     *   location in the status line is correct when the game ends. Yes, it
     *   matters!
     */
    dobjFor(Attack)
    {
        action()
        {
            gPlayerChar.makePosture(standing);
            if (gPlayerChar.location != self.location)
            {
                gPlayerChar.moveIntoForTravel(self.getOutermostRoom());
            }
        }
    }
    
    dobjFor(Push) asDobjFor(Attack)
    
    
    /* If a character is hidden, then no one can talk to her. */
    canTalkTo(actor)
    {
        if (actor.isHidden)
            return nil;
        
        return inherited(actor);
    }
    
    
    /* Touching the other characters is weird and creepy. */
    dobjFor(Feel)
    {
        check()
        {
            failCheck('{The dobj/she} probably wouldn\'t like that. ');
        }
    }
    
    
    /* 
     *   If the PC starts trying to order people about during the counting 
     *   scene, she may get caught.
     */
    obeyCommand(issuingActor, action)
    {
        if (issuingActor != self && caughtOutByTalking.condition)
        {
            caughtOutByTalking.exec();
        }
        
        return inherited(issuingActor, action);
    }
;



/* Possible roles for the characters when playing Hide and Seek */
enum roleCounting, roleHiding;



/* After a character takes an action, record it. */
modify Action
    condition = (gActor != gIssuingActor)
    
    afterAction()
    {
        gActor.lastAction = gAction;
    }
;



/* 
 *   A class to represent the NPCs when they don't happen to be around, so 
 *   that LOOK FOR EMMA etc. can give a reasonable response. Also should be 
 *   helpful if the player gets confused about the location of a character 
 *   (especially one who just disappeared into the hiding spot).
 */
class Unperson: MultiLoc, Unthing
    /* The NPC linked to this Unperson. */
    linkedPerson = nil
    
    /* Inherit vocabulary from the linked Person. */
    inheritVocab(target, done)
    {
        /* Do the normal processing. */
        inherited(target, done);
        
        /* Also inherit from the linked Person, if there is one. */
        if (linkedPerson)
        {
            linkedPerson.inheritVocab(target, done);
        }
    }
    
    name()
    {
        if (linkedPerson)
            return linkedPerson.name;
        return inherited;
    }
    
    
    /* 
     *   The Unpeople, like the NPCs, are all female. This shouldn't be 
     *   important, but I'm not taking any chances. 
     */
    isHer = true
    
    
    /* 
     *   If both the NPC and her Unperson are in scope, the NPC should 
     *   always be chosen as the object of commands.
     */
    vocabLikelihood = -50
    
    
    /* The Unpeople always hang out everywhere. */
    initialLocationClass = Room
    
    
    /* 
     *   If the player tries to refer to an NPC who is no longer around, say 
     *   the NPC's gone and mention where she was last seen.
     */
    notHereMsg()
    {
        /* 
         *   actions referring to someone who is no longer here should never
         *   take any time
         */
        gAction.actionTime = 0;
        
        local msg = '\^' + linkedPerson.theName + ' was around here a moment 
            ago, but you can\'t see her now. ';
        
        /* 
         *   Get info about the NPC's last location from the library's 
         *   following code.
         */
        local lastLoc = linkedPerson.lastKnownLocation;
        
        if (lastLoc)
        {
            msg += 'Last time you noticed her, she was ' +
                lastLoc.actorInName + '. ';
        }
        
        return msg;
    }
;


modify Person
    /* keep track of an NPC's last known location */
    lastKnownLocation = lawnWest
    
    noteLocation(traveler)
    {
        if (!traveler.isPlayerChar && 
            traveler == self && 
            gPlayerChar.canSee(self))
        {
            lastKnownLocation = location;
        }
    }
    
    afterTravel(traveler, connector)
    {
        noteLocation(traveler);
        
        inherited(traveler, connector);
    }
    
    beforeTravel(traveler, connector)
    {
        if (!gDobj || !gDobj.ofKind(HidingSpot))
        {
            noteLocation(traveler);
        }
        
        inherited(traveler, connector);
    }
    
    /* 
     *   an NPC entering a hiding spot isn't always simulated, so just brute-
     *   force this one
     */
    enterHidingSpot(hs)
    {
        if ((gPlayerChar.isPlayingHnS || gPlayerChar.knowsWhereEmmaIs) && 
            gPlayerChar.canSee(self))
        {
            lastKnownLocation = hs;
        }
        
        inherited(hs);
    }
;



modify Room
    /* Rooms are less likely targets for commands than other objects. */
    vocabLikelihood = -50
    
    /* Rooms don't list their contents when examined. */
    contentsListedInExamine = nil
;



/* Change a string to title case. */
modify String
    allToTitleCase()
    {
        local str = self;
        
        /* Uppercase the first character. */
        str = str.substr(1, 1).toTitleCase() + str.substr(2, str.length);
        
        local i = 0;
        
        /* Uppercase each character that follows a space. */
        while ((i = str.find(' ', i + 1)) && i < str.length)
        {
            /* uppercase the character after it */
            str = str.substr(1, i) + str.substr(i + 1, 1).toTitleCase()
                + str.substr(i + 2);
        }
        
        return str;
    }
;



modify AgendaItem
    /* 
     *   All AgendaItems are initially active; that way, they'll run as soon 
     *   as their conditions are true, without me having to tell them to.
     */
    initiallyActive = true
    
    /* 
     *   AgendaItems are by default ready when their linked StoryPoint (if 
     *   any) is active, their run condition is true, and their actor has not
     *   conversed this turn.
     *
     *   Additionally, Tiana doesn't do anything for a turn after talking to 
     *   the PC, because she was brung up to use manners.
     */
    isReady = ((!storyPoint || storyPoint.active_) && cond && 
               (getActor() != tiana || !getActor().conversedThisTurn()))
    
    /* 
     *   By default, the run condition is true and there is no linked 
     *   StoryPoint.
     */
    cond = true
    storyPoint = nil
    
    /* 
     *   invokeItem calls exec (which I override to do custom stuff for each 
     *   item) and sets isDone to true.
     */
    invokeItem()
    {
        exec();
        isDone = true;
    }
    
    /* By default, exec does nothing - it's just a hook. */
    exec() {}
;



/* 
 *   AgendaItems shared between Yvonne and Tiana.
 *
 *   This is the search-any-nearby-unsearched-hiding-spot AgendaItem. It 
 *   takes priority over movement AgendaItems, but only runs if the NPC isn't
 *   cheating, because if she is, she already knows where Emma is.
 */
class SearchAgendaItem: AgendaItem
    cond = (!getActor.isCheating && getActor.isNearUnsearchedHidingSpot &&
            !getActor.isPlayingHnS)
    agendaOrder = 50
    
    /* use invokeItem instead of exec so the item won't be marked as done */
    invokeItem()
    {
        local spots = getActor.getNearbyHidingSpots(true);
        local dobj = spots[1];
        newActorAction(getActor, Search, dobj);
    }
;

/* 
 *   The hide-with-Emma AgendaItem. It is activated after its owner searches 
 *   Emma's hiding spot (or approaches it while cheating), and takes 
 *   priority over (almost) all other AgendaItems.
 */
class HideAgendaItem: AgendaItem
    cond()
    {
        if (getActor.isPlayingHnS)
            return nil;
        
        return (bushNearCubby.searchedBy(getActor) ||
                (getActor.isCheating && getActor.canTouch(bushNearCubby)));
    }
    
    agendaOrder = 2
    
    exec()
    {
        getActor.enterHidingSpot(bushNearCubby);
    }
;


/* 
 *   A movement-type AgendaItem. It only applies when its owner is searching 
 *   for Emma and is not yet in the vicinity of Emma's hiding spot, and it 
 *   has the lowest priority. If its owner has done everything else that 
 *   needs to be done in a particular location, it moves her a step along 
 *   her pathway to the hiding spot.
 */
class MoveAgendaItem: AgendaItem
    cond = (!getActor.isPlayingHnS && !getActor.isIn(cubbyArea) &&
            gCurrStoryPoint is in (searching, playingHnS))
    
    agendaOrder = 150
    
    invokeItem()
    {
        local actor = getActor();
        local path = actor.pathToHidingSpot;
        
        /* 
         *   If the actor doesn't already have one, compute a pathway to 
         *   Emma's hiding spot.
         */
        if (!path)
        {
            path = actor.calcPathToHidingSpot();
        }
        
        /* 
         *   Find the actor's location in the path list, and take the next 
         *   room in the list as the next destination.
         */
        local i = path.indexOf(actor.location);
        local dest = path[i + 1];
        
        /* Attempt to travel to the destination. */
        actor.scriptedTravelTo(dest);
    }
;



/* 
 *   Special noun phrase class by Mike Roberts (for Tiana's original weird 
 *   name, mainly).
 */
class SpecialNounPhraseProd: NounPhraseWithVocab
    /* get the list of objects matching our special phrase */
    getMatchList = []

    /* resolve the objects */
    getVocabMatchList(resolver, results, flags)
    {
        /* return all of the in-scope matches */
        return getMatchList().subset({x: resolver.objInScope(x)})
            .mapAll({x: new ResolveInfo(x, flags)});
    }
;



/* Various modifications to Thing. */
modify Thing
    /* (Almost?) everything can be seen and heard clearly from a distance. */
    sightSize = large
    soundSize = large
    
    /* Examining something in the same location is slightly more likely. */
    dobjFor(Examine)
    {
        verify()
        {
            if (gActor.getOutermostRoom != self.getOutermostRoom)
                logicalRank(80, 'not in same location');
        }
    }
;



/* 
 *   Examining a non-portable object has the same likelihood as an ordinary 
 *   Thing.
 */
modify NonPortable
    dobjFor(Examine)
    {
        verify()
        {
            inherited Thing();
        }
    }
;


/* 
 *   If using WebUI, add the cover art folder as a legitimate place to find
 *   resources (e.g. images).
 */
#ifdef TADS_INCLUDE_NET
WebResourceResFile
   vpath = static new RexPattern('/coverart/')
;
#endif



/* 
 *   A special class for buildings (the house and the shed, which have 
 *   essentially the same behaviours).
 */
class Building: Fixture
    /* 
     *   Entering a building links to a TravelConnector of some sort, but 
     *   you can't get in from every side.
     */
    dobjFor(Enter)
    {
        verify()
        {
            logicalRank(150, 'is a building');
        }
        
        check()
        {
            if (!doorRoomList.valWhich({x: x == location}))
            {
                failCheck('There\'s no door on this side of {the dobj/him}. ');
            }
        }
        
        action()
        {
            replaceAction(TravelVia, connector);
        }
    }
    
    /* The list of places you can enter from. */
    doorRoomList = static new Vector(8)
    
    /* the TravelConnector it links to */
    connector = nil
    
    /* Instead of hiding in a building, enter it. */
    dobjFor(HideIn)
    {
        verify() 
        {
            /* less likely than hiding in a hiding spot */
            logicalRank(80, 'enterable but not a hiding spot');
        }
        
        action()
        {
            actionDobjEnter();
        }
    }
    
    notAHidingSpotMsg = '{The dobj/she} is out of bounds. No one would
        be hiding there. '
    
    cannotOpenMsg = 'There\'s no point. {The dobj/he} is out of bounds. '
;


/* During preinitialisation, fill in the doorRoomList of each Building. */
PreinitObject
    /* 
     *   This has to run after the adv3 preinitialisation, so it can consult 
     *   the list of directions.
     */
    execBeforeMe = [adv3LibPreinit]
    
    execute()
    {
        /* Link each Building's travel connector to its Building. */
        forEachInstance(Building, {x: x.connector.building = x});
        
        /* 
         *   Now run through the exits in each room. If it finds a connector 
         *   with a linked Building, add that room to the doorRoomList of 
         *   that Building.
         */
        for (local rm = firstObj(Room); rm != nil; rm = nextObj(rm, Room))
        {
            foreach(local dir in Direction.allDirections)
            {
                if (rm.(dir.dirProp) && rm.(dir.dirProp).building)
                {
                    rm.(dir.dirProp).building.doorRoomList.append(rm);
                }
            }
        }
    }
;


/* 
 *   I don't remember exactly what this code is doing, but commenting it out 
 *   stops the game from compiling, so I guess it's doing something 
 *   important. Evidently it's something to do with the code for the shed 
 *   and house.
 */
modify DeadEndConnector
    building = nil
;



/* 
 *   If the game is using the WebUI, MenuItem.removeStatusLine() isn't defined.
 *   Rather than #ifdeffing every call to it, I'll just reimplement it.
 */
#ifdef TADS_INCLUDE_NET
modify MenuItem
    removeStatusLine()
    {
        statuslineBanner.clearWindow();
    }
;
#endif
     


/* Don't show the status line until it gets "switched on". */
modify statusLine
    showStatusLine() 
    {
        /* 
         *   If the game is running in an interpreter that doesn't allow me to
         *   remove the status line, and either the status is active or the
         *   status has been active, update the status line.
         */
        if (!gameMain.usingWebUI && !systemInfo(SysInfoBanners) && 
            (statusHasBeenActive || gameMain.showStatus))
        {
            /* note that the status line has been switched on */
            statusHasBeenActive = true;
            inherited();
        }
        
        /* 
         *   If the game is running in an interpreter that does support removing
         *   the status line and the status is active, update the status line.
         */
        if ((gameMain.usingWebUI || systemInfo(SysInfoBanners))
            && gameMain.showStatus)
        {
            /* note that the status line has been switched on */
            statusHasBeenActive = true;
            inherited();
        }
        
        /* Otherwise, do nothing. */
    }
    
    /* Has the status line been active? */
    statusHasBeenActive = nil
;


/* Hide the status line when the game is restarted. */
PreRestartObject
    execute()
    {
        MenuItem.removeStatusLine();
    }
;


/* Hide the status line when the game is restored, if necessary. */
PostRestoreObject
    execute()
    {
        if (!gameMain.showStatus)
            MenuItem.removeStatusLine();
        
        /* If in darkness, stop the room description from printing. */
        if (gPlayerChar.isIn(dark))
        {
            "<.p>(Yvonne and {the tiana/she} are counting to 50.) ";
            exit;
        }
    }
;


/* 
 *   After an UNDO, if the status line should not be displayed, switch it 
 *   off. Also, don't perform a lookAround afterwards during the prologue.
 */
modify UndoAction
    performUndo(asCommand)
    {
        /* try undoing to the previous savepoint */
        if (undo())
        {
            local oldActor;
            local oldIssuer;
            local oldAction;

            /* notify all PostUndoObject instances */
            PostUndoObject.classExec();

            /* set up the globals for the command */
            oldActor = gActor;
            oldIssuer = gIssuingActor;
            oldAction = gAction;

            /* set the new globals */
            gActor = gPlayerChar;
            gIssuingActor = gPlayerChar;
            gAction = self;

            /* make sure we reset globals on the way out */
            try
            {
                /* success - mention what we did */
                gLibMessages.undoOkay(libGlobal.lastActorForUndo,
                                      libGlobal.lastCommandForUndo);
                
                /* 
                 *   look around, to refresh the player's memory - but only 
                 *   if not in the prologue
                 */
                if (gCurrStoryPoint not in (prologue, counting))
                    libGlobal.playerChar.lookAround(true);
            }
            finally
            {
                /* restore the parser globals to how we found them */
                gActor = oldActor;
                gIssuingActor = oldIssuer;
                gAction = oldAction;
            }
                
            /* 
             *   if this was an explicit 'undo' command, save the command
             *   to allow repeating it with 'again' 
             */
            if (asCommand)
                AgainAction.saveForAgain(gPlayerChar, gPlayerChar, nil, self);

            /* 
             *   Remove the status line. (MenuItem already has a 
             *   removeStatusLine method. Why reinvent the wheel?) Don't have to 
             *   check whether the status line is needed at this point, because 
             *   if it is, the library will bring it back anyway.
             */
            MenuItem.removeStatusLine();

            /* indicate success */
            return true;
        }
        else
        {
            /* no more undo information available */
            gLibMessages.undoFailed();

            /* indicate failure */
            return nil;
        }
    }
;



/* end the game with message "msg" */
function endGame(msg)
{
    /* 
     *   Remove any "Of course, that's the red-haired girl's name" report
     *   from the transcript - otherwise the output is buggy.
     */
    foreach (local rep in gTranscript.reports_)
    {
        local txt = rep.messageText_;
        
        /* is this the transcript I'm looking for? */
        if (txt && txt.startsWith('<.p>(Of course, <i>'))
        {
            /* get rid of it */
            gTranscript.reports_.removeElement(rep);
            
            /* been there, done that */
            break;
        }
    }
    
    /* end the game */
    finishGameMsg(msg, [finishOptionUndo, finishOptionCredits]);
}



/* 
 *   The map is pretty obvious, and all the rooms are visible from the rooms 
 *   next to them, so we'll make all destinations known.
 */
modify BasicLocation
    actorKnowsDestination(actor, conn)
    {
        return true;
    }
;



/* 
 *   improving the verbose exits lister, since "exits" sounds silly in an 
 *   outdoor context
 */
modify ExitLister
    showListPrefixWide(cnt, pov, parent) {
        if (me.location.ofKind(OutdoorRoom)) {
            if (cnt == 1)
                "From here you can only go ";
            else
                "From here you could go ";
        } else {
            inherited(cnt, pov, parent);
        }
    }
;


/* ditto for the status line exit lister */
modify statuslineExitLister
    showListEmpty(pov, parent)
    {
        "<br><b><<aHref('directions', nil, nil, AHREF_Plain)>>Directions:</a></b> 
        <i>None</i><br>";
    }
    showListPrefixWide(cnt, pov, parent)
    {
        "<br><b><<aHref('directions', nil, nil, AHREF_Plain)>>Directions:</a></b> ";
    }
;


/* ditto the terse room exit lister */
modify lookAroundTerseExitLister
    showListPrefixWide(cnt, pov, parent)
    {
        "<.roompara><.parser>Possible directions: ";
    }
;


/* 
 *   And the explicit EXITS lister. This particular message will only appear 
 *   when the player is in the dark, so we'll customise it for that.
 */
modify explicitExitLister
    showListEmpty(pov, parent)
    {
        "You can't see where to go with your eyes shut. ";
    }
;



/* 
 *   And the exits tip, which lets the player know what commands she can use 
 *   to adjust the exit lister settings.
 */
modify exitsTip
    desc = "You can control the possible direction listings with the DIRECTIONS 
        command. <<aHref('directions status', 'DIRECTIONS STATUS',
            'Turn on status line direction listings')>>
    shows the direction list in the status line,
    <<aHref('directions look', 'DIRECTIONS LOOK', 'List directions in room 
        descriptions')>> shows a full directions list in each room description,
    <<aHref('directions on', 'DIRECTIONS ON', 'Turn on all direction lists')>>
    shows both, and
    <<aHref('directions off', 'DIRECTIONS OFF', 'Turn off all direction lists')>>
    turns off both kinds of direction lists."
;



/* 
 *   Some StoryPoints (e.g. the prologue) may need to block certain actions 
 *   of the player's. Add a method to allow the game to check with them. By 
 *   default, they allow all actions.
 */
modify StoryPoint
    allowAction(action)
    {
        return true;
    }
;



/* A class for possible playthings. */
class PlayThing: Thing
    dobjFor(PlayOnWith)
    {
        verify()
        {
            logicalRank(120, 'is a plaything');
        }
    }
;



/* A class for possible hiding spots. */
class HidingSpot: Booth
    /* Who has searched this hiding spot? */
    searchedByList = []
    
    searchedBy(npc)
    {
        return searchedByList.indexOf(npc);
    }
    
    /* 
     *   Searching a hiding spot once produces a "nope-nothing-there" 
     *   message; searching it again produces an "already-looked-there" 
     *   message.
     */
    dobjFor(Search)
    {
        verify()
        {
            logicalRank(150, 'is a hiding spot');
        }
        
        check()
        {
            /* if playing Hide and Seek, there's no point in searching */
            if (gActor.isPlayingHnS)
            {
                if (gActor.roleInHnS == roleCounting)
                    failCheck('You can\'t start searching until you\'ve 
                        counted. ');
                else
                    failCheck('There\'s no point in searching now. 
                        {The tiana/she} is counting, and you\'re supposed
                        to be hiding. ');
            }
            
            /* 
             *   If the PC has already searched this spot, there's no point 
             *   in doing so again.
             */
            if (searchedBy(gActor))
            {
                /* This should only trigger for the PC, not the NPCs. */
                if (gActor.isPlayerChar())
                    failCheck('You already checked {the dobj/him}. Emma 
                        isn\'t there. ');
                else
                    failCheck('BUG: {the actor/she} is trying to search 
                        {the dobj/him} a second time. ');
            }
        }
        
        action()
        {
            /* 
             *   Note that the actor has now searched this spot, and print an
             *   appropriate message about the search.
             */
            searchedByList += gActor;
            if (gActor.isPlayerChar)
            {
                say(pcSearchMsg);
            }
            else
            {
                say(npcSearchMsg);
            }
        }
    }
    
    /* 
     *   After examining a hiding spot, automatically search it as well - 
     *   but only if it is close enough to touch and hasn't been searched yet.
     */
    dobjFor(Examine)
    {
        action()
        {
            inherited();
            
            if ((gDobj == bushNearCubby || !gActor.isPlayingHnS) && 
                !gActor.knowsWhereEmmaIs && 
                !searchedBy(gActor) && 
                gActor.canTouch(self))
            {
                /* Normally an examine takes no time, but searching does. */
                gAction.actionTime = 1;
                
                /* Print a blank line and search the hiding spot. */
                "<.p>";
                nestedAction(Search, gDobj);
            }
        }
    }
    
    /* 
     *   The default "there's nothing there" message. Should override this 
     *   on most instances; the game is bland enough as is.
     */
    pcSearchMsg = 'You search {the dobj/him}, but Emma isn\'t there. '
    
    /* 
     *   The default "[NPC] searches [the hiding spot]" message. Should 
     *   override this on most instances, but particularly those in the 
     *   north-east part of the map, where the player is likely to see Tiana 
     *   and Yvonne searching.
     */
    npcSearchMsg = '{The actor/she} search{es} {the dobj/him}. '
    
    /* 
     *   Change the message for entering a hiding spot. I should probably 
     *   create individual messages for many of the hiding spots. 
     */
    roomOkayPostureChangeMsg(posture, obj)
    {
        gMessageParams(obj);
        return '{You/he} hide{s} {in obj}. ';
    }
    
    /* 
     *   When a character enters a hiding spot, she usually hides in it 
     *   rather than taking up some other posture.
     */
    defaultPosture = hiding
    allowedPostures = static inherited + hiding
    obviousPostures = static inherited + hiding
    
    /* 
     *   Hiding in a hiding spot. According to the rules of the game, the 
     *   only place the PC should hide is in the bush with Emma, but the 
     *   player might get confused or just decide to try bending the rules. 
     *   Either way, hiding in a (non-Emma) hiding spot and then waiting 
     *   will trigger the "misunderstanding" ending.
     */
    dobjFor(HideIn)
    {
        preCond = (preCondForEntry(hiding))
        
        verify()
        {
            if (verifyEntry(hiding, '{You/he} {is} already hiding 
                {on dobj}. ', nil))
            {
                inherited();
            }
            
            if (gActor.canTouch(self))
                logicalRank(150, 'is a handy hiding spot');
        }
        
        action()
        {
            /* get the actor into the hiding spot */
            performEntry(hiding);
            
            /* 
             *   If the PC is hiding while playing Hide and Seek, trigger a 
             *   playing-Hide-and-Seek ending.
             */
            if (gActor.isPlayerChar && gActor.isPlayingHnS)
            {
                "You curl up {in dobj} to wait for {the tiana/her}.<.p>";
                    
                if (yvonne.isPlayingHnS) /* Tiana is too, but Emma isn't */
                {
                    if (emma.knowsEveryoneIsPlayingHnS)
                    {
                        "Peeking out of your hiding place, you see Emma 
                        running back to the house. ";
                    }
                    else
                    {
                        "{The tiana/she} has already found both you and Yvonne
                        by the time Emma turns up.<.p>";
                    }
                    
                    playingHnS.playingWithYvonne();
                }
                else /* only Tiana is */
                {
                    "It's not until after Tiana <<tiana.learnNameInPassing>> 
                    finds you that Emma and Yvonne turn up. ";
                    
                    /* end the game */
                    playingHnS.playingWithTiana();
                }
            }
        }
    }
    
    /* Instead of entering a hiding spot, hide in it. */
    dobjFor(Enter) asDobjFor(HideIn)
;



/* A new posture: hiding. */
hiding: Posture
    /* Putting the actor into a hiding position. */
    tryMakingPosture(loc) { return tryImplicitAction(HideIn, loc); }
    setActorToPosture(actor, loc) { nestedActorAction(actor, HideIn, loc); }

    /* Verbs for hiding */
    msgVerbIPresent = 'hide{s}'
    msgVerbIPast = 'hid'
    msgVerbTPresent = 'hide{s}'
    msgVerbTPast = 'hid'
    participle = 'hiding'
;



/* 
 *   Allow TopicEntry objects to match against classes as well as objects. 
 *   If the matchObj is a class, check each descendant of that class to see 
 *   if it matches.
 */
modify TopicMatchTopic
    matchTopic(fromActor, topic)
    {
        /* if the matchObj is a class, it needs special handling */
        if (matchObj != nil && matchObj.isClass())
        {
            /* make a list of all descendants of the class in question */
            local objs = new Vector(10);
            
            forEachInstance(matchObj, {x: objs += x});
            
            /* 
             *   There are no dynamically created objects in this game, so I 
             *   can safely stick the list into matchObj and let the 
             *   standard handling take over from here.
             */
            matchObj = objs.toList();
        }
        
        /* let the library code do its stuff */
        return inherited(fromActor, topic);
    }
;



/* 
 *   All the path objects in the game are a disambiguation nightmare, so 
 *   make sure they're only visible when the PC is in the same location as 
 *   they are.
 */
modify PathPassage
    sightSize = small
;



/* Class for the cars in the front garden. */
class Car: IndirectLockable, OpenableContainer, CustomFixture, Booth
    /* 
     *   The cars are locked and not unlockable, so the characters can't get 
     *   into them.
     */
    initiallyLocked = true
    cannotEnterMsg = '{The dobj/he} {is} locked. '
    cannotUnlockMsg = 'You don\'t have the key. '
    cannotLockMsg = (cannotUnlockMsg)
    
    /* It's pretty obvious the cars are locked. */
    cannotOpenLockedMsg = '{The dobj/he} {is} locked and you don\'t have the
        key. '
    
    /* Obviously you can't take/move the cars. */
    cannotTakeMsg = '{The dobj/he} {is} much too big and heavy. '
    
    /* Instead of searching an individual car, search all of them. */
    dobjFor(Search) remapTo(Search, cars)
    
    /* 
     *   The cars are all named after their owners, so there's no need for 
     *   definite or indefinite articles.
     */
    isQualifiedName = true
    
    /* Don't mention whether the car is open/closed in the description. */
    statusReportable = nil
    
    /* This is one of the group of cars. */
    collectiveGroups = [cars]
    
    /* Instead of hiding in an individual car, hide in the group of cars. */
    dobjFor(HideIn)
    {
        verify()
        {
            logicalRank(80, 'hiding in an individual car');
        }
        
        action()
        {
            replaceAction(HideIn, cars);
        }
    }
;



/* Fix bug with AskConnectors */
modify BasicProd
    lastTokenIndex = 0
    firstTokenIndex = 0
;



/* 
 *   Create a new SayToAction in place of another action and report the 
 *   change in action to the player. Don't actually replace the action, in 
 *   case I have some more processing to do first.
 */
function remapToSayTo(npc, topic)
{
    /* 
     *   I create my own action instead of simply calling 
     *   replaceAction so that I can save the action and pass it 
     *   to the command clarification report.
     */
    local action;
    action = SayToAction.createActionInstance();
    action.setResolvedObjects(npc, topic);
    
    /* report the remapping */
    local rep = new RemappedActionAnnouncement();
    rep.action_ = action;
    rep.messageText_ = '<./p0>\n<.assume>saying that to 
        {the tiana/her}<./assume>\n';
    gTranscript.addReport(rep);
    
    /* execute the new action */
    execNestedAction(nil, nil, gPlayerChar, action);
    
    /* return the new action, just in case I need to refer to it */
    return action;
}



/* Make the curly quotes filter aggressive. Grrr! */
modify cquoteOutputFilter
    aggressive = true
;



/* changes to various messages */
modify playerActionMessages
    /* 
     *   The only time the player will get the "It's too dark to do that" 
     *   message is when she has her eyes closed.
     */
    tooDarkMsg = 'You can\'t do that with your eyes closed. '
    
    /* 
     *   Since many of the objects in the game have a lot of synonyms, make 
     *   explicit what object the player is trying to manipulate. 
     */
    cannotTakeFixtureMsg = '{You/he} {can\'t} take {the dobj/him}. '
    cannotOpenMsg = '{The dobj/he} {is}n\'t something {you/he} {can} open. '
    okayTakeMsg = '{You/he} pick{s} up {the dobj/him}. '
    okayDropMsg = '{You/he} discard{s} {the dobj/him}. '
    cannotEnterMsg = '{The dobj/he} {is}n\'t something {you/he} {can} enter. '
    
    /* More childlike cannot eat message. */
    cannotEatMsg = '{The dobj/he} probably wouldn\'t taste very nice. '
    
    /* 
     *   Fix the ask-vague message, which is broken by the inclusion of the 
     *   SayQuery extension.
     */
    askVagueMsg = '<.parser>The story doesn&rsquo;t understand that command.
        Please use ASK ACTOR ABOUT TOPIC (or just A TOPIC).<./parser> '
;


modify libMessages
    /* 
     *   Change the default "Yvonne and the red-haired girl are standing 
     *   here" to something a bit more specific. (This should only be in use 
     *   during the counting scene - at other times, the NPCs will use their 
     *   specialDescs instead.)
     */
    actorHereGroupSuffix(posture, lst)
    {
        " <<lst.length() == 1 ? tSel('is', 'was') : tSel('are', 'were')>>
        waiting for Emma to hide. ";
    }
    
    
    /* 
     *   Change the exit-lister messages to refer to directions instead of 
     *   exits.
     */
    exitsOnOffOkay(stat, look)
    {
        if (stat && look)
            "<.parser>The list of possible directions will now be shown in both
            the status line and in each room description.<./parser> ";
        else if (!stat && !look)
            "<.parser>The list of possible directions will no longer be shown in
            either the status line or room description.<./parser> ";
        else
            "<.parser>The list of possible directions <<stat ? 'will' : 'will not'>> be
            shown in the status line, and <<look ? 'will' : 'won&rsquo;t'>>
            be included in room descriptions.<./parser> ";
    }

    /* describe the current EXITS settings */
    currentExitsSettings(statusLine, roomDesc)
    {
        "DIRECTIONS ";
        if (statusLine && roomDesc)
            "ON";
        else if (statusLine)
            "STATUSLINE";
        else if (roomDesc)
            "LOOK";
        else
            "OFF";
    }
    
    
    /* 
     *   The asterisk style of end-game message looks old-fashioned to me. 
     *   Change it to something a bit more stylish.
     */
    showFinishMsg(msg) { "<.p><b>--- <<msg>> ---</b><.p>"; }
    
    
    /* 
     *   Don't bother mentioning the actor's posture when up the tree, 
     *   because there's only one posture available in that situation anyway.
     */
    roomActorStatus(actor) 
    {
        if (actor.isIn(upTree))
            return;
        else
            inherited(actor);
    }
;



/* 
 *   Allow a single topic entry to fill in as a response to asking, telling, 
 *   and saying.
 */
class DefaultSayAskTellTopic: DefaultTopic
    includeInList = [&sayTopics, &askTopics, &tellTopics]
    matchScore = 1
;


class SayAskTellTopic: SayTopic
    includeInList = [&sayTopics, &askTopics, &tellTopics]
;



/* For convenience, all Actions offer the same failCheck method as Things do. */
modify Action
    failCheck(msg, [params])
    {
        delegated Thing(msg, params...);
    }
;



/* fix for adv3 bug */
modify NounPhraseWithVocab
    /*
     *   Run a set of resolved objects through matchName() or a similar
     *   routine.  Returns the filtered results.  
     */
    resolveNounsMatchName(results, resolver, matchList)
    {
        local origTokens;
        local adjustedTokens;
        local objVec;
        local ret;

        /* get the original token list for the command */
        origTokens = getOrigTokenList();

        /* get the adjusted token list for the command */
        adjustedTokens = getAdjustedTokens();

        /* set up to receive about the same number of results as inputs */
        objVec = new Vector(matchList.length());

        /* consider each preliminary match */
        foreach (local cur in matchList)
        {
            /* ask this object if it wants to be included */
            local newObj = resolver.matchName(
                cur.obj_, origTokens, adjustedTokens);

            /* check the result */
            if (newObj == nil)
            {
                /* 
                 *   it's nil - this means it's not a match for the name
                 *   after all, so leave it out of the results 
                 */
            }
            else if (newObj.ofKind(Collection))
            {
                /* 
                 *   it's a collection of some kind - add each element to
                 *   the result list, using the same flags as the original 
                 */
                foreach (local curObj in newObj)
                    objVec.append(new ResolveInfo(curObj, cur.flags_, self));
            }
            else
            {
                /* 
                 *   it's a single object - add it ito the result list,
                 *   using the same flags as the original 
                 */
                objVec.append(new ResolveInfo(newObj, cur.flags_, self));
            }
        }

        /* convert the result vector to a list */
        ret = objVec.toList();

        /* if our list is empty, note it in the results */
        if (ret.length() == 0)
        {
            /* 
             *   If the adjusted token list contains any tokens of type
             *   "miscWord", send the phrase to the results object for
             *   further consideration.  
             */
            if (adjustedTokens.indexOf(&miscWord) != nil)
            {
                /* 
                 *   we have miscWord tokens, so this is a miscWordList
                 *   match - let the results object process it specially.  
                 */
                ret = results.unknownNounPhrase(self, resolver);
            }
            
            if (!ret)
                ret = [];

            /* 
             *   if the list is empty, note that we have a noun phrase
             *   whose vocabulary words don't match anything in the game 
             */
            if (ret.length() == 0)
                results.noVocabMatch(resolver.getAction(), getOrigText());
        }

        /* return the result list */
        return ret;
    }
;




/* Conversation topics */

/* "name", for asking Tiana what she's called */
tName: Topic
    'her name'
;

/* "hiding spot", for generic references to Emma's hiding spot */
tHidingSpot: Topic
    'hiding spot/place'
;

/* the game */
tGame: Topic
    'playing (the) emma\'s sardines/game'
;

/* Hide and Seek (sometimes treated differently from Sardines) */
tHnS: Topic
    'playing proper hide (and) (&) go seek/hide-and-seek/hide-and-go-seek'
;

/* cheating */
tCheating: Topic
    'cheating/cheat/cheater'
;

/* 
 *   counting (mainly so that the player can answer Tiana's question by 
 *   SAYing COUNT
 */
tCounting: Topic
    '(i) (will) (i\'ll) count/counting'
;

/* 
 *   hiding (mainly so that the player can answer Tiana's question by SAYing 
 *   HIDE
 */
tHiding: Topic
    '(i) (will) (i\'ll) hide/hiding'
;

/* 
 *   birds (so I can give specific responses to asking about the baby bird 
 *   even when the PC hasn't encountered the bird yet)
 */
tBirds: Topic
    'Yvonne\'s her baby pet bird/birds/nest/nests/budgie/budgies/pet/pets'
    
    /* less likely than the actual bird, generally */
    vocabLikelihood = -50
;
