/* TABLE.C -- Command Table Routines

	Written March 1991 by Craig A. Finseth
	Copyright 1991 by Craig A. Finseth
*/

#include "freyja.h"

struct cmd {
	void (*proc)();		/* procedure to invoke */
	char *help;		/* help string */
	};

struct func_cmd {
	void (*proc)();		/* procedure to invoke */
	char *help;		/* help string */
	char *keyname;		/* key name */
	};

/* command table conventions are:

	0 - base
	1 - control-x
	2 - meta
	3 - function keys
*/

/* In help strings:

	@	no command
	^	self-insert
*/

	/* base commands */
static struct cmd base[] = {
{ RMarkSet,	"set the mark to the point" },			/* ^@ */
{ CLineA,	"move to the beginning of the line" },		/* ^A */
{ CCharB,	"move backward character" },			/* ^B */
{ WCaseRotate,	"rotate case of word lower->Capital->UPPER" },	/* ^C */
{ CCharFD,	"delete the following character" },		/* ^D */
{ CLineE,	"move to the end of the line" },		/* ^E */
{ CCharF,	"move forward character" },			/* ^F */
{ MAbort,	"abort, cancel" },				/* ^G */
{ CCharBD,	"delete the preceding character" },		/* ^H */
{ MInsChar,	"insert Tab" },					/* ^I */
{ WIndNext,	"insert Newline and indent" },			/* ^J */
{ CLineFD,	"kill / delete the rest of the line" },		/* ^K */
{ DScreenRange,	"recenter the screen" },			/* ^L */
{ WInsNL,	"insert Newline" },				/* ^M */
{ CLineF,	"move to the next line" },			/* ^N */
{ WInsNLA,	"open line" },					/* ^O */

{ CLineB,	"move to the previous line" },			/* ^P */
{ MQuote,	"quote the next command" },			/* ^Q */
{ MSearchB,	"reverse / backward search" },			/* ^R */
{ MSearchF,	"forward search" },				/* ^S */
{ CCharTran,	"interchange preceding characters" },		/* ^T */
{ MUArg,	"universal argument" },				/* ^U */
{ CScrnF,	"move to the next screen" },			/* ^V */
{ RRegDelete,	"delete region to kill buffer: cut" },		/* ^W */
{ MCtrlX,	"^X command prefix" },				/* ^X */
{ RYank,	"yank kill buffer: paste" },			/* ^Y */
{ CScrnB,	"move to the previous screen" },		/* ^Z */
{ MMeta,	"^[ meta command prefix" },			/* ^[ */
{ WIndDel,	"delete indentation on line" },			/* ^\ */
{ KFromMac,	"invoke keyboard macro" },			/* ^] */
{ MNotImpl,	"@" },						/* ^^ */
{ MNotImpl,	"@" },						/* ^_ */

{ MInsChar,	"^" },						/*   */
{ MInsChar,	"^" },						/* ! */
{ MInsChar,	"^" },						/* " */
{ MInsChar,	"^" },						/* # */
{ MInsChar,	"^" },						/* $ */
{ MInsChar,	"^" },						/* % */
{ MInsChar,	"^" },						/* & */
{ MInsChar,	"^" },						/* ' */
{ MInsChar,	"^" },						/* ( */
{ MInsChar,	"^" },						/* ) */
{ MInsChar,	"^" },						/* * */
{ MInsChar,	"^" },						/* + */
{ MInsChar,	"^" },						/* , */
{ MInsChar,	"^" },						/* - */
{ MInsChar,	"^" },						/* . */
{ MInsChar,	"^" },						/* / */

{ MInsChar,	"^" },						/* 0 */
{ MInsChar,	"^" },						/* 1 */
{ MInsChar,	"^" },						/* 2 */
{ MInsChar,	"^" },						/* 3 */
{ MInsChar,	"^" },						/* 4 */
{ MInsChar,	"^" },						/* 5 */
{ MInsChar,	"^" },						/* 6 */
{ MInsChar,	"^" },						/* 7 */
{ MInsChar,	"^" },						/* 8 */
{ MInsChar,	"^" },						/* 9 */
{ MInsChar,	"^" },						/* : */
{ MInsChar,	"^" },						/* ; */
{ MInsChar,	"^" },						/* < */
{ MInsChar,	"^" },						/* = */
{ MInsChar,	"^" },						/* > */
{ MInsChar,	"^" },						/* ? */

{ MInsChar,	"^" },						/* @ */
{ MInsChar,	"^" },						/* A */
{ MInsChar,	"^" },						/* B */
{ MInsChar,	"^" },						/* C */
{ MInsChar,	"^" },						/* D */
{ MInsChar,	"^" },						/* E */
{ MInsChar,	"^" },						/* F */
{ MInsChar,	"^" },						/* G */
{ MInsChar,	"^" },						/* H */
{ MInsChar,	"^" },						/* I */
{ MInsChar,	"^" },						/* J */
{ MInsChar,	"^" },						/* K */
{ MInsChar,	"^" },						/* L */
{ MInsChar,	"^" },						/* M */
{ MInsChar,	"^" },						/* N */
{ MInsChar,	"^" },						/* O */

{ MInsChar,	"^" },						/* P */
{ MInsChar,	"^" },						/* Q */
{ MInsChar,	"^" },						/* R */
{ MInsChar,	"^" },						/* S */
{ MInsChar,	"^" },						/* T */
{ MInsChar,	"^" },						/* U */
{ MInsChar,	"^" },						/* V */
{ MInsChar,	"^" },						/* W */
{ MInsChar,	"^" },						/* X */
{ MInsChar,	"^" },						/* Y */
{ MInsChar,	"^" },						/* Z */
{ MInsChar,	"^" },						/* [ */
{ MInsChar,	"^" },						/* \ */
{ MInsChar,	"^" },						/* ] */
{ MInsChar,	"^" },						/* ^ */
{ MInsChar,	"^" },						/* _ */

{ MInsChar,	"^" },						/* ` */
{ MInsChar,	"^" },						/* a */
{ MInsChar,	"^" },						/* b */
{ MInsChar,	"^" },						/* c */
{ MInsChar,	"^" },						/* d */
{ MInsChar,	"^" },						/* e */
{ MInsChar,	"^" },						/* f */
{ MInsChar,	"^" },						/* g */
{ MInsChar,	"^" },						/* h */
{ MInsChar,	"^" },						/* i */
{ MInsChar,	"^" },						/* j */
{ MInsChar,	"^" },						/* k */
{ MInsChar,	"^" },						/* l */
{ MInsChar,	"^" },						/* m */
{ MInsChar,	"^" },						/* n */
{ MInsChar,	"^" },						/* o */

{ MInsChar,	"^" },						/* p */
{ MInsChar,	"^" },						/* q */
{ MInsChar,	"^" },						/* r */
{ MInsChar,	"^" },						/* s */
{ MInsChar,	"^" },						/* t */
{ MInsChar,	"^" },						/* u */
{ MInsChar,	"^" },						/* v */
{ MInsChar,	"^" },						/* w */
{ MInsChar,	"^" },						/* x */
{ MInsChar,	"^" },						/* y */
{ MInsChar,	"^" },						/* z */
{ MInsChar,	"^" },						/* { */
{ MInsChar,	"^" },						/* | */
{ MInsChar,	"^" },						/* } */
{ MInsChar,	"^" },						/* ~ */
#if defined(UNIX)
{ CCharBD,	"delete the preceding character" } };		/* ^? */
#endif
#if defined(MSDOS)
{ WWordBD,	"delete the preceding word" } };		/* ^? */
#endif

	/* control-x commands */
static struct cmd ctlx[] = {
{ MNotImpl,	"@" },						/* ^@ */
{ MNotImpl,	"@" },						/* ^A */
{ FBufList,	"list the buffers" },				/* ^B */
{ MExit,	"exit the editor" },				/* ^C */
{ FDIRED,	"DIRED: directory display" },			/* ^D */
#if !defined(NOEXEC)
{ FCommand,	"execute system command" },			/* ^E */
#else
{ MNotImpl,	"@" },						/* ^E */
#endif
{ FFileFind,	"find a file, creating new buffer" },		/* ^F */
{ MAbort,	"abort, cancel" },				/* ^G */
{ WSentBD,	"delete the preceding sentence" },		/* ^H */
{ MNotImpl,	"@" },						/* ^I */
{ MNotImpl,	"@" },						/* ^J */
{ MNotImpl,	"@" },						/* ^K */
{ RLower,	"lowercase the region" },			/* ^L */
{ MNotImpl,	"@" },						/* ^M */
{ MNotImpl,	"@" },						/* ^N */
{ MNotImpl,	"@" },						/* ^O */

{ MNotImpl,	"@" },						/* ^P */
{ MNotImpl,	"@" },						/* ^Q */
{ FFileRead,	"read a file into the buffer" },		/* ^R */
{ FFileSave,	"save the buffer using current name" },		/* ^S */
{ MNotImpl,	"@" },						/* ^T */
{ RUpper,	"UPPERCASE the region" },			/* ^U */
{ CScrnFOW,	"move to next screen in other window" },	/* ^V */
{ FFileWrite,	"ask for filename and write the buffer" },	/* ^W */
{ RMarkSwap,	"interchange the point and mark" },		/* ^X */
{ MNotImpl,	"@" },						/* ^Y */
{ CScrnBOW,	"move to previous screen in other window" },	/* ^Z */
{ MNotImpl,	"@" },						/* ^[ */
{ RDelWhite,	"delete trailing whitespace on lines in the region" },/* ^\ */
{ MNotImpl,	"@" },						/* ^] */
{ MNotImpl,	"@" },						/* ^^ */
{ MNotImpl,	"@" },						/* ^_ */

{ MNotImpl,	"@" },						/*   */
{ MNotImpl,	"@" },						/* ! */
{ MNotImpl,	"@" },						/* " */
{ WPrintMar,	"display the margin settings" },		/* # */
{ MNotImpl,	"@" },						/* $ */
{ MNotImpl,	"@" },						/* % */
{ MNotImpl,	"@" },						/* & */
{ MNotImpl,	"@" },						/* ' */
{ KBegMac,	"start collecting keyboard macro" },		/* ( */
{ KEndMac,	"end collecting keyboard macro" },		/* ) */
{ MNotImpl,	"@" },						/* * */
{ MNotImpl,	"@" },						/* + */
{ WSetTabs,	"set this buffer's tab spacing to the argument" }, /* , */
{ MNotImpl,	"@" },						/* - */
{ WSetLeft,	"set this buffer's left margin to the argument" }, /* . */
{ WSetFill,	"set this buffer's fill mode" },		/* / */

{ MNotImpl,	"@" },						/* 0 */
{ DWindOne,	"set one window" },				/* 1 */
{ DWindTwo,	"set two windows" },				/* 2 */
{ DWindTwoO,	"set two windows, leave the point in other window" },/* 3 */
{ MNotImpl,	"@" },						/* 4 */
{ MNotImpl,	"@" },						/* 5 */
{ MNotImpl,	"@" },						/* 6 */
{ MNotImpl,	"@" },						/* 7 */
{ MIns8,	"insert the argument as a character" },		/* 8 */
{ MNotImpl,	"@" },						/* 9 */
{ MNotImpl,	"@" },						/* : */
{ MNotImpl,	"@" },						/* ; */
{ DLeft,	"scroll the window left" },			/* < */
{ WPrintPos,	"display position" },				/* = */
{ DRight,	"scroll the window right" },			/* > */
{ MNotImpl,	"@" },						/* ? */

{ MNotImpl,	"@" },						/* @ */
{ MNotImpl,	"@" },						/* A */
{ FBufNext,	"move to the next buffer in the ring" },	/* B */
{ MExit,	"exit / quit the editor" },			/* C */
{ MNotImpl,	"@" },						/* D */
{ KFromMac,	"invoke keyboard macro" },			/* E */
{ WSetRight,	"set this buffer's right margin to the argument" }, /* F */
{ MNotImpl,	"@" },						/* G */
{ HHelp,	"help" },					/* H */
{ MNotImpl,	"@" },						/* I */
{ MNotImpl,	"@" },						/* J */
{ FBufDel,	"delete a buffer" },				/* K */
{ MNotImpl,	"@" },						/* L */
{ MNotImpl,	"@" },						/* M */
{ MNotImpl,	"@" },						/* N */
{ DWindSwap,	"move / switch to the other window" },		/* O */

{ MNotImpl,	"@" },						/* P */
{ MQuote,	"quote the next command" },			/* Q */
{ MNotImpl,	"@" },						/* R */
{ MSearchF,	"forward search" },				/* S */
{ RTabify,	"tabify the region" },				/* T */
{ RUntabify,	"untabify the region" },			/* U */
{ FBufPrev,	"move to the previous buffer in the ring" },	/* P */
{ MNotImpl,	"@" },						/* W */
{ MNotImpl,	"@" },						/* X */
{ MNotImpl,	"@" },						/* Y */
{ MNotImpl,	"@" },						/* Z */
{ ROutdent,	"outdent the region" },				/* [ */
{ MNotImpl,	"@" },						/* \ */
{ RIndent,	"indent the region" },				/* ] */
{ DWindGrow,	"grow the current window" },			/* ^ */
{ MNotImpl,	"@" },						/* _ */

{ DTogVisGray,	"toggle whether grayspace is visible" },	/* ` */
{ MNotImpl,	"@" },						/* a */
{ FBufNext,	"move to the next buffer in the ring" },	/* b */
{ MExit,	"exit / quit the editor" },			/* c */
{ MNotImpl,	"@" },						/* d */
{ KFromMac,	"invoke keyboard macro" },			/* e */
{ WSetRight,	"set this buffer's right margin to the argument" }, /* f */
{ MNotImpl,	"@" },						/* g */
{ HHelp,	"help" },					/* h */
{ MNotImpl,	"@" },						/* i */
{ MNotImpl,	"@" },						/* j */
{ FBufDel,	"delete a buffer" },				/* k */
{ MNotImpl,	"@" },						/* l */
{ MNotImpl,	"@" },						/* m */
{ MNotImpl,	"@" },						/* n */
{ DWindSwap,	"move / switch to the other window" },		/* o */

{ MNotImpl,	"@" },						/* p */
{ MQuote,	"quote the next command" },			/* q */
{ MNotImpl,	"@" },						/* r */
{ MSearchF,	"forward search" },				/* s */
{ RTabify,	"tabify the region" },				/* t */
{ RUntabify,	"untabify the region" },			/* u */
{ FBufPrev,	"move to the previous buffer in the ring" },	/* p */
{ MNotImpl,	"@" },						/* w */
{ MNotImpl,	"@" },						/* x */
{ MNotImpl,	"@" },						/* y */
{ MNotImpl,	"@" },						/* z */
{ RHardToSoft,	"convert hard newlines to soft" },		/* { */
{ MNotImpl,	"@" },						/* | */
{ RSoftToHard,	"convert soft newlines to hard" },		/* } */
{ MNotImpl,	"@" },						/* ~ */
{ WSentBD,	"delete the preceding sentence" } };		/* ^? */

	/* meta commands */
static struct cmd meta[] = {
{ MNotImpl,	"@" },						/* ^@ */
#if !defined(NOCALC)
{ WNumMark,	"mark the current number" },			/* ^A */
{ WNumB,	"move backward number" },			/* ^B */
#else
{ MNotImpl,	"@" },						/* ^A */
{ MNotImpl,	"@" },						/* ^B */
#endif
{ DCal,		"display a calendar for the current month" },	/* ^C */
{ MNotImpl,	"@" },						/* ^D */
#if !defined(NOCALC)
{ UEnter,	"enter the current number into the calculator" },/* ^E */
{ WNumF,	"move forward number" },			/* ^F */
#else
{ MNotImpl,	"@" },						/* ^E */
{ MNotImpl,	"@" },						/* ^F */
#endif
{ MAbort,	"abort, cancel" },				/* ^G */
{ WWordBD,	"delete the preceding word" },			/* ^H */
{ MNotImpl,	"@" },						/* ^I */
{ MNotImpl,	"@" },						/* ^J */
{ CLineAED,	"delete the current line" },			/* ^K */
{ DNewDisplay,	"refresh and recenter the screen" },		/* ^L */
#if !defined(NOCALC)
{ UPrintX,	"insert the X register; if arg, do ^[ ^A ^W first" },/* ^M */
#else
{ MNotImpl,	"@" },						/* ^M */
#endif
{ DNext,	"advance the calendar a month" },		/* ^N */
{ MNotImpl,	"@" },						/* ^O */

{ DPrev,	"rewind the calendar a month" },		/* ^P */
{ MNotImpl,	"@" },						/* ^Q */
{ MReplaceQ,	"query global search and replace string" },	/* ^R */
{ MNotImpl,	"@" },						/* ^S */
{ MNotImpl,	"@" },						/* ^T */
#if !defined(NOCALC)
{ UCalc,	"RPN calculator" },				/* ^U */
#else
{ MNotImpl,	"@" },						/* ^U */
#endif
{ MNotImpl,	"@" },						/* ^V */
{ MMakeDelete,	"append the next kill to the kill buffer" },	/* ^W */
{ MNotImpl,	"@" },						/* ^X */
{ MNotImpl,	"@" },						/* ^Y */
{ MNotImpl,	"@" },						/* ^Z */
{ MNotImpl,	"@" },						/* ^[ */
{ MNotImpl,	"@" },						/* ^\ */
#if !defined(NOCALC)
{ ULoadMac,	"load or save the keyboard macro" },		/* ^] */
#else
{ MNotImpl,	"@" },						/* ^] */
#endif
{ MNotImpl,	"@" },						/* ^^ */
{ MNotImpl,	"@" },						/* ^_ */

{ RMarkSet,	"set the mark to the point" },			/*   */
{ MNotImpl,	"@" },						/* ! */
{ MNotImpl,	"@" },						/* " */
{ MNotImpl,	"@" },						/* # */
{ MNotImpl,	"@" },						/* $ */
{ MNotImpl,	"@" },						/* % */
{ MNotImpl,	"@" },						/* & */
{ MNotImpl,	"@" },						/* ' */
{ MNotImpl,	"@" },						/* ( */
{ MNotImpl,	"@" },						/* ) */
{ MNotImpl,	"@" },						/* * */
{ MNotImpl,	"@" },						/* + */
{ MNotImpl,	"@" },						/* , */
{ MNotImpl,	"@" },						/* - */
{ MNotImpl,	"@" },						/* . */
{ MNotImpl,	"@" },						/* / */

{ MNotImpl,	"@" },						/* 0 */
{ MNotImpl,	"@" },						/* 1 */
{ MNotImpl,	"@" },						/* 2 */
{ MNotImpl,	"@" },						/* 3 */
{ MNotImpl,	"@" },						/* 4 */
{ MNotImpl,	"@" },						/* 5 */
{ MNotImpl,	"@" },						/* 6 */
{ MNotImpl,	"@" },						/* 7 */
{ MNotImpl,	"@" },						/* 8 */
{ MNotImpl,	"@" },						/* 9 */
{ MNotImpl,	"@" },						/* : */
{ MNotImpl,	"@" },						/* ; */
{ FBufBeg,	"move to the beginning of the buffer" },	/* < */
{ WPrintLine,	"display line counts" },			/* = */
{ FBufEnd,	"move to the end of the buffer" },		/* > */
{ MNotImpl,	"@" },						/* ? */

{ MNotImpl,	"@" },						/* @ */
{ WSentB,	"move backward sentence" },			/* A */
{ WWordB,	"move backward word" },				/* B */
{ WWordCap,	"Capitalize word" },				/* C */
{ WWordFD,	"delete the following word" },			/* D */
{ WSentF,	"move forward sentence" },			/* E */
{ WWordF,	"move forward word" },				/* F */
{ MNotImpl,	"@" },						/* G */
{ WParaMark,	"mark the current paragraph" },			/* H */
{ MNotImpl,	"@" },						/* I */
{ MNotImpl,	"@" },						/* J */
{ WSentFD,	"delete the following sentence" },		/* K */
{ WWordLow,	"lowercase word" },				/* L */
{ MNotImpl,	"@" },						/* M */
{ MNotImpl,	"@" },						/* N */
{ MNotImpl,	"@" },						/* O */

{ MNotImpl,	"@" },						/* P */
{ WParaFill,	"fill pargraph (^U-justify, ^U^U-unjustify" },	/* Q */
{ MReplace,	"global search and replace string" },		/* R */
{ WLineCenter,	"center line" },				/* S */
{ WWordTran,	"interchange words" },				/* T */
{ WWordUp,	"UPPERCASE word" },				/* U */
{ CScrnB,	"move to the previous screen" },		/* V */
{ RRegCopy,	"copy region to kill buffer" },			/* W */
{ MNotImpl,	"@" },						/* X */
{ MNotImpl,	"@" },						/* Y */
{ MNotImpl,	"@" },						/* Z */
{ WParaB,	"move backward paragraph" },			/* [ */
{ WJoinGray,	"delete surrounding whitespace and insert space" },	/* \ */
{ WParaF,	"move forward paragraph" },			/* ] */
{ MNotImpl,	"@" },						/* ^ */
{ MNotImpl,	"@" },						/* _ */

{ MNotImpl,	"@" },						/* ` */
{ WSentB,	"move backward sentence" },			/* a */
{ WWordB,	"move backward word" },				/* b */
{ WWordCap,	"Capitalize word" },				/* c */
{ WWordFD,	"delete the following word" },			/* d */
{ WSentF,	"move forward sentence" },			/* e */
{ WWordF,	"move forward word" },				/* f */
{ MNotImpl,	"@" },						/* g */
{ WParaMark,	"mark the current paragraph" },			/* h */
{ MNotImpl,	"@" },						/* i */
{ MNotImpl,	"@" },						/* j */
{ WSentFD,	"delete the following sentence" },		/* k */
{ WWordLow,	"lowercase word" },				/* l */
{ MNotImpl,	"@" },						/* m */
{ MNotImpl,	"@" },						/* n */
{ MNotImpl,	"@" },						/* o */

{ MNotImpl,	"@" },						/* p */
{ WParaFill,	"fill pargraph (^U-justify, ^U^U-unjustify" },	/* q */
{ MReplace,	"global search and replace string" },		/* r */
{ WLineCenter,	"center line" },				/* s */
{ WWordTran,	"interchange words" },				/* t */
{ WWordUp,	"UPPERCASE word" },				/* u */
{ CScrnB,	"move to the previous screen" },		/* v */
{ RRegCopy,	"copy region to kill buffer" },			/* w */
{ MNotImpl,	"@" },						/* x */
{ MNotImpl,	"@" },						/* y */
{ MNotImpl,	"@" },						/* z */
{ MNotImpl,	"@" },						/* { */
{ MNotImpl,	"@" },						/* | */
{ MNotImpl,	"@" },						/* } */
{ BBufUnmod,	"clear the buffer modified flag" },		/* ~ */
{ WWordBD,	"delete the preceding word" } };		/* ^? */

	/* function keys */

#define NUMFUNCKEYS	(sizeof(func) / sizeof(func[0]))
#if defined(MSDOS)
static struct func_cmd func[] = {			/* 0 */
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ RMarkSet,	"set the mark to the point",			"Ctrl-2" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
							/* 10 */
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"Shift-Tab" },
{ WParaFill,	"fill pargraph (^U-justify, ^U^U-unjustify",	"Alt-Q" },
{ RRegCopy,	"copy region to kill buffer",			"Alt-W" },
{ WSentF,	"move forward sentence",			"Alt-E" },
{ MReplace,	"global search and replace string",		"Alt-R" },
							/* 20 */
{ WWordTran,	"interchange words",				"Alt-T" },
{ MNotImpl,	"@",						"Alt-Y" },
{ WWordUp,	"UPPERCASE word",				"Alt-U" },
{ MNotImpl,	"@",						"Alt-I" },
{ MNotImpl,	"@",						"Alt-O" },
{ MNotImpl,	"@",						"Alt-P" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
							/* 30 */
{ WSentB,	"move backward sentence",			"Alt-A" },
{ WLineCenter,	"center line",					"Alt-S" },
{ WWordFD,	"delete the following word",			"Alt-D" },
{ WWordF,	"move forward word",				"Alt-F" },
{ MNotImpl,	"@",						"Alt-G" },
{ WParaMark,	"mark the current paragraph",			"Alt-H" },
{ MNotImpl,	"@",						"Alt-J" },
{ WSentFD,	"delete the following sentence",		"Alt-K" },
{ WWordLow,	"lowercase word",				"Alt-L" },
{ MNotImpl,	"@",						"" },
							/* 40 */
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"Alt-Z" },
{ MNotImpl,	"@",						"Alt-X" },
{ WWordCap,	"Capitalize word",				"Alt-C" },
{ CScrnB,	"move to the previous screen",			"Alt-V" },
{ WWordB,	"move backward word",				"Alt-B" },
{ MNotImpl,	"@",						"Alt-N" },
							/* 50 */
{ MNotImpl,	"@",						"Alt-M" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ MNotImpl,	"@",						"" },
{ HHelp,	"help",						"F1" },
							/* 60 */
{ MNotImpl,	"@",						"F2" },
{ CLineAED,	"delete the current line",			"F3" },
{ MNotImpl,	"@",						"F4" },
{ FBufNext,	"move to the next buffer in the ring",		"F5" },
{ DWindSwap,	"move / switch to the other window",		"F6" },
{ WIndThis,	"open a line and indent it the same as current","F7" },
{ FFileSave,	"save the buffer using current name",		"F8" },
#if !defined(NOCALC)
{ UCalc,	"RPN calculator",				"F9" },
#else
{ MNotImpl,	"@",						"F9" },
#endif
{ DCal,		"display a calendar for the current month"	"F10" },
{ MNotImpl,	"@",						"" },
							/* 70 */
{ MNotImpl,	"@",						"" },
{ FBufBeg,	"move to the beginning of the buffer",		"Home" },
{ CLineB,	"move to the previous line",			"Up Arrow" },
{ CScrnB,	"move to the previous screen",			"PgUp" },
{ MNotImpl,	"@",						"" },
{ CCharB,	"move backward character",			"Left Arrow" },
{ MNotImpl,	"@",						"" },
{ CCharF,	"move forward character",			"Right Arrow" },
{ MNotImpl,	"@",						"" },
{ FBufEnd,	"move to the end of the buffer",		"End" },
							/* 80 */
{ CLineF,	"move to the next line",			"Down Arrow" },
{ CScrnF,	"move to the next screen",			"PgDn" },
{ WInsNLA,	"open line",					"Ins" },
{ CCharFD,	"delete the following character",		"Del" },
{ MNotImpl,	"@",						"Shift-F1" },
{ MNotImpl,	"@",						"Shift-F2" },
{ MNotImpl,	"@",						"Shift-F3" },
{ MNotImpl,	"@",						"Shift-F4" },
{ MNotImpl,	"@",						"Shift-F5" },
{ DWindTog,	"toggle between one and two windows",		"Shift-F6" },
							/* 90 */
{ MNotImpl,	"@",						"Shift-F7" },
{ FFileFind,	"find a file, creating new buffer",		"Shift-F8" },
{ MNotImpl,	"@",						"Shift-F9" },
{ MExit,	"exit / quit the editor",			"Shift-F10" },
{ MNotImpl,	"@",						"Ctrl-F1" },
{ RRegCopy,	"copy region to kill buffer",			"Ctrl-F2" },
{ RRegDelete,	"delete region to kill buffer: cut",		"Ctrl-F3" },
{ RYank,	"yank kill buffer: paste",			"Ctrl-F4" },
{ MNotImpl,	"@",						"Ctrl-F5" },
{ MNotImpl,	"@",						"Ctrl-F6" },
							/* 100 */
{ MNotImpl,	"@",						"Ctrl-F7" },
{ MNotImpl,	"@",						"Ctrl-F8" },
{ RMarkSet,	"set the mark to the point",			"Ctrl-F9" },
{ MNotImpl,	"@",						"Ctrl-F10" },
{ MNotImpl,	"@",						"Alt-F1" },
{ MNotImpl,	"@",						"Alt-F2" },
{ MNotImpl,	"@",						"Alt-F2" },
{ MNotImpl,	"@",						"Alt-F4" },
{ MNotImpl,	"@",						"Alt-F5" },
{ MNotImpl,	"@",						"Alt-F6" },
							/* 110 */
{ MNotImpl,	"@",						"Alt-F7" },
{ MNotImpl,	"@",						"Alt-F8" },
{ MNotImpl,	"@",						"Alt-F9" },
{ MNotImpl,	"@",						"Alt-F10" },
{ MNotImpl,	"@",						"Ctrl-PrtSc" },
{ WWordB,	"move backward word",				"Ctrl-Left Arrow" },
{ WWordF,	"move forward word",				"Ctrl-Right Arrow" },
{ MNotImpl,	"@",						"Ctrl-End" },
{ MNotImpl,	"@",						"Ctrl-PgDn" },
{ MNotImpl,	"@",						"Ctrl-Home" },
							/* 120 */
{ MNotImpl,	"@",						"Alt-1" },
{ MNotImpl,	"@",						"Alt-2" },
{ MNotImpl,	"@",						"Alt-3" },
{ MNotImpl,	"@",						"Alt-4" },
{ MNotImpl,	"@",						"Alt-5" },
{ MNotImpl,	"@",						"Alt-6" },
{ MNotImpl,	"@",						"Alt-7" },
{ MNotImpl,	"@",						"Alt-8" },
{ MNotImpl,	"@",						"Alt-9" },
{ MNotImpl,	"@",						"Alt-0" },
							/* 130 */
{ MNotImpl,	"@",						"Alt--" },
{ WPrintLine,	"display line counts",				"Alt-=" },
{ MNotImpl,	"@",						"Ctrl-PgUp" },
	/* the following are mapped in jaguar.c */
#if !defined(SYSMGR)
{ JMenu,	"bring up the menus",				"MENU" },
{ JSetup,	"invoke the set up application",		"SET UP" },
{ JFiler,	"invoke the Filer application",			"FILER" },
{ JComm,	"invoke the communications application",	"COMM" },
{ JAppt,	"invoke the appointment book application",	"APPT" },
{ JPhone,	"invoke the telephone book application",	"PHONE" },
{ JMemo,	"invoke the memo application",			"MEMO" },
							/* 140 */
{ JLotus,	"invoke Lotus 1-2-3",				"LOTUS" },
{ JHPCalc,	"invoke the HP calculator",			"HP CALC" } };
#else
{ JMenu,	"bring up the menus",				"MENU" } };
#endif
#else
static struct func_cmd func[] = {			/* 0 */
{ MNotImpl,	"@",						"" } };
#endif

static char descrbuf[SMALLBUFFSIZE];

/* ------------------------------------------------------------ */

/* Return the command's description. */

char *
TabDescr(key, table)
	int key;
	int table;
	{
	if (key == KEYQUIT) return("Quit");
	if (key == KEYABORT) return("Abort");

	if (key >= 256) table = 3;
	switch (table) {

	case 0:
		if (key > 127 && c.g.meta_handle == 'M')
			return(TabDescr(key & 0x7f, 2));
		return(TPrintChar(key & 0x7f, NULL));
		/* break; */

	case 1:
		key &= 0x7f;
		TPrintChar(ZCX, descrbuf);
		strcat(descrbuf, " ");
		TPrintChar(key, descrbuf + strlen(descrbuf));
		return(descrbuf);
		/* break; */

	case 2:
		key &= 0x7f;
		TPrintChar(ESC, descrbuf);
		strcat(descrbuf, " ");
		TPrintChar(key, descrbuf + strlen(descrbuf));
		return(descrbuf);
		/* break; */

	case 3:
		if (key >= 256) key &= 0xff;
		if (key >= NUMFUNCKEYS)
			return("???");
		else	return(func[key].keyname);
		/* break; */
		}
	}


/* ------------------------------------------------------------ */

/* Invoke the specifed key from the specified table. */

void
TabDispatch(key, table)
	int key;
	int table;
	{
	if (key == KEYQUIT) {
		MExit();
		return;
		}
	if (key == KEYABORT) {
		MAbort();
		return;
		}

	if (key >= 256) table = 3;
	switch (table) {

	case 0:
		if (key > 127) {
			if (c.g.meta_handle == 'M')
				(*meta[key & 0x7f].proc)();
			else if (c.g.meta_handle == 'I')
				MInsChar();
			else	MNotImpl();
			}		
		else	(*base[key & 0x7f].proc)();
		break;

	case 1:
		(*ctlx[key & 0x7f].proc)();
		break;

	case 2:
		(*meta[key & 0x7f].proc)();
		break;

	case 3:
		if (key >= 256) key &= 0xff;
		if (key >= NUMFUNCKEYS)
			MNotImpl();
		else	(*func[key].proc)();
		break;
		}
	}


/* ------------------------------------------------------------ */

/* Return a pointer to the help string for the specified command. */

char *
TabHelp(key, table)
	int key;
	int table;
	{
	if (key == KEYQUIT) return("exit / quit the editor");
	if (key == KEYABORT) return("abort, cancel");
	if (key >= 256) table = 3;
	switch (table) {

	case 0:
		if (key > 127) {
			if (c.g.meta_handle == 'M')
				return(meta[key & 0x7F].help);
			else if (c.g.meta_handle == 'I')
				return("^");
			else	return("@");
			}		
		return(base[key & 0x7f].help);
		/* break; */

	case 1:
		return(ctlx[key & 0x7f].help);
		/* break; */

	case 2:
		return(meta[key & 0x7f].help);
		/* break; */

	case 3:
		if (key >= 256) key &= 0xff;
		if (key >= NUMFUNCKEYS)
			return("@");
		else	return(func[key].help);
		/* break; */
		}
	}


/* ------------------------------------------------------------ */

/* Return the next command table or 0 if not a prefix. */

int
TabTable(key, table)
	int key;
	int table;
	{
	if (key == KEYQUIT) return(0);
	if (key == KEYABORT) return(0);

	if (key >= 256) return(3);
	if (table == 0) {
		if (key == ZCX) return(1);
		else if (key == ESC) return(2);
		else if (key >= 256) return(3);
		}
	return(0);
	}


/* ------------------------------------------------------------ */

/* Return True if the command is a deletion (kill) command or False if
not. */

FLAG
TabIsDelete(key, table)
	int key;
	int table;
	{
	void (*cmd)();

	if (key == KEYQUIT) return(FALSE);
	if (key == KEYABORT) return(FALSE);
	if (key >= 256) table = 3;
	switch (table) {

	case 0:		cmd = base[key & 0x7f].proc;	break;
	case 1:		cmd = ctlx[key & 0x7f].proc;	break;
	case 2:		cmd = meta[key & 0x7f].proc;	break;
	case 3:		if (key >= 256) key &= 0xff;
			if (key >= NUMFUNCKEYS)
				cmd = MNotImpl;
			else	cmd = func[key].proc;
			break;
		}
	return(cmd == CLineFD ||
		cmd == CLineAED ||
		cmd == MMakeDelete ||
		cmd == RRegDelete ||
		cmd == RRegCopy ||
		cmd == WSentFD ||
		cmd == WWordBD ||
		cmd == WWordFD);
	}


/* ------------------------------------------------------------ */

/* Return True if the command is a vertical motion command or False if
not. */

FLAG
TabIsVMove(key, table)
	int key;
	int table;
	{
	void (*cmd)();

	if (key == KEYQUIT) return(FALSE);
	if (key == KEYABORT) return(FALSE);
	if (key >= 256) table = 3;
	switch (table) {

	case 0:		cmd = base[key & 0x7f].proc;	break;
	case 1:		cmd = ctlx[key & 0x7f].proc;	break;
	case 2:		cmd = meta[key & 0x7f].proc;	break;
	case 3:		if (key >= 256) key &= 0xff;
			if (key >= NUMFUNCKEYS)
				cmd = MNotImpl;
			else	cmd = func[key].proc;
			break;
		}
	return(cmd == CLineB || cmd == CLineF);
	}


/* ------------------------------------------------------------ */

/* Return the number of function keys. */

int
TabNumFunc()
	{
	return(NUMFUNCKEYS);
	}


/* end of TABLE.C -- Command Table Routines */
