/*
// Abstract:
//	LIST---List Comment
//
//	The List Comment module displays portions of the comment files.
//
// Author:
//	Derek S. Nickel
//
// Creation date:
//	23 January 1991
//
// History:
// V01-001	Derek S. Nickel		23-JAN-1991
//	Original.
//
*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#include "list.h"
#include "index.h"
#include "memory.h"
#include "pager.h"
#include "tree.h"

#define MAX_LINE 256

typedef struct _listdata_t listdata_t;

struct _listdata_t {
	long matches;
	bin5_t offset;
	bin5_t start;
	bin5_t end;
	TextFile *tf;
	char cbuf[MAX_LINE];
};

static void search_port(int port_no, bin5_t start, bin5_t end,
long *matches);

static int write_one_line(node_t *node, listdata_t *db);

/***********************************************************************
	Messages.
***********************************************************************/

#define vgr__none \
"%%VOYAGER-I-NONE, no comments found\n"

#define vgr__listed \
"%%VOYAGER-I-LISTED, listed %ld comments\n"

/***********************************************************************
	list_comment
***********************************************************************/

void list_comment(bin5_t start, bin5_t end)
{
	long matches = 0;

	/*
	// Search each port in turn.
	*/

	if (ports[0].loaded) search_port(0, start, end, &matches);
	if (ports[1].loaded) search_port(1, start, end, &matches);
	if (ports[2].loaded) search_port(2, start, end, &matches);

	/*
	// Print summary.
	*/

	pager(0);
	if (matches == 0)
		printf(vgr__none);
	else
		printf(vgr__listed, matches);
}

/***********************************************************************
	search_port
***********************************************************************/

static void search_port(int port_no, bin5_t start, bin5_t end,
long *matches)
{
	bin5_t min_adr;
	bin5_t max_adr;
	listdata_t db;
	cond_value sys_status;

	/*
	// Define allowable search range for this port.
	*/

	min_adr = ports[port_no].min_adr;
	max_adr = ports[port_no].max_adr;

	/*
	// Verify correct sequencing.
	*/

	if (start > end) {
		bin5_t tmp;
		tmp = start;
		start = end;
		end = tmp;
	}

	/*
	// Search this port?
	*/

	if ((start < min_adr && end < min_adr) ||
	    (start > max_adr && end > max_adr))
		return;

	/*
	// Force start amd end into range.
	*/

	if (start < min_adr) start = min_adr;
	if (end > max_adr) end = max_adr;

	/*
	// Search for comment.
	*/

	db.matches = 0;
	db.offset = min_adr;
	db.start = start;
	db.end = end;
	db.tf = ports[port_no].ca_file;

	/*
	// Search for comments.
	*/

	sys_status = traverse_tree(
		&db.tf->root,
		write_one_line,
		&db );

	*matches += db.matches;
}

static int write_one_line(node_t *node, listdata_t *db)
{
	char *cbufp = db->cbuf;
	bin5_t adr = node->key + db->offset;

	if (db->start <= adr && adr <= db->end) {
		if (!node->deleted) {
			fseek(db->tf->f, node->value, SEEK_SET);
			fgets(cbufp, MAX_LINE, db->tf->f);
			db->cbuf[strlen(cbufp)-1] = '\0';
			cbufp += 6;
			db->matches++;
			pager(0);
			printf("%05lX %s\n", adr, cbufp);
		}
	}

	return 1;
}
