/* xsym.c -- SAD symbol extraction.

This file is part of SAD, the Saturn Disassembler package.

SAD is not distributed by Free Software Foundation. Do not ask them
for a copy or how to obtain new releases. Instead, send e-mail to the
address below. SAD is merely covered by the GNU General Public
License.

Please send your comments, ideas, and bug reports to
Jan Brittenson <bson@ai.mit.edu>

*/


/* Copyright (C) 1990 Jan Brittenson.

SAD is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 1, or (at your option) any later
version.

SAD is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with SAD; see the file COPYING.  If not, write to the Free
Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include "sad.h"

/* Options */
int
  opt_debug = 0,		/* Debug info */
  opt_rewrite = 0,		/* Rewrite .symbols */
  opt_supersede = 1;		/* Merge (0)/Supersede (1) */


/* Compare two symbols by id for ascending sort. */
static symidcmp(s1, s2)
  struct symbol **s1, **s2;
{
  return(strcmp((*s1)->id, (*s2)->id));
}


/* Look up symbol on name */
struct symbol *sym(symid)
  char *symid;
{
  static char symbuf[132];
  int bot, top, cent;
  struct symbol *closest;


  if(!symroot || !symref || !nsymbols)
    return(NULL);

  /* Look up symbol */
  top = nsymbols-1;
  bot = 0;
  cent = nsymbols >> 1;

  for(; top >= bot; cent = bot + ((top-bot) >> 1))
    {
      int stupid;

      stupid = strcmp(symid, symref[cent]->id);

      if(stupid > 0)
	bot = cent+1;
      else
	if(stupid < 0)
	  top = cent-1;
	else
	  return(symref[cent]);
    }

  /* Flaw: we can miss the last symbol */
  if(!strcmp(symref[top]->id, symid))
    return(symref[top]);

  return(NULL);
}


/* Print usage */
void usage(argc, argv)
  int argc;
  char **argv;
{
  fprintf(stderr, "usage: %s [-smrl]\n", *argv);
  exit(1);
}


/* Merge one line of input.
 * The patterns we look out for are:
 * 
 * [1] xxxxx <followed by any number of [2] and/or [3]>
 * [2] symbol:
 * [3] symbol=val
 *
 * All other lines are ignored.
 * 
 */
static void merge_sym_line()
{
  char lbuf[132], org_lbuf[sizeof lbuf], *lp, symid[132];
  int new_pc = 0;


  lbuf[0] = '\0';
  fgets(lbuf, sizeof lbuf, stdin);
  lineno++;

  strcpy(org_lbuf, lbuf);

  if(feof(stdin))
    return;

  if(opt_debug)
    fprintf(stderr,"line=%s", lbuf);
  
  lp = byspace(lbuf);
  
  if(!*lp)
    return;
  
  /* Explicit PC? */
  if(IS_HEX(*lp) || *lp == '#')
    if(hexstrtoi(&lp, &pc))
      {
	lp = byspace(lp);
	new_pc = TRUE;
      }
  
  if(!*lp)
    return;
  
  if(opt_debug && new_pc)
    fprintf(stderr, "--> pc=%x\n", pc);
     
  /* Scan symbol ids */
  while(symbolidp(&lp, symid, sizeof symid - 1))
    {
      struct symbol *symp;


      if(opt_debug)
	fprintf(stderr, "--> id=\"%s\", rest=%s", symid, lp);

      if((symid[0] != 'L' || symid[1] != '_') && 
	 (*lp == ':' || *lp == '=' || *lp == '/'))
	{
	  int eraseflag = (*lp == '/');
	  char marker = *lp;

	  lp = byspace(lp+1);
	  
	  if(symp = sym(symid))
	    {
	      if(eraseflag)
		{
		  if(symp->type & T_SYM_ISDEF)
		    warning("Attempted deletion of new symbol %s.", symp->id);
		  
		  symp->type |= T_SYM_ERASE;

		  if(opt_debug)
		    fputs("(Erased)\n", stderr);
		}
	      else
		{
		  int newval, newtype = T_SYM_CODE;

		  if(marker == ':')
		    if(new_pc)
		      newval = pc;
		    else
		      newval = determine_pc(stdin);
		  else
		    if(!intstrtoi(&lp, &newval))
		      {
			error("Bad integer syntax:\n%s", org_lbuf);
			return;
		      }
		    else
		      newtype = T_SYM_DATA;
		  
		  symp->type |= T_SYM_ISDEF;
		  symp->type &= ~T_SYM_TYPE;
		  symp->type |= newtype;
		  
		  if(opt_supersede)
		    {
		      symp->val = newval;
		      if(symp->type & T_SYM_ERASE)
			{
			  warning("Redefinition of deleted symbol %s.",
				  symp->id);
			  
			  symp->type &= ~T_SYM_ERASE;
			}
		    }
		  else
		    if(symp->val != newval)
		      warning("Symbol %s already exists - use '-s' to supersede.",
			      symp->id);

		  if(opt_debug)
		    fputs("(Updated)\n", stderr);

		}
	    }
	  else
	    /* Symbol does not exist - create new */
	    {
	      extern char *malloc();
	      struct symbol
		*newsym =
		  (struct symbol *) malloc(sizeof(struct symbol));
	      
	      
	      if(!newsym)
		{
		  perror("symbol alloc");
		  return;
		}
	      
	      newsym->link = symroot;
	      symroot = newsym;
	      
	      newsym->seq = newsym->ref = 0;
	      newsym->id  = strdup(symid);
	      newsym->xrefhead = newsym->xreftail = NULL;
	      newsym->type = T_SYM_ISDEF;

	      switch(marker)
		{
		case ':':
		  
		  if(new_pc)
		    newsym->val = pc;
		  else
		    newsym->val = determine_pc(stdin);

		  newsym->type |= T_SYM_CODE;
		  break;
		  
		case '=':
		  
		  intstrtoi(&lp, &newsym->val);
		  newsym->type |= T_SYM_DATA;
		  break;
		  
		default:
		  
		  symroot = newsym->link;
		  free(newsym);
		  return;
		  
		}

	      if(opt_debug)
		fputs("(Added)\n");
	    }
	}
      lp = byspace(lp);
    }
}


/* Parse arguments */
static void args(argc, argv)
  int argc;
  char **argv;
{
  char *cp1;
  int flagsp = 0;

  if(argc > 1 && argv[1][0] == '-')
    {
      flagsp = 1;

      for(cp1 = argv[1]+1; *cp1; cp1++)
	switch(*cp1)
	  {
	  case 'Q':		/* Debug info */

	    opt_debug = !opt_debug;
	    break;

	  case 's':		/* Supersede */

	    opt_supersede = 1;
	    break;

	  case 'm':		/* Merge */

	    opt_supersede = 0;
	    break;

	  case 'r':		/* Rewrite .symbols */

	    opt_rewrite = !opt_rewrite;
	    break;

	  case 'l':		/* Line numbers in errors and warnings */

	    opt_lines = !opt_lines;
	    break;

	  default:

	    usage(argc, argv);
	  }
    }

  if(argc != 1+flagsp)
    usage(argc, argv);

  if(opt_debug)
    fprintf(stderr, "Debugging is enabled.\nThis is SAD %s %s\n\n", 
	    sad_version, *argv);
}  


main(argc, argv)
  int argc;
  char **argv;
{
  init();			/* Initialize tables */
  
  args(argc, argv);		/* Decode arguments */

  load_symbols(SAD_SYMBOLS, symidcmp);	/* Load symbol table */
  
  pc = 0;
  while(!feof(stdin))
    merge_sym_line();		/* Merge/supersede one line of input */

  /* Print new symbol file */
  print_symfile((opt_rewrite ? SAD_SYMBOLS : NULL));
}
