/*
** Program fdb - family tree database generation and manipulation
**
** Copyright (C) 1994 Andy Burrows 
**
**            email: cadellin@corum.me.man.ac.uk (130.88.29.14)
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public Licence as published by the Free
** Software Foundation; either version 1, or any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
** or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
** more details. It should be included in this distribution in a file 
** called COPYING
**
*/

/*
   this file contains the functions relating to the graphical display
   of the database
*/

/*#define DEBUG */

/* standard headers */

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/* XView headers */

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/canvas.h>
#include <xview/font.h>
#include <xview/xv_xrect.h>

/* fdb database definitions */

#include "definitions.h"

/*
   diagnostic dump of descendants structure 
*/

dump_descendants()
{
int  i,j,k;

for(i=0;i<number_of_descendants;i++)
    {
    printf("\ndescendant %d, %s %s at (%d,%d)\n",i,
                          descendant[i]->forenames,
                          descendant[i]->surname,
                          descendant[i]->location.x,
                          descendant[i]->location.y);
    printf("%d spouse(s)\n",descendant[i]->number_of_spouses);
    for(j=0;j<descendant[i]->number_of_spouses;j++)
        {
        printf("    spouse %d, %s, at (%d,%d) has %d children\n",j,
                          descendant[i]->spouse_name[j],
                          descendant[i]->spouse_location[j].x,
                          descendant[i]->spouse_location[j].y,
                          descendant[i]->num_children_of_spouse[j]);
        for(k=0;k<descendant[i]->num_children_of_spouse[j];k++)
            {
            printf("        child %d, %s %s, at (%d,%d)\n",k,
                          descendant[i]->children[j][k]->forenames,
                          descendant[i]->children[j][k]->surname,
                          descendant[i]->children[j][k]->location.x,
                          descendant[i]->children[j][k]->location.y);
            }
        }
    }
}

/*
   write text centred on specified x coordinate
*/

void  write_centred(dpy, xwin, x, y, text, size)

Display        *dpy;
Window          xwin;
int             x, y;
char           *text;
int             size; /* bit of a fudge! size = 0 => small_font
                                         size = 1 => font       */
{
int  x_start;

/* find the length of the string */

if(size == 0)
    (void)xv_get(small_font, FONT_STRING_DIMS, text, &dimensions);
else
    (void)xv_get(font, FONT_STRING_DIMS, text, &dimensions);

/* move the x location to centre the string */

x_start = x - (dimensions.width/2);

/* draw the string, using the appropriate GC */

#ifdef DEBUG
printf("writing string: %s at (%d,%d)\n",text,x,y);
#endif

if(size == 0)
    XDrawString(dpy, xwin, gc_small, x_start, y, text, strlen(text));
else
    XDrawString(dpy, xwin, gc, x_start, y, text, strlen(text));

}

/*
   draw the current line on the gen canvas
*/

void  draw_line(canvas, paint_window, dpy, xwin, area)

Canvas          canvas;
Xv_Window       paint_window;
Display        *dpy;
Window          xwin;
Xv_xrectlist   *area;
{
int   i, j, num_lines;
/* 
   first check that there is some data to plot i.e. at least one
   display struct is non-NULL (they are used sequentially, so
   if the first one is NULL they all are
*/

if(generation[0] == (struct db_generation *) NULL)
    {
#ifdef DEBUG
    printf("canvas redraw procedure called but no data available\n");
#endif
    return;
    }

#ifdef DEBUG
    printf("canvas redraw procedure called\n");
#endif

/* synchronise X calls */

#ifdef DEBUG
/*
XSynchronize(dpy, TRUE);
*/
#endif

/* clear the window */

XClearWindow(dpy, xwin); 

/* loop for all people in the tree */

for(i=0;i<number_of_generations;i++)
    {
    for(j=0;j<generation[i]->number_of_children;j++)
        {
        /* write the strings into the window, centred on the x hot spot */

        /* add the forenames if not empty */
        num_lines = 0;
        if(strcmp(generation[i]->forenames[j],""))
            {
            write_centred(dpy, xwin, generation[i]->x[j], generation[i]->y[j], 
                    generation[i]->forenames[j], 1);
            num_lines++;
            }

        /* add the surname if not empty */
        if(strcmp(generation[i]->surname[j],""))
            {
            write_centred(dpy, xwin, generation[i]->x[j], 
                    generation[i]->y[j] + num_lines * vertical_text_separation, 
                    generation[i]->surname[j], 1);
            num_lines++;
            }

        /* add the date of birth of known, otherwise try the date of death */
        if(strcmp(generation[i]->birth_date[j],"b. "))
            {
            write_centred(dpy, xwin, generation[i]->x[j], 
                    generation[i]->y[j] + num_lines * vertical_text_separation, 
                    generation[i]->birth_date[j], 0);
            num_lines++;
            }
        else
            {
            if(strcmp(generation[i]->death_date[j],"d. "))
                {
                write_centred(dpy, xwin, generation[i]->x[j], 
                        generation[i]->y[j] + num_lines * vertical_text_separation, 
                        generation[i]->death_date[j], 0);
                num_lines++;
                }
            }

        /* if this is the traced person, add their spouse if known */
        if((j == generation[i]->location) && (generation[i]->spouse != -1))
            {
            write_centred(dpy, xwin, generation[i]->x[j], 
                    generation[i]->y[j] + num_lines * vertical_text_separation, 
                    generation[i]->spouse_name, 0);
            }

        /* 
           draw the little stub line up to the cross line except for
           the last generation
        */

        if(i < number_of_generations - 1)
            XDrawLine(dpy, xwin, gc, generation[i]->x[j], 
                 generation[i]->y[j] - vertical_text_separation,
                 generation[i]->x[j],
                 generation[i]->y[j] - 2 * vertical_text_separation);
        }
    /* 
       draw the line linking the children and the line to the next generation 
       (except for the last generation) 
       Note: the length of the line to the next generation depends on the 
       number of non-empty entries in the traced person of the generation above
    */
    if(i < number_of_generations - 1)
        {
        XDrawLine(dpy, xwin, gc, generation[i]->x[0], 
                 generation[i]->y[0] - 2 * vertical_text_separation,
                 generation[i]->x[generation[i]->number_of_children - 1],
                 generation[i]->y[generation[i]->number_of_children - 1] - 
                             2 * vertical_text_separation);

        num_lines = 0;
        if(strcmp(generation[i+1]->forenames[generation[i+1]->location],""))
            num_lines++;
        if(strcmp(generation[i+1]->surname[generation[i+1]->location],""))
            num_lines++;
        if(strcmp(generation[i+1]->birth_date[generation[i+1]->location],"b. "))
            num_lines++;
        if(generation[i+1]->spouse != -1)
            num_lines++;

        XDrawLine(dpy, xwin, gc, generation[i]->x[generation[i]->location], 
                 generation[i]->y[generation[i]->location] - 2 * vertical_text_separation,
                 generation[i]->x[generation[i]->location],
                 generation[i]->y[generation[i]->location] - 
                             vertical_generation_separation +
                             num_lines * vertical_text_separation);
        }
    }
}

/*
   find the maximum length (in pixels) in the appropriate font of the
   text to be displayed for the specified person in the generation
*/

int  longest_text_gen(gen, member)

int  gen, member;
{
int  longest, i;
int  length[4];

/* zero the length array (bloody 4.1.1 compiler!!) */

for(i=0;i<4;i++)
    length[i] = 0;

/* find the number of pixels needed for the strings */

(void) xv_get(font, FONT_STRING_DIMS, generation[gen]->forenames[member], &dimensions);
length[0] = dimensions.width;
(void) xv_get(font, FONT_STRING_DIMS, generation[gen]->surname[member], &dimensions);
length[1] = dimensions.width;
(void) xv_get(small_font, FONT_STRING_DIMS, generation[gen]->birth_date[member], &dimensions);
length[2] = dimensions.width;

/* if the person is one of the trace people, add the spouse */

if(member == generation[gen]->location)
    {
    (void) xv_get(small_font, FONT_STRING_DIMS, generation[gen]->spouse_name, &dimensions);
    length[3] = dimensions.width;
    }

/* find the largest */

longest = length[0];
for(i=1;i<4;i++)
    if(length[i] > longest)
        longest = length[i];

/* return it */

return longest;
}

/*
   order the children of the specified generation in descending age order
*/

void  order_children_in_generation(level,parent)

int  level, parent;
{
long int  dob[MAX_CHILDREN], temp_dob;
int       i, j, child[MAX_CHILDREN], temp_child;

/* 
   set the number of people in this generation
   this routine should never be called with an invalid parent value!
*/

generation[level]->number_of_children = entry[parent]->number_of_children;

/* load the children and their encoded birth dates into work arrays */

for(i=0;i<generation[level]->number_of_children;i++)
    {
    child[i] = entry[parent]->child[i];
    dob[i]   = entry[entry[parent]->child[i]]->encoded_date;
    }

#ifdef DEBUG
/*
printf("\nbefore sorting:\n");
for(i=0;i<generation[level]->number_of_children;i++)
    {
    printf("index: %d - dob: %ld\n",child[i], dob[i]);
    }
*/
#endif

/* simple bubble sort (not needed if there is only one child) */

if(generation[level]->number_of_children > 1)
    {
    for(j=0;j<generation[level]->number_of_children - 1;j++)
        {
        for(i=0;i<generation[level]->number_of_children - j - 1;i++)
            {
            if(dob[i] > dob[i+1])
                {
                temp_dob   = dob[i];
                dob[i]     = dob[i+1];
                dob[i+1]   = temp_dob;
                temp_child = child[i];
                child[i]   = child[i+1];
                child[i+1] = temp_child;
                }
            }
        }
    }

/* assign the sorted children to the array */

for(i=0;i<generation[level]->number_of_children;i++)
    generation[level]->children[i] = child[i];

#ifdef DEBUG
/*
printf("\nafter sorting:\n");
for(i=0;i<generation[level]->number_of_children;i++)
    {
    printf("index: %d - dob: %ld\n",child[i], dob[i]);
    }
*/
#endif

}

/*
   build the data structures required for displaying the 
   required descent (MALE or FEMALE) of the current person
*/

void  build_line(required_gender)

int  required_gender;
{
int  i, j, max_x, min_x;
int  cumulative_x, old_string_length, new_string_length, max_y, min_y;
int  x_translate, y_translate;
int  return_code;
char temp_string[MAX_STRING_LENGTH];

/* zero the in use array */

for(i=0;i<MAX_INDICES;i++)
    in_use[i] = 0;

/* 
   calculate the vertical separation between text items and
   generations based on the larger font size 
*/

(void) xv_get(font, FONT_STRING_DIMS, "X", &dimensions);
vertical_text_separation = dimensions.height + DELTA_Y;
vertical_generation_separation = VERTICAL_SEPARATION * vertical_text_separation;

#ifdef DEBUG
printf("vertical text separation = %d\n",vertical_text_separation);
#endif

/*
   create the male or female line as far back as it goes
   or up to the max. number of generations in a plot, whichever
   is smaller
*/

descent_array[0] = current_index;
in_use[current_index] = 1;
number_of_generations = 1;
if(required_gender == MALE)
    {
    while(((descent_array[number_of_generations] = 
           entry[descent_array[number_of_generations - 1]]->father) != -1) &&
           (number_of_generations < max_gen_generations))
        {
        if(in_use[descent_array[number_of_generations]])
            {
            /* warn the user of the problem and stop the search */
            sprintf(error_message,"Warning! - Apparent Circular Path in Requested Descent Line at %s %s - Stopping Before First Repeated Person!",
                    entry[descent_array[number_of_generations - 1]]->forenames,
                    entry[descent_array[number_of_generations - 1]]->surname);
            show_error();
            break;
            }
        else
            {
            in_use[descent_array[number_of_generations]] = 1;
            number_of_generations++;
            }
        }
    }
else
    {
    while(((descent_array[number_of_generations] = 
           entry[descent_array[number_of_generations - 1]]->mother) != -1) &&
           (number_of_generations < max_gen_generations))
        {
        if(in_use[descent_array[number_of_generations]])
            {
            /* warn the user of the problem and stop the search */
            sprintf(error_message,"Warning! - Apparent Circular Path in Requested Descent Line at %s %s - Stopping Before First Repeated Person!",
                    entry[descent_array[number_of_generations - 1]]->forenames,
                    entry[descent_array[number_of_generations - 1]]->surname);
            show_error();
            break;
            }
        else
            {
            in_use[descent_array[number_of_generations]] = 1;
            number_of_generations++;
            }
        }
    }

#ifdef DEBUG
for(i=0;i<number_of_generations;i++)
    printf("generation %d: %s %s\n",i,entry[descent_array[i]]->forenames,
                                      entry[descent_array[i]]->surname);
#endif

/* empty any existing display structures */

for(i=0;i<MAX_INDICES;i++)
    {
    if(generation[i] != (struct db_generation *) NULL)
        {
        free(generation[i]);
        generation[i] = (struct db_generation *) NULL;
        }
    }

/* build the display structures */

for(i=0;i<number_of_generations;i++)
    {
    /* allocate the memory */
    if((generation[i] = (struct db_generation *) calloc((size_t) 1, 
                       (size_t) sizeof(struct db_generation))) == NULL)
        {
        /* 
            something has gone badly wrong so try to tidy up as best
            as possible and suggest the user backs up and bails out!
        */
        for(j=0;j<i;j++)
            free(generation[j]);

        strcpy(error_message,"Unable to Allocate Display Memory - Suggest Back Up Current Work and Bail Out!");
        show_error();
        return;
        }
    /* set the person through whom we are tracing the descent */
    generation[i]->person = descent_array[i];

    /* 
       if this is not the first element (i.e. the current person) then 
       set the appropriate spouse of the person through whom we are tracing 
       the descent
       check that the other parent of the person at the previous level is one of
       the spouses of their trace parent, if not leave the field empty
    */

    generation[i]->spouse = -1;
    if(i != 0)
        {
        if(required_gender == MALE)
            {
            for(j=0;j<entry[descent_array[i]]->number_of_spouses;j++)
                if(entry[descent_array[i]]->spouse[j] == entry[descent_array[i-1]]->mother)
                    generation[i]->spouse = entry[descent_array[i]]->spouse[j];
            }
        else
            {
            for(j=0;j<entry[descent_array[i]]->number_of_spouses;j++)
                if(entry[descent_array[i]]->spouse[j] == entry[descent_array[i-1]]->father)
                    generation[i]->spouse = entry[descent_array[i]]->spouse[j];
            }
        }

    /*
       order the children so that the eldest is at the start of the array
       don't try this with the furthest generation 'cos by definition we
       don't know the relevant parent so we can't work out who the children are
       let alone how many there are of them!
    */

    if(i < number_of_generations - 1)
        {
        order_children_in_generation(i,descent_array[i+1]);
        }
    else
        /*
           if this is the last generation under consideration, then
           there is only one entry
        */
        {
        generation[i]->children[0] = descent_array[i];
        generation[i]->number_of_children = 1;
        }

    /*
        store the names of the generation members
    */

    /* if there is a spouse, find the name otherwise use empty string */

    if(generation[i]->spouse != -1)
        {
        /* initialise the spouse's name with their forenames */
        sprintf(generation[i]->spouse_name,"m. %s ",entry[generation[i]->spouse]->forenames);

        /*
           if the spouse's surname is different from the persons, add it on 
        */
        if(strcmp(entry[generation[i]->spouse]->surname,entry[generation[i]->person]->surname))
            strcat(generation[i]->spouse_name,entry[generation[i]->spouse]->surname);

        /* 
           if the spouse is female, they may have a maiden name, if so (and it is
           different from their surname) add it on
        */
        if((entry[generation[i]->spouse]->gender == FEMALE) && 
                (strcmp(entry[generation[i]->spouse]->maiden_name,"")) &&
                (entry[generation[i]->spouse]->surname,entry[generation[i]->spouse]->maiden_name))
            {
            sprintf(temp_string," (nee %s)", entry[generation[i]->spouse]->maiden_name);
            strcat(generation[i]->spouse_name,temp_string);
            }
        }
    else
        {
        strcpy(generation[i]->spouse_name,"");
        }

    /* name, birth and death dates of the children */

    for(j=0;j<generation[i]->number_of_children;j++)
        {
        strcpy(generation[i]->forenames[j],entry[generation[i]->children[j]]->forenames);
        /* 
           if the person is female, add their surname (and maybe maiden name)
        */
        if((entry[generation[i]->children[j]]->gender == FEMALE))
            {
            /* 
               if the person has a maiden name (and it is different from their surname) add it on
            */
            if(strcmp(entry[generation[i]->children[j]]->surname,
                    entry[generation[i]->children[j]]->maiden_name) &&
                    strcmp(entry[generation[i]->children[j]]->maiden_name,""))
                {
                sprintf(generation[i]->surname[j],"%s (nee %s)", 
                        entry[generation[i]->children[j]]->surname,
                        entry[generation[i]->children[j]]->maiden_name);
                }
            else
                {
                sprintf(generation[i]->surname[j],"%s", 
                        entry[generation[i]->children[j]]->surname);
                }
            }
        else
            /* the person is male so just use their surname */
            {
            strcpy(generation[i]->surname[j],entry[generation[i]->children[j]]->surname);
            }
        sprintf(generation[i]->birth_date[j],"b. %s",
                               entry[generation[i]->children[j]]->birth_date);
        sprintf(generation[i]->death_date[j],"d. %s",
                               entry[generation[i]->children[j]]->death_date);
        }
    
#ifdef DEBUG
    printf("spouse is %s\n", generation[i]->spouse_name);
    for(j=0;j<generation[i]->number_of_children;j++)
        {
        printf("child %d is %s %s\n",j,
                generation[i]->forenames[j],generation[i]->surname[j]);
        printf("birthdate %s\n", generation[i]->birth_date[j]);
        }
#endif

    /* 
       locate the person being traced through in the generation currently 
       to allow the canvas locations to be calculated
    */

    generation[i]->location = 0;
    while(generation[i]->children[generation[i]->location] != generation[i]->person)
        generation[i]->location++;

#ifdef DEBUG
    printf("person being traced in generation %d is child index %d\n",i,
            generation[i]->location);
#endif
    /* 
       calculate the x and y locations of the generation members relative the 
       trace person and the generation number
    */

    old_string_length = longest_text_gen(i,generation[i]->location);

#ifdef DEBUG
    printf("length of string %s is %d pixels\n",
            generation[i]->forenames[generation[i]->location], old_string_length);
#endif

    /* put the traced person in the middle */

    generation[i]->y[generation[i]->location] = - vertical_generation_separation * i;
    generation[i]->x[generation[i]->location] = 0;

#ifdef DEBUG
    printf("generation: %d  child:%d  at(%d,%d)\n",i,generation[i]->location,
               generation[i]->x[generation[i]->location], 
               generation[i]->y[generation[i]->location]);
#endif

    cumulative_x = -0.5 * old_string_length;
    for(j=generation[i]->location + 1;j<generation[i]->number_of_children;j++)
        {
        new_string_length = longest_text_gen(i,j);

#ifdef DEBUG
    printf("length of string %s is %d pixels\n",
            generation[i]->forenames[j], new_string_length);
#endif

        cumulative_x += old_string_length + HORIZONTAL_SEPARATION;

        /* set the drawing coordinates for the forenames string */

        generation[i]->x[j] = cumulative_x + 0.5 * new_string_length;
        generation[i]->y[j] = - vertical_generation_separation * i;

        old_string_length = new_string_length;
#ifdef DEBUG
        printf("generation: %d  child:%d  at(%d,%d)\n",i,j,
                   generation[i]->x[j], generation[i]->y[j]);
#endif
        }

    old_string_length = longest_text_gen(i,generation[i]->location);
    cumulative_x = -0.5 * old_string_length;
    for(j=generation[i]->location - 1;j>=0;j--)
        {
        new_string_length = longest_text_gen(i,j);
        cumulative_x -=  HORIZONTAL_SEPARATION + new_string_length;

        /* set the drawing coordinates for the forenames string */

        generation[i]->x[j] = cumulative_x + 0.5 * new_string_length;
        generation[i]->y[j] = - vertical_generation_separation * i;

        old_string_length = new_string_length;
#ifdef DEBUG
        printf("generation: %d  child:%d  at(%d,%d)\n",i,j,
                   generation[i]->x[j], generation[i]->y[j]);
#endif
        }
    }

/*
   find the extent of the plot in both the +ve and -ve x directions
*/

max_x = -10000;
min_x =  10000;
for(i=0;i<number_of_generations;i++)
    {
    for(j=0;j<generation[i]->number_of_children;j++) 
        {
        if(generation[i]->x[j] + 0.5 * longest_text_gen(i,j) > max_x)
            {
            max_x = generation[i]->x[j] + 0.5 * longest_text_gen(i,j);
            }
        if(generation[i]->x[j] - 0.5 * longest_text_gen(i,j) < min_x)
            {
            min_x = generation[i]->x[j] - 0.5 * longest_text_gen(i,j);
            }
        }
    }

/* add on the margins */

max_x += ((int) VIEW_MARGIN);
min_x -= ((int) VIEW_MARGIN);

/* calculate the y range */

max_y = VIEW_MARGIN + 4 * vertical_text_separation;
min_y = - VIEW_MARGIN - (number_of_generations - 1) * vertical_generation_separation;

#ifdef DEBUG
printf("plot ranges: x in [%d,%d], y in [%d,%d]\n",min_x,max_x,min_y,max_y);
#endif

/* 
   set the offset values required to put the plot in the right place 
   on the canvas
*/

x_translate = - min_x;
y_translate = - min_y;

/* 
   set the variables which hold the current total size of the plot
*/

gen_canvas_width  = x_translate + max_x;
gen_canvas_height = y_translate + max_y;

/* 
   if the new canvas paint window size is below the max, reduce the canvas, panel
   and frame sizes otherwise set them to the max.
   allow space in the x-direction for the width of the scrollbar and 
   in the y-direction for the button panel and the width of the scrollbar
   also position the dismiss display button centrally
*/

if(gen_canvas_width < max_window_width - 25)
    {
    xv_set(gen_frame,  XV_WIDTH,  gen_canvas_width + 25, NULL);
    xv_set(gen_canvas, XV_WIDTH,  gen_canvas_width + 25, NULL);
    xv_set(gen_panel,  XV_WIDTH,  gen_canvas_width + 25, NULL);
    xv_set(dismiss_gen_button, XV_X, ((gen_canvas_width + 25)/2) - 50, NULL);
    }
else
    {
    xv_set(gen_frame,  XV_WIDTH,  max_window_width, NULL);
    xv_set(gen_canvas, XV_WIDTH,  max_window_width, NULL);
    xv_set(gen_panel,  XV_WIDTH,  max_window_width, NULL);
    xv_set(dismiss_gen_button, XV_X, (max_window_width/2) - 50, NULL);
    }

if(gen_canvas_height < max_window_height - 55)
    {
    xv_set(gen_frame,  XV_HEIGHT,  gen_canvas_height + 55, NULL);
    xv_set(gen_canvas, XV_HEIGHT,  gen_canvas_height + 25, NULL);
    xv_set(gen_panel,  XV_Y,       gen_canvas_height + 25,
                       XV_HEIGHT,  30, 
                       NULL);
    }
else
    {
    xv_set(gen_frame,  XV_HEIGHT,  max_window_height, NULL);
    xv_set(gen_canvas, XV_HEIGHT,  max_window_height - 30, NULL);
    xv_set(gen_panel,  XV_Y,       max_window_height - 30,
                       XV_HEIGHT,  30, 
                       NULL);
    }


/* set the size of the paint window appropriately */

xv_set(gen_canvas, CANVAS_WIDTH,  gen_canvas_width, NULL);
xv_set(gen_canvas, CANVAS_HEIGHT, gen_canvas_height, NULL);

/* add the offsets to all coordinate entries in the display */

for(i=0;i<number_of_generations;i++)
    {
    for(j=0;j<generation[i]->number_of_children;j++) 
        {
        generation[i]->x[j] += x_translate;
        generation[i]->y[j] += y_translate;
        }
    }
/* return, the canvas will be drawn on the next redraw callback to draw_line() */

}

/*
   calculates and returns the distance between two points
*/

int   distance(x1,y1,x2,y2)

int  x1, y1, x2, y2;
{
return (int) floor(sqrt( (double) ((x2 - x1)*(x2 - x1) + (y2 - y1)*(y2 - y1)) ));
}

/*
   generation user event in canvas callback
*/

void  user_gen_event(window, event)

Xv_window   window;
Event      *event;
{
int  i, j, min_so_far, r, gen, member;

#ifdef DEBUG
printf("user generation event procedure called\n");
#endif
/* act on the event */
switch (event_action(event))
    {
    case ACTION_SELECT:
#ifdef DEBUG
        printf("left mouse button push at (%d,%d)\n",event_x(event),event_y(event));
#endif

    /* update the text entries before a possible change of current person */

    update_text();

    /* 
       find the nearest person to the location of the button push 
       check first if there is any information available
    */

        if(generation[0] != (struct db_generation *) NULL)
            { 
            min_so_far = 10000;
            for(i=0;i<number_of_generations;i++)
                {
                for(j=0;j<generation[i]->number_of_children;j++)
                    {
                    if((r = distance(generation[i]->x[j],generation[i]->y[j],
                                     event_x(event), event_y(event))) < min_so_far)
                        {
                        min_so_far = r;
                        gen = i;
                        member = j;
                        }
                    }
                }
            /* 
               set the current person equal to the nearest person and 
               update the display if necessary
            */

            if(current_index != generation[gen]->children[member])
                {
                current_index = generation[gen]->children[member];
                update_panel_items();
                }
            }

        break;
    default:;  /* the ; is to satisfy a snotty Sun C compiler! */
#ifdef DEBUG
        printf("unknown event\n");
#endif
    }
}

/*
   descendants user event in canvas callback
*/

void  user_des_event(window, event)

Xv_window   window;
Event      *event;
{
int  i, j, min_so_far, r, des;

#ifdef DEBUG
printf("user descendants event procedure called\n");
#endif
/* act on the event */
switch (event_action(event))
    {
    case ACTION_SELECT:
#ifdef DEBUG
        printf("left mouse button push at (%d,%d)\n",event_x(event),event_y(event));
#endif

    /* update the text entries before a possible change of current person */

    update_text();

    /* 
       find the nearest person to the location of the button push 
       check first if there is any information available
    */

        if(descendant[0] != (struct db_descendant *) NULL)
            { 
            min_so_far = 10000;
            for(i=0;i<number_of_descendants;i++)
                {
                if((r = distance(descendant[i]->location.x,descendant[i]->location.y,
                                 event_x(event), event_y(event))) < min_so_far)
                    {
                    min_so_far = r;
                    des = i;
                    }
                }
            /* 
               set the current person equal to the nearest person and 
               update the display if necessary
            */

            if(current_index != descendant[des]->index)
                {
                current_index = descendant[des]->index;
                update_panel_items();
                }
            }
        break;
    default:;  /* the ; is to satisfy a snotty Sun C compiler! */
#ifdef DEBUG
        printf("unknown event\n");
#endif
    }
}


/*
   find the maximum length (in pixels) in the appropriate font of the
   text to be displayed for the specified person in the descendant struct 
*/

int  longest_text_des(des)

int  des;
{
int  longest, i;
int  length[3];

/* zero the length array (bloody 4.1.1 compiler!!) */

for(i=0;i<3;i++)
    length[i] = 0;

/* find the number of pixels needed for the strings */

(void) xv_get(font, FONT_STRING_DIMS, descendant[des]->forenames, &dimensions);
length[0] = dimensions.width;
(void) xv_get(font, FONT_STRING_DIMS, descendant[des]->surname, &dimensions);
length[1] = dimensions.width;
(void) xv_get(small_font, FONT_STRING_DIMS, descendant[des]->birth_date, &dimensions);
length[2] = dimensions.width;

/* find the largest */

longest = length[0];
for(i=1;i<3;i++)
    if(length[i] > longest)
        longest = length[i];

/* return it */

return longest;
}

/*
   order the children of the specified descendant, eldest first
*/

void  order_children_of_descendant(child_array, number_in_array)

int  *child_array, number_in_array;
{
long int  dob[MAX_CHILDREN], temp_dob;
int       i, j, temp_child;

/* load the childrens encoded birth dates into the work array */

for(i=0;i<number_in_array;i++)
    dob[i]   = entry[child_array[i]]->encoded_date;

#ifdef DEBUG
/*
printf("\nbefore sorting:\n");
for(i=0;i<number_in_array;i++)
    {
    printf("index: %d - dob: %ld\n",child_array[i], dob[i]);
    }
*/
#endif

/* simple bubble sort (not needed if there is only one child) */

if(number_in_array > 1)
    {
    for(j=0;j<number_in_array - 1;j++)
        {
        for(i=0;i<number_in_array - j - 1;i++)
            {
            if(dob[i] > dob[i+1])
                {
                temp_dob   = dob[i];
                dob[i]     = dob[i+1];
                dob[i+1]   = temp_dob;
                temp_child = child_array[i];
                child_array[i]   = child_array[i+1];
                child_array[i+1] = temp_child;
                }
            }
        }
    }

#ifdef DEBUG
/*
printf("after sorting\n");
for(i=0;i<number_in_array;i++)
    {
    printf("index: %d - dob: %ld\n",child_array[i], dob[i]);
    }
*/
#endif

}

/*
   the recursive descendants tree builder
   assumes only that descendant[index]->index has been set
   returns the coordinates of the person in descendant[struct_index]
   to be set in the level above
*/

struct point  build_node(struct_index, recursion_depth)

int  struct_index, recursion_depth;
{
struct point  return_coord;
int           children[MAX_CHILDREN], left_most_coord, right_most_coord;
int           i, j, k, person_index, counter, text_length;
int           leaf_text_length, spouse_text_length;
int           allocated, initialise, num_dummy_kids, dummy_spouse_index;
char          appendage[MAX_STRING_LENGTH];

/* read the index of the descendant */

person_index = descendant[struct_index]->index;

/* set the display information of the descendant */

strcpy(descendant[struct_index]->forenames,entry[person_index]->forenames);

/* if there is a maiden name (and it is different from their surname), use it */

if(strcmp(entry[person_index]->maiden_name,"") && 
            strcmp(entry[person_index]->surname,entry[person_index]->maiden_name))
    sprintf(descendant[struct_index]->surname,"%s (nee %s)",entry[person_index]->surname,
            entry[person_index]->maiden_name);
else
    strcpy(descendant[struct_index]->surname,entry[person_index]->surname);

/* if there is a birth date, set it */

if(strcmp(entry[person_index]->birth_date,""))
    sprintf(descendant[struct_index]->birth_date,"b. %s",entry[person_index]->birth_date);
else
    strcpy(descendant[struct_index]->birth_date,"");

/* if there is a death date, set it */

if(strcmp(entry[person_index]->death_date,""))
    sprintf(descendant[struct_index]->death_date,"d. %s",entry[person_index]->death_date);
else
    strcpy(descendant[struct_index]->death_date,"");

#ifdef DEBUG
printf("current person is %s %s (%s)\n",descendant[struct_index]->forenames,
                descendant[struct_index]->surname, descendant[struct_index]->birth_date);
#endif
 
/* set the total number of children of the descendant */

descendant[struct_index]->number_of_children = entry[person_index]->number_of_children;

/*
    if we have reached the maximum allowed depth, force this person to be a leaf 
    by setting the number of children to 0
*/

if((recursion_depth + 1) >= max_des_generations)
    descendant[struct_index]->number_of_children = 0;

/* 
   check all the children of the person to see if they have already been used
   during the creation of this tree. if so there is a loop in the tree!
*/

for(i=0;i<entry[person_index]->number_of_children;i++)
    {
    if(in_use[entry[person_index]->child[i]])
        {
        /* inform the user of the problem */
        sprintf(error_message,"Warning! - Apparent Circular Path in Requested Descendants Tree Following %s %s - Stopping Before First Repeated Person!",
                entry[person_index]->forenames,entry[person_index]->surname);
        show_error();

        /* force this person to be a leaf */
        descendant[struct_index]->number_of_children = 0;
        }
    }

/* set the number of spouses */

descendant[struct_index]->number_of_spouses = entry[person_index]->number_of_spouses;

/* 
   if there are any children, loop through the spouses, allocating them 
   to the temporary array and ordering them before allocating them to
   the permanent array
*/ 

if(descendant[struct_index]->number_of_children > 0)
    {

#ifdef DEBUG
    printf("current person has %d children\n",descendant[struct_index]->number_of_children);
#endif

#ifdef DEBUG
    printf("current person has %d spouses\n",descendant[struct_index]->number_of_spouses);
#endif

    for(i=0;i<descendant[struct_index]->number_of_spouses;i++)
        {
        /* 
           set the index of the spouse 
           these may get sorted in marriage date order in a later version
        */

        descendant[struct_index]->spouse[i] = entry[person_index]->spouse[i];

        /* set the name of the spouse */

        sprintf(descendant[struct_index]->spouse_name[i],"m. %s ",
                  entry[entry[person_index]->spouse[i]]->forenames);

        /* if the spouse has a different surname, add it on */

        if(strcmp(entry[entry[person_index]->spouse[i]]->surname,
                    entry[person_index]->surname))
            strcat(descendant[struct_index]->spouse_name[i],
                        entry[entry[person_index]->spouse[i]]->surname); 
        
        /* if there is a maiden name (and it is different from the surname), add it on */

        if(strcmp(entry[entry[person_index]->spouse[i]]->maiden_name,"") &&
                    strcmp(entry[entry[person_index]->spouse[i]]->surname,
                    entry[entry[person_index]->spouse[i]]->maiden_name))
            {
            sprintf(appendage," (nee %s)",
                    entry[entry[person_index]->spouse[i]]->maiden_name);
            strcat(descendant[struct_index]->spouse_name[i],appendage);
            }

#ifdef DEBUG
        printf("spouse (**%s**)",descendant[struct_index]->spouse_name[i]);
#endif

        /* 
           allocate the children of the currently considered spouse to the 
           temporary array
        */

        counter = 0;
        for(j=0;j<entry[person_index]->number_of_children;j++)
            {
            if(entry[person_index]->gender == MALE)
                {
                if(entry[entry[person_index]->child[j]]->mother == 
                                        entry[person_index]->spouse[i])
                    {
                    children[counter++] = entry[person_index]->child[j]; 
                    }
                }
            else
                {
                if(entry[entry[person_index]->child[j]]->father == 
                                       entry[person_index]->spouse[i])
                    {
                    children[counter++] = entry[person_index]->child[j]; 
                    }
                }
 
            }

#ifdef DEBUG
        printf(" has %d children\n",counter);
#endif

        /*
           set the number of children for this spouse 
        */

        descendant[struct_index]->num_children_of_spouse[i] = counter;

        /*
           sort them into eldest first order (if there are any!)
           if not, remove this spouse from the reckoning (will need to
           display spouses with no children in a later revision, probably
           using current_x to determine position)
        */

        if(counter > 0)
            {
            order_children_of_descendant(children, counter);

            /*
               allocate them to the permanent array
            */

            for(j=0;j<counter;j++)
                descendant[struct_index]->children_of_spouse[i][j] = children[j];

            }
        }

    /*
       need to check if there are any unallocated children. if there are, create
       an extra dummy spouse entry and attach the children to it
    */

    initialise = 1;
    for(i=0;i<entry[person_index]->number_of_children;i++)
        {
        allocated = 0;
        for(j=0;j<descendant[struct_index]->number_of_spouses;j++)
            {
            for(k=0;k<descendant[struct_index]->num_children_of_spouse[j];k++)
                {
                if(entry[person_index]->child[i] == 
                              descendant[struct_index]->children_of_spouse[j][k])
                    allocated = 1;
                }
            }
        if(!allocated)
            {
#ifdef DEBUG
            printf("child %s %s is not linked to a spouse of the current person\n",
                entry[entry[current_index]->child[i]]->forenames,
                entry[entry[current_index]->child[i]]->surname);
#endif
            if(initialise)
                /* create an extra dummy spouse entry */
                {
                descendant[struct_index]->number_of_spouses++;
                dummy_spouse_index = descendant[struct_index]->number_of_spouses - 1;
#ifdef DEBUG
                printf("number of spouses of current person set to %d\n",dummy_spouse_index + 1);
#endif
                strcpy(descendant[struct_index]->spouse_name[dummy_spouse_index],""); 
                num_dummy_kids = 0;
                descendant[struct_index]->num_children_of_spouse[dummy_spouse_index] = 
                                                      num_dummy_kids;
                initialise = 0;
                }
            /* allocate the child to the dummy array ready for ordering */
            children[num_dummy_kids++] = entry[person_index]->child[i]; 
            }
        }

    if(!initialise)
        {
        /* if there are any dummy spouse children order them by age as for the normal spouses */

        order_children_of_descendant(children, num_dummy_kids);

        /* allocate them to the permanent array, creating a dummy first spouse entry */

        descendant[struct_index]->num_children_of_spouse[dummy_spouse_index] = num_dummy_kids;
        strcpy(descendant[struct_index]->spouse_name[dummy_spouse_index],"");

        for(j=0;j<num_dummy_kids;j++)
            {
            descendant[struct_index]->children_of_spouse[dummy_spouse_index][j] = children[j];
#ifdef DEBUG
            printf("child %d of dummy spouse is %s %s\n",j,
                    entry[descendant[struct_index]->
                            children_of_spouse[dummy_spouse_index][j]]->forenames,
                    entry[descendant[struct_index]->
                            children_of_spouse[dummy_spouse_index][j]]->surname);
#endif
            }
        }
                 
    /*
       loop through the spouses, creating a descendant entry for each
       of their children, calling the current routine recursively for
       each in turn, the return code is the coordinate pair for that
       descendant
    */

    for(i=0;i<descendant[struct_index]->number_of_spouses;i++)
        {
#ifdef DEBUG
        printf("processing spouse %d (**%s**)\n",i,descendant[struct_index]->spouse_name[i]);
#endif
        for(j=0;j<descendant[struct_index]->num_children_of_spouse[i];j++)
            {
#ifdef DEBUG
            printf("processing child %d (**%s %s**)\n",j,
                    entry[descendant[struct_index]->
                            children_of_spouse[i][j]]->forenames,
                    entry[descendant[struct_index]->
                            children_of_spouse[i][j]]->surname);
#endif
            /* allocate memory for the structure */

            if((descendant[number_of_descendants++] = 
               (struct db_descendant *) calloc((size_t) 1, 
                             (size_t) sizeof(struct db_descendant))) == NULL)
                {
                /* big trouble, try to free allocated space so far, and warn user */
                for(k=0;k<MAX_INDICES;k++)
                    {
                    if(descendant[k] != (struct db_descendant *) NULL)
                        {
                        free(descendant[k]);
                        descendant[k] = (struct db_descendant *) NULL;
                        }
                    }
                strcpy(error_message,"Unable to Allocate Display Memory - Suggest Back Up Current Work and Bail Out!");
                show_error();
                return return_coord; /* RETURNS WHAT????? */
                }

            /* attach the newly created child struct to the current struct */

            descendant[struct_index]->children[i][j] = 
                            descendant[number_of_descendants - 1];

            /* set its index entry in the main database */

            descendant[struct_index]->children[i][j]->index = 
                            descendant[struct_index]->children_of_spouse[i][j];

            /* call the recursive routine to set the coordinates of each child */

#ifdef DEBUG
            printf("calling build_node for %s %s at depth %d\n",
                entry[descendant[struct_index]->children_of_spouse[i][j]]->forenames,
                entry[descendant[struct_index]->children_of_spouse[i][j]]->surname,
                recursion_depth + 1);
#endif

            descendant[struct_index]->children[i][j]->location = 
                          build_node(number_of_descendants - 1, recursion_depth + 1);
#ifdef DEBUG
            printf("returned from build node\nplacing %s %s at coords (%d,%d)\n",
                    descendant[struct_index]->children[i][j]->forenames, 
                    descendant[struct_index]->children[i][j]->surname,
                    descendant[struct_index]->children[i][j]->location.x, 
                    descendant[struct_index]->children[i][j]->location.y);
#endif

            /* check the coords limits */

            if((descendant[struct_index]->children[i][j]->location.x + 
                  0.5 * longest_text_des(number_of_descendants - 1)) > max_desc_x)
                max_desc_x = descendant[struct_index]->children[i][j]->location.x + 
                    0.5 * longest_text_des(number_of_descendants - 1);
            if((descendant[struct_index]->children[i][j]->location.y + 
                    vertical_generation_separation) > max_desc_y)
                max_desc_y = descendant[struct_index]->children[i][j]->location.y + 
                    vertical_generation_separation;

            }
        /* 
           set the coordinates of the spouse 
           the x-coord is set to be the average of the first and last of
           their children (if there were any children) 
           otherwise use the value of current_x and increment it
        */

        if(descendant[struct_index]->num_children_of_spouse[i] > 0)
            {
            descendant[struct_index]->spouse_location[i].x = 
                (descendant[struct_index]->children[i][0]->location.x +
                descendant[struct_index]->children[i][descendant[struct_index]->
                           num_children_of_spouse[i] - 1]->location.x)/2;
            descendant[struct_index]->spouse_location[i].y = 
                (recursion_depth * vertical_generation_separation) + 
                (4 * vertical_text_separation);
#ifdef DEBUG
            printf("spouses 1st  child is at %d\n",
                    descendant[struct_index]->children[i][0]->location.x);
            printf("spouses last child is at %d\n",
                    descendant[struct_index]->children[i][descendant[struct_index]->
                           num_children_of_spouse[i] - 1]->location.x);
            printf("so placing spouse %d (who has children) at (%d,%d)\n",i,
                    descendant[struct_index]->spouse_location[i].x,
                    descendant[struct_index]->spouse_location[i].y);
#endif
            }
        else
            {
            /* 
               find the length of the spouse's name and use this
               to calculate the spouse's position and the increment 
               to current_x
            */
            (void) xv_get(small_font, FONT_STRING_DIMS, 
                        descendant[struct_index]->spouse_name[i], &dimensions);
            text_length = dimensions.width;

            /* check that text length is >= the minimum allowed */

            if(text_length < min_text_width)
                text_length = min_text_width;

            current_x += 0.5 * text_length;

            descendant[struct_index]->spouse_location[i].x = current_x;
            descendant[struct_index]->spouse_location[i].y = 
                (recursion_depth * vertical_generation_separation) + 
                (4 * vertical_text_separation);

            /* move current_x on including the horizontal spacing factor */

            current_x += 0.5 * text_length + HORIZONTAL_SEPARATION;

#ifdef DEBUG
            printf("placing spouse %d (who has no children) at (%d,%d)\n",i,
                    descendant[struct_index]->spouse_location[i].x,
                    descendant[struct_index]->spouse_location[i].y);
#endif
            }
        }
    /*
      return the coords of the current person
      the value will be something like the average of the highest and lowest
      child coords which should all have been set by now
    */
#ifdef DEBUG
    printf("dealt with all children of %s %s\n",
            descendant[struct_index]->forenames, descendant[struct_index]->surname);
#endif

    /* 
        find the extremes of the positions of the children of the current person
        if either the first or the last spouse has no children by the current
        person then the spouse's location is used
    */

#ifdef DEBUG
    printf("number of children of first spouse is %d\n", descendant[struct_index]->
               num_children_of_spouse[0]);
#endif
    if(descendant[struct_index]->num_children_of_spouse[0] != 0)
        left_most_coord = descendant[struct_index]->children[0][0]->location.x;
    else
        left_most_coord = descendant[struct_index]->spouse_location[0].x;

#ifdef DEBUG
    printf("left most coord x-coord set to %d\n",left_most_coord);
    printf("number of children of last spouse is %d\n", descendant[struct_index]->
               num_children_of_spouse[descendant[struct_index]->number_of_spouses - 1]);
#endif
    if(descendant[struct_index]->
            num_children_of_spouse[descendant[struct_index]->number_of_spouses - 1] != 0)
        /* bit of a monster, this next line! sorry! */
        right_most_coord = descendant[struct_index]->
                children[descendant[struct_index]->number_of_spouses - 1]
                [descendant[struct_index]->num_children_of_spouse[descendant[struct_index]->
                number_of_spouses - 1] - 1]->location.x;
    else
        right_most_coord = descendant[struct_index]->
                spouse_location[descendant[struct_index]->number_of_spouses - 1].x;

#ifdef DEBUG
    printf("right most coord x-coord set to %d\n",right_most_coord);
#endif

    return_coord.x = (left_most_coord + right_most_coord)/2;

    return_coord.y = recursion_depth * vertical_generation_separation;
#ifdef DEBUG
    printf("returning coords (%d,%d)\n", return_coord.x, return_coord.y);
#endif
    return return_coord;
    }
else
    {
    /*
       the current person is a leaf!
       the difficult bit! allocate coordinates to this child and return them
       they will be set in the call above this
    */
    /*
       loop through any spouses of the leaf person, allocating
       coordinates based on current_x
    */
    for(i=0;i<descendant[struct_index]->number_of_spouses;i++)
        {
        /* 
           set the index of the spouse 
           these may get sorted in marriage date order in a later version
        */

        descendant[struct_index]->spouse[i] = entry[person_index]->spouse[i];

        /* set the name of the spouse */

        sprintf(descendant[struct_index]->spouse_name[i],"m. %s ",
                  entry[entry[person_index]->spouse[i]]->forenames);

        /* if the spouse has a different surname, add it on */

        if(strcmp(entry[entry[person_index]->spouse[i]]->surname,
                    entry[person_index]->surname))
            strcat(descendant[struct_index]->spouse_name[i],
                        entry[entry[person_index]->spouse[i]]->surname); 
        
        /* if there is a maiden name (and it is different from the surname), add it on */

        if(strcmp(entry[entry[person_index]->spouse[i]]->maiden_name,"") &&
                    strcmp(entry[entry[person_index]->spouse[i]]->surname,
                    entry[entry[person_index]->spouse[i]]->maiden_name))
            {
            sprintf(appendage," (nee %s)",
                    entry[entry[person_index]->spouse[i]]->maiden_name);
            strcat(descendant[struct_index]->spouse_name[i],appendage);
            }

#ifdef DEBUG
        printf("spouse (**%s**)",descendant[struct_index]->spouse_name[i]);
#endif
        /* 
           find the length of the spouse's name and use this
           to calculate the spouse's position and the increment 
           to current_x (by the width of the leaf's text or the
           spouses, whichever is larger)
        */
        leaf_text_length = longest_text_des(struct_index);
        (void) xv_get(small_font, FONT_STRING_DIMS, 
            descendant[struct_index]->spouse_name[i], &dimensions);
        spouse_text_length = dimensions.width;
        text_length = leaf_text_length > spouse_text_length ? leaf_text_length :
                                                                 spouse_text_length;

        /* check that text length is >= the minimum allowed */

        if(text_length < min_text_width)
            text_length = min_text_width;

        current_x += 0.5 * text_length;

        descendant[struct_index]->spouse_location[i].x = current_x;
        descendant[struct_index]->spouse_location[i].y = 
            (recursion_depth * vertical_generation_separation) + 
            (4 * vertical_text_separation);

        /* move current_x on including the horizontal spacing factor */

        current_x += 0.5 * text_length + HORIZONTAL_SEPARATION;

#ifdef DEBUG
        printf("placing spouse %d (who has no children) at (%d,%d)\n",i,
                descendant[struct_index]->spouse_location[i].x,
                descendant[struct_index]->spouse_location[i].y);
#endif
        }

    /*
       if there were any spouses, set the current persons position
       to the middle position, otherwise use the value of current_x
       and the width of the text entry for the person
    */
    if(descendant[struct_index]->number_of_spouses > 0)
        {
        return_coord.x = (descendant[struct_index]->spouse_location[0].x + 
                   descendant[struct_index]->spouse_location[descendant[struct_index]->
                   number_of_spouses - 1].x)/2;
        return_coord.y = recursion_depth * vertical_generation_separation;
        }
    else
        {
        /* 
           find the longest piece of text in the persons entry and use this
           to calculate the person's position
        */
    
        text_length = longest_text_des(struct_index);

        /* check that text length is >= the minimum allowed */

        if(text_length < min_text_width)
            text_length = min_text_width;

        current_x += 0.5 * text_length;

        return_coord.x = current_x;
        return_coord.y = recursion_depth * vertical_generation_separation;

        /* move current_x on including the horizontal spacing factor */

        current_x += 0.5 * text_length + HORIZONTAL_SEPARATION;
        }

#ifdef DEBUG
    printf("%s %s has no children (is a leaf), returning coords (%d,%d)\n",
            descendant[struct_index]->forenames, descendant[struct_index]->surname,
            return_coord.x, return_coord.y);
#endif
    return return_coord;
    }
}

/*
   the recursive descendants tree drawer 
*/

void  draw_node(person, dpy, xwin)

struct db_descendant  *person;
Display        *dpy;
Window          xwin;
{
int  person_index, i, j;

/* read the index of the descendant */

person_index = person->index;

/* 
   display the information of the descendant
*/

/* add the forenames if not empty, otherwise put a line in */

if(strcmp(person->forenames,""))
    {
    write_centred(dpy, xwin, person->location.x,person->location.y,person->forenames, 1);
    }
else
    {
    XDrawLine(dpy, xwin, gc, 
              person->location.x, 
              person->location.y + 
                  ((int) floor(-0.8 * ((double) vertical_text_separation))),
              person->location.x,
              person->location.y + 
                  ((int) floor(0.2 * ((double) vertical_text_separation))));
    }


/* add the surname if not empty, otherwise put a line in */
if(strcmp(person->surname,""))
    {
    write_centred(dpy, xwin, person->location.x, 
            person->location.y + 1 * vertical_text_separation, 
            person->surname, 1);
    }
else
    {
    XDrawLine(dpy, xwin, gc, 
              person->location.x, 
              person->location.y + 
                  ((int) floor(0.2 * ((double) vertical_text_separation))),
              person->location.x,
              person->location.y + 
                  ((int) floor(1.2 * ((double) vertical_text_separation))));

    }


/* 
   add the date of birth of known, otherwise try date of death, 
   otherwise put a line in 
*/
if(strcmp(person->birth_date,""))
    {
    write_centred(dpy, xwin, person->location.x, 
            person->location.y + 2 * vertical_text_separation, 
            person->birth_date, 0);
    }
else
    {
    if(strcmp(person->death_date,""))
        {
        write_centred(dpy, xwin, person->location.x, 
                person->location.y + 2 * vertical_text_separation, 
                person->death_date, 0);
        }
    else
        {
#ifdef DEBUG
        printf("no birth-date or death-date for %s %s - drawing line from (%d,%d) to (%d,%d)\n",
            person->forenames,person->surname,person->location.x,
                  person->location.y + 
                      ((int) floor(1.2 * ((double) vertical_text_separation))),
                  person->location.x,
                  person->location.y + 
                      ((int) floor(2.2 * ((double) vertical_text_separation))));
#endif
        XDrawLine(dpy, xwin, gc, 
                  person->location.x, 
                  person->location.y + 
                      ((int) floor(1.2 * ((double) vertical_text_separation))),
                  person->location.x,
                  person->location.y + 
                      ((int) floor(2.2 * ((double) vertical_text_separation))));
        }
    }

/* 
   loop through the spouses, displaying the relevant info 
*/ 

/* draw the little stub down from the person to the spouse across line */

if(person->number_of_spouses > 0)
    {
    XDrawLine(dpy, xwin, gc, 
              person->location.x, 
              person->location.y + 
                  ((int) floor(2.2 * ((double) vertical_text_separation))),
              person->location.x,
              person->location.y + 
                  ((int) floor(2.5 * ((double) vertical_text_separation))));
    }

if(person->number_of_spouses > 0)
    {
    /* 
       actually there should always be at least one spouse if there are any
       children, even if it is only a single dummy entry
    */
    /* draw a line above the spouses from first to last */

    XDrawLine(dpy, xwin, gc, 
            person->spouse_location[0].x, 
            person->spouse_location[0].y - 
                ((int) floor((double) 1.5 * vertical_text_separation)),
            person->spouse_location[person->number_of_spouses - 1].x,
            person->spouse_location[person->number_of_spouses - 1].y - 
                ((int) floor((double) 1.5 * vertical_text_separation)));
#ifdef DEBUG
    printf("spouses (0 to %d) line drawn from (%d,%d) to (%d,%d)\n",
            person->number_of_spouses - 1,
            person->spouse_location[0].x, 
            person->spouse_location[0].y - 
                ((int) floor((double) 1.5 * vertical_text_separation)),
            person->spouse_location[person->number_of_spouses - 1].x,
            person->spouse_location[person->number_of_spouses - 1].y - 
                ((int) floor((double) 1.5 * vertical_text_separation)));
#endif

    for(i=0;i<person->number_of_spouses;i++)
        {
        /* 
           if there is actually a spouse (not a dummy) display the name 
           otherwise draw a line through
        */

        if(strcmp(person->spouse_name[i],""))
            {
            write_centred(dpy, xwin, person->spouse_location[i].x, 
                    person->spouse_location[i].y, person->spouse_name[i], 0);
            }
        else
            {
#ifdef DEBUG
            printf("dummy spouse, drawing line from (%d,%d) to (%d,%d)\n",
                  person->spouse_location[i].x, 
                  person->spouse_location[i].y + 1 * vertical_text_separation,
                  person->spouse_location[i].x, 
                  person->spouse_location[i].y - 1 * vertical_text_separation);
#endif
            XDrawLine(dpy, xwin, gc, 
                  person->spouse_location[i].x, 
                  person->spouse_location[i].y + 1 * vertical_text_separation,
                  person->spouse_location[i].x, 
                  person->spouse_location[i].y - 1 * vertical_text_separation);
            }

        /* draw the stub under the spouse */

#ifdef DEBUG
        printf("drawing spouse under stub from (%d,%d) to %d,%d)\n",
            person->spouse_location[i].x, 
            person->spouse_location[i].y + 
                ((int) floor(0.25 *((double) vertical_text_separation))),
            person->spouse_location[i].x, 
            person->spouse_location[i].y + 
                ((int) floor(1.0 *((double) vertical_text_separation))));
#endif
        XDrawLine(dpy, xwin, gc, 
            person->spouse_location[i].x, 
            person->spouse_location[i].y + 
                ((int) floor(0.25 *((double) vertical_text_separation))),
            person->spouse_location[i].x, 
            person->spouse_location[i].y + 
                ((int) floor(1.0 *((double) vertical_text_separation))));
          
            /* draw the stub over the spouse */

#ifdef DEBUG
        printf("drawing spouse over stub from (%d,%d) to %d,%d)\n",
            person->spouse_location[i].x, 
            person->spouse_location[i].y - 
                ((int) floor(0.9 *((double) vertical_text_separation))),
            person->spouse_location[i].x, 
            person->spouse_location[i].y -
                ((int) floor(1.5 *((double) vertical_text_separation))));
#endif
        XDrawLine(dpy, xwin, gc, 
            person->spouse_location[i].x, 
            person->spouse_location[i].y - 
                ((int) floor(1.0 *((double) vertical_text_separation))),
            person->spouse_location[i].x, 
            person->spouse_location[i].y -
                ((int) floor(1.5 *((double) vertical_text_separation))));
          

        /* draw the children`s cross bar (if there are any) */

        if(person->num_children_of_spouse[i] > 0)
            {
#ifdef DEBUG
            printf("drawing child cross bar from (%d,%d) to %d,%d)\n",
                  person->children[i][0]->location.x, 
                  person->spouse_location[i].y + vertical_text_separation,
                  person->children[i][person->num_children_of_spouse[i] - 1]->location.x,
                  person->spouse_location[i].y + vertical_text_separation);
#endif
            XDrawLine(dpy, xwin, gc, 
                  person->children[i][0]->location.x, 
                  person->spouse_location[i].y + vertical_text_separation,
                  person->children[i][person->num_children_of_spouse[i] - 1]->location.x,
                  person->spouse_location[i].y + vertical_text_separation);
            }
          
        /* loop through the children, plotting them and so on and on and on... */

        for(j=0;j<person->num_children_of_spouse[i];j++)
            {
            /* draw the stub to the cross bar above the child */

            XDrawLine(dpy, xwin, gc, 
                person->children[i][j]->location.x, 
                person->children[i][j]->location.y - 
                  ((int) floor(1.0 *((double) vertical_text_separation))),
                person->children[i][j]->location.x, 
                person->children[i][j]->location.y -
                  ((int) floor(3.0 *((double) vertical_text_separation))));
          
            draw_node(person->children[i][j], dpy, xwin);
            }
        }
    }
else
    {
    /*
       if there are no known spouses then simply plot the children
    */

    for(j=0;j<person->number_of_children;j++)
        draw_node(person->children[0][j], dpy, xwin);

    }
}

/* 
   the callback for the descendants tree drawer 
*/

void  draw_descendants(canvas, paint_window, dpy, xwin, area)

Canvas          canvas;
Xv_Window       paint_window;
Display        *dpy;
Window          xwin;
Xv_xrectlist   *area;
{
int   i, j;
/* 
   first check that there is some data to plot i.e. at least one
   display struct is non-NULL (they are used sequentially, so
   if the first one is NULL they all are
*/

if(descendant[0] == (struct db_descendant *) NULL)
    {
#ifdef DEBUG
    printf("descendants canvas redraw procedure called but no data available\n");
#endif
    return;
    }

#ifdef DEBUG
    printf("descendants canvas redraw procedure called\n");
#endif

/* synchronise X calls */

#ifdef DEBUG
/*
XSynchronize(dpy, TRUE);
*/
#endif

/* clear the window */

XClearWindow(dpy, xwin); 

/* call the recursive draw routine */

draw_node(descendant[0], dpy, xwin);

}

/*
   build the data structures required for displaying all the 
   descendants of the current person
*/

void  build_descendants()
{
struct point   position;
int            i, j, recursion_depth;
int            x_translate, y_translate;

/* zero the in use array */

for(i=0;i<MAX_INDICES;i++)
    in_use[i] = 0;

/* 
   calculate the vertical separation between text items and
   generations based on the larger font size 
*/

(void) xv_get(font, FONT_STRING_DIMS, "X", &dimensions);
vertical_text_separation = dimensions.height + DELTA_Y;
vertical_generation_separation = VERTICAL_SEPARATION * vertical_text_separation;

#ifdef DEBUG
printf("called build_descendants\n************************\n");
printf("vertical text separation = %d\n",vertical_text_separation);
#endif

/* empty any existing descendants display structures */

for(i=0;i<MAX_INDICES;i++)
    {
    if(descendant[i] != (struct db_descendant *) NULL)
        {
        free(descendant[i]);
        descendant[i] = (struct db_descendant *) NULL;
        }
    }

/* create a descendant entry for the current person */

number_of_descendants = 0;
if((descendant[number_of_descendants++] = (struct db_descendant *) calloc((size_t) 1, 
                       (size_t) sizeof(struct db_descendant))) == NULL)
    {
    /* 
        something has gone badly wrong so try to tidy up as best
        as possible and suggest the user backs up and bails out!
    */
    strcpy(error_message,"Unable to Allocate Display Memory - Suggest Back Up Current Work and Bail Out!");
    show_error();
    return;
    }
descendant[0]->index = current_index;
in_use[current_index] = 1;

/* 
   initialise the x position indicator for leaf values (leaves are at the
   end of the branches) the actual value is arbitrary since they will be
   translated to centre them on the canvas
*/

current_x = VIEW_MARGIN;

/* 
   initialise the max and min values of the plot in each direction 
   the min values won't change and therefore don't need checking
*/

min_desc_x = 0;
min_desc_y = - VIEW_MARGIN;
max_desc_x = max_desc_y =  -10000;

/*
   call the recursive descendant tree builder with the current person's struct index! 
   the function returns the position of the entry in a point struct
*/

recursion_depth = 0;

#ifdef DEBUG
printf("calling build_node for %s %s at depth %d\n",entry[current_index]->forenames,
                    entry[current_index]->surname,recursion_depth);
#endif

descendant[0]->location = build_node(0, recursion_depth);

#ifdef DEBUG
printf("placing %s %s at coords (%d,%d)\n",
        descendant[0]->forenames, descendant[0]->surname,
        descendant[0]->location.x, descendant[0]->location.y);
#endif

/* check the coords limits for all descendants */

for(i=0;i<number_of_descendants;i++)
    {
    if((descendant[i]->location.x + 0.5 * longest_text_des(i)) > max_desc_x)
        max_desc_x = descendant[i]->location.x + 
                         (int) floor((double) 0.5 * ((float) longest_text_des(i)));
    if((descendant[i]->location.y + vertical_generation_separation) > max_desc_y)
        max_desc_y = descendant[i]->location.y + 2 * vertical_text_separation;

    /* also check the spouses x locations */
    for(j=0;j<descendant[i]->number_of_spouses;j++)
        {
        (void)xv_get(small_font, FONT_STRING_DIMS, 
                     descendant[i]->spouse_name[j], &dimensions);
        if((descendant[i]->spouse_location[j].x + 0.5 * dimensions.width) > max_desc_x)
            max_desc_x = descendant[i]->spouse_location[j].x + 
                             (int) floor((double) 0.5 * dimensions.width);
        }
    }

/* 
   set the variables which hold the current total size of the plot
*/

/* add the view margin to the max values */

max_desc_x += VIEW_MARGIN;
max_desc_y += VIEW_MARGIN;

des_canvas_width  = max_desc_x - min_desc_x;
des_canvas_height = max_desc_y - min_desc_y;

#ifdef DEBUG
printf("min, max x = %d, %d\n",min_desc_x,max_desc_x);
printf("min, max y = %d, %d\n",min_desc_y,max_desc_y);
#endif

/* 
   set the offset values required to put the plot in the right place 
   on the canvas
*/

x_translate = - min_desc_x;
y_translate = - min_desc_y;

/* add the offsets to all coordinates in the plot */

for(i=0;i<number_of_descendants;i++)
    {
    descendant[i]->location.x += x_translate;
    descendant[i]->location.y += y_translate;
    for(j=0;j<descendant[i]->number_of_spouses;j++)
        {
        descendant[i]->spouse_location[j].x += x_translate;
        descendant[i]->spouse_location[j].y += y_translate;
        }
    }

/* 
   if the new canvas paint window size is below the max, reduce the canvas, panel
   and frame sizes otherwise set them to the max.
   allow space in the x-direction for the width of the scrollbar and 
   in the y-direction for the button panel and the width of the scrollbar
   also position the dismiss display button centrally
*/

if(des_canvas_width < max_window_width - 25)
    {
    xv_set(des_frame,  XV_WIDTH,  des_canvas_width + 25, NULL);
    xv_set(des_canvas, XV_WIDTH,  des_canvas_width + 25, NULL);
    xv_set(des_panel,  XV_WIDTH,  des_canvas_width + 25, NULL);
    xv_set(dismiss_des_button, XV_X, ((des_canvas_width + 25)/2) - 50, NULL);
    }
else
    {
    xv_set(des_frame,  XV_WIDTH,  max_window_width, NULL);
    xv_set(des_canvas, XV_WIDTH,  max_window_width, NULL);
    xv_set(des_panel,  XV_WIDTH,  max_window_width, NULL);
    xv_set(dismiss_des_button, XV_X, (max_window_width/2) - 50, NULL);
    }

if(des_canvas_height < max_window_height - 55)
    {
    xv_set(des_frame,  XV_HEIGHT,  des_canvas_height + 55, NULL);
    xv_set(des_canvas, XV_HEIGHT,  des_canvas_height + 25, NULL);
    xv_set(des_panel,  XV_Y,       des_canvas_height + 25,
                       XV_HEIGHT,  30, 
                       NULL);
    }
else
    {
    xv_set(des_frame,  XV_HEIGHT,  max_window_height, NULL);
    xv_set(des_canvas, XV_HEIGHT,  max_window_height - 30, NULL);
    xv_set(des_panel,  XV_Y,       max_window_height - 30,
                       XV_HEIGHT,  30, 
                       NULL);
    }

/* set the size of the paint window appropriately */

xv_set(des_canvas, CANVAS_WIDTH,  des_canvas_width, NULL);
xv_set(des_canvas, CANVAS_HEIGHT, des_canvas_height, NULL);

#ifdef DEBUG
printf("paint window size set to %d by %d in descendants builder\n",des_canvas_width,des_canvas_height);
#endif

/* diagnostic */
/*dump_descendants(); */
}

/*
   when a descent view is split, this attaches the event and repaint callbacks
*/

void  new_view_gen(orig_view, new_view, pos)
Xv_Window  orig_view, new_view;
int        pos;
{
Xv_Window   paint_window;

#ifdef DEBUG
printf("setting descent line callbacks to new view\n");
#endif

/* locate the paint window of the new view */

paint_window = (Xv_Window) xv_get(new_view, CANVAS_VIEW_PAINT_WINDOW);

/* attach the callbacks and event mask */

xv_set(paint_window,
       CANVAS_REPAINT_PROC,        draw_line,
       WIN_EVENT_PROC,             user_gen_event,
       WIN_X_EVENT_MASK,
           ButtonPressMask, NULL,
       NULL);
}

/*
   when a descendants view is split, this attaches the event and repaint callbacks
*/

void  new_view_des(orig_view, new_view, pos)
Xv_Window  orig_view, new_view;
int        pos;
{
Xv_Window   paint_window;

#ifdef DEBUG
printf("setting descendants tree callbacks to new view\n");
#endif

/* locate the paint window of the new view */

paint_window = (Xv_Window) xv_get(new_view, CANVAS_VIEW_PAINT_WINDOW);

/* attach the callbacks and event mask */

xv_set(paint_window,
       CANVAS_REPAINT_PROC,        draw_descendants,
       WIN_EVENT_PROC,             user_des_event,
       WIN_X_EVENT_MASK,
           ButtonPressMask, NULL,
       NULL);
}

