/*
** Program fdb - family tree database generation and manipulation
**
** Copyright (C) 1994 Andy Burrows 
**
**            email: cadellin@corum.me.man.ac.uk (130.88.29.14)
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public Licence as published by the Free
** Software Foundation; either version 1, or any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
** or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
** more details. It should be included in this distribution in a file 
** called COPYING
**
*/

/*
   this file contains the family tree database functions
*/

/*#define DEBUG */

/* standard headers */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

/* XView headers */

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/font.h>

/* fdb database definitions */

#include "definitions.h"

/*
   show the current error message in the error display window
   sorting out the size of the window and positioning of its contents first
*/

void  show_error()

{
int  error_panel_width;

/* set the error message item */
xv_set(error_message_item, PANEL_LABEL_STRING, error_message, NULL);

/* reset the width of the display panel to accomodate the new string */
window_fit_width(error_panel);

/* read in the new panel width */
error_panel_width = (int) xv_get(error_panel, XV_WIDTH);

/* locate the OK button in the middle of the new panel */
xv_set(error_ok_button, XV_X, (error_panel_width/2) - 45, NULL);

/* fit the frame around the panel */
window_fit(error_frame);

/* display the error */
xv_set(error_frame, XV_SHOW, TRUE, NULL);
}

/*
   take a date string and encode it as a long int YYYYMMDD 
   where YYYY is the year
         MM   is the month
         DD   is the day
   returns -1 if it can't get anything out
   returns  0 if string is empty
*/

long int  encode_date(string)

char  *string;
{
long int  day, month, year, encoded, temp_day, temp_month, temp_year, temp;
int       i, j, num_items;
char      working[MAX_DATE], substring[MAX_DATE], testing[MAX_DATE];

/* initialise the variables */

day = month = year = encoded = -1;

#ifdef DEBUG
/* printf("attempting to encode date string: %s\n",string); */
#endif

/* check for the empty string */

if(!strcmp(string,""))
    return 0;

/* read the date string into the work space, converting to lower case on the way */

i = 0;
while((working[i] = tolower(string[i])) != '\0')
    i++;

#ifdef DEBUG
/* printf("translated to: %s\n",string); */
#endif

/* read in substrings and try to work out what they are */

j = 0;

while( ((num_items = sscanf((working + j),"%s",substring)) != EOF) &&
       (j < (int) strlen(working)) )
    {

    /* increment the distance along the string counter */

    j += strlen(substring) + 1;

#ifdef DEBUG
/*    printf("substring: %s\n",substring); */
#endif

    /* first check for the simplest possible options */

    strcpy(testing,substring);

    if((num_items = sscanf(testing,"%ld/%ld/%ld",&temp_day,&temp_month,&temp_year)) == 3)
        {
        /* check if values are sensible */
        if((temp_day <= 31) && (temp_day >= 1))
            {
            day = temp_day;
            }
        if((temp_month <= 12) && (temp_month >= 1))
            {
            month = temp_month;
            }
        if(temp_year < 9999)    /* this will break in the year 9999!! */
            {
            year = temp_year;
            }
        }

    strcpy(testing,substring);

    if((num_items = sscanf(testing,"%ld\\%ld\\%ld",&temp_day,&temp_month,&temp_year)) == 3)
        {
        /* check if values are sensible */
        if((temp_day <= 31) && (temp_day >= 1))
            {
            day = temp_day;
            }
        if((temp_month <= 12) && (temp_month >= 1))
            {
            month = temp_month;
            }
        if(temp_year < 9999)    /* this will break in the year 9999!! */
            {
            year = temp_year;
            }
        }

    /* now get a bit more clever! */

    strcpy(testing,substring);

    if(strstr(testing,"jan") != NULL)
        month = 1;
    else if(strstr(testing,"feb") != NULL)
        month = 2;
    else if(strstr(testing,"mar") != NULL)
        month = 3;
    else if(strstr(testing,"apr") != NULL)
        month = 4;
    else if(strstr(testing,"may") != NULL)
        month = 5;
    else if(strstr(testing,"jun") != NULL)
        month = 6;
    else if(strstr(testing,"jul") != NULL)
        month = 7;
    else if(strstr(testing,"aug") != NULL)
        month = 8;
    else if(strstr(testing,"sep") != NULL)
        month = 9;
    else if(strstr(testing,"oct") != NULL)
        month = 10;
    else if(strstr(testing,"nov") != NULL)
        month = 11;
    else if(strstr(testing,"dec") != NULL)
        month = 12;

    strcpy(testing,substring);

/*
    if(strstr(testing,"1st") != NULL)
        day = 1;
    else if(strstr(testing,"2nd") != NULL)
        day = 2;
    else if(strstr(testing,"3rd") != NULL)
        day = 3;
    else if(strstr(testing,"21st") != NULL)
        day = 21;
    else if(strstr(testing,"22nd") != NULL)
        day = 22;
    else if(strstr(testing,"23rd") != NULL)
        day = 23;
    else if(strstr(testing,"31st") != NULL)
        day = 31;

    strcpy(testing,substring);
*/

    if((num_items = sscanf(testing,"%ld",&temp)) == 1)
        {
        /* 
            the following check line will break in the year 9999
        */
        if((temp > 0) && (temp < 9999))
            {
            /* 
               if there's a th, an st, an nd or an rd in the string, 
               then it's probably a day 
            */
            if((strstr(testing,"th") != NULL) ||
               (strstr(testing,"st") != NULL) ||
               (strstr(testing,"nd") != NULL) ||
               (strstr(testing,"rd") != NULL))
                {
                if(temp < 32)
                    day = temp;
                }
            else
                {
                /* if its too big to be a day, make it a year */
                if(temp > 31)
                    {
                    year = temp;
                    }
                else
                    /* it could be either, so make some intelligent(!) guesses */
                    {
                    if(day != -1)
                        {
                        /* if we already have a day and no year, call it a year */
                        if(year == -1)
                            {
                            year = temp;
                            }
                        }
                    else
                        {
                        /* if we have no day, call it a day */
                        day = temp;
                        }
                    }
                }
            }
        }
#ifdef DEBUG
/*
    printf("day: %ld\n",day);
    printf("month: %ld\n",month);
    printf("year: %ld\n",year);
*/
#endif
    }

/* make sure there is a century attached to the year */

if((year < 100) && (year >= 0))
   year += 1900;  /* assume this century if not specified */

#ifdef DEBUG
/*    printf("year: %ld\n",year); */
#endif

/* form the encode date and return it */

/* if we couldn't find the year, set it to 0! */

if(year == -1)
    year = 0;

/* add the month and day info if available */

encoded = year * 10000;
if(month != -1)
    encoded += month * 100;
if(day != -1)
    encoded += day;
    
#ifdef DEBUG
/* printf("encoded date: %ld\n",encoded); */
#endif

return encoded;
}

/*
   free up the memory used by the database
*/

void  free_database()
{
int   i;

for(i=0;i<MAX_INDICES;i++)
    if(entry[i] != (struct db_record *) NULL)
        {
        free(entry[i]);
        entry[i] = (struct db_record *) NULL;
        }

/* zero the database variables */

num_entries = max_index = current_index = 0;

/* create the new status display string and set it */

strcpy(status_string,"Currently 0 Entries in Database");
xv_set(fdb_frame, FRAME_RIGHT_FOOTER, status_string, NULL);

}

/* 
   initialise the database, from the specified file if available 
*/

void  initialise_database()
{
FILE   *fp;
int     i, j, index, num_items_read, dummy;
char    dummy_string[255];

/* check new file is readable */

if(strcmp(new_file_name,"") != 0)
    {
    if((fp = fopen(new_file_name,"r")) == NULL)
        {
        /* 
           if this is the very first time in here i.e. starting up, then
           exit the program, otherwise just return and carry on with the
           current database
        */
        if(first_time)
            {
            printf("fdb V%s - unable to open file %s\n",FDB_VERSION,new_file_name);
            exit(1);
            }
        else
            {
            /* indicate the failure */
            sprintf(error_message,"Unable to Open File: %s",new_file_name);
            show_error();
            return;
            }
        }
    }

/* unset the first time flag */

first_time = 0;

/* copy the new file name to the current file name */

strcpy(file_name, new_file_name);

/* free up the database entries ready for re-use */

free_database();

if(strcmp(file_name,"") != 0)
    {
    /*
       read in the first line and search for the string "cadellinsoft"
    */

    fgets(dummy_string, 255, fp);
    if(strstr(dummy_string,"cadellinsoft") == NULL)
        {
        sprintf(error_message,"file %s does not appear to a database file!",file_name);
        show_error();
        goto FILE_FORMAT_WRONG;
        }

    /* 
       read in the number of entries in the database, the maximum database index 
       and the index of the current person
    */

    num_items_read = fscanf(fp,
        "number of entries:%d max index in use:%d current active index:%d\n",
        &num_entries,&max_index,&current_index);
#ifdef DEBUG
    printf("number of entries: %d, max index in use: %d, current active index: %d\n",
        num_entries, max_index, current_index);
#endif 
    if((num_items_read == EOF) || (num_items_read < 3))
        {
        sprintf(error_message,"error reading first line of database file %s, bailing out!\n", file_name);
        show_error();
        goto FILE_FORMAT_WRONG;
        }

    /*
       check that the values read in are sensible
    */

    /*
       check number of children does not exceed max. (could have been created with
       another version of FDB with a bigger MAX_CHILDREN)
    */

    if(num_entries > ((int) MAX_INDICES) )
        {
        printf("There appear to be %d entries in database file %s, increase MAX_INDICES to at least %d in definitions.h and recompile\n",
            num_entries, file_name, num_entries);
        exit(1);
        }

    if(max_index < 0)
        {
        sprintf(error_message,"the max. index used in your database appears to be %d, some mistake surely!\n", max_index);
        show_error();
        goto FILE_FORMAT_WRONG;
        }

    if(num_entries < 1)
        {
        sprintf(error_message,"there appear to be %d entries in your database, some mistake surely!", num_entries);
        show_error();
        goto FILE_FORMAT_WRONG;
        }

    if(num_entries > max_index + 1)
        {
        printf("your database seems to have more entries (%d) than would appear sensible from the max. index used (%d)\n", num_entries, max_index);
        show_error();
        goto FILE_FORMAT_WRONG;
        }

    if((current_index < 0) || (current_index > max_index))
        {
        printf("the currently active database element (%d) appears to be outside\n",
            current_index);
        printf("the current database, which appears to have %d elements, setting\n",
            max_index);
        printf("current active element to %d\n",max_index);
        current_index = max_index;
        }

    /*
       allocate sufficient database entries
    */

    for(i=0;i<=max_index;i++)
        {
        if((entry[i] = (struct db_record *) calloc((size_t) 1,
                       (size_t) sizeof(struct db_record))) == NULL)
            {
            printf("unable to calloc memory for database, bailing out!\n");
            exit(1);
            }
        }

    /* create the new status display string and set it */

    if(num_entries == 1)
        strcpy(status_string,"Currently 1 Entry in Database");
    else
        sprintf(status_string,"Currently %d Entries in Database",num_entries);

    xv_set(fdb_frame, FRAME_RIGHT_FOOTER, status_string, NULL);

    /*
       read in the database information
    */

    for(i=0;i<num_entries;i++)
        {
        /* read in the index of the entry */

        num_items_read = fscanf(fp,"entry:%d index:%d\n",&dummy,&index);

#ifdef DEBUG
        printf("reading entry: %d, index: %d\n",dummy, index);
#endif

        if(num_items_read != 2)
            {
            sprintf(error_message,"unable to read index for entry %d in file\n",dummy);
            show_error();
            goto FILE_FORMAT_WRONG;
            }
        if((index < 0) || (index > max_index))
            {
            sprintf(error_message,"index for entry %d is %d, not in range [0,%d]\n",i+1,index,max_index);
            show_error();
            goto FILE_FORMAT_WRONG;
            }

        /* 
           read in the database items 
        */

        /* read and discard the name/gender info header */
        fgets(dummy_string, 255, fp);

        /* surname, removing the trailing carriage return */
        fgets(entry[index]->surname, MAX_SURNAME, fp);
        entry[index]->surname[strlen(entry[index]->surname)-1] = '\0';
#ifdef DEBUG
        printf("surname: %s\n",entry[index]->surname);
#endif
            
        /* forenames, removing the trailing carriage return */
        fgets(entry[index]->forenames, MAX_FORENAMES, fp);
        entry[index]->forenames[strlen(entry[index]->forenames)-1] = '\0';
#ifdef DEBUG
        printf("forenames: %s\n",entry[index]->forenames);
#endif
            
        /* title, removing the trailing carriage return */
        fgets(entry[index]->title, MAX_TITLE, fp);
        entry[index]->title[strlen(entry[index]->title)-1] = '\0';
#ifdef DEBUG
        printf("title: %s\n",entry[index]->title);
#endif
            
        /* maiden name, removing the trailing carriage return */
        fgets(entry[index]->maiden_name, MAX_SURNAME, fp);
        entry[index]->maiden_name[strlen(entry[index]->maiden_name)-1] = '\0';
#ifdef DEBUG
        printf("maiden name: %s\n",entry[index]->maiden_name);
#endif
            
        /* read in the gender string and decode */
        j=0;
        fgets(dummy_string, 255, fp);

        /* lose any leading spaces */
        while(dummy_string[j] == ' ')
            j++;

        /* if first non-space is m or M assume male, otherwise female */
        if((dummy_string[j] == 'm') || (dummy_string[j] == 'M'))
            entry[index]->gender = MALE;
        else
            entry[index]->gender = FEMALE;
#ifdef DEBUG
        if(entry[index]->gender == MALE)
            printf("gender: Male\n");
        else
            printf("gender: Female\n");
#endif
 
        /* read and discard the birth info header */
        fgets(dummy_string, 255, fp);

        /* date of birth, removing the trailing carriage return */
        fgets(entry[index]->birth_date, MAX_DATE, fp);
        entry[index]->birth_date[strlen(entry[index]->birth_date)-1] = '\0';

        /* 
           encode the date, if there was a problem with the encoding, 
           indicate the failure 
        */

        if((entry[index]->encoded_date = 
                  encode_date(entry[index]->birth_date)) == -1)
            {
            sprintf(error_message,"Unable to Understand Date of Birth Entry '%s' for %s %s",
                    entry[index]->birth_date, entry[index]->forenames,
                    entry[index]->surname);
            show_error();
            }
            
        /* place of birth, removing the trailing carriage return */
        fgets(entry[index]->birth_place, MAX_STRING_LENGTH, fp);
        entry[index]->birth_place[strlen(entry[index]->birth_place)-1] = '\0';
            
        /* read and discard the baptism info header */
        fgets(dummy_string, 255, fp);

        /* date of baptism, removing the trailing carriage return */
        fgets(entry[index]->baptism_date, MAX_DATE, fp);
        entry[index]->baptism_date[strlen(entry[index]->baptism_date)-1] = '\0';
            
        /* place of baptism, removing the trailing carriage return */
        fgets(entry[index]->baptism_place, MAX_STRING_LENGTH, fp);
        entry[index]->baptism_place[strlen(entry[index]->baptism_place)-1] = '\0';
            
        /* read and discard the marriage info header */
        fgets(dummy_string, 255, fp);

        /* date of marriage, removing the trailing carriage return */
        fgets(entry[index]->marriage_date, MAX_DATE, fp);
        entry[index]->marriage_date[strlen(entry[index]->marriage_date)-1] = '\0';
            
        /* place of marriage, removing the trailing carriage return */
        fgets(entry[index]->marriage_place, MAX_STRING_LENGTH, fp);
        entry[index]->marriage_place[strlen(entry[index]->marriage_place)-1] = '\0';
            
        /* read in the status string and decode */
        j=0;
        fgets(dummy_string, 255, fp);

        /* lose any leading spaces */
        while(dummy_string[j] == ' ')
            j++;

        /* if first non-space is l or L assume living, otherwise deceased */
        if((dummy_string[j] == 'l') || (dummy_string[j] == 'L'))
            entry[index]->status = LIVING;
        else
            entry[index]->status = DECEASED;
#ifdef DEBUG
        if(entry[index]->status == LIVING)
            printf("status: Living\n");
        else
            printf("status: Deceased\n");
#endif

        /*
           if deceased, read in the extra info 
        */
 
        if(entry[index]->status == DECEASED)
            {
            /* read and discard the death info header */
            fgets(dummy_string, 255, fp);

            /* date of death, removing the trailing carriage return */
            fgets(entry[index]->death_date, MAX_DATE, fp);
            entry[index]->death_date[strlen(entry[index]->death_date)-1] = '\0';
            
            /* place of death, removing the trailing carriage return */
            fgets(entry[index]->death_place, MAX_STRING_LENGTH, fp);
            entry[index]->death_place[strlen(entry[index]->death_place)-1] = '\0';
            
            /* place of rest, removing the trailing carriage return */
            fgets(entry[index]->resting_place, MAX_STRING_LENGTH, fp);
            entry[index]->resting_place[strlen(entry[index]->resting_place)-1] = '\0';
            }

        /* index of mothers entry */
        num_items_read = fscanf(fp,"index of mother:%d\n",&dummy);
        if(num_items_read == 1)
            {
            entry[index]->mother = dummy;
            }
        else
            {
            sprintf(error_message,"problem reading index of mother for database entry %d\n",i+1);
            show_error();
            goto FILE_FORMAT_WRONG;
            }

#ifdef DEBUG
        printf("index of mother: %d\n",dummy);
#endif

        /* index of fathers entry */
        num_items_read = fscanf(fp,"index of father:%d\n",&dummy);
        if(num_items_read == 1)
            {
            entry[index]->father = dummy;
            }
        else
            {
            sprintf(error_message,"problem reading index of father for database entry %d\n",i+1);
            show_error();
            goto FILE_FORMAT_WRONG;
            }

#ifdef DEBUG
        printf("index of father: %d\n",dummy);
#endif

        /* number of spouses */
        num_items_read = fscanf(fp,"number of spouses:%d\n",&dummy);
        if(num_items_read == 1)
            {
            entry[index]->number_of_spouses = dummy;
            }
        else
            {
            sprintf(error_message,"problem reading number of spouses for database entry %d\n",i+1);
            show_error();
            goto FILE_FORMAT_WRONG;
            }
#ifdef DEBUG
        printf("number of spouses is %d\n",dummy);
#endif
        /* 
           check number of spouses does not exceed max. (could have been created with
           another version of FDB with a bigger MAX_SPOUSES)
        */

        if(entry[index]->number_of_spouses > (((int) MAX_SPOUSES) - 1) )
            {
            printf("%s %s has %d spouses - increase MAX_SPOUSES in definitions.h to at least %d and recompile\n",
                entry[index]->forenames,entry[index]->surname,dummy,dummy+1);
            exit(1);
            }

        /* if there are any spouses, read and discard the spouse indices header */
        if(entry[index]->number_of_spouses > 0)
            fgets(dummy_string, 255, fp);

        /* read in the spouse indices */
        for(j=0;j<entry[index]->number_of_spouses;j++)
            {
            num_items_read = fscanf(fp,"%d\n",&dummy);
            if(num_items_read == 1)
                {
                entry[index]->spouse[j] = dummy;
                }
            else
                {
                sprintf(error_message,"problem reading index of spouse %d for database entry %d\n",
                        j+1,i+1);
                show_error();
                goto FILE_FORMAT_WRONG;
                }
            }

        /* number of children */
        num_items_read = fscanf(fp,"number of children:%d\n",&dummy);
        if(num_items_read == 1)
            {
            entry[index]->number_of_children = dummy;
            }
        else
            {
            sprintf(error_message,"problem reading number of children for database entry %d\n",i+1);
            show_error();
            goto FILE_FORMAT_WRONG;
            }
#ifdef DEBUG
        printf("number of children is %d\n",dummy);
#endif
        /* 
           check number of children does not exceed max. (could have been created with
           another version of FDB with a bigger MAX_CHILDREN)
        */

        if(entry[index]->number_of_children > ((int) MAX_CHILDREN) )
            {
            printf("%s %s has %d children - increase MAX_CHILDREN in definitions.h to at least %d and recompile\n",
                entry[index]->forenames,entry[index]->surname,dummy,dummy);
            exit(1);
            }

        /* if there are any children, read and discard the children indices header */
        if(entry[index]->number_of_children > 0)
            fgets(dummy_string, 255, fp);

        /* read in the child indices */
        for(j=0;j<entry[index]->number_of_children;j++)
            {
            num_items_read = fscanf(fp,"%d\n",&dummy);
            if(num_items_read == 1)
                {
                entry[index]->child[j] = dummy;
                }
            else
                {
                sprintf(error_message,"problem reading index of child %d for database entry %d\n",
                        j+1,i+1);
                show_error();
                goto FILE_FORMAT_WRONG;
                }
            }
        /* read and discard the occupation info header */
        fgets(dummy_string, 255, fp);

        /* occupation, removing the trailing carriage return */
        fgets(entry[index]->occupation, MAX_OCCUPATION, fp);
        entry[index]->occupation[strlen(entry[index]->occupation)-1] = '\0';
#ifdef DEBUG
        printf("occupation: **%s*\n",entry[index]->occupation);
#endif
            
        /* read and discard the notes info header */
        fgets(dummy_string, 255, fp);

        /* notes, removing the trailing carriage return */
        fgets(entry[index]->notes, MAX_STRING_LENGTH, fp);
        entry[index]->notes[strlen(entry[index]->notes)-1] = '\0';
#ifdef DEBUG
        printf("notes: **%s*\n",entry[index]->notes);
#endif

        /* read and discard the source info header */
        fgets(dummy_string, 255, fp);
#ifdef DEBUG
        printf("discarding source info header: **%s*\n",dummy_string);
#endif

        /* birth source of info, removing the trailing carriage return */
        fgets(entry[index]->birth_source, MAX_STRING_LENGTH, fp);
        entry[index]->birth_source[strlen(entry[index]->birth_source)-1] = '\0';
#ifdef DEBUG
        printf("birth source info: **%s*\n",entry[index]->birth_source);
#endif

        /* baptism source of info, removing the trailing carriage return */
        fgets(entry[index]->baptism_source, MAX_STRING_LENGTH, fp);
        entry[index]->baptism_source[strlen(entry[index]->baptism_source)-1] = '\0';
#ifdef DEBUG
        printf("baptism source info: **%s*\n",entry[index]->baptism_source);
#endif

        /* marriage source of info, removing the trailing carriage return */
        fgets(entry[index]->marriage_source, MAX_STRING_LENGTH, fp);
        entry[index]->marriage_source[strlen(entry[index]->marriage_source)-1] = '\0';
#ifdef DEBUG
        printf("marriage source info: **%s*\n",entry[index]->marriage_source);
#endif

        /* death source of info, removing the trailing carriage return */
        fgets(entry[index]->death_source, MAX_STRING_LENGTH, fp);
        entry[index]->death_source[strlen(entry[index]->death_source)-1] = '\0';
#ifdef DEBUG
        printf("death source info: **%s*\n",entry[index]->death_source);
#endif
        }

    /* set the default file menu selection to load since this is a new database */

    xv_set(file_menu, MENU_DEFAULT, 1, NULL);

    return;
    }

/*
   code jumps here of file format is wrong
*/

FILE_FORMAT_WRONG:

/* 
   if no database file, create a new entry
*/

#ifdef DEBUG
printf("generating a new database, with one element initially\n");
#endif

current_index = create_new_entry();

#ifdef DEBUG
printf("index of entry: %d\n",current_index);
#endif

/* set the default file menu selection to load since this is a new database */

xv_set(file_menu, MENU_DEFAULT, 1, NULL);

return;
}

/*
   save the database to the current file
*/

void  save_database()
{
FILE   *fp;
int     i, j, entry_count;

if((fp = fopen(new_file_name,"w")) == NULL)
    {
    /* indicate the failure */
    sprintf(error_message,"Unable to Open File: %s",new_file_name);
    show_error();
    return;
    }

/* copy the new file name to the current file name */

strcpy(file_name, new_file_name);

/* write the title */

fprintf(fp,"Database File Created by fdb V%s - cadellinsoft '94\n",FDB_VERSION);

/* write the header information */

fprintf(fp,"number of entries: %d max index in use: %d current active index: %d\n",
        num_entries, max_index, current_index);

/* loop for all entries, writing out those which are non-NULL */

entry_count = 1;
for(i=0;i<=max_index;i++)
    {
    if(entry[i] != (struct db_record *) NULL)
        {
#ifdef DEBUG
        printf("writing out index %d\n",i);
#endif
        fprintf(fp,"entry: %d index: %d\n",entry_count,i);
        fprintf(fp,"**name, title, maiden name and gender**\n");
        fprintf(fp,"%s\n",entry[i]->surname);
        fprintf(fp,"%s\n",entry[i]->forenames);
        fprintf(fp,"%s\n",entry[i]->title);
        fprintf(fp,"%s\n",entry[i]->maiden_name);
        if(entry[i]->gender == MALE)
            fprintf(fp,"Male\n");
        else
            fprintf(fp,"Female\n");
        fprintf(fp,"**birth date and place**\n");
        fprintf(fp,"%s\n",entry[i]->birth_date);
        fprintf(fp,"%s\n",entry[i]->birth_place);
        fprintf(fp,"**baptism date and place**\n");
        fprintf(fp,"%s\n",entry[i]->baptism_date);
        fprintf(fp,"%s\n",entry[i]->baptism_place);
        fprintf(fp,"**marriage date and place**\n");
        fprintf(fp,"%s\n",entry[i]->marriage_date);
        fprintf(fp,"%s\n",entry[i]->marriage_place);
        if(entry[i]->status == LIVING)
            {
            fprintf(fp,"Living\n");
            }
        else
            {
            fprintf(fp,"Deceased\n");
            fprintf(fp,"**date and place of death and resting place**\n");
            fprintf(fp,"%s\n",entry[i]->death_date);
            fprintf(fp,"%s\n",entry[i]->death_place);
            fprintf(fp,"%s\n",entry[i]->resting_place);
            }
        fprintf(fp,"index of mother: %d\n",entry[i]->mother);
        fprintf(fp,"index of father: %d\n",entry[i]->father);
        fprintf(fp,"number of spouses: %d\n",entry[i]->number_of_spouses);
        if(entry[i]->number_of_spouses > 0)
            fprintf(fp,"**spouse(s) indices**\n");
        for(j=0;j<entry[i]->number_of_spouses;j++)
            fprintf(fp,"%d\n",entry[i]->spouse[j]);
        fprintf(fp,"number of children: %d\n",entry[i]->number_of_children);
        if(entry[i]->number_of_children > 0)
            fprintf(fp,"**child(ren) indices**\n");
        for(j=0;j<entry[i]->number_of_children;j++)
            fprintf(fp,"%d\n",entry[i]->child[j]);
        fprintf(fp,"**occupation**\n");
        fprintf(fp,"%s\n",entry[i]->occupation);
        fprintf(fp,"**notes**\n");
        fprintf(fp,"%s\n",entry[i]->notes);
        fprintf(fp,"**sources of info for birth, baptism, marriage and death respectively**\n");
        fprintf(fp,"%s\n",entry[i]->birth_source);
        fprintf(fp,"%s\n",entry[i]->baptism_source);
        fprintf(fp,"%s\n",entry[i]->marriage_source);
        fprintf(fp,"%s\n",entry[i]->death_source);
        entry_count++;
        }
    }
fclose(fp);

/* set the default file menu selection to load since database has just been saved */

xv_set(file_menu, MENU_DEFAULT, 1, NULL);

}

/*
   update all the text items, choices, buttons etc. to reflect the
   new value of current_item
*/

void  update_panel_items()
{
int   selected_title, i, num_choices;
char  mothers_name[MAX_SURNAME + MAX_FORENAMES + 1];
char  fathers_name[MAX_SURNAME + MAX_FORENAMES + 1];
char  spouse_name[MAX_SURNAME + MAX_FORENAMES + 1];
char  child_name[MAX_SURNAME + MAX_FORENAMES + 1];

/* select the title number to use in the choice item */
selected_title = 0;
for(i=1;i<num_titles;i++)
    {
    if(strcmp(titles[i],entry[current_index]->title) == 0)
        selected_title = i;
    }

xv_set(title_choice, PANEL_VALUE, selected_title, NULL); 
xv_set(surname_text, PANEL_VALUE, entry[current_index]->surname, NULL);
xv_set(forenames_text, PANEL_VALUE, entry[current_index]->forenames, NULL);
xv_set(gender_choice, PANEL_VALUE, entry[current_index]->gender, NULL);
xv_set(maiden_name_text, PANEL_VALUE, entry[current_index]->maiden_name, NULL);

/*
   if the current person is male, or if they never married, disable the
   maiden name field otherwise enable it
*/

if((entry[current_index]->gender == MALE) || (entry[current_index]->number_of_spouses == 0))
    xv_set(maiden_name_text, PANEL_INACTIVE, TRUE, NULL);
else
    xv_set(maiden_name_text, PANEL_INACTIVE, FALSE, NULL);

xv_set(birth_date_text, PANEL_VALUE, entry[current_index]->birth_date, NULL);
xv_set(birth_place_text, PANEL_VALUE, entry[current_index]->birth_place, NULL);
xv_set(baptism_date_text, PANEL_VALUE, entry[current_index]->baptism_date, NULL);
xv_set(baptism_place_text, PANEL_VALUE, entry[current_index]->baptism_place, NULL);
xv_set(marriage_date_text, PANEL_VALUE, entry[current_index]->marriage_date, NULL);
xv_set(marriage_place_text, PANEL_VALUE, entry[current_index]->marriage_place, NULL);

/*
   if there are no spouses, deactivate the marriage date and place
   otherwise activate them
*/

if(entry[current_index]->number_of_spouses == 0)
    {
    xv_set(marriage_date_text, PANEL_INACTIVE, TRUE, NULL);
    xv_set(marriage_place_text, PANEL_INACTIVE, TRUE, NULL);
    }
else
    {
    xv_set(marriage_date_text, PANEL_INACTIVE, FALSE, NULL);
    xv_set(marriage_place_text, PANEL_INACTIVE, FALSE, NULL);
    }

xv_set(status_choice, PANEL_VALUE, entry[current_index]->status, NULL);
xv_set(death_date_text, PANEL_VALUE, entry[current_index]->death_date, NULL);
xv_set(death_place_text, PANEL_VALUE, entry[current_index]->death_place, NULL);
xv_set(resting_place_text, PANEL_VALUE, entry[current_index]->resting_place, NULL);

/*
   if the current person is still living 
   deactivate the place of death and resting place elements
*/

if(entry[current_index]->status == LIVING)
    {
    xv_set(death_date_text, PANEL_INACTIVE, TRUE, NULL);
    xv_set(death_place_text, PANEL_INACTIVE, TRUE, NULL);
    xv_set(resting_place_text, PANEL_INACTIVE, TRUE, NULL);
    }
else
    {
    xv_set(death_date_text, PANEL_INACTIVE, FALSE, NULL);
    xv_set(death_place_text, PANEL_INACTIVE, FALSE, NULL);
    xv_set(resting_place_text, PANEL_INACTIVE, FALSE, NULL);
    }

/* create the mothers name string */

if(entry[current_index]->mother != -1)
    sprintf(mothers_name,"Mother: %s %s",entry[entry[current_index]->mother]->forenames,
                                 entry[entry[current_index]->mother]->surname);
else
    strcpy(mothers_name,"Mother:");

xv_set(display_mother_button, PANEL_LABEL_STRING, mothers_name, NULL);

/* if there is no mother defined, deactivate the button otherwise activate it */

if(entry[current_index]->mother == -1)
    xv_set(display_mother_button, PANEL_INACTIVE, TRUE, NULL);
else
    xv_set(display_mother_button, PANEL_INACTIVE, FALSE, NULL);

/* create the fathers name string */

if(entry[current_index]->father != -1)
    sprintf(fathers_name,"Father: %s %s",entry[entry[current_index]->father]->forenames,
                                 entry[entry[current_index]->father]->surname);
else
    strcpy(fathers_name,"Father:");

xv_set(display_father_button, PANEL_LABEL_STRING, fathers_name, NULL);

/* if there is no father defined, deactivate the button otherwise activate it */

if(entry[current_index]->father == -1)
    xv_set(display_father_button, PANEL_INACTIVE, TRUE, NULL);
else
    xv_set(display_father_button, PANEL_INACTIVE, FALSE, NULL);

/* find out how many elements currently in the spouse scrolling list */

num_choices = (int) xv_get(spouse_list, PANEL_LIST_NROWS);

#ifdef DEBUG
printf("deleting %d elements from spouse scrolling list\n",num_choices);
#endif

/* clear the current contents of the spouse scrolling list */

if(num_choices > 0)
    {
    for( i = num_choices - 1 ; i >= 0 ; i-- )
        xv_set(spouse_list, PANEL_LIST_DELETE, i, NULL);
    }

/* 
   add the spouse(s) of the current entry to the list 
*/

for( i = 0; i < entry[current_index]->number_of_spouses ; i++ )
    {
    sprintf(spouse_name,"%s %s",entry[entry[current_index]->spouse[i]]->forenames,
                                entry[entry[current_index]->spouse[i]]->surname);
    xv_set(spouse_list, 
            PANEL_LIST_INSERT,        i, 
            PANEL_LIST_STRING,        i,    spouse_name,
            PANEL_LIST_SELECT,        i,    FALSE,
            NULL);
    }

/* if there are no spouses, deactivate the list, otherwise activate it */

if(entry[current_index]->number_of_spouses == 0)
    xv_set(spouse_list, PANEL_INACTIVE, TRUE, NULL);
else
    xv_set(spouse_list, PANEL_INACTIVE, FALSE, NULL);

/* find out how many elements currently in the child scrolling list */

num_choices = (int) xv_get(child_list, PANEL_LIST_NROWS);

#ifdef DEBUG
printf("deleting %d elements from child scrolling list\n",num_choices);
#endif

/* clear the current contents of the child scrolling list */

if(num_choices > 0)
    {
    for( i = num_choices - 1 ; i >= 0 ; i-- )
        xv_set(child_list, PANEL_LIST_DELETE, i, NULL);
    }

/* 
   add the child(ren) of the current entry to the list 
*/

for( i = 0; i < entry[current_index]->number_of_children ; i++ )
    {
    sprintf(child_name,"%s %s",entry[entry[current_index]->child[i]]->forenames,
                                entry[entry[current_index]->child[i]]->surname);
    xv_set(child_list, 
            PANEL_LIST_INSERT,        i, 
            PANEL_LIST_STRING,        i,    child_name,
            PANEL_LIST_SELECT,        i,    FALSE,
            NULL);
    }

/* if there are no children, deactivate the list, otherwise activate it */

if(entry[current_index]->number_of_children == 0)
    xv_set(child_list, PANEL_INACTIVE, TRUE, NULL);
else
    xv_set(child_list, PANEL_INACTIVE, FALSE, NULL);

/* set the occupation and notes fields */

xv_set(occupation_text, PANEL_VALUE, entry[current_index]->occupation, NULL);
xv_set(notes_text, PANEL_VALUE, entry[current_index]->notes, NULL);

/* set the date source information fields */

xv_set(birth_source_text, PANEL_VALUE, entry[current_index]->birth_source, NULL);
xv_set(baptism_source_text, PANEL_VALUE, entry[current_index]->baptism_source, NULL);
xv_set(marriage_source_text, PANEL_VALUE, entry[current_index]->marriage_source, NULL);
xv_set(death_source_text, PANEL_VALUE, entry[current_index]->death_source, NULL);

/* 
   if the person has no spouse, deactivate the marriage source field,
   otherwise activate it
*/

if(entry[current_index]->number_of_spouses == 0)
    xv_set(marriage_source_text, PANEL_INACTIVE, TRUE, NULL);
else
    xv_set(marriage_source_text, PANEL_INACTIVE, FALSE, NULL);

/* 
   if the person is still alive, deactivate the death source field,
   otherwise activate it
*/

if(entry[current_index]->status == LIVING)
    xv_set(death_source_text, PANEL_INACTIVE, TRUE, NULL);
else
    xv_set(death_source_text, PANEL_INACTIVE, FALSE, NULL);

}

/*
   create a new database entry, selecting the lowest unused index
   initialise the entries and return the index number
*/

int  create_new_entry()
{
int     index;

/* find the lowest index not currently in use */

index=0;
while((entry[index] != (struct db_record *) NULL) && (index < MAX_INDICES))
    index++;

/* 
   check for database overflow, if so inform the user and return the 
   current index
*/

if(index == MAX_INDICES)
    {
    /* indicate the failure */
    strcpy(error_message,"Database Full! - Please Increment MAX_INDICES in definitions.h and Recompile!");
    show_error();
    return current_index;
    }

#ifdef DEBUG
printf("generating a new entry, index %d\n",index);
#endif

/* calloc the memory, display error message and return the current index if failure */

if((entry[index] = (struct db_record *) calloc((size_t) 1,
               (size_t) sizeof(struct db_record))) == NULL)
    {
    /* indicate the failure */
    strcpy(error_message,"Unable to Allocate Memory for New Entry!!");
    show_error();
    return current_index;
    }

/* initialise the entry with empty values */

strcpy(entry[index]->surname,"");
strcpy(entry[index]->forenames,"");
strcpy(entry[index]->title,"");
strcpy(entry[index]->birth_date,"");
strcpy(entry[index]->birth_place,"");
strcpy(entry[index]->baptism_date,"");
strcpy(entry[index]->baptism_place,"");
strcpy(entry[index]->marriage_date,"");
strcpy(entry[index]->marriage_place,"");
strcpy(entry[index]->death_date,"");
strcpy(entry[index]->death_place,"");
strcpy(entry[index]->resting_place,"");
entry[index]->gender = FEMALE;
entry[index]->mother = -1;
entry[index]->father = -1;
entry[index]->number_of_spouses = 0;
entry[index]->number_of_children = 0;
strcpy(entry[index]->occupation,"");
strcpy(entry[index]->notes,"");

/* increment the number of entries counter */

num_entries++;

/* create the new status display string and set it */

if(num_entries == 1)
    strcpy(status_string,"Currently 1 Entry in Database");
else
    sprintf(status_string,"Currently %d Entries in Database",num_entries);

xv_set(fdb_frame, FRAME_RIGHT_FOOTER, status_string, NULL);

/* check if we have a new highest index */

if(index > max_index)
    {
    max_index = index;
#ifdef DEBUG
    printf("new max index: %d\n",max_index);
#endif
    }

/* set the default file menu selection to save since something has changed */

xv_set(file_menu, MENU_DEFAULT, 3, NULL);

/* return the index of the new entry */

return index;
}

/*
   select an entry from the database using a scrolling list
   the gender is specified by the argument, if -ve all entries
   are displayed
*/

int  select_entry(required_gender)
int required_gender;
{
int   i, j, num_choices, list_index;
char  entry_name[2*MAX_SURNAME + MAX_FORENAMES + MAX_DATE + 25];
char  temp_string[MAX_STRING_LENGTH + 15];

/* find out how many elements currently in the entries scrolling list */

num_choices = (int) xv_get(entries_list, PANEL_LIST_NROWS);

#ifdef DEBUG
printf("deleting %d elements from entries scrolling list\n",num_choices);
#endif

/* clear the current contents of the entries scrolling list */

if(num_choices > 0)
    {
    for( i = num_choices - 1 ; i >= 0 ; i-- )
        xv_set(entries_list, PANEL_LIST_DELETE, i, NULL);
    }

/* 
   add all non-NULL database entries of the appropriate gender
   to the scrolling list
   screening out 1) the person themselves
*/

list_index = 0;
for( i = 0; i <= max_index; i++ )
    {
    /* rule out NULL entries, wrong gender and self */
    if((entry[i] != (struct db_record *) NULL) &&
            ((required_gender < 0) || (entry[i]->gender == required_gender)) &&
            (i != current_index))
        {

        /* rule out the persons mother and father */
        if((entry[current_index]->mother == i) || (entry[current_index]->father == i))
            goto next_please;

        /* rule out any spouses of the current person */
        for(j=0;j<entry[current_index]->number_of_spouses;j++)
            if(entry[current_index]->spouse[j] == i)
                goto next_please;
        
        /* rule out any children of the current person */
        for(j=0;j<entry[current_index]->number_of_children;j++)
            if(entry[current_index]->child[j] == i)
                goto next_please;
        
        /* create the initial string with the persons name */
        sprintf(entry_name,"%s %s",entry[i]->forenames, entry[i]->surname);

        /* add their date of birth if available */
        if(strcmp(entry[i]->birth_date,""))
            {
            sprintf(temp_string," (b. %s)",entry[i]->birth_date);
            strcat(entry_name,temp_string);
            }

        /* if no date of birth, try date of death if available */
        if(!strcmp(entry[i]->birth_date,"") && strcmp(entry[i]->death_date,""))
            {
            sprintf(temp_string," (d. %s)",entry[i]->death_date);
            strcat(entry_name,temp_string);
            }

        /* 
           if there is no birth or death date info available, try to add the name of 
           the father or the mother to distinguish the person from any others
           with the same name
        */

        if(!strcmp(entry[i]->birth_date,"") && !strcmp(entry[i]->death_date,""))
            {
            if(entry[i]->father != -1)
                {
                if(strcmp(entry[entry[i]->father]->forenames,""))
                    {
                    /* get the terms correct depending on persons gender */
                    if(entry[i]->gender == MALE)
                        sprintf(temp_string," (son of %s %s)",entry[entry[i]->father]->forenames,
                                                              entry[entry[i]->father]->surname);
                    else
                        sprintf(temp_string," (daughter of %s %s)",
                                                              entry[entry[i]->father]->forenames,
                                                              entry[entry[i]->father]->surname);
                    strcat(entry_name,temp_string);
                    }
                }
            else
                {
                /* 
                   if no father, check for mother (a bit sexist but it had to be one
                   of the two ways round!)
                */
                if(entry[i]->mother != -1)
                    {
                    if(strcmp(entry[entry[i]->mother]->forenames,""))
                        {
                        /* get the terms correct depending on persons gender */
                        if(entry[i]->gender == MALE)
                            sprintf(temp_string," (son of %s %s)",
                                                  entry[entry[i]->mother]->forenames,
                                                  entry[entry[i]->mother]->surname);
                        else
                            sprintf(temp_string," (daughter of %s %s)",
                                                  entry[entry[i]->mother]->forenames,
                                                  entry[entry[i]->mother]->surname);
                        strcat(entry_name,temp_string);
                        }
                    }
                }
            }

        /* 
            last ditch attempt to find a unique identifier. if no birth date,
            death date, father or mother, add their maiden name if available 
            this is a long shot since if there is no parent name info it is
            unlikely that the maiden name will be known
        */

        if(!strcmp(entry[i]->birth_date,"") && !strcmp(entry[i]->death_date,"") &&
                (entry[i]->father == -1) && (entry[i]->mother == -1))
            {
            if(strcmp(entry[i]->maiden_name,""))
                {
                sprintf(temp_string," (nee %s)",entry[i]->maiden_name);
                strcat(entry_name,temp_string);
                }
            else
                {
                /* absolute last ditch attempt, try the notes field */
                if(strcmp(entry[i]->notes,""))
                    {
                    sprintf(temp_string," (%s)",entry[i]->notes);
                    strcat(entry_name,temp_string);
                    }
                else
                    {
                    /* absolute absolute last ditch attempt, try the occupation field */
                    if(strcmp(entry[i]->occupation,""))
                        {
                        sprintf(temp_string," (%s)",entry[i]->occupation);
                        strcat(entry_name,temp_string);
                        }
                    }
                }
            }

        /* add the entry to the list */
        xv_set(entries_list, 
                PANEL_LIST_INSERT,        list_index, 
                PANEL_LIST_STRING,        list_index,    entry_name,
                PANEL_LIST_SELECT,        list_index,    FALSE,
                NULL);
        /* keep a track of the indices of the list entries */
        entries_index[list_index] = i;

        /* increment the list size counter */
        list_index++;
        }
    next_please:;   /* the extra semi-colon is to keep the Sun C compiler happy! */
    }

/*
   show the entries list to allow a selection to be made
*/

xv_set(entries_frame, XV_SHOW, TRUE, NULL);

}

/*
   select an entry from the database (for deselection) using a scrolling 
   list whether the search is for spouse or child is specified by the 
   argument
*/

int  deselect_entry(required_type)
int required_type;
{
int   i, j, num_choices, list_index;
char  entry_name[MAX_SURNAME + MAX_FORENAMES + 1];

/* find out how many elements currently in the entries scrolling list */

num_choices = (int) xv_get(entries_list, PANEL_LIST_NROWS);

#ifdef DEBUG
printf("deleting %d elements from entries scrolling list\n",num_choices);
#endif

/* clear the current contents of the entries scrolling list */

if(num_choices > 0)
    {
    for( i = num_choices - 1 ; i >= 0 ; i-- )
        xv_set(entries_list, PANEL_LIST_DELETE, i, NULL);
    }

/* 
   create the list of spouses or children as required
*/

list_index = 0;
if(required_type == SPOUSE)
    {
    for( i = 0; i < entry[current_index]->number_of_spouses; i++ )
        {
        sprintf(entry_name,"%s %s",entry[entry[current_index]->spouse[i]]->forenames, 
                                   entry[entry[current_index]->spouse[i]]->surname);
        xv_set(entries_list, 
                PANEL_LIST_INSERT,        list_index, 
                PANEL_LIST_STRING,        list_index,    entry_name,
                PANEL_LIST_SELECT,        list_index,    FALSE,
                NULL);
        /* keep a track of the indices of the list entries */
        entries_index[list_index] = entry[current_index]->spouse[i];

        /* increment the list size counter */
        list_index++;
        }
    }
else /* must be a child deselection if it isn`t a spouse */
    {
    for( i = 0; i < entry[current_index]->number_of_children; i++ )
        {
        sprintf(entry_name,"%s %s",entry[entry[current_index]->child[i]]->forenames, 
                                   entry[entry[current_index]->child[i]]->surname);
        xv_set(entries_list, 
                PANEL_LIST_INSERT,        list_index, 
                PANEL_LIST_STRING,        list_index,    entry_name,
                PANEL_LIST_SELECT,        list_index,    FALSE,
                NULL);
        /* keep a track of the indices of the list entries */
        entries_index[list_index] = entry[current_index]->child[i];

        /* increment the list size counter */
        list_index++;
        }
    }

/*
   show the entries list to allow a selection to be made
*/

xv_set(entries_frame, XV_SHOW, TRUE, NULL);

}

/*
   this function ensures that all text fields are up to date
   it is called whenever the current person is about to change
   and is designed to catch the cases where the user has entered
   text in a field and then moved to another field using the mouse
*/

void  update_text()
{

/*
    copy the text from each text field to the appropriate database location
*/

strcpy(entry[current_index]->surname, (char *) xv_get(surname_text, PANEL_VALUE));
strcpy(entry[current_index]->forenames, (char *) xv_get(forenames_text, PANEL_VALUE));
strcpy(entry[current_index]->maiden_name, (char *) xv_get(maiden_name_text, PANEL_VALUE));
strcpy(entry[current_index]->birth_date, (char *) xv_get(birth_date_text, PANEL_VALUE));
strcpy(entry[current_index]->birth_place, (char *) xv_get(birth_place_text, PANEL_VALUE));
strcpy(entry[current_index]->baptism_date, (char *) xv_get(baptism_date_text, PANEL_VALUE));
strcpy(entry[current_index]->baptism_place, (char *) xv_get(baptism_place_text, PANEL_VALUE));
strcpy(entry[current_index]->marriage_date, (char *) xv_get(marriage_date_text, PANEL_VALUE));
strcpy(entry[current_index]->marriage_place, (char *) xv_get(marriage_place_text, PANEL_VALUE));
strcpy(entry[current_index]->death_date, (char *) xv_get(death_date_text, PANEL_VALUE));
strcpy(entry[current_index]->death_place, (char *) xv_get(death_place_text, PANEL_VALUE));
strcpy(entry[current_index]->resting_place, (char *) xv_get(resting_place_text, PANEL_VALUE));
strcpy(entry[current_index]->birth_source, (char *) xv_get(birth_source_text, PANEL_VALUE));
strcpy(entry[current_index]->baptism_source, (char *) xv_get(baptism_source_text, PANEL_VALUE));
strcpy(entry[current_index]->marriage_source, (char *) xv_get(marriage_source_text, PANEL_VALUE));
strcpy(entry[current_index]->death_source, (char *) xv_get(death_source_text, PANEL_VALUE));
strcpy(entry[current_index]->occupation, (char *) xv_get(occupation_text, PANEL_VALUE));
strcpy(entry[current_index]->notes, (char *) xv_get(notes_text, PANEL_VALUE));

/* set the default file menu selection to save since something has changed */

xv_set(file_menu, MENU_DEFAULT, 3, NULL);
}
