/*============================================================
 * browse.c -- Implements the browse command
 * Copyright(c) 1992-4 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 25 Aug 93
 *   2.3.6 - 01 Nov 93    3.0.0 - 24 Sep 94
 *==========================================================*/

#include "standard.h"
#include "table.h"
#include "indiseq.h"
#include "gedcom.h"

extern STRING idsbrs, idsrmv, idfbrs, idcbrs, idcrmv, iscnew, issnew;
extern STRING idfcop, ntprnt, nofath, nomoth, nospse, noysib, noosib;
extern STRING noprnt, nohusb, nowife, hasbth, hasnei, nocinf, nocofp;
extern STRING idpnxt, ids2fm, idc2fm, idplst, idp2br, crtcfm, crtsfm;
extern STRING ronlye, ronlya;

NODE family_to_browse_to();
INDISEQ ask_for_indiseq();
INDISEQ name_to_indiseq();

#define ALLPARMS &indi1, &indi2, &fam1, &fam2, &seq

/*=========================================
 * browse -- Main loop of browse operation.
 *=======================================*/
browse (indi1)
NODE indi1;
{
	INT code, len, rc;
	NODE indi2, fam1, fam2;
	STRING key, name;
	INDISEQ seq = NULL;

	if (!indi1 && (seq = ask_for_indiseq(idplst, &rc))) {
		if ((len = length_indiseq(seq)) < 1) return;
		if (len == 1) {
			element_indiseq(seq, 0, &key, &name);
			indi1 = key_to_indi(key);
			remove_indiseq(seq, FALSE);
			seq = NULL;
		}
	}
	if (!indi1 && !seq) return;
	code = indi1 ? BROWSE_INDI : BROWSE_LIST;
	while (code != BROWSE_QUIT) {
		switch (code) {
		case BROWSE_INDI:
			code = browse_indi(ALLPARMS); break;
		case BROWSE_FAM:
			code = browse_fam(ALLPARMS); break;
		case BROWSE_PED:
			code = browse_pedigree(ALLPARMS); break;
		case BROWSE_TAND:
			code = browse_tandem(ALLPARMS); break;
		case BROWSE_2FAM:
			code = browse_2fam(ALLPARMS); break;
		case BROWSE_LIST:
			code = browse_list(ALLPARMS); break;
		}
	}
}
/*================================================
 * browse_indi -- Handle person browse operations.
 *==============================================*/
INT browse_indi (pindi1, pindi2, pfam1, pfam2, pseq)
NODE *pindi1, *pindi2, *pfam1, *pfam2;
INDISEQ *pseq;
{
	STRING key, name, addstrings[2];
	INT i, c, len, rc;
	NODE node, save = NULL, indi = *pindi1;
	INDISEQ seq = NULL;
	char scratch[100];

	addstrings[0] = crtcfm;
	addstrings[1] = crtsfm;
	if (!indi) return BROWSE_QUIT;
	while (TRUE) {
		c = indi_browse(indi);
		if (c != 'a') save = NULL;
		switch (c) {
		case 'e':	/* Edit this person */
			indi = edit_indi(indi);
			break;
		case 'g': 	/* Browse to person's family */
			if (*pfam1 = choose_family(indi, ntprnt, idfbrs))
				return BROWSE_FAM;
			else
				message(ntprnt);
			break;
		case 'f': 	/* Browse to person's father */
			if (!(node = indi_to_fath(indi)))
				message(nofath);
			else
				indi = node;
			break;
		case 'm':	/* Browse to person's mother */
			if (!(node = indi_to_moth(indi)))
				message(nomoth);
			else
				indi = node;
			break;
		case 'z':	/* Zip browse another person */
			node = ask_for_indi(idpnxt, FALSE, FALSE);
			if (node) indi = node;
			break;
		case 's':	/* Browse to person's spouse */
			node = choose_spouse(indi, nospse, idsbrs);
			if (node) indi = node;
			break;
		case 'c':	/* Browse to person's child */
			node = choose_child(indi, NULL, nocofp,
			    idcbrs, FALSE);
			if (node) indi = node;
			break;
		case 'p':	/* Switch to pedigree mode */
			*pindi1 = indi;
			return BROWSE_PED;
		case 'o':	/* Browse to older sib */
			if (!(node = indi_to_prev_sib(indi)))
				message(noosib);
			else
				indi = node;
			break;
		case 'y':	/* Browse to younger sib */
			if (!(node = indi_to_next_sib(indi)))
				message(noysib);
			else 
				indi = node;
			break;
		case 'u':	/* Browse to parents' family */
			if (!(*pfam1 = indi_to_famc(indi)))
				message(noprnt);
			else
				return  BROWSE_FAM;
			break;
		case 'b': 	/* Browse new list of persons */
			seq = ask_for_indiseq(idplst, &rc);
			if (!seq) break;
			if ((len = length_indiseq(seq)) == 1) {
				element_indiseq(seq, 0, &key, &name);
				indi = key_to_indi(key);
				remove_indiseq(seq, FALSE);
				break;
			}
			*pseq = seq;
			return BROWSE_LIST;
			break;
		case 'n':	/* Add new person */
			if (!(node = add_indi_by_edit())) break;
			save = indi;
			indi = node;
			break;
		case 'a':	/* Add family for current person */
			if (readonly) {
				message(ronlya);
				break;
			}
			i = choose_from_list(idfcop, 2, addstrings);
			if (i == -1) break;
			if (i == 0) node = add_family(NULL, NULL, indi);
			else if (save) {
				if (keyflag)
					sprintf(scratch, "%s%s (%s)", issnew,
				    	    indi_to_name(save, 55),
					    rmvat(nxref(save))+1);
				else
					sprintf(scratch, "%s%s", issnew,
					    indi_to_name(save, 55));
				if (ask_yes_or_no(scratch))
					node = add_family(indi, save, NULL);
				else
					node = add_family(indi, NULL, NULL);
			} else
				node = add_family(indi, NULL, NULL);
			save = NULL;
			if (!node) break;
			*pfam1 = node;
			return BROWSE_FAM;
		case 't':	/* Switch to tandem browsing */
			node = ask_for_indi(idp2br, FALSE, FALSE);
			if (node) {
				*pindi1 = indi;
				*pindi2 = node;
				return BROWSE_TAND;
			}
			break;
		case 'x': 	/* Swap families of current person */
			swap_families(indi);
			break;
		case 'h':	/* Add person as spouse */
			add_spouse(indi, NULL, TRUE);
			break;
		case 'i':	/* Add person as child */
			add_child(indi, NULL);
			break;
		case 'r':	/* Remove person as spouse */
			remove_spouse(indi, NULL, FALSE);
			break;
		case 'd':	/* Remove person as child */
			remove_child(indi, FALSE);
			break;
		case 'q':
		default:
			return BROWSE_QUIT;
		}
	}
}
/*===============================================
 * browse_fam -- Handle family browse selections.
 *=============================================*/
INT browse_fam (pindi, pdum, pfam1, pfam2, pseq)
NODE *pfam1, *pindi, *pdum, *pfam2;
INDISEQ *pseq;
{
	INT i, c, len, rc;
	NODE save = NULL, fam = *pfam1, node, husb, wife, chil, rest;
	NODE spnodes[2];
	INDISEQ seq;
	static STRING hstring = NULL, wstring = NULL;
	STRING key, name, spstrings[2];
	char scratch[100];

	if (!fam) return BROWSE_QUIT;
	while (TRUE) {
		c = fam_browse(fam);
		if (c != 'a' && c != 's') save = NULL;
		switch (c) {
		case 'e':	/* Edit family's record */
			fam = edit_family(fam);
			break;
		case 'f':	/* Browse to family's father */
			if (!(*pindi = fam_to_husb(fam)))
				message(nohusb);
			else
				return BROWSE_INDI;
			break;
		case 'm':	/* Browse to family's mother */
			if (!(*pindi = fam_to_wife(fam)))
				message(nowife);
			else
				return BROWSE_INDI;
			break;
		case 'c':	/* Browse to a child */
			*pindi = choose_child(NULL, fam, nocinf,
			    idcbrs, FALSE);
			if (*pindi) return BROWSE_INDI;
			break;
		case 'd':	/* Remove a child */
			if (readonly) {
				message(ronlye);
				break;
			}
			*pindi = choose_child(NULL, fam, nocinf,
			    idcrmv, TRUE);
			if (*pindi) remove_child(*pindi, TRUE);
			break;
		case 's':	/* Add spouse to family */
			if (readonly) {
				message(ronlye);
				break;
			}
			split_fam(fam, &husb, &wife, &chil, &rest);
			join_fam(fam, husb, wife, chil, rest);
			if (husb && wife) {
				message(hasbth);
				break;
			}
			if (save) {
				if (keyflag)
					sprintf(scratch, "%s%s (%s)", issnew,
					    indi_to_name(save, 56),
					    rmvat(nxref(save))+1);
				else
					sprintf(scratch, "%s%s", issnew,
					    indi_to_name(save, 56));
				if (ask_yes_or_no(scratch)) {
					add_spouse(save, fam, FALSE);
					save = NULL;
					break;
				}
			}
			add_spouse(NULL, fam, TRUE);
			save = NULL;
			break;
		case 'r':	/* Remove spouse from family */
			if (readonly) {
				message(ronlye);
				break;
			}
			split_fam(fam, &husb, &wife, &chil, &rest);
			join_fam(fam, husb, wife, chil, rest);
			if (!husb && !wife) {
				message(hasnei);
				break;
			}
			if (hstring) stdfree(hstring);
			if (wstring) stdfree(wstring);
			hstring = wstring = NULL;
			i = 0;
			if (husb) {
				husb = key_to_indi(rmvat(nval(husb)));
				spstrings[i] = indi_to_list_string(husb,
				    NULL, 66);
				spnodes[i++] = husb;
			}
			if (wife) {
				wife = key_to_indi(rmvat(nval(wife)));
				spstrings[i] = indi_to_list_string(wife,
				    NULL, 66);
				spnodes[i++] = wife;
			}
			i = choose_from_list(idsrmv, i, spstrings);
			if (i == -1) break;
			remove_spouse(spnodes[i], fam, TRUE);
			break;
		case 'n':	/* Add person to database */
			save = add_indi_by_edit();
			break;
		case 'a':	/* Add child to family */
			if (readonly) {
				message(ronlye);
				break;
			}
			if (save) {
				if (keyflag)
					sprintf(scratch, "%s%s (%s)", iscnew,
					    indi_to_name(save, 56),
					    rmvat(nxref(save))+1);
				else
					sprintf(scratch, "%s%s", iscnew,
					    indi_to_name(save, 56));
				if (ask_yes_or_no(scratch)) {
					add_child(save, fam);
					save = NULL;
					break;
				}
			}
			add_child(NULL, fam);
			save = NULL;
			break;
		case 'b': 	/* Browse to new list of persons */
			seq = ask_for_indiseq(idplst, &rc);
			if (!seq) break;
			if ((len = length_indiseq(seq)) == 1) {
				element_indiseq(seq, 0, &key, &name);
				*pindi = key_to_indi(key);
				remove_indiseq(seq, FALSE);
				return BROWSE_INDI;
				break;
			}
			*pseq = seq;
			return BROWSE_LIST;
			break;
		case 'z':	/* Zip browse to new person */
			*pindi = ask_for_indi(idpnxt, FALSE, FALSE);
			if (*pindi) return BROWSE_INDI;
			break;
		case 't':	/* Enter family tandem mode */
			node = ask_for_fam(ids2fm, idc2fm);
			if (node) {
				*pfam1 = fam;
				*pfam2 = node;
				return BROWSE_2FAM;
			}
			break;
		case 'x':	/* Swap two children */
			swap_children(NULL, fam);
			break;
		case 'q':
		default:
			return BROWSE_QUIT;
		}
	}
}
/*======================================================
 * browse_pedigree -- Handle pedigree browse selections.
 *====================================================*/
INT browse_pedigree (pindi, pdum1, pfam, pdum2, pseq)
NODE *pindi, *pfam, *pdum1, *pdum2;
INDISEQ *pseq;
{
	NODE node, indi = *pindi;
	INT rc, len;
	STRING key, name;
	INDISEQ seq = NULL;
	if (!indi) return BROWSE_QUIT;
	while (TRUE) {
		switch (ped_browse(indi)) {
		case 'e':	/* Edit person */
			indi = edit_indi(indi);
			break;
		case 'i':	/* Switch to person browse mode */
			*pindi = indi;
			return BROWSE_INDI;
		case 'f':	/* Browse to father */
			if (!(node = indi_to_fath(indi)))
				message(nofath);
			else
				indi = node;
			break;
		case 'm':	/* Browse to mother */
			if (!(node = indi_to_moth(indi)))
				message(nomoth);
			else
				indi = node;
			break;
		case 's':	/* Browse to spouse */
			node = choose_spouse(indi, nospse, idsbrs);
			if (node) indi = node;
			break;
		case 'c':	/* Browse to children */
			if (node = choose_child(indi, NULL, nocofp,
			    idcbrs, FALSE))
				indi = node;
			break;
		case 'g':	/* Switch to family mode */
			if (*pfam = choose_family(indi, ntprnt, idfbrs))
				return BROWSE_FAM;
			else
				message(ntprnt);
			break;
		case 'b': 	/* Browse new list of persons */
			seq = ask_for_indiseq(idplst, &rc);
			if (!seq) break;
			if ((len = length_indiseq(seq)) == 1) {
				element_indiseq(seq, 0, &key, &name);
				indi = key_to_indi(key);
				remove_indiseq(seq, FALSE);
				break;
			}
			*pseq = seq;
			return BROWSE_LIST;
			break;
		case 'q':
		default:
			return BROWSE_QUIT;
		}
	}
}
/*==================================================
 * choose_child -- Choose child of person or family.
 *================================================*/
NODE choose_child (indi, fam, msg0, msgn, ask1)
NODE indi, fam;
STRING msg0, msgn;
BOOLEAN ask1;
{
	INDISEQ seq;
	NODE node;
	INT i = 0;
	if (indi) seq = indi_to_children(indi);
	if (!indi && fam) seq = fam_to_children(fam);
	if (!seq) {
		message(msg0);
		return NULL;
	}
	node = format_and_choose_indi(seq, FALSE, FALSE, ask1, msgn, msgn);
	remove_indiseq(seq, FALSE);
	return node;
}
/*=========================================
 * choose_spouse -- Choose person's spouse.
 *=======================================*/
NODE choose_spouse (indi, msg0, msgn)
NODE indi;
STRING msg0, msgn;
{
	INDISEQ seq;
	NODE node;
	INT i = 0, len;
	if (!indi) return NULL;
	seq = indi_to_spouses(indi);
	if (!seq) {
		message(msg0);
		return NULL;
	}
	node = format_and_choose_indi(seq, FALSE, TRUE, FALSE, NULL, msgn);
	remove_indiseq(seq, FALSE);
	return node;
}
/*=========================================================
 * choose_family -- Choose family from person's FAMS lines.
 *=======================================================*/
NODE choose_family (indi, msg0, msgn)
NODE indi;
STRING msg0, msgn;
{
	NODE node, fam, spouse;
	INT i = 0, len;
	char scratch[12];
	INDISEQ seq = indi_to_families(indi);
	if (!seq) {
		message(msg0);
		return NULL;
	}
	node = format_and_choose_indi(seq, TRUE, FALSE, FALSE, NULL, msgn);
	remove_indiseq(seq, FALSE);
	return node;
}
/*==================================================
 * format_indiseq -- Format print lines of sequence.
 *================================================*/
format_indiseq (seq, famp, marr)
INDISEQ seq;	/* sequence */
BOOLEAN famp;	/* seq of fams? */
BOOLEAN marr;	/* try to give marriage? */
{
	NODE indi, fam, spouse;
	INT ifkey;
	char scratch[20];
	STRING p;
	if (famp) {
		FORINDISEQ(seq, el, num)
			fam = key_to_fam(skey(el));
			if (sval(el)) {
				sprintf(scratch, "I%d", sval(el));
				spouse = key_to_indi(scratch);
			} else
				spouse = NULL;
			sprn(el) = indi_to_list_string(spouse, fam, 70);
		ENDINDISEQ
	} else {
		FORINDISEQ(seq, el, num)
			indi = key_to_indi(skey(el));
			if (marr) {
				sprintf(scratch, "F%d", sval(el));
				fam = key_to_fam(scratch);
			} else
				fam = NULL;
			sprn(el) = indi_to_list_string(indi, fam, 70);
		ENDINDISEQ
	}
}
