/*=============================================================
 * record.c -- Routines to handle BTREE records
 * Copyright(c) 1991-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 01 Jul 93
 *   3.0.0 - 24 Sep 94
 *===========================================================*/

#include "standard.h"
#include "btree.h"

/*=================================
 * addrecord -- Add record to BTREE
 *===============================*/
BOOLEAN addrecord (btree, rkey, record, len)
BTREE btree;	/* btree to add record to */
RKEY rkey;	/* key of record */
RECORD record;	/* record to add */
INT len;	/* record length */
{
	INDEX index;
	BLOCK old, new, xtra;
	FKEY nfkey, last = 0, parent;
	SHORT i, j, k, l, n, lo, hi;
	BOOLEAN found = FALSE;
	INT off = 0;
	FILE *fo, *fn1, *fn2;
	char scratch1[200], scratch2[200], *p = record;

/*wprintf("ADD: rkey = %s\n", rkey2str(rkey)); dumpbtree(btree); /*DEBUG*/
/* search for data block that does/should hold record */
	ASSERT(bwrite(btree));
	ASSERT(index = bmaster(btree));
	while (itype(index) == BTINDEXTYPE) {

/* maintain "lazy" parent chaining in btree */
		if (iparent(index) != last) {
			ASSERT(index != bmaster(btree));
			iparent(index) = last;
			writeindex(bbasedir(btree), index);
		}
		last = iself(index);
		n = nkeys(index);
		nfkey = fkeys(index, 0);
		for (i = 1; i <= n; i++) {
			if (strncmp(rkey.r_rkey, rkeys(index, i).r_rkey, 8) < 0)
				break;
			nfkey = fkeys(index, i);
		}
		ASSERT(index = getindex(btree, nfkey));
	}

/* have block that may hold older version of record */
	iparent(index) = last;
	old = (BLOCK) index;
	ASSERT(nkeys(old) < NORECS);

/* see if block has earlier version of record */
	lo = 0;
	hi = nkeys(old) - 1;
	found = FALSE;
	while (lo <= hi) {
		SHORT md = (lo + hi)/2;
		INT rel = strncmp(rkey.r_rkey, rkeys(old, md).r_rkey, 8);
		if (rel < 0)
			hi = --md;
		else if (rel > 0)
			lo = ++md;
		else {
			found = TRUE;
			lo = md;
			break;
		}
	}

/* construct header for updated data block */
	new = allocblock();
	itype(new) = itype(old);
	iparent(new) = iparent(old);
	iself(new) = iself(old);
	n = nkeys(new) = nkeys(old);

/* put info about all records up to new one in new header */
	for (i = 0; i < lo; i++) {
		rkeys(new, i) = rkeys(old, i);
		lens(new, i) = lens(old, i);
		offs(new, i) = off;
		off += lens(old, i);
	}

/* put info about added record in new header; may be new record */
	rkeys(new, lo) = rkey;
	lens(new, lo) = len;
	offs(new, lo) = off;
	off += len;

/* put info about all records after new one in new header */
	if (found)
		j = 0, i++;
	else
		j = 1;
	for (; i < n; i++) {
		rkeys(new, i + j) = rkeys(old, i);
		lens(new, i + j) = lens(old, i);
		offs(new, i + j) = off;
		off += lens(old, i);
	}
	if (!found) nkeys(new) = n + 1;

/* must rewrite data block with new record; open original and new */
	sprintf(scratch1, "%s/%s", bbasedir(btree), fkey2path(iself(old)));
	ASSERT(fo = fopen(scratch1, "r"));
	sprintf(scratch1, "%s/tmp1", bbasedir(btree));
	ASSERT(fn1 = fopen(scratch1, "w"));

/* see if new record must cause data block split */
	if (!found && n == NORECS - 1) goto splitting;

/* no split; write new header and preceding records to temp file */
	ASSERT(fwrite(new, BUFLEN, 1, fn1) == 1);
	putheader(btree, new);
	for (i = 0; i < lo; i++) {
		if (fseek(fo, (long)(offs(old, i) + BUFLEN), 0))
			FATAL();
		filecopy(fo, lens(old, i), fn1);
	}

/* write new record to temp file */
	p = record;
	while (len >= BUFLEN) {
		ASSERT(fwrite(p, BUFLEN, 1, fn1) == 1);
		len -= BUFLEN;
		p += BUFLEN;
	}
	if (len && fwrite(p, len, 1, fn1) != 1) FATAL();

/* write rest of records to temp file */
	if (found) i++;
	for ( ; i < n; i++) {
		if (fseek(fo, (long)(offs(old, i)+BUFLEN), 0)) FATAL();
		filecopy(fo, lens(old, i), fn1);
	}

/* make changes permanent in database */
	fclose(fn1);
	fclose(fo);
	sprintf(scratch1, "%s/tmp1", bbasedir(btree));
	sprintf(scratch2, "%s/%s", bbasedir(btree), fkey2path(iself(old)));
	stdfree(old);
	movefiles(scratch1, scratch2);
	return TRUE;	/* return point for non-splitting case */

/* data block must be split for new record; open second temp file */
splitting:
	sprintf(scratch1, "%s/tmp2", bbasedir(btree));
	ASSERT(fn2 = fopen(scratch1,"w"));

/* write header and 1st half of records; don't worry where new record goes */
	nkeys(new) = n/2;	/* temporary */
	ASSERT(fwrite(new, BUFLEN, 1, fn1) == 1);
	putheader(btree, new);
	for (i = j = 0; j < n/2; j++) {
		if (j == lo) {
			p = record;
			while (len >= BUFLEN) {
				ASSERT(fwrite(p, BUFLEN, 1, fn1) == 1);
				len -= BUFLEN;
				p += BUFLEN;
			}
			if (len && fwrite(p, len, 1, fn1) != 1)
				FATAL();
		} else {
			if (fseek(fo, (long)(offs(old, i) + BUFLEN), 0))
				FATAL();
			filecopy(fo, lens(old, i), fn1);
			i++;
		}
	}

/* create and write new block header */
	nfkey = iself(new);
	parent = iparent(new);
	xtra = crtblock(btree);
	iparent(xtra) = parent;
	off = 0;
	for (k = 0, l = n/2; k < n - n/2 + 1; k++, l++) {
		rkeys(xtra, k) = rkeys(new, l);
		lens(xtra, k) = lens(new, l);
		offs(xtra, k) = off;
		off += lens(new, l);
	}
	nkeys(xtra) = n - n/2 + 1;
	ASSERT(fwrite(xtra, BUFLEN, 1, fn2) == 1);
	putheader(btree, xtra);

/* write second half of records to second temp file */
	for (j = n/2; j <= n; j++) {
		if (j == lo) {
			p = record;
			while (len >= BUFLEN) {
				ASSERT(fwrite(p, BUFLEN, 1, fn2) == 1);
				len -= BUFLEN;
				p += BUFLEN;
			}
			if (len && fwrite(p, len, 1, fn2) != 1)
				FATAL();
		} else {
			if (fseek(fo, (long)(offs(old, i) + BUFLEN), 0))
				FATAL();
			filecopy(fo, lens(old, i), fn2);
			i++;
		}
	}

/* make changes permanent in database */
	fclose(fo);
	fclose(fn1);
	fclose(fn2);
	stdfree(old);
	sprintf(scratch1, "%s/tmp1", bbasedir(btree));
	sprintf(scratch2, "%s/%s", bbasedir(btree), fkey2path(nfkey));
	movefiles(scratch1, scratch2);
	sprintf(scratch1, "%s/tmp2", bbasedir(btree));
	sprintf(scratch2, "%s/%s", bbasedir(btree), fkey2path(iself(xtra)));
	movefiles(scratch1, scratch2);

/* add index of new data block to its parent (may cause more splitting) */
	addkey(btree, parent, rkeys(xtra, 0), iself(xtra));
	return TRUE;
}
/*======================================================
 * filecopy -- Copy record from one data file to another
 *====================================================*/
filecopy (fo, len, fn)
FILE *fo, *fn;
INT len;
{
	char buffer[BUFLEN];
	while (len >= BUFLEN) {
		ASSERT(fread(buffer, BUFLEN, 1, fo) == 1);
		ASSERT(fwrite(buffer, BUFLEN, 1, fn) == 1);
		len -= BUFLEN;
	}
	if (len) {
		ASSERT(fread((char *)buffer, len, 1, fo) == 1);
		ASSERT(fwrite(buffer, len, 1, fn) == 1);
	}
}
/*===================================
 * getrecord -- Get record from BTREE
 *=================================*/
RECORD getrecord (btree, rkey, plen)
BTREE btree;
RKEY rkey;
INT *plen;
{
	INDEX index;
	SHORT i, n, lo, hi;
	FKEY nfkey;
	BLOCK block;
	BOOLEAN found = FALSE;
	char scratch[200];
	FILE *fr;
	RECORD record;
	INT len;

/*wprintf("GETRECORD: rkey: %s\n", rkey2str(rkey)); /*DEBUG*/
	*plen = 0;
	ASSERT(index = bmaster(btree));

/* search for data block that does/should hold record */
	while (itype(index) == BTINDEXTYPE) {
		n = nkeys(index);
		nfkey = fkeys(index, 0);
		for (i = 1; i <= n; i++) {
			if (strncmp(rkey.r_rkey, rkeys(index, i).r_rkey, 8) < 0)
				break;
			nfkey = fkeys(index, i);
		}
		ASSERT(index = getindex(btree, nfkey));
	}

/* Found block that may hold record - search for key */
	block = (BLOCK) index;
	lo = 0;
	hi = nkeys(block) - 1;
	while (lo <= hi) {
		SHORT md = (lo + hi)/2;
		INT rel = strncmp(rkey.r_rkey, rkeys(block, md).r_rkey, 8);
		if (rel < 0)
			hi = --md;
		else if (rel > 0)
			lo = ++md;
		else {
			found = TRUE;
			lo = md;
			break;
		}
	}
	if (!found) return NULL;

	sprintf(scratch, "%s/%s", bbasedir(btree), fkey2path(iself(block)));
	ASSERT(fr = fopen(scratch, "r"));
	if (fseek(fr, (long)(offs(block, lo) + BUFLEN), 0)) FATAL();
	if ((len = lens(block, lo)) == 0) {
		*plen = 0;
		fclose(fr);
		return NULL;
	}
	record = (RECORD) stdalloc(len + 1);
	ASSERT(fread(record, len, 1, fr) == 1);
	fclose(fr);
	record[len] = 0;
	*plen = len;
	return record;
}
/*=======================================
 * movefiles -- Move first file to second
 *=====================================*/
movefiles (from, to)
STRING from, to;
{
	unlink(to);
	link(from, to);
	unlink(from);
}
