/*=================================================================
 * lbrowse.c -- Handle list browse mode.
 * Copyright (c) 1993 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 01 Sep 93 - modified
 *=================================================================
 */
#include <stdio.h>
#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "indiseq.h"

#define VIEWABLE 12

extern STRING lstnam, lstnon, lstwht, lstnad, lstpad, lstbot, lsttop;
extern STRING idplst, lstnew, mrkper;

INDISEQ current_seq = NULL;
LIST browse_lists;

typedef struct {
	STRING bl_name;
	INDISEQ bl_seq;
} *BLEL, BLEL_struct;

/*=====================================================
 *  init_browse_lists -- Initialize named browse lists.
 *===================================================*/
init_browse_lists ()
{
	browse_lists = create_list();
}
/*===========================================
 *  add_browse_list -- Add named browse list.
 *=========================================*/
add_browse_list (name, seq)
STRING name;
INDISEQ seq;
{
	BLEL blel;
	BOOLEAN done = FALSE;
	if (!name) return;
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		if (blel->bl_name && !strcmp(name, blel->bl_name)) {
			remove_indiseq(blel->bl_seq, FALSE);
			blel->bl_seq = seq;
			done = TRUE;
			break;
		}
	ENDLIST
	if (done) return;
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		if (!blel->bl_name) {
			blel->bl_name = name;
			blel->bl_seq = seq;
			return;
		}
	ENDLIST
 	blel = (BLEL) stdalloc(sizeof(BLEL_struct));
	blel->bl_name = name;
	blel->bl_seq = seq;
	enqueue_list(browse_lists, blel);
}
/*=================================================
 *  remove_browse_list -- Remove named browse list.
 *===============================================*/
remove_browse_list (name, seq)
STRING name;
INDISEQ seq;
{
	BLEL blel;
	remove_indiseq(seq, FALSE);
	if (!name) return;
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		if (blel->bl_name && !strcmp(name, blel->bl_name)) {
			blel->bl_name = NULL;
			blel->bl_seq = NULL;
		}
	ENDLIST
}
/*===========================================
 *  find_named_seq -- Find named browse list.
 *=========================================*/
INDISEQ find_named_seq (name)
STRING name;
{
	BLEL blel;
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		if (!strcmp(name, blel->bl_name))
			return copy_indiseq(blel->bl_seq);
	ENDLIST
	return NULL;
}
/*===================================================
 *  new_name_browse_list -- Rename named browse list.
 *=================================================*/
new_name_browse_list (old, new)
STRING old, new;
{
	BLEL blel;
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		if (!strcmp(old, blel->bl_name)) {
			stdfree(blel->bl_name);
			blel->bl_name = new;
			return;
		}
	ENDLIST
}
/*===================================================
 *  update_browse_list -- Assign name to browse list.
 *=================================================*/
update_browse_list (name, seq)
STRING name;
INDISEQ seq;
{
	BLEL blel;
	if (!name) {	/* remove anonymous lists */
		remove_indiseq(seq, FALSE);
		return;
	}
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		if (!strcmp(name, blel->bl_name))
			blel->bl_seq = seq;
	ENDLIST
}
/*==============================================================
 * remove_from_browse_lists -- Remove stale elements from lists.
 *============================================================*/
remove_from_browse_lists (key)
STRING key;
{
	BLEL blel;
	INDISEQ seq;
	extern INDISEQ current_seq;
	if (current_seq) {
		seq = current_seq;
		while (delete_indiseq(seq, key, NULL, 0))
			;
	}
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		seq = blel->bl_seq;
		while (delete_indiseq(seq, key, NULL, 0))
			;
	ENDLIST
}
/*================================================================
 * rename_from_browse_lists -- Re-figures name of possible element
 *   in browse lists.
 *==============================================================*/
rename_from_browse_lists (key)
STRING key;
{
	INDISEQ seq;
	BLEL blel;
	if (current_seq) {
		seq = current_seq;
		rename_indiseq(seq, key);
	}
	FORLIST(browse_lists, e)
		blel = (BLEL) e;
		seq = blel->bl_seq;
		rename_indiseq(seq, key);
	ENDLIST
}
/*========================================
 * browse_list -- Handle list browse mode.
 *======================================*/
INT browse_list (pindi1, pindi2, pfam1, pfam2, pseq)
NODE *pindi1, *pindi2, *pfam1, *pfam2;
INDISEQ *pseq;
{
	INT c, top, cur, mark, len, tmp, rc;
	STRING key, name, newname, lname;
	NODE indi;
	INDISEQ seq, newseq;

	current_seq = NULL;
	if (!pseq || !(seq = *pseq) || (len = length_indiseq(seq)) <= 0)
		return  BROWSE_QUIT;
	top = cur = 0;
	mark =  -1;
	element_indiseq(seq, cur, &key, &name);
	indi = key_to_indi(key);
	current_seq = seq;

	while (TRUE) {
		switch (c = list_browse(seq, top, cur, mark)) {
		case 'j':	/* Move down line */
			if (cur >= len - 1) {
				message(lstbot);
				break;
			}
			cur++;
			element_indiseq(seq, cur, &key, &name);
			indi = key_to_indi(key);
			if (cur >= top + VIEWABLE) top++;
			break;
		case 'k':	/* Move up line */
			if (cur <= 0) {
				message(lsttop);
				break;
			}
			cur--;
			element_indiseq(seq, cur, &key, &name);
			indi = key_to_indi(key);
			if (cur + 1 == top) top--;
			break;
		case 'e':	/* Edit current person */
			indi = edit_indi(indi);
	    		if ((len = length_indiseq(seq)) <= 0) {
				remove_browse_list(lname, seq);
				current_seq = NULL;
				lname = NULL;
				return BROWSE_QUIT;
			}
			if (cur >= len) cur = len - 1;
			break;
		case 'i':	/* Browse current person */
			*pindi1 = indi;
			if (current_seq)
				remove_indiseq(current_seq, FALSE);
			current_seq = NULL;
			return BROWSE_INDI;
		case 'm':	/* Mark current person */
			mark = (cur == mark) ? -1: cur;
			break;
		case 'd':	/* Delete person from list */
			if (len <= 1) {
				if (current_seq)
					remove_indiseq(current_seq, FALSE);
				current_seq = NULL;
				return BROWSE_QUIT;
			}
			delete_indiseq(seq, NULL, NULL, cur);
			len--;
			if (mark == cur) mark = -1;
			if (mark > cur) mark--;
			if (cur == len)
				cur--;
				element_indiseq(seq, cur, &key, &name);
				indi = key_to_indi(key);
			if (cur < top) top = cur;
			break;
		case 't':	/* Enter tandem mode */
			if (mark == -1 || cur == mark) {
				message(mrkper);
				break;
			}
			*pindi2 = indi;
			element_indiseq(seq, mark, &key, &name);
			*pindi1 = key_to_indi(key);
			current_seq = NULL;
			return BROWSE_TAND;
		case 'b':	/* Browse new persons */
			newseq = (INDISEQ) ask_for_indiseq(idplst, &rc);
			if (!newseq) break;
			current_seq = seq = newseq;
			element_indiseq(seq, 0, &key, &name);
			indi = key_to_indi(key);
			if ((len = length_indiseq(seq)) == 1) {
				*pindi1 = indi;
				remove_indiseq(newseq, FALSE);
				current_seq = NULL;
				return BROWSE_INDI;
			}
			top = cur = 0;
			mark = -1;
			break;
		case 'a':	/* Add persons to current list */
			newseq = (INDISEQ) ask_for_indiseq(lstpad, &rc);
			if (!newseq) {
				message(lstnad);
				break;
			}
			FORINDISEQ(newseq, e, i)
				append_indiseq(seq, skey(e), snam(e), NULL,
				    FALSE, FALSE);
			ENDINDISEQ
			namesort_indiseq(seq);
			cur = top = 0;
			mark = -1;
			len = length_indiseq(seq);
			remove_indiseq(newseq, FALSE);
			message(lstnew);
			break;
		case 'n':	/* Name this list */
			newname = ask_for_string(lstwht, "enter name: ");
			if (!newname)
				message(lstnon);
			else {
				newname = strsave(newname);
				add_browse_list (newname, copy_indiseq(seq));
				mprintf(lstnam, newname);
			}
			break;
		case 'x':	/* Swap current with marked */
			if (mark == -1) break;
			tmp = mark;
			mark = cur;
			cur = tmp;
			element_indiseq(seq, cur, &key, &name);
			indi = key_to_indi(key);
			if (cur < top) top = cur;
			if (cur > top + VIEWABLE - 1) top = cur;
			break;
		case 'q':	/* Return to main menu */
		default:
			current_seq = NULL;
			return BROWSE_QUIT;
		}
	}
}
