/*==============================================================
 * alloc.c -- Memory allocation part of the program interpreter.
 * Copyright(c) by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 17 Aug 93 - modified
 *===============================================================
 */
#include <stdio.h>
#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "interp.h"

extern STRING ierror;

/*==========================================
 * alloc_interp -- Allocates an INTERP node.
 *========================================*/
INTERP alloc_interp (type)
char type;
{
	INTERP node = (INTERP) stdalloc(sizeof(*node));
	itype(node) = type;
	node->i_word1 = node->i_word2 = node->i_word3 = NULL;
	node->i_word4 = node->i_word5 = NULL;
	node->i_next = NULL;
	node->i_line = rplineno;
	return node;
}
/*===================================================
 * literal_node -- Create an INTERP node for literal.
 *=================================================*/
INTERP literal_node (str)
STRING str;
{
	INTERP node = alloc_interp(ILITERAL);
	iliteral(node) = strsave(str);
	return node;
}
/*==========================================================
 * children_node -- Create an INTERP node for children loop.
 *========================================================*/
INTERP children_node (fexpr, cvar, nvar, body)
INTERP fexpr;		/* expr */
STRING cvar, nvar;	/* child & counter */
INTERP body;		/* loop body */
{
	INTERP node = alloc_interp(ICHILDREN);
	ifamily(node) = (WORD) fexpr;
	ichild(node) = (WORD) cvar;
	inum(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*========================================================
 * spouses_node -- Create an INTERP node for spouses loop.
 *======================================================*/
INTERP spouses_node (pexpr, svar, fvar, nvar, body)
INTERP pexpr;	/* expr */
STRING svar;	/* spouse */
STRING fvar;	/* family */
STRING nvar;	/* counter */
INTERP body;	/* loop body */
{
	INTERP node = alloc_interp(ISPOUSES);
	iprinc(node) = (WORD) pexpr;
	ispouse(node) = (WORD) svar;
	ifamvar(node) = (WORD) fvar;
	inum(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*==========================================================
 * families_node -- Create an INTERP node for families loop.
 *========================================================*/
INTERP families_node (pexpr, fvar, svar, nvar, body)
INTERP pexpr;	/* expr */
STRING fvar;	/* family */
STRING svar;	/* spouse */
STRING nvar;	/* counter */
INTERP body;	/* loop body */
{
	INTERP node = alloc_interp(IFAMILIES);
	iprinc(node) = (WORD) pexpr;
	ifamvar(node) = (WORD) fvar;
	ispouse(node) = (WORD) svar;
	inum(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*=========================================================
 * forindiset_node -- Create an INTERP node for index loop.
 *=======================================================*/
INTERP forindiset_node (iexpr, ivar, vvar, nvar, body)
INTERP iexpr;	/* expr */
STRING ivar;	/* person */
STRING vvar;	/* value */
STRING nvar;	/* counter */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(IINDICES);
	iindex(node) = (WORD) iexpr;
	iindivar(node) = (WORD) ivar;
	ivalvar(node) = (WORD) vvar;
	inum(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*=====================================================
 * forlist_node -- Create an INTERP node for list loop.
 *===================================================*/
INTERP forlist_node (iexpr, evar, nvar, body)
INTERP iexpr;	/* expr */
STRING evar;	/* element */
STRING nvar;	/* counter */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(ILIST);
	ilist(node) = (WORD) iexpr;
	ielement(node) = (WORD) evar;
	inum(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*========================================================
 * forindi_node -- Create an INTERP node for forindi loop.
 *======================================================*/
INTERP forindi_node (ivar, nvar, body)
STRING ivar;	/* person */
STRING nvar;	/* counter */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(IINDI);
	iindivar(node) = (WORD) ivar;
	inum(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*======================================================
 * forfam_node -- Create an INTERP node for forfam loop.
 *====================================================*/
INTERP forfam_node (fvar, nvar, body)
STRING fvar;	/* family */
STRING nvar;	/* counter */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(IFAM);
	iindivar(node) = (WORD) fvar;
	inum(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*==========================================================
 * fornotes_node -- Create an INTERP node for fornotes loop.
 *========================================================*/
INTERP fornotes_node (nexpr, vvar, body)
INTERP nexpr;	/* expr */
STRING vvar;	/* value */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(INOTES);
	inode(node) = (WORD) nexpr;
	istrng(node) = (WORD) vvar;
	ibody(node) = (WORD) body;
	return node;
}
/*==========================================================
 * fornodes_node -- Create an INTERP node for fornodes loop.
 *========================================================*/
INTERP fornodes_node (nexpr, nvar, body)
INTERP nexpr;	/* expr */
STRING nvar;	/* node (next level) */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(INODES);
	inode(node) = (WORD) nexpr;
	isubnode(node) = (WORD) nvar;
	ibody(node) = (WORD) body;
	return node;
}
/*==========================================================
 * traverse_node -- Create an INTERP node for traverse loop.
 *========================================================*/
INTERP traverse_node (nexpr, snode, levv, body)
INTERP nexpr;	/* node */
STRING snode;	/* subnode */
STRING levv;	/* level */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(ITRAV);
	inode(node) = (WORD) nexpr;
	isubnode(node) = (WORD) snode;
	ilev(node) = (WORD) levv;
	ibody(node) = (WORD) body;
	return node;
}
/*===================================================
 * iden_node -- Create an INTERP node for identifier.
 *=================================================*/
INTERP iden_node (iden)
STRING iden;
{
	INTERP node = alloc_interp(IIDENT);
	iident(node) = (WORD) iden;
	return node;
}
/*=================================================
 * icons_node -- Create an INTERP node for integer.
 *===============================================*/
INTERP icons_node (ival)
INT ival;
{
	INTERP node = alloc_interp(IICONS);
	iicons(node) = (WORD) ival;
	return node;
}
/*==================================================
 * proc_node -- Create an INTERP node for procedure.
 *================================================*/
INTERP proc_node (name, parms, body)
STRING name;	/* proc name */
INTERP parms;	/* param/s */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(IPROC);
	iname(node) = (WORD) name;
	iparams(node) = (WORD) parms;
	ibody(node) = (WORD) body;
	return node;
}
/*=====================================================
 * func_node -- Create an INTERP node for builtin call.
 *===================================================*/
INTERP func_node (name, elist)
STRING name;	/* proc name */
INTERP elist;	/* param/s */
{
	INTERP node;
	INT lo = 0, hi = nobuiltins - 1, md, n, r;
	BOOLEAN found = FALSE;
	while (lo <= hi) {
		md = (lo + hi) >> 1;
		if ((r = strcmp(name, builtins[md].ft_name)) < 0)
			hi = md - 1;
		else if (r > 0)
			lo = md + 1;
		else {
			found = TRUE;
			break;
		}
	}
	if (!found) {
		wprintf(ierror, rplineno);
		wprintf("%s is not a built-in function.\n", name);
		semerrors++;
	} else if ((n = num_params(elist)) < builtins[md].ft_nparms_min
		    && n > builtins[md].ft_nparms_max) {
		wprintf(ierror, rplineno);
		wprintf("%s: should have %d to %d parameters.\n", name,
		    builtins[md].ft_nparms_min, builtins[md].ft_nparms_max);
		semerrors++;
	}
	node = alloc_interp(IFUNC);
	iname(node) = (WORD) name;
	ielist(node) = (WORD) elist;
	ifunc(node) = (WORD) builtins[md].ft_eval;
	return node;
}
/*===================================================
 * if_node -- Create an INTERP node for if structure.
 *=================================================*/
INTERP if_node (cond, tnode, enode)
INTERP cond;	/* cond expr */
INTERP tnode;	/* then */
INTERP enode;	/* else */
{
	INTERP node = alloc_interp(IIF);
	icond(node) = (WORD) cond;
	ithen(node) = (WORD) tnode;
	ielse(node) = (WORD) enode;
	return node;
}
/*=========================================================
 * while_node -- Create an INTERP node for while structure.
 *=======================================================*/
INTERP while_node (cond, body)
INTERP cond;	/* cond expr */
INTERP body;	/* body */
{
	INTERP node = alloc_interp(IWHILE);
	icond(node) = (WORD) cond;
	ibody(node) = (WORD) body;
	return node;
}
/*=======================================================
 * call_node -- Create an INTERP node for procedure call.
 *=====================================================*/
INTERP call_node (name, args)
STRING name;	/* proc name */
INTERP args;	/* arg/s */
{
	INTERP node = alloc_interp(ICALL);
	iname(node) = (WORD) name;
	iargs(node) = (WORD) args;
	return node;
}
