/*================================================================
 * xreffile.c -- Handles the xref file.
 * Copyright(c) 1991 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled 
 *   Version 2.3.5 - 02 Sep 93 - modified 
 *================================================================
 */
#include "standard.h"
#include "btree.h"
#include "table.h"
#include "gedcom.h"

extern BTREE BTR;

/*===================================================================
 * First two words in xrefs file are number of INDI and FAM keys in
 *   file; remaining words are keys, INDIs first, then FAMs; first in
 *   each group is next unused key; rest are keys of deleted records.
 *=================================================================*/

static INT nixrefs;	/* num of INDI keys */
static INT nfxrefs;	/* num of FAM keys */
static INT *ixrefs;	/* list of INDI keys */
static INT *fxrefs;	/* list of FAM keys */
static INT maxixrefs = 0;
static INT maxfxrefs = 0;
static FILE *xreffp;	/* open xref file pointer */
static BOOLEAN xrefopen = FALSE;

/*=============================== 
 * initxref -- Create xrefs file.
 *=============================*/
initxref ()
{
	char scratch[100];
	INT i = 1;
	ASSERT(!xrefopen);
	sprintf(scratch, "%s/xrefs", BTR->b_basedir);
	ASSERT(xreffp = fopen(scratch, "w"));
	if (fwrite(&i, sizeof(INT), 1, xreffp) != 1 ||
	    fwrite(&i, sizeof(INT), 1, xreffp) != 1 ||
	    fwrite(&i, sizeof(INT), 1, xreffp) != 1 ||
	    fwrite(&i, sizeof(INT), 1, xreffp) != 1)
		FATAL();
	fclose(xreffp);
}
/*=============================
 * openxref -- Open xrefs file.
 *===========================*/
BOOLEAN openxref ()
{
	char scratch[100];
	ASSERT(!xrefopen);
	sprintf(scratch, "%s/xrefs", BTR->b_basedir);
	if (!(xreffp = fopen(scratch, "r+"))) return FALSE;
	xrefopen = TRUE;
	return readxrefs();
}
/*===============================
 * closexref -- Close xrefs file.
 *=============================*/
closexref ()
{
	fclose(xreffp);
	xrefopen = FALSE;
}
/*=====================================
 * getixref -- Return next ixref value.
 *===================================*/
STRING getixref ()
{
	INT n;
	static char scratch[10];
	if (!xrefopen || nixrefs < 1) FATAL();
	if (nixrefs == 1)
		n = (ixrefs[0])++;
	else
		n = ixrefs[--nixrefs];
	ASSERT(writexrefs());
	sprintf(scratch, "@I%d@", n);
	return scratch;
}
/*=====================================
 * getfxref -- Return next fxref value.
 *===================================*/
STRING getfxref ()
{
	INT n;
	static char scratch[10];
	if (!xrefopen || nfxrefs < 1) FATAL();
	if (nfxrefs == 1)
		n = (fxrefs[0])++;
	else
		n = fxrefs[--nfxrefs];
	ASSERT(writexrefs());
	sprintf(scratch, "@F%d@", n);
	return scratch;
}
/*==============================
 * readxrefs -- Read xrefs file.
 *============================*/
BOOLEAN readxrefs ()
{
	ASSERT(xrefopen);
	if (fread(&nixrefs, sizeof(INT), 1, xreffp) != 1 ||
	    fread(&nfxrefs, sizeof(INT), 1, xreffp) != 1) return FALSE;
	if (nixrefs > maxixrefs) growixrefs();
	if (nfxrefs > maxfxrefs) growfxrefs();
	if (nixrefs && fread(ixrefs, sizeof(INT), nixrefs, xreffp) != nixrefs)
		return FALSE;
	if (nfxrefs && fread(fxrefs, sizeof(INT), nfxrefs, xreffp) != nfxrefs)
		return FALSE;
	return TRUE;
}
/*================================
 * writexrefs -- Write xrefs file.
 *==============================*/
writexrefs ()
{
	ASSERT(xrefopen);
	rewind(xreffp);
	if (fwrite(&nixrefs, sizeof(INT), 1, xreffp) != 1 ||
	    fwrite(&nfxrefs, sizeof(INT), 1, xreffp) != 1 ||
	    fwrite(ixrefs, sizeof(INT), nixrefs, xreffp) != nixrefs ||
	    fwrite(fxrefs, sizeof(INT), nfxrefs, xreffp) != nfxrefs)
		FATAL();
	fflush(xreffp);
	return TRUE;
}
/*============================================
 * addixref -- Add deleted INDI key to ixrefs.
 *==========================================*/
addixref (key)
INT key;
{
	if (key <= 0 || !xrefopen || nixrefs < 1) FATAL();
	if (nixrefs >= maxixrefs) growixrefs();
	ixrefs[nixrefs++] = key;
	ASSERT(writexrefs());
}
/*===========================================
 * addfxref -- Add deleted FAM key to fxrefs.
 *=========================================*/
addfxref (key)
INT key;
{
	if (key <= 0 || !xrefopen || nfxrefs < 1) FATAL();
	if (nfxrefs >= maxfxrefs) growfxrefs();
	fxrefs[nfxrefs++] = key;
	ASSERT(writexrefs());
}
/*============================================
 * growixrefs -- Grow memory for ixrefs array.
 *==========================================*/
growixrefs ()
{
	INT i, m = maxixrefs, *newp;
	maxixrefs = nixrefs + 10;
	newp = (INT *) stdalloc(maxixrefs*sizeof(INT));
	if (m) {
		for (i = 0; i < nixrefs; i++)
			newp[i] = ixrefs[i];
		stdfree(ixrefs);
	}
	ixrefs = newp;
}
/*============================================
 * growfxrefs -- Grow memory for fxrefs array.
 *==========================================*/
growfxrefs ()
{
	INT i, m = maxfxrefs, *newp;
	maxfxrefs = nfxrefs + 10;
	newp = (INT *) stdalloc(maxfxrefs*sizeof(INT));
	if (m) {
		for (i = 0; i < nfxrefs; i++)
			newp[i] = fxrefs[i];
		stdfree(fxrefs);
	}
	fxrefs = newp;
}
/*===================================================
 * num_indis -- Return number of persons in database.
 *=================================================*/
INT num_indis ()
{
	return ixrefs[0] - nixrefs;
}
/*===================================================
 * num_fams -- Return number of families in database.
 *=================================================*/
INT num_fams ()
{
	return fxrefs[0] - nfxrefs;
}
