#ifndef LINT
static char * sccsdef = "@(#)socket1.c	1.1	(Alex Crain) 6/20/89";
#endif

/*
 *  socket.c - high level socket routines
 *
 *  Written by Alex Crain.
 *
 *  This file is based in the Berkeley file uipc_socket.c,
 *  but is *not* guarenteed to be in any way compatable. It is
 *  close enough to the Berkeley code that the following applies...
 *
 *  Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 *  All rights reserved.
 * 
 *  Redistribution and use in source and binary forms are permitted
 *  provided that this notice is preserved and that due credit is given
 *  to the University of California at Berkeley. The name of the University
 *  may not be used to endorse or promote products derived from this
 *  software without specific prior written permission. This software
 *  is provided "as is" without express or implied warranty.
 *
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/file.h>
#include <sys/var.h>
#include <sys/errno.h>
#include <uipc/mbuf.h>
#include <uipc/socket.h>
#include <uipc/socketvar.h>
#include <uipc/domain.h>
#include <uipc/protosw.h>
#include <uipc/fproto.h>


int
socreate (domain, sop, type, proto)
  int domain;
  struct socket ** sop;
  int type, proto;
{
   register struct protosw * prp;
   struct socket * so;
   struct mbuf * m;
   int error = 0;
   
   if (proto)
       prp = pffindproto (domain, proto, type);
   else
       prp = pffindtype (domain, type);
   
   if (prp == 0)
       return EPROTONOSUPPORT;

   if (prp->pr_type != type)
       return EPROTOTYPE;

   m = m_getclr (M_WAIT, MT_SOCKET);
   so = mtod (m, struct socket *);
   
   so->so_options = 0;
   so->so_state = (suser () ? SS_PRIV : 0);
   so->so_type = type;
   so->so_proto = prp;

   if (error = (* prp->pr_usrreq) (so, PRU_ATTACH, (struct mbuf *) 0,
			       (struct mbuf *) proto, (struct mbuf *) 0))
    {
       so->so_state |= SS_NOFDREF;
       sofree(so);
       return error;
    }
   * sop = so;
   return 0;
}

int
sobind (so, nam)
  struct socket * so;
  struct mbuf * nam;
{
   int s = splnet ();
   int error = (* so->so_proto->pr_usrreq) (so, PRU_BIND, 
		(struct mbuf *) 0, nam, (struct mbuf *) 0);

   splx (s);
   return error;
}

int
solisten (so, backlog)
  struct socket * so;
  int backlog;
{
   int s = splnet ();
   int error;
   if (error = (* so->so_proto->pr_usrreq) (so, PRU_LISTEN,
		    (struct mbuf *) 0, (struct mbuf *) 0, (struct mbuf *) 0))
       goto bad;
   if (so->so_q == 0)
    {
       so->so_q = so;
       so->so_q0 = so;
       so->so_options |= SO_ACCEPTCONN;
    }

   if (backlog < 0)
       backlog = 0;
   so->so_qlimit = MIN (backlog, SOMAXCONN);
 bad:
   splx (s);
   return error;
}    

void   
sofree (so)
  struct socket * so;
{
   if (so->so_pcb || (so->so_state & SS_NOFDREF) == 0)
       return;
   if (so->so_head)
    {
       if (! soqremque (so, 0) && ! soqremque (so, 1))
	   panic ("sofree dq");
       so->so_head = 0;
    }
   sbrelease (&so->so_snd);
   sorflush (so);
   (void) m_free (dtom (so));
}

int
soclose (so)
  struct socket * so;
{
   int s = splnet ();
   int error = 0;

   if (so->so_options & SO_ACCEPTCONN)
    {
       while (so->so_q0 != so)
	   (void) soabort (so->so_q0);
       while (so->so_q != so)
	   (void) soabort (so->so_q);
    }
   if (so->so_pcb == 0)
       goto discard;
   if (so->so_state & SS_ISCONNECTED)
    {
       if ((so->so_state & SS_ISDISCONNECTING) == 0)
	   if (error = sodisconnect (so))
	       goto drop;
       if (so->so_options & SO_LINGER)
	{
	   if ((so->so_state & SS_ISDISCONNECTING) &&
	       (so->so_state & SS_NBIO))
	       goto drop;
	   while (so->so_state & SS_ISCONNECTED)
	       (void) sleep ((caddr_t) &so->so_timeo, PZERO + 1);
	}
    }

 drop:
   if (so->so_pcb)
    {
       int error2 = (* so->so_proto->pr_usrreq) (so, PRU_DETACH,
		      (struct mbuf *) 0, (struct mbuf *) 0, (struct mbuf *) 0);
       if (error == 0)
	   error = error2;
    }

 discard:
   if (so->so_state & SS_NOFDREF)
       panic ("soclose: NODEREF");
   so->so_state |= SS_NOFDREF;
   sofree (so);
   splx (s);
   return error;
}

int
soabort (so)
  struct socket * so;
{
   return (* so->so_proto->pr_usrreq) (so, PRU_ABORT,
		(struct mbuf *) 0, (struct mbuf *) 0, (struct mbuf *) 0);
}

int
soaccept (so, nam)
  struct socket * so;
  struct mbuf * nam;
{
   int s = splnet ();
   int error;

   if ((so->so_state & SS_NOFDREF) == 0)
       panic ("soaccept: !NOFDREF");
   so->so_state &= ~SS_NOFDREF;
   error = (* so->so_proto->pr_usrreq) (so, PRU_ACCEPT,
		(struct mbuf *) 0, nam, (struct mbuf *) 0);
   splx (s);
   return (error);
}

int
soconnect (so, nam)
  struct socket * so;
  struct mbuf * nam;
{
   int s;
   int error;

   if (so->so_options & SO_ACCEPTCONN)
       return EOPNOTSUPP;
   s = splnet ();
   if (so->so_state & (SS_ISCONNECTED | SS_ISCONNECTING) &&
       ((so->so_proto->pr_flags & PR_CONNREQUIRED) ||
	(error = sodisconnect (so))))
       error = EISCONN;
   else
       error = (* so->so_proto->pr_usrreq) (so, PRU_CONNECT,
		(struct mbuf *) 0, nam, (struct mbuf *) 0);
   splx (s);
   return error;
}

int
soconnect2 (so1, so2)
  struct socket * so1, * so2;
{
   int s = splnet ();
   int error = (* so1->so_proto->pr_usrreq) (so1, PRU_CONNECT2,
		(struct mbuf *) 0, (struct mbuf *) so2, (struct mbuf *) 0);
   splx (s);
   return error;
}

int
sodisconnect (so)
  struct socket * so;
{
   int s = splnet ();
   int error;

   if ((so->so_state & SS_ISCONNECTED) == 0)
    {
       error = ENOTCONN;
       goto bad;
    }
   if (so->so_state & SS_ISDISCONNECTING)
    {
       error = EALREADY;
       goto bad;
    }
   error = (* so->so_proto->pr_usrreq) (so, PRU_DISCONNECT,
		(struct mbuf *) 0, (struct mbuf *) 0, (struct mbuf *) 0);
 bad:
   splx (s);
   return error;
}

int
sosend (so, nam, flags, rights)
  struct socket * so;
  struct mbuf * nam;
  int flags;
  struct mbuf * rights;
{
   int space, s, rlen = 0, dontroute;
   int len, error = 0, first = 1;
   struct mbuf ** mb, * top = 0, * m;

   /*
    *  barf if we want to send one big chunk and don't have the space.
    */
   if (sendallatonce (so) && u.u_count > so->so_snd.sb_hiwat)
       return (EMSGSIZE);

   dontroute = (flags & MSG_DONTROUTE) && 
	       (so->so_options & SO_DONTROUTE) == 0 &&
	       (so->so_proto->pr_flags & PR_ATOMIC);

   if (rights)
       rlen = rights->m_len;
    
#define snderr(errno)	{ error = errno; splx (s); goto release; }

 restart:
   sblock (&so->so_snd);
   
   do {
       s = splnet ();

       /* check out our basic requirements. */

       if (so->so_state & SS_CANTSENDMORE)
	   snderr (EPIPE);

       if (so->so_error)
	{
	   error = so->so_error;
	   so->so_error = 0;
	   splx (s);
	   goto release;
	}
       
       if ((so->so_state & SS_ISCONNECTED) == 0)
	{
	   if (so->so_proto->pr_flags & PR_CONNREQUIRED)
	       snderr (ENOTCONN);
	   if (nam == 0)
	       snderr (EDESTADDRREQ);
	}
       if (flags & MSG_OOB)
	   space = 1024;
       else
	{
	   space = sbspace (&so->so_snd);

	   /*
	    *  If we need more room, wait for it.
	    */
	   if (space <= rlen || 
	       sendallatonce (so) && space < u.u_count + rlen)
	    {
	       if (so->so_state & SS_NBIO)
		{
		   if (first)
		       error = EWOULDBLOCK;
		   splx (s);
		   goto release;
		}
	       sbunlock (&so->so_snd);
	       sbwait (&so->so_snd);
	       splx (s);
	       goto restart;
	    }
	}

       splx (s);
       

       /*
	*  We have the room, we've done sanity checks.
	*  Now make an mbuf chain out of our data, waiting if necessarry.
	*/
       mb = &top;
       space -= rlen;
       while (space > 0 && u.u_count)
	{
	   MGET (m, M_WAIT, MT_DATA);
	   len = MIN (MIN (MLEN, u.u_count), space);
	   space -= len;
	   iomove (mtod (m, caddr_t), len, IO_WRITE);
	   m->m_len = len;
	   * mb = m;
	   if (error = u.u_error)
	       goto release;
	   mb = &m->m_next;
#ifdef DEBUG
	   dump_mbuf(m,"sosendit");
#endif
	}
       if (dontroute)
	   so->so_options |= SO_DONTROUTE;

       /*
	*  write mbuf to socket.
	*/
       s = splnet ();
       error = (* so->so_proto->pr_usrreq) (so, 
			  (flags & MSG_OOB) ? PRU_SENDOOB : PRU_SEND,
			  top, (caddr_t) nam, rights);
       splx (s);
       if (dontroute)
	   so->so_options &= ~SO_DONTROUTE;
       rights = top = (struct mbuf *) 0;
       rlen = first = 0;
    } while (error == 0 && u.u_count);

 release:
   sbunlock (&so->so_snd);

   if(select_sleep) {
      select_sleep = 0;
      wakeup((caddr_t) &select_sleep_addr);
   }

   if (top)
       m_freem (top);
   if (error == EPIPE)
       psignal (u.u_procp, SIGPIPE);
   return error;
}

int
soreceive (so, nam, flags, rights)
  struct socket * so;
  struct mbuf ** nam;
  int flags;
  struct mbuf ** rights;
{
   struct mbuf * m, * nextrecord;
   int len, error = 0, s, moff, offset;
   struct protosw * pr = so->so_proto;

   if (rights)
       * rights = (struct mbuf *) 0; /* if we have a rights, zero it */
   if (nam)
       * nam = (struct mbuf *) 0; /* zero from buffer address */
   if (flags & MSG_OOB)
    {
       m = m_get (M_WAIT, MT_DATA);
       if (error = (* pr->pr_usrreq) (so, PRU_RCVOOB, m,
				      (struct mbuf *) (flags & MSG_PEEK), 
				      (struct mbuf *) 0))
	   goto bad;
       do {
	   len = MIN (u.u_count, m->m_len);
	   iomove (mtod (m, caddr_t), len, IO_READ);
	   m = m_free (m);
	} while (u.u_count && (error = u.u_error) == 0 && m);
    bad:
       if (m)
	   m_freem (m);
       return error;
    }
	   
#ifdef DEBUG
   dump_mbuf (dtom (so),"sorecieve");
#endif
 restart:
   sblock (&so->so_rcv);
   s = splnet ();

   if (so->so_rcv.sb_cc == 0) /* if there is no chars in buffer */
    {
       if (so->so_error) /* if there is an error affecting connection */
	{
	   error = so->so_error;
	   so->so_error = 0;
	   goto release;
	}
/*       if ((so->so_state & SS_ISCONNECTED) == 0 &&
	   (pr->pr_flags & PR_CONNREQUIRED))
	{
	   error = ENOTCONN;
	   goto release;
	} */
       if (so->so_state & SS_CANTRCVMORE || u.u_count == 0)
	   goto release;
       if (so->so_state & SS_NBIO) /* if nonblocking return EWOULDBLOCK */
	{
	   error = EWOULDBLOCK;
	   goto release;
	}
       sbunlock (&so->so_rcv);
       sbwait (&so->so_rcv);
       splx (s);
       goto restart;
    }

   /* this checks the mbuf chain to see if there is data */
   if ((m = so->so_rcv.sb_mb) == (struct mbuf *) 0)
       panic ("receive 1");
   nextrecord = m->m_act;
   if (pr->pr_flags & PR_ADDR)
    {
       if (m->m_type != MT_SONAME)
	   panic ("receive 1a");
       if (flags & MSG_PEEK)
	{
	   if (nam)
	       * nam = m_copy (m, 0, m->m_len);
           m = m->m_next;
	}
       else
	{
	   sbfree (&so->so_rcv, m);
	   if (nam)
	    {
	       * nam = m;
	       m = m->m_next;
	       (* nam)->m_next = (struct mbuf *) 0;
	       so->so_rcv.sb_mb = m;
	    }
	   else
	    {
	       MFREE (m, so->so_rcv.sb_mb);
	       m = so->so_rcv.sb_mb;
	    }
	   if (m)
	       m->m_act = nextrecord;
	}
    }
   if (m && m->m_type == MT_RIGHTS)
    {
       if ((pr->pr_flags & PR_RIGHTS) == 0)
	   panic ("receive 2");
       if (flags & MSG_PEEK)
	{
	   if (rights)
	       * rights = m_copy (m, 0, m->m_len);
	   m = m->m_next;
	} 
       else
	{
	   sbfree (&so->so_rcv, m);
	   if (rights)
	    {
	       * rights = m;
	       so->so_rcv.sb_mb = m->m_next;
	       m->m_next = (struct mbuf *) 0;
	       m = so->so_rcv.sb_mb;
	    }
	   else
	    {
	       MFREE (m, so->so_rcv.sb_mb);
	       m = so->so_rcv.sb_mb;
	    }
	   if (m)
	       m->m_act = nextrecord;
	}
    }
   moff = 0;
   offset = 0;
   while (m && u.u_count != 0 && error == 0)
    {
       if (m->m_type != MT_DATA && m->m_type != MT_HEADER)
	   panic ("receive 3");
       len = u.u_count;
       so->so_state &= ~SS_RCVATMARK;
       if (so->so_oobmark && len > so->so_oobmark - offset)
	   len = so->so_oobmark - offset;
       if (len > m->m_len - moff)
	   len = m->m_len - moff;
       splx (s);

       iomove (mtod (m, caddr_t) + moff, (int) len, IO_READ);
       error = u.u_error;

       s = splnet ();
       if (len == m->m_len - moff)
	{
	   if (flags & MSG_PEEK)
	    {
	       m = m->m_next;
	       moff = 0;
	    }
	   else
	    {
	       nextrecord = m->m_act;
	       sbfree (&so->so_rcv, m);
	       MFREE (m, so->so_rcv.sb_mb);
	       if (m = so->so_rcv.sb_mb)
		   m->m_act = nextrecord;
	    }
	}
       else
	{
	   if (flags & MSG_PEEK)
	       moff += len;
	   else
	    {
	       m->m_off += len;
	       m->m_len -= len;
	       so->so_rcv.sb_cc -= len;
	    }
	}
       if (so->so_oobmark)
	{
	   if ((flags & MSG_PEEK) == 0)
	    {
	       so->so_oobmark -= len;
	       if (so->so_oobmark == 0)
		{
		   so->so_state |= SS_RCVATMARK;
		   break;
		}
	    }
	   else
	       offset += len;
	}
    }
   if ((flags & MSG_PEEK) == 0)
    {
       if (m == 0)
	   so->so_rcv.sb_mb = nextrecord;
       else if (pr->pr_flags & PR_ATOMIC)
	   (void) sbdroprecord (&so->so_rcv);
       if (pr->pr_flags & PR_WANTRCVD && so->so_pcb)
	   (* pr->pr_usrreq) (so, PRU_RCVD, (struct mbuf *) 0,
			      (struct mbuf *) 0, (struct mbuf *) 0);
       if (error == 0 && rights && * rights &&
	   pr->pr_domain->dom_externalize)
	   error = (* pr->pr_domain->dom_externalize) (* rights);
    }

 release:
   sbunlock (&so->so_rcv);
   splx (s);
   return error;
}

void	       
sorflush (so)
  struct socket * so;
{
   struct sockbuf * sb = &so->so_rcv;
   struct protosw * pr = so->so_proto;
   int s;
   struct sockbuf asb;

   sblock (sb);
   s = splimp ();
   socantrcvmore (so);
   sbunlock (sb);

   if(select_sleep) {
	select_sleep = 0;
	wakeup((caddr_t) &select_sleep_addr);
   }

   asb = * sb;
   bzero ((caddr_t) sb, sizeof (* sb));
   splx (s);
   if (pr->pr_flags & PR_RIGHTS && pr->pr_domain->dom_dispose)
     (* pr->pr_domain->dom_dispose) (asb.sb_mb);
   sbrelease (&asb);
}

int
sosetopt (so, level, optname, m0)
  struct socket * so;
  int level, optname;
  struct mbuf * m0;
{
   int error = 0;
   struct mbuf * m = m0;
   
   if (level != SOL_SOCKET)
    {
       if (so->so_proto && so->so_proto->pr_ctloutput)
	   return (* so->so_proto->pr_ctloutput) 
	             (PRCO_SETOPT, so, level, optname, &m0);
       error = ENOPROTOOPT;
    }
   else
    {
       switch (optname)
	{
	case SO_LINGER:
	   if (m == NULL || m->m_len != sizeof (struct linger))
	    {
	       error = EINVAL;
	       goto bad;
	    }
	   so->so_linger = mtod (m, struct linger *)->l_linger;
	   /* fall through ... */

	case SO_DEBUG:
	case SO_KEEPALIVE:
	case SO_DONTROUTE:
	case SO_USELOOPBACK:
	case SO_BROADCAST:
	case SO_REUSEADDR:
	case SO_OOBINLINE:
	   if (m == (struct mbuf *) 0 || m->m_len < sizeof (int))
	    {
	       error = EINVAL;
	       goto bad;
	    }
	   if (* mtod (m, int *))
	       so->so_options |= optname;
	   else
	       so->so_options &= ~optname;
	   break;

	case SO_SNDBUF:
	case SO_RCVBUF:
	case SO_SNDLOWAT:
	case SO_RCVLOWAT:
	case SO_SNDTIMEO:
	case SO_RCVTIMEO:

	   if (m == (struct mbuf *) 0 || m->m_len < sizeof (int))
	    {
	       error = EINVAL;
	       goto bad;
	    }
	   switch (optname)
	    {
	    case SO_SNDBUF:
	       if (sbreserve (&so->so_rcv, * mtod (m, int *)) == 0)
		{
		   error = ENOBUFS;
	           goto bad;
		}
	       break;

	    case SO_RCVBUF:
	       if (sbreserve (&so->so_rcv, * mtod (m, int *)) == 0)
		{
		   error = ENOBUFS;
	           goto bad;
		}
	       break;

	    case SO_SNDLOWAT:
	       so->so_snd.sb_lowat = * mtod (m, int *);
	       break;

	    case SO_RCVLOWAT:
	       so->so_rcv.sb_lowat = * mtod (m, int *);
	       break;

	    case SO_SNDTIMEO:
	       so->so_snd.sb_timeo = * mtod (m, int *);
	       break;

	    case SO_RCVTIMEO:
	       so->so_rcv.sb_timeo = * mtod (m, int *);
	       break;
	    }
	   break;
	   
	default:
	   error = ENOPROTOOPT;
	   break;
	}
    }
 bad:
   if (m)
       (void) m_free (m);
   return error;
}

int
sogetopt (so, level, optname, mp)
  struct socket * so;
  int level, optname;
  struct mbuf ** mp;
{
   struct mbuf * m;
   
   if (level != SOL_SOCKET)
    {
       if (so->so_proto && so->so_proto->pr_ctloutput)
	   return (* so->so_proto->pr_ctloutput)
			(PRCO_GETOPT, so, level, optname, mp);
       else
	   return ENOPROTOOPT;
    }
   else
    {
       m = m_get (M_WAIT, MT_SOOPTS);
       m->m_len = sizeof (int);
       
       switch (optname)
	{
	case SO_LINGER:
	   m->m_len = sizeof (struct linger);
	   mtod (m, struct linger *)->l_onoff = so->so_options & SO_LINGER;
	   mtod (m, struct linger *)->l_linger = so->so_linger;
	   break;

	    
	case SO_DEBUG:
	case SO_KEEPALIVE:
	case SO_DONTROUTE:
	case SO_USELOOPBACK:
	case SO_BROADCAST:
	case SO_REUSEADDR:
	case SO_OOBINLINE:
	   * mtod (m, int *) = so->so_options & optname;
	   break;

	case SO_TYPE:
	   * mtod (m, int *) = so->so_type;
	   break;

	case SO_ERROR:
	   * mtod (m, int *) = so->so_error;
	   break;

	case SO_SNDBUF:
	   * mtod (m, int *) = so->so_snd.sb_hiwat;
	   break;

	case SO_RCVBUF:
	   * mtod (m, int *) = so->so_rcv.sb_hiwat;
	   break;

	case SO_SNDLOWAT:
	   * mtod (m, int *) = so->so_snd.sb_lowat;
	   break;

	case SO_RCVLOWAT:
	   * mtod (m, int *) = so->so_rcv.sb_lowat;
	   break;

	case SO_SNDTIMEO:
	   * mtod (m, int *) = so->so_snd.sb_timeo;
	   break;

	case SO_RCVTIMEO:
	   * mtod (m, int *) = so->so_rcv.sb_timeo;
	   break;

	default:
	   (void) m_free (m);
	   return ENOPROTOOPT;
	}
       * mp = m;
       return 0;
    }
}

void
sohasoutofband (so)
  struct socket * so;
{
   struct proc * p;

   if (so->so_pgrp < 0)
       signal (-so->so_pgrp, SIGURG);
   else if (so->so_pgrp)
       for (p = proc; p < (struct proc *) v.ve_proc; p = p->p_xlink)
	   if (p->p_pid == so->so_pgrp)
	    {
	       psignal (p, SIGURG);
	       break;
	    }
#ifdef SB_COLL
   if (so->so_rcv.sb_sel)
    {
       selwakeup (so->so_rcv.sb_sel, so->so_rcv.sb_flags & SB_COLL);
       so->so_rcv,sb_sel = 0;
       so->so_rcv.sb_flags &= ~SB_COLL;
    }
#endif
}
