/*
 * pty.c - Berkeley style pseudo tty driver for system V
 *
 * Copyright (c) 1987, Jens-Uwe Mager, FOCUS Computer GmbH
 * Not derived from licensed software.
 *
 * Permission is granted to freely use, copy, modify, and redistribute
 * this software, provided that no attempt is made to gain profit from it,
 * the author is not construed to be liable for any results of using the
 * software, alterations are clearly marked as such, and this notice is
 * not modified.
 */

/*
 * Modified for use on the UnixPC by:
 * Eric H. Herrin II
 * University of Kentucky Mathematical Sciences Laboratories
 * eric@ms.uky.edu, eric@ms.uky.csnet, !cbosgd!ukma!eric
 *
 * See README.3b1 for details of port and installation.
 * Version 2.1
 */

/*
 * the following are arbitrary 3 unused bits from t_state
 * in sys/tty.h
 */
/* The UnixPC does not have any extra bits in t_state, thus
 * one must provide other means of storing the state.
 */
#define MRWAIT	01	/* master waiting in read */
#define t_rloc	t_cc[0]		/* wchannel */
#define MWWAIT	02	/* master waiting in write */
#define t_wloc	t_cc[1]		/* wchannel */
#define MOPEN	04	/* master is open */

#include <uipc/pty.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/sysmacros.h>
#include <sys/systm.h>
#include <sys/file.h>
#include <sys/conf.h>
#include <sys/proc.h>
#include <sys/dir.h>
#include <sys/tty.h>
#include <sys/signal.h>
#include <sys/user.h>
#include <sys/errno.h>
#include <sys/termio.h>
#include <sys/ttold.h>

/* The tty structures must be local to this driver.  One doesn't have
 * conf.c
 */
struct tty pts_tty[PTYCNT];
int pts_cnt = PTYCNT;
int ptystate[PTYCNT];
/* This is needed so that we know what the major device is. This
 * is need by socket driver. So we keep the Major device number
 * here for socket, this is used by the select function 
 */
int pty_major = 0; /* if zero then no pty's are opened */

ptyopen(dev, flag)
	register dev_t		dev;
	register int		flag;
{
	register struct tty *tp;

	dev = minor(dev);
	if (Master(dev) == True) {
#		ifdef DEBUG
		eprintf("open(master): \n");
#		endif
		dev -= PTYCNT; 
		tp = &pts_tty[dev];
		if (dev >= pts_cnt) {
			u.u_error = ENXIO;
			return;
		}
		/*
	 	* allow only one controlling process
	 	*/
		if (ptystate[dev] & MOPEN) {
			u.u_error = EBUSY;
			return;
		}
		if (tp->t_state & WOPEN)
			wakeup((caddr_t)&tp->t_canq);
		tp->t_state |= CARR_ON;
		ptystate[dev] |= MOPEN;
	} else {
#		ifdef DEBUG
		eprintf("open(slave): \n");
#		endif
		tp = &pts_tty[dev];
		if (dev >= pts_cnt) {
			u.u_error = ENXIO;
			return;
		}
		if ((tp->t_state & (ISOPEN|WOPEN)) == 0) {
			ttinit(tp);
			tp->t_proc = ptsproc;
		}
		/*
	  	 * if master is still open, don't wait for carrier
	 	 */
		if (ptystate[dev] & MOPEN)
			tp->t_state |= CARR_ON;
		if (!(flag & FNDELAY)) {
			while ((tp->t_state & CARR_ON) == 0) {
				tp->t_state |= WOPEN;
				sleep((caddr_t)&tp->t_canq, TTIPRI);
			}
		}
		(*linesw[tp->t_line].l_open)(tp);
	}
}

ptyclose(dev, flag)
	register dev_t		dev;
	register int		flag;
{
        register struct tty     *tp;

	dev = minor(dev);
	if (Master(dev) == True) {
#		ifdef DEBUG
		eprintf("close(master): \n");
#		endif
		dev -= PTYCNT;
		tp = &pts_tty[dev];
		if (tp->t_state & ISOPEN) {
			signal(tp->t_pgrp, SIGHUP);
			ttyflush(tp, FREAD|FWRITE);
		}
		/*
	 	 * virtual carrier gone
	 	 */
		tp->t_state &= ~(CARR_ON);
		ptystate[dev] &= ~MOPEN;
	} else {
#		ifdef DEBUG
		eprintf("close(slave): \n");
#		endif
		tp = &pts_tty[dev];
		(*linesw[tp->t_line].l_close)(tp);
		tp->t_state &= ~CARR_ON;
	}
}

ptyread(dev)
        register dev_t		dev;
{
	register struct tty     *tp;
	register                n;

	dev = minor(dev);
	if (Master(dev) == True) {
#		ifdef DEBUG
		eprintf("read(master): \n");
#		endif
		dev -= PTYCNT;		
                tp = &pts_tty[dev];

		/* added fix for hanging master side when the slave hangs
		 * up too early.  Fix by Michael Bloom (mb@ttidca.tti.com).
		 */
		if ((tp->t_state & (ISOPEN|TTIOW)) == 0) {
			u.u_error = EIO;
			return;
		}
		while (u.u_count > 0) {
			ptsproc(tp, T_OUTPUT);
			if ((tp->t_state & (TTSTOP|TIMEOUT))
			    || tp->t_tbuf.c_ptr == NULL || tp->t_tbuf.c_count == 0) {
				if (u.u_fmode & FNDELAY)
					break;
#				ifdef DEBUG
				eprintf("read(master): master going to sleep\n");
#				endif
				ptystate[dev] |= MRWAIT;
				sleep((caddr_t)&tp->t_rloc, TTIPRI);
#				ifdef DEBUG
				eprintf("read(master): master woke up\n");
#				endif

				continue;
			}
			n = min(u.u_count, tp->t_tbuf.c_count);
			if (n) {
#				ifdef DEBUG
				eprintf("read(master): got some stuff\n");
#				endif
				if (copyout(tp->t_tbuf.c_ptr, u.u_base, n)) {
					u.u_error = EFAULT;
					break;
				}
				tp->t_tbuf.c_count -= n;
				tp->t_tbuf.c_ptr += n;
				u.u_base += n;
				u.u_count -= n;
			}
		}
	} else {
#		ifdef DEBUG
		eprintf("read(slave): \n");
#		endif
		tp = &pts_tty[dev];
#		ifdef DEBUG
		eprintf("read(slave): got some stuff\n");
#		endif
		(*linesw[tp->t_line].l_read)(tp);
	}
}
	
ptywrite(dev)
	register dev_t		dev;
{
	register struct tty     *tp;
	register                n;

	dev = minor(dev);
	if (Master(dev) == True) {
#		ifdef DEBUG
		eprintf("write(master): \n");
#		endif
		dev -= PTYCNT;
		tp = &pts_tty[dev];
		
		if ((tp->t_state & ISOPEN) == 0) {
			u.u_error = EIO;
			return;
		}
		while (u.u_count > 0) {
			if ((tp->t_state & TBLOCK) || tp->t_rbuf.c_ptr == NULL) {
				if (u.u_fmode & FNDELAY)
					break;
				ptystate[dev] |= MWWAIT;
#				ifdef DEBUG
				eprintf("write(master): going to sleep\n");
#				endif

				sleep((caddr_t)&tp->t_wloc, TTOPRI);

#				ifdef DEBUG
				eprintf("write: waking up\n");
#				endif

				continue;
			}
			n = min(u.u_count, tp->t_rbuf.c_count);
			if (n) {
#				ifdef DEBUG
				eprintf("write(master): sending some stuff\n");
#				endif
				if (copyin(u.u_base,tp->t_rbuf.c_ptr, n)) {
					u.u_error = EFAULT;
					break;
				}
				tp->t_rbuf.c_count -= n;
				u.u_base += n;
				u.u_count -= n;
			}
			(*linesw[tp->t_line].l_input)(tp);
		}
	} else {
#		ifdef DEBUG
		eprintf("write(slave): \n");
#		endif
		tp = &pts_tty[dev];
#		ifdef DEBUG
		eprintf("write(slave): sending some stuff\n");
#		endif
		(*linesw[tp->t_line].l_write)(tp);
	}
}

ptyioctl(dev, cmd, arg, mode)
	dev_t		dev;
	int		cmd, arg, mode;
{
	register struct tty *tp;

	dev = minor(dev);
	if (Master(dev) == True) {
#		ifdef DEBUG
		eprintf("ioctl(master): \n");
#		endif
		dev -= PTYCNT;
		tp = &pts_tty[dev];
		/*
	 	 * sorry, but we can't fiddle with the tty struct without
	 	 * having done LDOPEN
	 	 */
		if (tp->t_state & ISOPEN) {
			if (cmd == TCSBRK && arg ==  NULL) {
				signal(tp->t_pgrp, SIGINT);
				if ((tp->t_iflag & NOFLSH) == 0)
					ttyflush(tp, FREAD|FWRITE);
			} else {
				/*
			 	 * we must flush output to avoid hang in ttywait
			 	 */
				if (cmd == TCSETAW || cmd == TCSETAF || 
				   cmd == TCSBRK || cmd == TIOCSETP)
					ttyflush(tp, FWRITE);
				ttiocom(tp, cmd, arg, mode);
			}
		}
	} else {
#		ifdef DEBUG
		eprintf("ioctl(slave): \n");
#		endif
		tp = &pts_tty[dev];
		ttiocom(tp, cmd, arg, mode);
	}
}

ptsproc(tp, cmd)
register struct tty *tp;
{
	register struct ccblock *tbuf;
	extern ttrstrt();

	switch (cmd) {
	case T_TIME:
#		ifdef DEBUG
		eprintf("ptsproc: T_TIME:\n");
#		endif
		tp->t_state &= ~TIMEOUT;
		goto start;
	case T_WFLUSH:
#		ifdef DEBUG
		eprintf("ptsproc: T_WFLUSH:\n");
#		endif
		tp->t_tbuf.c_size  -= tp->t_tbuf.c_count;
		tp->t_tbuf.c_count = 0;
		/* fall through */
	case T_RESUME:
#		ifdef DEBUG
		eprintf("ptsproc: T_RESUME:\n");
#		endif
		tp->t_state &= ~TTSTOP;
		/* fall through */
	case T_OUTPUT:
start:
#		ifdef DEBUG
		eprintf("ptsproc: T_OUTPUT:\n");
#		endif
		if (tp->t_state & (TTSTOP|TIMEOUT))
			break;
#		ifdef DEBUG
		eprintf("ptsproc: T_OUTPUT: past(TTSTOP|TIMEOUT)");
#		endif
		tbuf = &tp->t_tbuf;
		if (tbuf->c_ptr == NULL || tbuf->c_count == 0) {
#		ifdef DEBUG
		eprintf("ptsproc: T_OUTPUT: tbuf empty, may break\n");
#		endif
			if (tbuf->c_ptr)
				tbuf->c_ptr -= tbuf->c_size;
			if (!(CPRES & (*linesw[tp->t_line].l_output)(tp)))
				break;
		}
		if (tbuf->c_count && (ptystate[tp-pts_tty] & MRWAIT)) {
#ifdef DEBUG
			eprintf("ptsproc: T_OUTPUT: waking up master\n");
#endif
			ptystate[tp-pts_tty] &= ~MRWAIT;
			wakeup((caddr_t)&tp->t_rloc);
		}
#		ifdef DEBUG
		eprintf("ptsproc: T_OUTPUT: leaving end\n");
#		endif
		break;
	case T_SUSPEND:
#		ifdef DEBUG
		eprintf("ptsproc: T_SUSPEND:\n");
#		endif
		tp->t_state |= TTSTOP;
		break;
	case T_BLOCK:
#		ifdef DEBUG
		eprintf("ptsproc: T_BLOCK:\n");
#		endif
		/*
		 * the check for ICANON appears to be neccessary
		 * to avoid a hang when overflowing input
		 */
		if ((tp->t_iflag & ICANON) == 0)
			tp->t_state |= TBLOCK;
		break;
	case T_BREAK:
#		ifdef DEBUG
		eprintf("ptsproc: T_BREAK:\n");
#		endif
		tp->t_state |= TIMEOUT;
		timeout(ttrstrt, tp, HZ/4);
		break;
#ifdef T_LOG_FLUSH
	case T_LOG_FLUSH:
#endif
	case T_RFLUSH:
#		ifdef DEBUG
		eprintf("ptsproc: T_RFLUSH:\n");
#		endif
		if (!(tp->t_state & TBLOCK))
			break;
		/* fall through */
	case T_UNBLOCK:
#		ifdef DEBUG
		eprintf("ptsproc: T_UNBLOCK:\n");
#		endif
		tp->t_state &= ~(TTXOFF|TBLOCK);
		/* fall through */
	case T_INPUT:
#		ifdef DEBUG
		eprintf("ptsproc: T_INPUT:\n");
#		endif
		if (ptystate[tp-pts_tty] & MWWAIT) {
			ptystate[tp-pts_tty] &= ~MWWAIT;
#			ifdef DEBUG
			eprintf("ptsproc: T_INPUT: waking up master\n");
#			endif
			wakeup((caddr_t)&tp->t_wloc);
		}
		break;
	default:
#		ifdef DEBUG
		eprintf("ptsproc: default:\n");
#		else
		;
#		endif
	}
}
	
/* This routine used to be a stub, however, an industrious soul found
 * the release routine caused a panic whenever the driver is released
 * and some ptys are still open.  The simple 'for' loop fixes this 
 * problem.
 *
 * Credit should be given to:
 * Mike "Ford" Ditto
 * kenobi!ford@crash.CTS.COM, ...!crash!kenobi!ford
 * for finding the bug and writing the for loop.
 *
 * [Eric H. Herrin II, 10-7-87]
 */
ptyrelease()
{
	register int 	i;

#	ifdef DEBUG
	eprintf("ptyrelease:\n");
#	endif
	for (i=0; i<PTYCNT; i++)
		if ((ptystate[i] & (ISOPEN|MOPEN)) || 
                   (pts_tty[i].t_state & WOPEN)) {
			u.u_error = EBUSY;
			return;
		}
	/* next release socket driver */
	i = uipcrelease();
	return i;
}
