#ifndef LINT
static char * sccsdef = "@(#)syscalls.c	1.1	(Alex Crain) 6/20/89";
#endif

/*
 *  syscalls.c - system call kernal interface routines.
 *
 *  Written by Alex Crain.
 *
 *  This file is based in the Berkeley file uipc_syscalls.c,
 *  but is *not* guarenteed to be in any way compatable. It is
 *  close enough to the Berkeley code that the following applies...
 *
 *  Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 *  All rights reserved.
 * 
 *  Redistribution and use in source and binary forms are permitted
 *  provided that this notice is preserved and that due credit is given
 *  to the University of California at Berkeley. The name of the University
 *  may not be used to endorse or promote products derived from this
 *  software without specific prior written permission. This software
 *  is provided "as is" without express or implied warranty.
 *
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/user.h>
#include <sys/file.h>
#include <sys/buf.h>
#include <sys/errno.h>
#include <sys/systm.h>
#include <uipc/mbuf.h>
#include <uipc/socket.h>
#include <uipc/socketvar.h>
#include <uipc/domain.h>
#include <uipc/protosw.h>
#include <uipc/un.h>
#include <uipc/fproto.h>

struct file * getsock ();

/*
 * socket (domain, type, protocol)
 *
 *   Create a socket and add it to the processes open file table. This involves
 * some creativity, because the file structure is really too small for our 
 * uses. The kernal only knows about inodes, so there is no f_type slot. 
 * Instead, the kernal looks for a NULL f_inode, which means that we are a 
 * socket. Unfortunately, this means that there is no room in the file 
 * structure for the socket address, so we keep all of our sockets in a linear
 * table, and store the table offset in f_offset, which has no meaning here 
 * anyway. (see the macros filesock() and sockoffet() in conf.h).
 */

void
socket ()
{
   register struct a {
      int	domain;
      int 	type;
      int	proto;
   } * uap = (struct a *) u.u_ap;

   struct socket * so;
   struct file *fp;

   if ((fp = falloc ((struct inode *) 0, FREAD| FWRITE)) == NULL)
       return;

   if (u.u_error = socreate (uap->domain, &so, uap->type, uap->proto))
       goto bad;

   fp->f_offset = sockoffset (so);
   return;

 bad:
   u.u_ofile[u.u_rval1] = 0;
   fp->f_count = 0;
   fp->f_next = ffreelist;
   ffreelist = fp;
}

void
bind ()
{
   struct a {
      int	s;
      caddr_t	name;
      int	namelen;
   } * uap = (struct a *) u.u_ap;

   struct file * fp;
   struct mbuf * nam;

   if ((fp = getsock (uap->s)) == 0)
       return;

   if (u.u_error = sockargs (&nam, uap->name, uap->namelen, MT_SONAME))
       return;

   u.u_error = sobind (filesock (fp), nam);
   m_freem (nam);
}

void
listen ()
{
   struct a {
      int	s;
      int	backlog;
   } * uap = (struct a *) u.u_ap;
   struct file * fp;

   if ((fp = getsock (uap->s)) == 0)
       return;

   u.u_error = solisten (filesock (fp), uap->backlog);
}

void
accept ()
{
   struct a {
      int	s;
      caddr_t	name;
      int	* anamelen;
   } * uap = (struct a *) u.u_ap;
   struct file * fp;
   struct mbuf * nam;
   int namelen;
   int s;
   struct socket * so;

   if (uap->name == 0)
       goto noname;

   if (u.u_error = copyin ((caddr_t) uap->anamelen, (caddr_t) &namelen, 
			   sizeof (namelen)))
       return;

   if (useracc ((caddr_t) uap->name, (u_int) namelen, UACC_WRITE) == 0)
    {
       u.u_error = EFAULT;
       return;
    }

 noname:
   if ((fp = getsock (uap->s)) == 0)
       return;
   s = splnet ();
   so = filesock (fp);
   if ((so->so_options & SO_ACCEPTCONN) == 0)
    {
       u.u_error = EINVAL;
       goto bad;
    }
   if ((so->so_state & SS_NBIO) && so->so_qlen == 0)
    {
       u.u_error = EWOULDBLOCK;
       goto bad;
    }
   while (so->so_qlen == 0 && so->so_error == 0)
    {
       if (so->so_state & SS_CANTRCVMORE)
	{
	   so->so_error = ECONNABORTED;
	   break;
	}
       sleep ((caddr_t) &so->so_timeo, PZERO+1);
    }

   if (so->so_error)
    {
       u.u_error = so->so_error;
       so->so_error = 0;
       goto bad;
    }

   if ((fp = falloc ((struct inode *) 0, FREAD| FWRITE)) == 0)
       goto bad;
   else
    {
       struct socket * so2 = so->so_q;
       if (soqremque (so2, 1) == 0)
	   panic ("accept");
       so = so2;
    }

   fp->f_offset = sockoffset (so);
   nam = m_get (M_WAIT, MT_SONAME);
   (void) soaccept (so, nam);

   if (uap->name)
    {
       if (namelen > nam->m_len)
	   namelen = nam->m_len;
       (void) copyout (mtod (nam, caddr_t), (caddr_t) uap->name, 
		       (u_int) namelen);
       (void) copyout ((caddr_t) &namelen, (caddr_t) uap->anamelen,
		       sizeof (*uap->anamelen));
    }
   m_freem (nam);

 bad:
   splx (s);
   return;
}

void
connect ()
{
   struct a {
      int	s;
      caddr_t	name;
      int	namelen;
   } * uap = (struct a *) u.u_ap;
   struct file * fp;
   struct socket * so;
   struct mbuf * nam;
   int s;

   if ((fp = getsock (uap->s)) == 0)
       return;

   so = filesock (fp);

   if ((so->so_state & SS_NBIO) &&
       (so->so_state & SS_ISCONNECTING))
    {
       u.u_error = EALREADY;
       return;
    }

   if (u.u_error = sockargs (&nam, uap->name, uap->namelen, MT_SONAME))
       return;

   if (u.u_error = soconnect (so, nam))
       goto bad;

   if ((so->so_state & SS_NBIO) &&
       (so->so_state & SS_ISCONNECTING))
    {
       u.u_error = EINPROGRESS;
       m_freem (nam);
       return;
    }

   s = splnet ();

   if (setjmp (u.u_qsav))
    {
       if (u.u_error == 0)
	   u.u_error = EINTR;
       goto bad2;
    }

   while ((so->so_state & SS_ISCONNECTING) && so->so_error == 0)
       sleep ((caddr_t) &so->so_timeo, PZERO + 1);

   u.u_error = so->so_error;
   so->so_error = 0;

 bad2:
   splx (s);

 bad:
   so->so_state &= ~SS_ISCONNECTING;
   m_freem (nam);
}

void   
socketpair ()
{
   struct a {
      int	domain;
      int	type;
      int	proto;
      int	* rsv;
   } * uap = (struct a *) u.u_ap;

   register struct file * fp1, * fp2;
   struct socket * so1, * so2;
   int sv[2];

   /*
    *  verify that uap->rsv is in the users address space & writeable.
    *  UACC_READ and UACC_WRITE are defined in <uipc/conf.h>.
    */
   if (useracc ((caddr_t) uap->rsv, sizeof (int) * 2, UACC_WRITE) == 0)
    {
       u.u_error = EFAULT;
       return;
    }

   /*
    *  Create some sockets (2).
    */
   if (u.u_error = socreate (uap->domain, &so1, uap->type, uap->proto))
       return;

   if (u.u_error = socreate (uap->domain, &so2, uap->type, uap->proto))
       goto free1;

   /*
    *  assign them to file structures in the open file table.
    */
   if ((fp1 = falloc ((struct inode *) 0, FREAD | FWRITE)) == NULL)
       goto free2;
   sv[0] = u.u_rval1;
   fp1->f_offset = sockoffset (so1);

   if ((fp2 = falloc ((struct inode *) 0, FREAD | FWRITE)) == NULL)   
       goto free3;
   sv[1] = u.u_rval1;
   fp2->f_offset = sockoffset (so2);

   /* 
    *  Connect them together.
    */

   if (u.u_error = soconnect2 (so1, so2))
       goto free4;

   /*
    *  DATAGRAMS need to be connected both ways
    */
   if (uap->type == SOCK_DGRAM)
       if (u.u_error = soconnect2 (so2, so1))
	   goto free4;

   /*
    *  done, return 0 and pass the file descriptors back.
    */
   u.u_rval1 = 0;
   copyout ((caddr_t) sv, (caddr_t) uap->rsv, 2 * sizeof (int));
   return;

 free4:
   fp2->f_count = 0;
   fp2->f_next = ffreelist;
   ffreelist = fp2;

 free3:
   fp1->f_count = 0;
   fp1->f_next = ffreelist;
   ffreelist = fp1;

 free2:
   (void) soclose (so2);
   
 free1:
   (void) soclose (so1);
}

void
sendto ()
{
   struct a {
      int	s;
      caddr_t	buf;
      int	len;
      int	flags;
      caddr_t	to;
      int	tolen;
   } * uap = (struct a *) u.u_ap;

   struct msghdr msg;
   
   msg.msg_name = uap->to;
   msg.msg_namelen = uap->tolen;
   msg.msg_accrights = (caddr_t) 0;
   msg.msg_accrightslen = 0;

   u.u_base = uap->buf;
   u.u_count = uap->len;
   u.u_segflg = 0;

   sendit (uap->s, &msg, uap->flags);
}

void
send ()
{
   struct a {
      int	s;
      caddr_t	buf;
      int	len;
      int	flags;
   } * uap = (struct a *) u.u_ap;

   struct msghdr msg;

   msg.msg_name = (caddr_t) 0;
   msg.msg_namelen = 0;
   msg.msg_accrights = (caddr_t) 0;
   msg.msg_accrightslen = 0;

   u.u_base = uap->buf;
   u.u_count = uap->len;
   u.u_segflg = 0;
      
   sendit (uap->s, &msg, uap->flags);
}

void
sendit (s, mp, flags)
  int s;
  struct msghdr * mp;
  int flags;
{
   struct file * fp;
   struct mbuf * to, * rights;

   if ((fp = getsock (s)) == 0)
       return;

   if (u.u_count != 0 && useracc (u.u_base, u.u_count, UACC_READ) == 0)
    {
       u.u_error = EFAULT;
       return;
    }

   if (mp->msg_name)
    {
       if (u.u_error = sockargs (&to, mp->msg_name, mp->msg_namelen,MT_SONAME))
	   return; 
    }
   else
      to = (struct mbuf *) 0;

   if (mp->msg_accrights)
    {
       if (u.u_error = sockargs (&to, mp->msg_accrights, mp->msg_accrightslen,
				 MT_SONAME))
	   goto bad;
    }
   else
       rights = (struct mbuf *) 0;

   u.u_error = sosend (filesock (fp), to, flags, rights);

   if (rights)
       m_freem (rights);

 bad:
   if (to)
       m_freem (to);
}

void
recvfrom ()
{
   struct a {
      int	s;
      caddr_t	buf;
      int	len;
      int	flags;
      caddr_t	from;
      int	* fromlenaddr;
   } * uap = (struct a *) u.u_ap;

   struct msghdr msg;

   msg.msg_name = uap->from;
   if (u.u_error = copyin ((caddr_t) uap->fromlenaddr, 
			   (caddr_t) &msg.msg_namelen, 
			   sizeof (msg.msg_namelen)))
       return;

   msg.msg_accrights = (caddr_t) 0;
   msg.msg_accrightslen = 0;

   u.u_base = uap->buf;
   u.u_count = uap->len;
   u.u_segflg = 0;

   recvit (uap->s, &msg, uap->flags, (caddr_t) uap->fromlenaddr, (caddr_t) 0);
}

void
recv ()
{
   struct a {
      int	s;
      caddr_t	buf;
      int	len;
      int	flags;
   } * uap = (struct a *) u.u_ap;

   struct msghdr msg;

   msg.msg_name = (caddr_t) 0;
   msg.msg_namelen = 0;
   msg.msg_accrights = (caddr_t) 0;
   msg.msg_accrightslen = 0;

   u.u_base = uap->buf;
   u.u_count = uap->len;
   u.u_segflg = 0;

   recvit (uap->s, &msg, uap->flags, (caddr_t) 0, (caddr_t) 0);
}

void
recvit (s, mp, flags, namelenp, rightslenp)
  int s;
  struct msghdr * mp;
  int flags;
  caddr_t namelenp, rightslenp;
{
   struct file * fp;
   struct mbuf * from, * rights;
   int len;

   if ((fp = getsock (s)) == 0)
       return;

   if (u.u_count != 0 && useracc (u.u_base, u.u_count, UACC_WRITE) == 0)
    {
       u.u_error = EFAULT;
       return;
    }

   u.u_error = soreceive (filesock (fp), &from, flags, &rights);

   if (mp->msg_name)
    {
       len = mp->msg_namelen;
       if (len <= 0 || from == (struct mbuf *) 0)
	   len = 0;
       else
	{
	   if (len > from->m_len)
	        len = from->m_len;
	   (void) copyout ((caddr_t) mtod (from, caddr_t), 
			   (caddr_t) mp->msg_name, (unsigned) len);
	}
       (void) copyout ((caddr_t) &len, namelenp, sizeof (int));
    }

   if (mp->msg_accrights)
    {
       len = mp->msg_accrightslen;
       if (len <= 0 || rights == (struct mbuf *) 0)
	   len = 0;
       else
	{
	   if (len > rights->m_len)
	        len = rights->m_len;
	   (void) copyout ((caddr_t) mtod (rights, caddr_t), 
			   (caddr_t) mp->msg_accrights, (unsigned) len);
	}
       (void) copyout ((caddr_t) &len, rightslenp, sizeof (int));
    }

   if (rights)
       m_freem (rights);
   if (from)
       m_freem (from);
}

void
setsockopt ()
{
   struct a {
      int	s;
      int	level;
      int	name;
      caddr_t	val;
      int	valsize;
   } * uap = (struct a *) u.u_ap;
   struct file * fp;
   struct mbuf * m = (struct mbuf *) 0;

   if ((fp = getsock (uap->s)) == 0)
       return;

   if (uap->valsize > MLEN)
    {
       u.u_error = EINVAL;
       return;
    }
   if (uap->val)
    {
       m = m_get (M_WAIT, MT_SOOPTS);
       if (m == (struct mbuf *) 0)
	{
	   u.u_error = ENOBUFS;
	   return;
	}
       if (u.u_error = copyin (uap->val, mtod (m, caddr_t), 
			       (u_int) uap->valsize))
	{
	   (void) m_freem (m);
	   return;
	}
       m->m_len = uap->valsize;
    }
   u.u_error = sosetopt (filesock (fp), uap->level, uap->name, m);
}

void
getsockopt ()
{
   struct a {
      int	s;
      int	level;
      int	name;
      caddr_t	val;
      int	* avalsize;
   } * uap = (struct a *) u.u_ap;
   struct file * fp;
   struct mbuf * m = (struct mbuf *) 0;
   int valsize;

   if ((fp = getsock (uap->s)) == 0)
       return;

   if (uap->val)
    {
       if (u.u_error = copyin ((caddr_t) uap->avalsize, (caddr_t) &valsize,
			   sizeof (valsize)))
	   return;
    }
   else
       valsize = 0;

   if (u.u_error = sogetopt (filesock (fp), uap->level, uap->name, &m))
       goto bad;

   if (uap->val && valsize && m != (struct mbuf *) 0)
    {
       if (valsize > m->m_len)
	   valsize = m->m_len;
       if (u.u_error = copyout (mtod (m, caddr_t), uap->val, (u_int) valsize))
	   goto bad;
       u.u_error = copyout ((caddr_t) &valsize, (caddr_t) uap->avalsize,
			    sizeof (valsize));
     }
 bad:
   if (m != (struct mbuf *) 0)
       (void) m_freem (m);
}

void
sockpipe ()
{
   register struct file * fpr, * fpw;
   struct socket * sor, * sow;
   int r;

   /*
    *  Create some sockets (2).
    */
   if (u.u_error = socreate (AF_UNIX, &sor, SOCK_STREAM, 0))
       return;

   if (u.u_error = socreate (AF_UNIX, &sow, SOCK_STREAM, 0))
       goto free1;

   /*
    *  assign them to file structures in the open file table.
    */
   if ((fpr = falloc ((struct inode *) 0, FREAD | FWRITE)) == NULL)
       goto free2;
   fpr->f_offset = sockoffset (sor);
   r = u.u_rval1;

   if ((fpw = falloc ((struct inode *) 0, FREAD | FWRITE)) == NULL)   
       goto free3;
   fpw->f_offset = sockoffset (sow);
   u.u_rval2 = u.u_rval1;
   u.u_rval1 = r;

   /* 
    *  Connect them together.
    */

   if (u.u_error = unp_connect2 (sow, sor))
       goto free4;

   /*
    *  Close one direction.
    */

   sor->so_state |= SS_CANTSENDMORE;
   sow->so_state |= SS_CANTRCVMORE;
   return;

 free4:
   fpw->f_count = 0;
   fpw->f_next = ffreelist;
   ffreelist = fpw;

 free3:
   fpr->f_count = 0;
   fpr->f_next = ffreelist;
   ffreelist = fpr;

 free2:
   (void) soclose (sow);
   
 free1:
   (void) soclose (sor);
}

void
getsockname ()
{
   struct a {
      int	fdes;
      caddr_t	asa;
      int	* alen;
   } * uap = (struct a *) u.u_ap;
   struct file * fp;
   struct socket * so;
   struct mbuf * m;
   int len;

   if ((fp = getsock (uap->fdes)) == 0)
       return;
   if (u.u_error = copyin ((caddr_t) uap->alen, (caddr_t) &len, sizeof (len)))
       return;
   so = filesock (fp);
   if ((m = m_getclr (M_WAIT, MT_SONAME)) == (struct mbuf *) 0)
    {
       u.u_error = ENOBUFS;
       return;
    }
   if (u.u_error = (* so->so_proto->pr_usrreq) (so, PRU_SOCKADDR,
		       (struct mbuf *) 0, m, (struct mbuf *) 0))
       goto bad;
   if (len > m->m_len)
       len = m->m_len;
   if (u.u_error = copyout (mtod (m, caddr_t), (caddr_t) uap->asa,
			    (u_int) len))
       goto bad;
   u.u_error = copyout ((caddr_t) &len, (caddr_t) uap->alen, 
			    sizeof (len));
 bad:
   m_freem (m);
}

      
/*
 *  System call helper functions
 */

int
sockargs (aname, name, namelen, type)
  struct mbuf ** aname;
  caddr_t name;
  int namelen, type;
{
   struct mbuf * m;
   int error;

   if (namelen > MLEN)
       return EINVAL;

   if ((m = m_get (M_WAIT, type)) == NULL)
       return ENOBUFS;

   m->m_len = namelen;

   if (error = copyin (name, mtod (m, caddr_t), (u_int) namelen))
       (void) m_free (m);
   else
       * aname = m;

   return error;
}

struct file *
getsock (fd)
  int fd;
{
   struct file * fp;

   if ((fp = getf (fd)) == NULL)
       return 0;
   
   if (fp->f_inode)
    {
       u.u_error = ENOTSOCK;
       return 0;
    }
   
   return fp;
}
