#ifndef LINT
static char * sccsdef = "@(#)socket2.c	1.1	(Alex Crain) 6/20/89";
#endif

/*
 *  socket2.c - low level socket routines
 *
 *  Written by Alex Crain.
 *
 *  This file is based in the Berkeley file uipc_socket2.c,
 *  but is *not* guarenteed to be in any way compatable. It is
 *  close enough to the Berkeley code that the following applies...
 *
 *  Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 *  All rights reserved.
 * 
 *  Redistribution and use in source and binary forms are permitted
 *  provided that this notice is preserved and that due credit is given
 *  to the University of California at Berkeley. The name of the University
 *  may not be used to endorse or promote products derived from this
 *  software without specific prior written permission. This software
 *  is provided "as is" without express or implied warranty.
 *
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/proc.h>
#include <sys/var.h>
#include <uipc/mbuf.h>
#include <uipc/socket.h>
#include <uipc/socketvar.h>
#include <uipc/protosw.h>
#include <uipc/domain.h>
#include <uipc/fproto.h>

void
soisconnecting (so)
  struct socket * so;
{
   so->so_state &= ~(SS_ISCONNECTED | SS_ISDISCONNECTING);
   so->so_state |= SS_ISCONNECTING;
   wakeup ((caddr_t) &so->so_timeo);
}

void
soisconnected (so)
  struct socket * so;
{
   struct socket * head = so->so_head;
   
   if (head)
    {
       if (soqremque (so, 0) == 0)
	   panic ("soisconnected");
       soinsque (head, so, 1);
       sorwakeup (head);
       wakeup ((caddr_t) &head->so_timeo);
    }

   so->so_state &= ~(SS_ISCONNECTING | SS_ISDISCONNECTING);
   so->so_state |= SS_ISCONNECTED;
   wakeup ((caddr_t) &so->so_timeo);
   sorwakeup (so);
   sowwakeup (so);
}

void
soisdisconnecting (so)
  struct socket * so;
{
   so->so_state &= ~SS_ISCONNECTING;
   so->so_state |= (SS_ISDISCONNECTING | SS_CANTRCVMORE | SS_CANTSENDMORE);
   wakeup ((caddr_t) &so->so_timeo);
   sorwakeup (so);
   sowwakeup (so);
}

void
soisdisconnected (so)
  struct socket * so;
{
   so->so_state &= ~(SS_ISCONNECTING | SS_ISCONNECTED | SS_ISDISCONNECTING);
   so->so_state |= (SS_CANTRCVMORE | SS_CANTSENDMORE);
   wakeup ((caddr_t) &so->so_timeo);
   sorwakeup (so);
   sowwakeup (so);
}

struct socket *
sonewconn (head)
  struct socket * head;
{
   struct socket * so;
   struct mbuf * m;

   if (head->so_qlen + head->so_q0len > 3 * head->so_qlimit / 2)
       goto bad;
   if ((m = m_getclr(M_DONTWAIT, MT_SOCKET)) == (struct mbuf *) 0)
       goto bad;
   so = mtod (m, struct socket *);
   so->so_type = head->so_type;
   so->so_options = head->so_options & ~SO_ACCEPTCONN;
   so->so_linger = head->so_linger;
   so->so_state = head->so_state | SS_NOFDREF;
   so->so_proto = head->so_proto;
   so->so_timeo = head->so_timeo;
   so->so_pgrp = head->so_pgrp;
   soinsque (head, so, 0);
   if ((* so->so_proto->pr_usrreq) (so, PRU_ATTACH,
		(struct mbuf *) 0, (struct mbuf *) 0, (struct mbuf *) 0))
    {
       (void) soqremque (so, 0);
       (void) m_free (m);
       goto bad;
    }
   return (so);
 bad:
   return (struct socket *) 0;
}

void
soinsque (head, so, q)
  struct socket * head, * so;
  int q;
{
   so->so_head = head;
   if (q == 0)
    {
       head->so_q0len++;
       so->so_q0 = head->so_q0;
       head->so_q0 = so;
    }
   else
    {
       head->so_qlen++;
       so->so_q = head->so_q;
       head->so_q = so;
    }
}

int
soqremque (so, q)
  struct socket * so;
  int q;
{
   struct socket * head, * prev, * next;

   head = so->so_head;
   prev = head;
   for (;;) 
    {
       next = q ? prev->so_q : prev->so_q0;
       if (next == so)
	   break;
       if (next == head)
	   return 0;
       prev = next;
    }
   if (q == 0)
    {
       prev->so_q0 = next->so_q0;
       head->so_q0len--;
    }
   else
    {
       prev->so_q = next->so_q;
       head->so_qlen--;
    } 
   next->so_q0 = next->so_q = 0;
   next->so_head = 0;
   return 1;
}

void
socantsendmore (so)
  struct socket * so;
{
   so->so_state |= SS_CANTSENDMORE;
   sowwakeup (so);
}

void
socantrcvmore (so)
  struct socket * so;
{
   so->so_state |= SS_CANTRCVMORE;
   sorwakeup (so);
}

#ifdef SB_COLL
void
sbselqueue (sb)
  struct sockbuf * sb;
{
   struct proc * p;

   if ((p = sb->sb_sel) && p->p_wchan == (caddr_t) &selwait)
       sb->sb_flags |= SB_COLL;
   else
       sb->sb_sel = u.u_procp;
}
#endif

void
sbwait (sb)
  struct sockbuf * sb;
{
   sb->sb_flags |= SB_WAIT;
   (void) sleep ((caddr_t) &sb->sb_cc, PZERO + 1);
}

void
sbwakeup (sb)
  struct sockbuf * sb;
{
#ifdef SB_COLL
   if (sb->sb_sel)
    {
       selwakeup (sb->sb_sel, sb->sb_flags & SB_COLL);
       sb->sb_sel = 0;
       sb->sb_flags &= ~SB_COLL;
    }
#endif
   if (sb->sb_flags & SB_WAIT)
    {
       sb->sb_flags &= ~SB_WAIT;
       wakeup ((caddr_t) &sb->sb_cc);
    }
}

/* ARGSUSED */
void
sowakeup (so, sb)
  struct socket * so;
  struct sockbuf * sb;
{
   struct proc * p;

   sbwakeup (sb);

   if (so->so_state & SS_ASYNC)
    {
       if (so->so_pgrp < 0)
	    signal (-so->so_pgrp, SIGIO);
       else if (so->so_pgrp)
	   for (p = proc; p < (struct proc *) v.ve_proc; p = p->p_xlink)
	       if (p->p_pid == so->so_pgrp)
		{
		   psignal (p, SIGURG);
		   break;
		}
    }
}

/*
 *  Socket buffer utiliy routines.
 */

int
soreserve (so, sndcc, rcvcc)
  struct socket * so;
  int sndcc, rcvcc;
{
   if (sbreserve (&so->so_snd, sndcc) == 0)
       goto bad1;
   if (sbreserve (&so->so_rcv, rcvcc) == 0)
       goto bad2;
   return 0;
 bad2:
   sbrelease (&so->so_snd);
 bad1:
   return ENOBUFS;
}

/*
 *  Reserve mbufs for a socketbuf.
 */

int
sbreserve (sb, cc)
  struct sockbuf * sb;
{
   sb->sb_hiwat = cc;
   sb->sb_mbmax = cc * 2;
   return 1;
}

void
sbrelease (sb)
  struct sockbuf * sb;
{
   sbflush (sb);
   sb->sb_hiwat = sb->sb_mbmax = 0;
}

void
sbappend (sb, m)
  struct sockbuf * sb;
  struct mbuf * m;
{
   struct mbuf * n;

   if (m == 0)
       return;
   if (n = sb->sb_mb)
    {
       while (n->m_act)
	   n = n->m_act;
       while (n->m_next)
	   n = n->m_next;
    }
   sbcompress (sb, m, n);
}

void
sbappendrecord (sb, m0)
  struct sockbuf * sb;
  struct mbuf * m0;
{
   struct mbuf * m;

   if (m0 == 0)
       return;

   if (m = sb->sb_mb)
       while (m->m_act)
	   m = m->m_act;

   sballoc (sb, m0);
   if (m)
       m->m_act = m0;
   else
       sb->sb_mb = m0;
   m = m0->m_next;
   m0->m_next = 0;
   sbcompress (sb, m, m0);
}

int
sbappendaddr (sb, asa, m0, rights0)
  struct sockbuf * sb;
  struct sockaddr * asa;
  struct mbuf * rights0, * m0;
{
   struct mbuf * m, * n;
   int space = sizeof (* asa);

   for (m = m0; m; m = m->m_next)
       space += m->m_len;
   if (rights0)
       space += rights0->m_len;
   if (space > sbspace (sb))
       return 0;
   MGET (m, M_DONTWAIT, MT_SONAME);
   if (m == 0)
       return 0;
   * mtod (m, struct sockaddr *) = * asa;
   m->m_len = sizeof (* asa);
   if (rights0 && rights0->m_len)
    {
       if ((m->m_next = m_copy (rights0, 0, rights0->m_len)) == 0)
	{
	   m_freem (m);
	   return 0;
	}
       sballoc (sb, m->m_next);
    }
   sballoc (sb, m);
   if (n = sb->sb_mb)
    {
       while (n->m_act)
	   n = n->m_act;
       n->m_act = m;
    }
   else
       sb->sb_mb = m;
   if (m->m_next)
       m = m->m_next;
   if (m0)
       sbcompress (sb, m0, m);
   return 1;
}

int
sbappendrights (sb, m0, rights)
  struct sockbuf * sb;
  struct mbuf * rights, * m0;
{
   struct mbuf * m, * n;
   int space = 0;

   if (rights == 0)
       panic ("sbappendrights");
   for (m = m0; m; m = m->m_next)
       space += m->m_len;
   space += rights->m_len;
   if (space > sbspace (sb))
       return 0;
   if ((m = m_copy (rights, 0, rights->m_len)) == 0)
       return 0;
   sballoc (sb, m);
   if (n = sb->sb_mb)
    {
       while (n->m_act)
	   n = n->m_act;
       n->m_act = m;
    }
   else
       sb->sb_mb = m;
   if (m0)
       sbcompress (sb, m0, m);
   return 1;
}

void
sbcompress (sb, m, n)
  struct sockbuf * sb;
  struct mbuf * m, * n;
{
   while (m)
    {
       if (m->m_len == 0)
	{
	   m = m_free (m);
	   continue;
	}
       if (n && n->m_off <= MMAXOFF && m->m_off <= MMAXOFF &&
	   (n->m_off + n->m_len + m->m_len <= MMAXOFF &&
	    n->m_type == m->m_type))
	{
	   bcopy (mtod (m, caddr_t), mtod (n, caddr_t) + n->m_len,
		  (unsigned) m->m_len);
	   n->m_len += m->m_len;
	   sb->sb_cc += m->m_len;
	   m = m_free (m);
	   continue;
	}
       sballoc (sb, m);
       if (n)
	   n->m_next = m;
       else
	   sb->sb_mb = m;
       n = m;
       m = m->m_next;
       n->m_next = 0;
    }
}

void
sbflush (sb)
  struct sockbuf * sb;
{
   if (sb->sb_flags & SB_LOCK)
       panic ("sbflush");
   while (sb->sb_mbcnt)
       sbdrop (sb, (int) sb->sb_cc);
   if (sb->sb_cc || sb->sb_mbcnt || sb->sb_mb)
       panic ("sbflush 2");
}

/*
 *  Throw away len bytes from sb, starting at the beginning.
 */

void
sbdrop (sb, len)
  struct sockbuf * sb;
  int len;
{
   register struct mbuf * m, * mn;
   struct mbuf * next;

   next = (m = sb->sb_mb) ? m->m_act : 0;

   while (len > 0)
    {
       if (m == 0)
	{
	   if (next == 0)
	       panic ("sbdrop");
	   m = next;
	   next = m->m_act;
	   continue;
	}

       if (m->m_len > len)
	{
	   m->m_len -= len;
	   m->m_off += len;
	   sb->sb_cc -= len;
	   break;
	}

       len -= m->m_len;
       sbfree (sb, m);
       MFREE (m, mn);
       m = mn;
    }

   while (m && m->m_len == 0)	/* when is this case necessary? */
    {
       sbfree (sb, m);
       MFREE (m, mn);
       m = mn;
    }

   if (m)
    {
       sb->sb_mb = m;
       m->m_act = next;
    }
   else
       sb->sb_mb = next;
}

void
sbdroprecord (sb)
  struct sockbuf * sb;
{
   struct mbuf * m, * mn;

   m = sb->sb_mb;
   if (m)
    {
       sb->sb_mb = m->m_act;
       do 
	{
	   sbfree (sb, m);
	   MFREE (m, mn);
	} 
       while (m - mn);
    }
}
