/* mroff.c - main program and file input routines */

#include <errno.h>
#include <stdio.h>

#include <local/argparse.h>

#define MROFF_C

#include "mroff.h"

public char CCVAL;			/* current escape character */
public char C2VAL;			/* nobreak character */

private FILE *infp;			/* current input file descriptor */
public FILE *outfp;			/* output file descriptor */
public FILE *nullfp;			/* for -o option */
public char LINE[MAXLINE];		/* current input line */
public char pages[40];			/* which pages should be printed */

#define FMAX		8		/* # of additional files open */
private int FPTR;
private FILE *FSTACK[FMAX];

private char BACKBUF[MAXLINE];
private int BINP;			/* position in above */

/* initialize all variables for processing one file */
private void initvar()
{
  /* module mroff.c */
  BINP=0;
  FPTR=0;
  CCVAL='\\';
  C2VAL='\'';
  
  /* initialize all modules */
  fill_init();
  font_init();
  trap_init();
  string_init();
  numreg_init();
  macro_init();
  divers_init();
}

/* put back char on input */
public void putback(c) char c;
{
  if (++BINP >= MAXLINE)
  mrofferror("too many characters pushed back","",0,INTERNALERROR);
  BACKBUF[BINP]=c;
}

/* put back string on input */
public void pbstr(s) char *s;
{
  int i;

  for (i=strlen(s); i; ) putback(s[--i]);
}

public int ngetc()
{
  int c;

  if (BINP) c=BACKBUF[BINP];
  else
  {
    if ((c=getc(infp)) == EOF)
    {
      BINP=1;
      return EOF;
    }
    BACKBUF[BINP=1]=c;
  }
  if (c!=EOF) BINP--;
  return(c);
}

public char *getline()
{
  char *l;
  int c;
  
  l=LINE;
  while ((c=ngetc(infp))!=EOF && c!='\n')
  {
    if (c!='\t') *l++=c;
    else { *l++=CCVAL; *l++='t'; }
  }
  *l='\0';
  return (l!=LINE || c!=EOF ? LINE : NULL);
}

/* .so name */
public void source(s) char *s;
{
  s+=3; while (*s==' ') s++;
  if (*s=='\0') mrofferror("missing filename","",0,0);
  else
  {
    if(FPTR<FMAX) FSTACK[FPTR++]=infp;
    else mrofferror("too many source files",s,0,INTERNALERROR);
    if ((infp=fopen(s,"r"))==NULL)
    mrofferror("can\'t open file",s,errno,INTERNALERROR);
  }
}

/* .rd prompt */
public void doread(s) char *s;
{
  char buf[MAXLINE],*input;
  int l;
  
  s+=3;
  while (*s==' ') s++;
  if (isatty(fileno(stdin)))
  if (*s) fprintf(stderr,"%s",s); else fprintf(stderr,"\007");
  input=buf; *input='\0'; l = -1;
  while (l!=0)
  {
    gets(input);
    l=strlen(input);
    input+=l;
  }
  pbstr(buf);
}

/* shall this page be printed ? */
public bool selectpage(n) int n;
{
  char *s = pages;
  int p,q;

  if (*s=='\0') return TRUE;
  while (*s)
  {  
    p=0; while (*s>='0' && *s<='9') p=10*p+*s++-'0';
    if (p==n) return TRUE;
    if (*s=='-')
    {
      s++;
      q=0; while (*s>='0' && *s<='9') q=10*q+*s++-'0';
      if (n>=p && n<=q) return TRUE;
    }
    if (*s==',') s++;
  }
  return FALSE;
}

/* process input from 'infp' and its include files */
private void dofile(textfile,lastfile) bool textfile,lastfile;
{
  char *pc,*e;

  while ((e=getline())!=NULL || FPTR)
  {
    if (e!=NULL)
    {
      if (LINE[0]==COMMAND || LINE[0]==C2VAL)
      {
        if (!execmac(LINE)) request(LINE);
      }
      else if (textfile) text(LINE);
    }
    else
    {
      fclose(infp);
      infp=FSTACK[--FPTR];
      BINP=0;
    }
  }
  if (lastfile) { dobreak(); checkpos(); }
}

private char maclib[10];

private void readmaclib()
{
  char macfile[100];
  
  strcpy(macfile,MACPATH);
  strcat(macfile,maclib);
  if (*maclib)
  {
    if ((infp=fopen(macfile,"r"))==NULL) mrofferror("unknow macro lib",maclib,errno,USERERROR);
    BINP=0;
    dofile(FALSE,FALSE);
    fclose(infp);
  }
}

private bool mac_debug=FALSE;
private int pl=0;
private bool usage=FALSE;

#define ARGTABSIZE 5
private ARG argtab[ARGTABSIZE] =
{
  { 'm', STRING, (int*)maclib },
  { 'o', STRING, (int*)pages },
  { 'M', BOOLEAN, (int*)&mac_debug },
  { 'p', INTEGER, &pl },
  { '?', BOOLEAN, (int*)&usage }
};

public main(argc,argv) int argc; char **argv;
{
  int i;
  
  *maclib= *pages='\0';
  argc=argparse(argc,argv,argtab,ARGTABSIZE);
  if (usage)
  {
    fprintf(stderr,"Usage: mroff {-p<pagelength>} {-m<macrolib>} {-o<pageenum>} {-M} {file ...}\n");
    exit(0);
  }
  nullfp=fopen("/dev/null","w");
  initvar();
  if (pl) PLVAL=72*pl;
  vdopen();
  outfp=(selectpage(1) ? stdout : nullfp);
  readmaclib();
  setfont(0);
  i=1;
  if (argc==1)
  {
    infp=stdin;
    dofile(TRUE,TRUE);
  }
  else while (i<argc)
  {
    BINP=0;
    if ((infp=fopen(argv[i],"r"))==NULL) 
    {
      fprintf(stderr,"mroff: Can't open %s\n", argv[i]);
      break;
    }
    else
    {
      dofile(TRUE,i==argc-1);
      fclose(infp);
    }
    i++;
  }
  doend();
  dofile(TRUE);
  vdclose();
  closedivs();
  if (mac_debug) dumpmacros();
  exit(0);
}
