/* oki240.c - grafic output filter for okidata ML 391 printers, 240 x 120 dpi

   SYNOPSIS: oki {-in} {-f} <input >output
   OPTIONS: -in pagelength equals n inchs
            -f  send a formfeed after each page

   December 30th 1990  -  PC-MINIX 1.5.10  -  Version 1.4
*/

#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>

#include <local/bool.h>
#include <local/argparse.h>

#define RESOLUTION 240		/* resolution in dpi */
#define DOTS_PER_ROW 3264	/* for a line of 16 inch */
#define MAXROWNUM 180		/* for a pagelength of 12 inch */

int ROWNUM;
bool formfeed=TRUE;
bool just_opened;

/* Name and filedescriptor of virtual memory (oh, if I really have it...) */
#define VIRTUAL "/usr/tmp/okivirt"
int virtual;

loadrow(num,row) int num; char *row;
{
  lseek(virtual,(long)num*(long)DOTS_PER_ROW,0);
  read(virtual,row,DOTS_PER_ROW);
}

saverow(num,row) int num; char *row;
{
  lseek(virtual,(long)num*(long)DOTS_PER_ROW,0);
  write(virtual,row,DOTS_PER_ROW);
}

/* number of buffered blocks */
#define BUFFERS 10

struct
{
  int num;
  char *row;
} queue[BUFFERS];

char rowbuf[BUFFERS][DOTS_PER_ROW];

/* prepares an access to a row and places it first in the queue */
char *getrow(num) int num;
{
  int i;
  char *row;

  for (i=0; i<BUFFERS; i++)
  if (queue[i].num==num)
  {
    row=queue[i].row;
    while (i) { queue[i]=queue[i-1]; i--; }
    queue[0].row=row;
    queue[0].num=num;
    return row;
  }
  row=queue[BUFFERS-1].row;
  saverow(queue[BUFFERS-1].num,queue[BUFFERS-1].row);
  for (i=BUFFERS-1; i; i--) queue[i]=queue[i-1];
  loadrow(num,row);
  queue[0].num=num; queue[0].row=row;
  return row;
}

int color;
int overlay;

char setp[8]={128,64,32,16,8,4,2,1};
char clrp[8]={127,191,223,239,247,251,253,254};

/* set a dot on the internal screen */
setdot(x,y) unsigned int x,y;
{
  int offset;

  offset=(y&1);
  
  if (x<DOTS_PER_ROW && y<ROWNUM*8)
  {
    if (color) *(getrow(offset+(y/16)*2) + x) |= setp[(y>>1)&7];
    else *(getrow(offset+(y/16)*2) + x) &= clrp[(y>>1)&7];
  }
}

/* get a dot from the internal screen */
int getdot(x,y) unsigned int x,y;
{
  int offset;

  if (y&1) offset=3; else offset=0;
  if (x<DOTS_PER_ROW && y<ROWNUM*8)
    return *(getrow(offset+((y/48)*6+((y%48)>>4))) + x) & setp[(y>>1)&7];
}

/* draws a line in physical coordinates */
void do_line(x1,y1,x2,y2) int x1,y1,x2,y2;
{
  int x,y,z,dx,dy,dz,i1,i2,ox,oy;

  y1=y1>>1; y2=y2>>1;
  dx=x2-x1; if (dx<0) dx = -dx;
  dy=y2-y1; if (dy<0) dy = -dy;
  if (dx+dy==0) return;
  if (x1<x2)
  {
    x=x1; y=y1;
    if (y1>y2) z = -1; else z=1;
  }
  else
  {
    x=x2; y=y2;
    if (y2>y1) z = -1; else z=1;
  }
  if (dx>dy) i2=dx; else i2=dy;
  for (ox=0; ox<=overlay; ox++) for (oy=0; oy<=overlay; oy++) setdot(x+ox,y-oy);
  if (dx==0 || dy==0)
  for (i1=1; i1<=i2; i1++)
  {
    if (dx==0) y+=z; else x++;
    for (ox=0; ox<=overlay; ox++) for (oy=0; oy<=overlay; oy++) setdot(x+ox,y-oy);
  }
  else
  {
    dz=i2 >> 1;
    for (i1=1; i1<=i2; i1++)
    {
      if (dz<dx) { dz+=dy; x++; }
      if (dz>=dx) { dz-=dx; y+=z; }
      for (ox=0; ox<=overlay; ox++) for (oy=0; oy<=overlay; oy++) setdot(x+ox,y-oy);
    }
  }
}

#include "procfont.c"

/* ------------------------ Filter commands ----------------------------- */
/* We only speak of points or dots and no longer of pixel, inch or s.e.   */

/* cursor position. Note that there is only one cursor for both text and
grafics. */
int cursx,cursy;

/* update the device, whichs means that all data send to it will appear */
void updatepl()
{
  int i,j,lastcol;
  char *row;

  for (i=0; i<ROWNUM; i+=2)
  {

    row=getrow(i)+DOTS_PER_ROW;
    for (lastcol=DOTS_PER_ROW; lastcol; lastcol--) if (*--row) break;
    if (lastcol>0)
    {
      /* 240 dpi */
      printf("\033Z%c%c",lastcol % 256,lastcol / 256);
      row=getrow(i);
      for (j=0; j<lastcol; j++)
      {
        putchar(*row++);
      }
    }
    /* 1/180 inch line feed */
    printf("\033J\01\015");

    row=getrow(i+1)+DOTS_PER_ROW;
    for (lastcol=DOTS_PER_ROW; lastcol; lastcol--) if (*--row) break;
    if (lastcol>0)
    {
      /* 240 Pixel/Zoll */
      printf("\033Z%c%c",lastcol % 256,lastcol / 256);
      row=getrow(i+1);
      for (j=0; j<lastcol; j++)
      {
        putchar(*row++);
      }
    }
    /* 23/180 inch line feed */
    printf("\033J\027\015");

  }
  if (formfeed) putchar('\f');
}

void erasepl()
{
  int i;

  if (just_opened) return;
  cursx=0; cursy=0;
  color=1;
  for (i=0; i<BUFFERS; i++) 
  {
    queue[i].row = rowbuf[i];
    queue[i].num = i;
  }
  memset(rowbuf,0,sizeof(rowbuf));
  lseek(virtual,0L,0);
  for (i=0; i<ROWNUM; i++) write(virtual,rowbuf,DOTS_PER_ROW);
}

/* open the device and set some parameters to default values.  The grafic
area is cleared and updated. */
void openpl()
{
  int i;	

  virtual=open(VIRTUAL,O_CREAT|O_RDWR);
  erasepl();
  just_opened=TRUE;
}

/* update and close a device */
void closepl()
{
  close(virtual);
  unlink(VIRTUAL);
}

/* Initialize the font 'name' with an pointsize of 'points'. */
void font(name,points,b) char *name; int points;
{
  int W;

  if (access(name,0) != -1)
  {
    W=readfont(name);
    makefont(points*RESOLUTION,72*W);
    /* compute overlay for lower resolution, because each pixel overlaps
    its neighbour */
    overlay=(int)((points*(2*RESOLUTION)/3)/(72*W));
  }
}

/* Move cursor to x,y */
void move(x,y)
{
  cursx=x; cursy=y;
}

#define DOT2PIX(x) ((int)(((long)(x)*(long)RESOLUTION)/72L))
#define PIX2DOT(p) ((int)(((long)(p)*72L)/(long)RESOLUTION))

/* Draws a line from the cursor to x,y.  Then move the cursor to this point. */
cont(x,y) int x,y;
{
  do_line(DOT2PIX(cursx),DOT2PIX(cursy),DOT2PIX(x),DOT2PIX(y));
}

line(x1,y1,x2,y2) int x1,y1,x2,y2;
{
  move(x1,y1); cont(x2,y2);
}

/* Draws a string in the current font at the current position. */
text(s) unsigned char *s;
{
  int x,y;

  x=DOT2PIX(cursx); y=DOT2PIX(cursy);
  while (*s>=' ')
  {
    x=drawchar(x,y,*s-' ');
    s++;
  }
  cursx=PIX2DOT(x);
}

int pagelength=11;

#define argtabsize 2
ARG argtab[argtabsize] =
{
  { 'l', INTEGER, &pagelength },
  { 'f', BOOLEAN, (int *) &formfeed }
};

main(argc,argv) int argc; char **argv;
{
  char name[512], ln[512];
  int size,x1,y1,x2,y2;
  
  argc=argparse(argc,argv,argtab,argtabsize);
  /* warning: next line needs short circuit evaluation. */
  /* Do not change the computation of pagelength because of integer arithmetic */
  ROWNUM=((pagelength=((pagelength*180)/24)*2) >=15 && pagelength<=MAXROWNUM ? pagelength : MAXROWNUM);
  /* unidirectional printing */
  printf("\033U1");
  while (gets(ln)!=NULL)
  {
    switch (*ln)
    {
      case 'o' : openpl(); break;

      case 'e' : erasepl(); break;

      case 'q' : closepl(); break;

      case 'u' : updatepl(); break;

      case 'f' :
      sscanf(ln+1,"%s %d",name,&size);
      font(name,size);
      break;

      case 't' : text(ln+1); break;
 
      case 'm' : sscanf(ln+1,"%d %d",&x1,&y1); move(x1,y1); break;
      
      case 'l' :
      {
        int o;
      	
        o=overlay; overlay=0;
        sscanf(ln+1,"%d %d %d %d",&x1,&y1,&x2,&y2);
        line(x1,y1,x2,y2);
        overlay=o;
        break;
      }

      case 'c' :
      {
        int o;
      	
        o=overlay; overlay=0;
        sscanf(ln+1,"%d %d",&x1,&y1);
        cont(x1,y1);
        overlay=o;
        break;
      }
      
    }
    if (*ln!='o') just_opened=FALSE;
  }
  /* reset the printer */
  printf("\033U0\033@");
}
