/* hgc.c - grafic output filter for hercules cards

   December 6th 1990  -  PC-MINIX 1.5.10  -  Version 1.2
*/

#include <stdio.h>

#define RESOLUTION 72	/* resolution in dpi */

/* -----------------------------------------------------------------------
   hardware dependent things for hercules cards.  do not look at the code,
   we really need grafics support in the console task of the kernel.
*/

/* internal screen buffer */
char screen[0x8000];
int color;

/* clear the internal screen */
void clrscr()
{
  unsigned n;
  char *s=screen;

  for (n=0; n<0x8000; n++) *s++=0;
}

char setp[8]={128,64,32,16,8,4,2,1};
char clrp[8]={127,191,223,239,247,251,253,254};

/* set a dot on the internal screen */
void setdot(x,y) unsigned int x,y;
{
  if (x<720 && y<348)
  {
    if (color) screen[((y&3)<<13) + 90 * (y >> 2) + (x >> 3)] |= setp[x&7];
    else screen[((y&3)<<13) + 90 * (y >> 2) + (x >> 3)] &= clrp[x&7];
  }
}

/* get a dot from the internal screen, maybe we need this function in later
   version.
*/
int getdot(x,y) unsigned int x,y;
{
  if (x<720 && y<348)
  return (screen[((y&3)<<13) + 90 * (y >> 2) + (x >> 3)] & setp[x&7]);
}

#define TEXT_PAGE  0
#define GRAF_PAGE0 1
#define GRAF_PAGE1 2

/* this are the video ports */
#define INDEX_REG  0x3b4
#define DATA_REG   0x3b5
#define MODE_REG   0x3b8
#define CONFIG_REG 0x3bf

/* activate video signal */
#define ACTIVE     8

graficmode(mode) int mode;
{
  static int
  grfval[12]={0x35,0x2d,0x2e,0x7,0x5b,0x02,0x57,0x57,0x2,0x3,0x0,0x0},
  txtval[12]={0x61,0x50,0x52,0xf,0x19,0x06,0x19,0x19,0x2,0xd,0x0,0xc};
  int regsel;

  switch (mode)
  {
    case GRAF_PAGE0 :
    case GRAF_PAGE1 :
    {
      port_out(MODE_REG,(mode==GRAF_PAGE0 ? 0x02 : 0x82));
      for (regsel=0; regsel<12; regsel++)
      {
        port_out(INDEX_REG,regsel);
        port_out(DATA_REG,grfval[regsel]);
      }
      port_out(CONFIG_REG,3);
      port_out(MODE_REG,(mode==GRAF_PAGE0 ? 0x02 : 0x82) | ACTIVE);
      break;
    }

    case TEXT_PAGE:
    {
      port_out(MODE_REG,0x20);
      for (regsel=0; regsel<12; regsel++)
      {
        port_out(INDEX_REG,regsel);
        port_out(DATA_REG,txtval[regsel]);
      }
      port_out(CONFIG_REG,1);
      port_out(MODE_REG,0x20 | ACTIVE);
    }

  }
}

/* copy the internal screen to the video memory */
refreshvideo(mode)
{
  int outfd;

  outfd = open("/dev/mem", 1);
  if (outfd >=0)
  {
    lseek(outfd, (long)(mode == GRAF_PAGE0 ? 0xb0000 : 0xb8000), 0);
    /* write can't handle blocks larger than 32767 bytes */
    write(outfd, screen, 16384);
    write(outfd, screen+16384, 16384);
    close(outfd);
  }
}

/* this part is not included in procfont.c, because it is only for raster
   devices, not for things like plotters or other vector devices
*/

int overlay;

/* draws a line in physical coordinates */
void do_line(x1,y1,x2,y2) int x1,y1,x2,y2;
{
  int x,y,z,dx,dy,dz,i1,i2,ox,oy;
  
  dx=x2-x1; if (dx<0) dx = -dx;
  dy=y2-y1; if (dy<0) dy = -dy;
  if (dx+dy==0) return;
  if (x1<x2)
  {
    x=x1; y=y1;
    if (y1>y2) z = -1; else z=1;
  }
  else
  {
    x=x2; y=y2;
    if (y2>y1) z = -1; else z=1;
  }
  if (dx>dy) i2=dx; else i2=dy;
  for (ox=0; ox<=overlay; ox++) for (oy=0; oy<=overlay; oy++) setdot(x+ox,y-oy);
  if (dx==0 || dy==0)
  for (i1=1; i1<=i2; i1++)
  {
    if (dx==0) y+=z; else x++;
    for (ox=0; ox<=overlay; ox++) for (oy=0; oy<=overlay; oy++) setdot(x+ox,y-oy);
  }
  else
  {
    dz=i2 >> 1;
    for (i1=1; i1<=i2; i1++)
    {
      if (dz<dx) { dz+=dy; x++; }
      if (dz>=dx) { dz-=dx; y+=z; }
      for (ox=0; ox<=overlay; ox++) for (oy=0; oy<=overlay; oy++) setdot(x+ox,y-oy);
    }
  }
}

/* --------------------------------------------------------------------------
   you can open your eyes, the cruel hardware handling function end here.
*/

#include "procfont.c"

/* cursor position. Note that there is only one cursor for both text and
grafics. */
int cursx,cursy;

/* update a device, whichs means that all data send to it will appear */
void updatepl()
{
  refreshvideo(GRAF_PAGE1);
}

void erasepl()
{
  cursx=0; cursy=0;
  color=1;
  clrscr();
}

/* open the device and set some parameters to default values.  The grafic
area is cleared and updated. */
void openpl()
{
  graficmode(GRAF_PAGE1);
  erasepl();
  refreshvideo(GRAF_PAGE1);
}

/* update and close a device */
void closepl()
{
  graficmode(TEXT_PAGE);
}

/* Initialize the font 'name' with an pointsize of 'points' and compute overlay
factor for getting filled letters in each size. */
void font(name,points) char *name; int points;
{
  int M;
  
  if (access(name,0) != -1)
  {
    M=readfont(name);
    makefont(points*RESOLUTION,72*M);
    overlay=(int)((points*RESOLUTION)/(72*M));
  }
}

/* Move cursor to x,y */
void move(x,y) int x,y;
{
  /* negative values caused by preview pager are out of range */
  cursx=(x<0 ? 800 : x);
  cursy=(y<0 ? 800 : y);
}

#define DOT2PIX(x) ((int)(((long)(x)*(long)RESOLUTION)/72L))
#define PIX2DOT(p) ((int)(((long)(p)*72L)/(long)RESOLUTION))

/* Draws a line from the cursor to x,y.  Then move the cursor to this point. */
cont(x,y) int x,y;
{
  do_line(DOT2PIX(cursx),DOT2PIX(cursy),DOT2PIX(x),DOT2PIX(y));
}

line(x1,y1,x2,y2) int x1,y1,x2,y2;
{
  move(x1,y1); cont(x2,y2);
}

/* Draws a string in the current font at the current position. */
text(s) unsigned char *s;
{
  int x,y;

  x=DOT2PIX(cursx); y=DOT2PIX(cursy);
  while (*s>=' ') x=drawchar(x,y,*s++-' ');
  cursx=PIX2DOT(x);
}

main(argc,argv) int argc; char **argv;
{
  char name[512], ln[512];
  int size,x1,y1,x2,y2;
  
  while (gets(ln)!=NULL)
  {
    switch (*ln)
    {
      case 'o' : openpl(); break;

      case 'e' : erasepl(); break;

      case 'q' : closepl(); break;

      case 'u' : updatepl(); break;

      case 'f' : sscanf(ln+1,"%s %d",name,&size); font(name,size); break;

      case 't' : text(ln+1); break;
 
      case 'm' : sscanf(ln+1,"%d %d",&x1,&y1); move(x1,y1); break;
      
      case 'l' :
      {
        int o;
      	
        o=overlay;
        overlay=0;
        sscanf(ln+1,"%d %d %d %d",&x1,&y1,&x2,&y2);
        line(x1,y1,x2,y2);
        overlay=o;
        break;
      }
      
      case 'c' :
      {
        int o;
      	
        o=overlay;
        overlay=0;
        sscanf(ln+1,"%d %d",&x1,&y1);
        cont(x1,y1);
        overlay=o;
        break;
      }
    }
  }
}
