/* tar - tape archiver			After: Michiel Huisjes */
/* 					adapted by : H. Salomons */

/*
 * Usage: tar [c|x|t][olv] tapefile [[-]files]
 * 
 * c creates a new tapefile with named files x extracts from existing tapefile
 * of named files t table of contents of tapefile for named files o assign
 * files to process owner rather than the original owner l extract files to
 * the local directory rather than the original directory v verbose
 * 
 * if no files are given for extract and table of contents, all files are
 * processed. if the minus sign is used before the filename, the file is
 * taken to contain a list of filenames that has to be processed.
 * 
 * Bugs: This tapefiler should only be used as a program to read or make
 * simple tape archives. Its basic goal is to read (or build) UNIX V7 tape
 * archives and extract the named files. It is not wise to use it on raw
 * magnetic tapes. It doesn't know anything about linked files, except when
 * the involved fields are filled in.
 * 
 * to achieve some of the above features the program has become set-uid-root.
 */
#define NO_EXTENDED_LIST

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#define NULLD ((struct dirent *) 0)

#include <pwd.h>
#include <grp.h>
extern struct passwd *getpwuid();
extern struct group *getgrgid();
#define NIL_PWD ((struct passwd *) 0)
#define NIL_GRP ((struct group *) 0)

typedef char BOOL;

#define TRUE	1
#define FALSE	0
#define DONT_USE_READ_TO_SKIP
#define HEADER_SIZE	512
#define NAME_SIZE	100
#define LIST_SIZE   512
#define LINE_SIZE   256

#define BLOCK_BOUNDARY	 20

typedef union
{
    char        hdr_block[HEADER_SIZE];
    struct m
    {
	char        m_name[NAME_SIZE];
	char        m_mode[8];
	char        m_uid[8];
	char        m_gid[8];
	char        m_size[12];
	char        m_time[12];
	char        m_checksum[8];
	char        m_linked;
	char        m_link[NAME_SIZE];
    }
                member;
}
            HEADER;

HEADER      header;

#define INT_TYPE	(sizeof(header.member.m_uid))
#define LONG_TYPE	(sizeof(header.member.m_size))

#define MKDIR		"/bin/mkdir"

#define NIL_HEADER	((HEADER *) 0)
#define NIL_PTR		((char *) 0)
#define BLOCK_SIZE	512
#define INTER_COUNT 18

#define OK 0
#define IS_DIR  0
#define NO_PATH 1
#define NO_DIR  2
#define CR_ERROR 5

BOOL        show_fl, creat_fl, ext_fl, verb_fl, int_fl, loc_fl, owner_fl;
char       *progname;
int         isfloppy;
int         tar_fd;
int         group_id;
int         passw_id;
char        io_buffer[BLOCK_SIZE];
char        path[NAME_SIZE];
char        pathname[NAME_SIZE];
char       *namelist[LIST_SIZE];
int         namecount;
int         interval = INTER_COUNT;	/* this program knows too much */

int         total_blocks;
long        convert();
char       *malloc();
char       *strsave();
char       *local();

#define block_size()	(int) ((convert(header.member.m_size, LONG_TYPE) \
+ (long) BLOCK_SIZE - 1) / (long) BLOCK_SIZE)

int         main(argc, argv)
    int         argc;
    register char *argv[];
{
    register char *ptr;
    struct stat st;
    int         i;

    group_id = getgid();
    passw_id = getuid();
    show_fl = FALSE;
    creat_fl = FALSE;
    ext_fl = FALSE;
    verb_fl = FALSE;
    owner_fl = TRUE;
    total_blocks = 0;
    isfloppy = 0;
    int_fl = 0;
    loc_fl = 0;

    progname = *argv;
    if (argc < 3)
	pr_usage();

    for (ptr = argv[1]; *ptr; ptr++)
    {
	switch (*ptr)
	{
	case 'c':
	    creat_fl = TRUE;
	    break;
	case 'x':
	    ext_fl = TRUE;
	    break;
	case 't':
	    show_fl = TRUE;
	    break;
	case 'v':
	    verb_fl = TRUE;
	    break;
	case 'i':
	    int_fl = TRUE;
	    break;
	case 'l':
	    loc_fl = TRUE;
	    break;
	case 'o':
	    owner_fl = FALSE;
	    break;
	default:
	    pr_usage();
	}
    }

    if (creat_fl + ext_fl + show_fl != 1)
	pr_usage();

    if (creat_fl)
    {
	tar_fd = creat(argv[2], 0644);
	chown(argv[2], passw_id, group_id);
    }
    else
	tar_fd = open(argv[2], 0);

    if (tar_fd < 0)
    {
	fprintf(stderr, "can't open %s\n", argv[2]);
	exit(1);
    }
    if (creat_fl && int_fl)
    {
	if (stat(argv[2], &st) < 0)
	{
	    fprintf(stderr, "cannot stat %s\n", argv[2]);
	    exit(1);
	}
	if ((st.st_mode & S_IFMT) == S_IFBLK)
	    isfloppy = 1;
    }
    namecount = 0;
    for (i = 3; i < argc; i++)
    {
	ptr = argv[i];
	if (*ptr == '-')
	    add_list(++ptr);
	else
	    next_arg(ptr);
    }

    if (creat_fl)
    {
	for (i = 0; i < namecount; i++)
	{
	    add_file(namelist[i]);
	    path[0] = '\0';
	}
	adjust_boundary();
    }
    else
	tarfile();

    close(tar_fd);
    exit(0);
}

int         pr_usage()
{
    fprintf(stderr, "Usage: %s [cxlto][v] tarfile [[-]files]\n", progname);
    exit(1);
}

int         add_list(name)
    char       *name;
{
    FILE       *list_fd;
    char        line[LINE_SIZE];
    char       *fgets();

    if ((list_fd = fopen(name, "r")) == (FILE *) 0)
    {
	fprintf(stderr, "can't open list %s\n", name);
	exit(1);
    }
    while (fgets(line, LINE_SIZE, list_fd) != NIL_PTR)
    {
	line[strlen(line) - 1] = '\0';
	next_arg(line);
    }
    fclose(list_fd);
}

BOOL        get_header()
{
    register int check;

    mread(tar_fd, &header, sizeof(header));
    if (header.member.m_name[0] == '\0')
	return FALSE;

    check = (int) convert(header.member.m_checksum, INT_TYPE);

    if (check != checksum())
    {
	fprintf(stderr, "%s: header checksum error\n", progname);
	exit(1);
    }
    return TRUE;
}

int         show_name()
{
    int         blocks;
    char        linkflg;
    char       *mem_name;
    char       *un;
    struct passwd *pwd;
    struct group *grp;
#ifdef EXTENDED_LIST
    int         id;
#endif

    blocks = block_size();

    linkflg = header.member.m_linked;
    if (linkflg == 'Z')
    {
	mem_name = header.member.m_link;
	un = "Z";
    }
    else
    {
	mem_name = header.member.m_name;
	un = " ";
    }
    if (verb_fl)
    {
	fprintf(stdout, "%-40s", local(mem_name));
#ifdef EXTENDED_LIST
	date(convert(header.member.m_time, LONG_TYPE));
#endif
	if (linkflg == '1')
	    fprintf(stdout, " linked to %s", header.member.m_link);
	else
	    fprintf(stdout, " %4d %s", blocks, un);
#ifdef EXTENDED_LIST
	fprintf(stdout, " %s", header.member.m_mode);
	id = convert(header.member.m_gid, INT_TYPE);
	if ((grp = getgrgid(id)) == NIL_GRP)
	    fprintf(stdout, " (%d,", id);
	else
	    fprintf(stdout, " (%s,", grp->gr_name);

	id = convert(header.member.m_uid, INT_TYPE);
	if ((pwd = getpwuid(id)) == NIL_PWD)
	    fprintf(stdout, "%d) ", id);
	else
	    fprintf(stdout, "%s) ", pwd->pw_name);
#endif
    }
    else
	fprintf(stdout, "%s", local(mem_name));
    fprintf(stdout, "\n");
}

int         tarfile()
{
    register char *ptr;
    register char *mem_name;

    while (get_header())
    {
	if (header.member.m_linked == 'Z')
	    mem_name = header.member.m_link;
	else
	    mem_name = header.member.m_name;
	if (ext_fl)
	{
	    if (is_dir(mem_name))
	    {
		if (in_list(mem_name))
		{
		    for (ptr = mem_name; *ptr; ptr++);
		    *(ptr - 1) = '\0';
#ifdef DEBUG_PATHS
		    fprintf(stderr, "path_exists(%s)\n", mem_name);
#endif
		    if (path_exists(mem_name) == NO_PATH)
		    {
			if (verb_fl)
			    fprintf(stdout, "%-40s %4d\n", mem_name, 0);
			else
			    fprintf(stdout, "%s\n", mem_name);
			mkdir(mem_name);
			owner_mode(mem_name, 0);
		    }
		}
	    }
	    else
	    {
		if (in_list(mem_name))
		    extract(mem_name);
		else
		    skip_entry();
	    }
	}
	else
	{
	    if ( /* !is_dir(mem_name) && */ in_list(mem_name))
	    {
		show_name();
		skip_entry();
	    }
	}
    }
}

int         skip_entry()
{
    register int blocks = block_size();

#ifdef USE_READ_TO_SKIP
    while (blocks--)
	(void) read(tar_fd, io_buffer, BLOCK_SIZE);
#else
    lseek(tar_fd, ((long) blocks * (long) BLOCK_SIZE), 1);
#endif
}

char       *local(name)
    char       *name;
{
    register char *result;

    if (loc_fl)
    {
	if ((result = strrchr(name, '/')) != NIL_PTR)
	    return (++result);
    }
    return (name);
}

int         extract(file)
    register char *file;
{
    register int fd;
    char       *fname;
    char       *lfile;
    char        command[LINE_SIZE];

    if (header.member.m_linked == '1')
    {
	if (link(header.member.m_link, file) < 0)
	    fprintf(stderr, "Cannot link %s to %s\n",
		    header.member.m_link, file);
	else
	    fprintf(stderr, "Linked %s to %s\n", header.member.m_link, file);
	return;
    }
    lfile = local(file);
    if (header.member.m_linked == 'Z')
	fname = "/tmp/tmp.Z";
    else
	fname = lfile;

    if ((fd = path_creat(fname, 0755)) < 0)
    {
	fprintf(stderr, "Cannot create %s\n", fname);
	skip_entry();
	return;
    }
    copy(fname, tar_fd, fd, convert(header.member.m_size, LONG_TYPE));
    (void) close(fd);

    if (header.member.m_linked == 'Z')
    {
	/* uncompress file to original */
	sprintf(command, "compress -cd %s > %s", fname, lfile);
	system(command);
	unlink(fname);
    }
    owner_mode(lfile, 0);
}

int         owner_mode(name, overrule)
    char       *name;
    int         overrule;
{
    int         mode;
    int         uid;
    int         gid;
    long        ltim[2];

    mode = convert(header.member.m_mode, INT_TYPE);
    uid = convert(header.member.m_uid, INT_TYPE);
    gid = convert(header.member.m_gid, INT_TYPE);
    if (!owner_fl || (getpwuid(uid) == NIL_PWD) || (getgrgid(gid) == NIL_GRP))
    {
	uid = passw_id;
	gid = group_id;
    }
    ltim[0] = ltim[1] = convert(header.member.m_time, LONG_TYPE);

    chown(name, uid, gid);
    chmod(name, mode);
    utime(name, ltim);
}

int         copy(file, from, to, bytes)
    char       *file;
    int         from, to;
    register long bytes;
{
    register int rest;
    int         blocks = (int) ((bytes + (long) BLOCK_SIZE - 1) / (long) BLOCK_SIZE);

    show_name();
    while (blocks--)
    {
	(void) read(from, io_buffer, BLOCK_SIZE);
	rest = (bytes > (long) BLOCK_SIZE) ? BLOCK_SIZE : (int) bytes;
	mwrite(to, io_buffer, (to == tar_fd) ? BLOCK_SIZE : rest);
	bytes -= (long) rest;
    }
}

long        convert(str, type)
    char        str[];
int         type;
{
    register long ac = 0L;
    register int i;

    for (i = 0; i < type; i++)
    {
	if (str[i] >= '0' && str[i] <= '7')
	{
	    ac <<= 3;
	    ac += (long) (str[i] - '0');
	}
    }

    return ac;
}

int         mkdir(dir_name)
    char       *dir_name;
{
    register int pid, w;
    int         result;

    if ((pid = fork()) < 0)
    {
	fprintf(stderr, "%s: can't fork\n", progname);
	exit(1);
    }
    if (pid == 0)
    {
	execl(MKDIR, "mkdir", dir_name, NIL_PTR);
	{
	    fprintf(stderr, "%s: can't exec 'mkdir'\n", progname);
	    exit(1);
	}
    }
    do
	w = wait(&result);
    while (w != -1 && w != pid);
    return (result);
}

int         checksum()
{
    register char *ptr = header.member.m_checksum;
    register int ac = 0;

    while (ptr < &header.member.m_checksum[INT_TYPE])
	*ptr++ = ' ';

    ptr = header.hdr_block;
    while (ptr < &header.hdr_block[BLOCK_SIZE])
	ac += *ptr++;

    return ac;
}

int         is_dir(file)
    register char *file;
{
    while (*file++ != '\0');

    return (*(file - 2) == '/');
}


char       *path_name(file)
    register char *file;
{
    sprintf(pathname, "%s%s", path, file);
    return pathname;
}

int         add_path(name)
    register char *name;
{
    register char *path_ptr = path;

    while (*path_ptr)
	path_ptr++;

    if (name == NIL_PTR)
    {
	while (*path_ptr-- != '/');
	while (*path_ptr != '/' && path_ptr != path)
	    path_ptr--;
	if (*path_ptr == '/')
	    path_ptr++;
	*path_ptr = '\0';
    }
    else
    {
	while (*name)
	{
	    if (path_ptr == &path[NAME_SIZE])
	    {
		fprintf(stderr, "%s: pathname too long\n", progname);
		exit(1);
	    }
	    *path_ptr++ = *name++;
	}
	*path_ptr++ = '/';
	*path_ptr = '\0';
    }
}

int         add_file(file)
    register char *file;
{
    char        orig[NAME_SIZE];
    char        back[NAME_SIZE];
    char       *cp;
    char       *cq;
    struct stat st;
    struct dirent *dir;
    DIR        *dp;
    register int fd;

#define LPAR '('
#define RPAR ')'
    strcpy(orig, file);
    if ((cp = strchr(orig, RPAR)) != NIL_PTR)
	*cp = '\0';
    if ((cp = strchr(orig, LPAR)) != NIL_PTR)
    {
	*cp = '\0';
	strcpy(back, ++cp);
	cp = back;
	cq = orig;
    }
    else
    {
	cp = orig;
	cq = NIL_PTR;
    }
    if (stat(cp, &st) < 0)
    {
	fprintf(stderr, "Cannot find %s\n", cp);
	return;
    }
    if (st.st_mode & S_IFREG)
    {
	if ((fd = open(cp, 0)) < 0)
	{
	    fprintf(stderr, "Can't open %s\n", cp);
	    return;
	}
    }
    else if (st.st_mode & S_IFDIR)
    {
	if ((dp = opendir(cp)) == (DIR *) 0)
	{
	    fprintf(stderr, "Can't opendir(%s)\n", cp);
	    return;
	}
    }
    make_header(path_name(cp), cq, &st);

    mwrite(tar_fd, &header, sizeof(header));

    if (st.st_mode & S_IFREG)
	copy(path_name(cp), fd, tar_fd, st.st_size);
    else if (st.st_mode & S_IFDIR)
    {
	if (chdir(file) < 0)
	    fprintf(stderr, "Can't chdir to %s\n", file);
	else
	{
	    add_path(file);
	    if ((dir = readdir(dp)) == NULLD)	/* "." */
		fprintf(stderr, "read error on '.'\n");
	    else if ((dir = readdir(dp)) == NULLD)	/* ".." */
		fprintf(stderr, "read error on '..'\n");
	    else
	    {
		while ((dir = readdir(dp)) != NULLD)
		    if (dir->d_ino)
			add_file(dir->d_name);
	    }
	    chdir("..");
	    add_path(NIL_PTR);
	}
    }
    else
	fprintf(stderr, " Tar: unknown file type. Not added.\n");

    if (st.st_mode & S_IFREG)
	(void) close(fd);
    else if (st.st_mode & S_IFDIR)
	(void) closedir(dp);
}

int         make_header(file, back, st)
    char       *file;
    char       *back;
    register struct stat *st;
{
    register char *ptr = header.member.m_name;
    struct stat st_orig;
    struct stat *sp;
    long        size;

    clear_header();
    sp = st;
    size = st->st_size;

    while (*ptr++ = *file++);
    if (back != NIL_PTR)
    {
	sp = &st_orig;
	header.member.m_linked = 'Z';
	ptr = header.member.m_link;
	while (*ptr++ = *back++);
	ptr = header.member.m_link;
	if (stat(ptr, sp) != 0)
	{
	    fprintf(stderr, "lost original file '%s'\n", ptr);
	    exit(1);
	}
    }
    else
	header.member.m_linked = ' ';

    if (st->st_mode & S_IFDIR)
    {
	*(ptr - 1) = '/';
	size = 0L;
    }
    sprintf(header.member.m_mode, "%05o ", sp->st_mode & 07777);
    sprintf(header.member.m_uid, "%05o ", sp->st_uid);
    sprintf(header.member.m_gid, "%05o ", sp->st_gid);
    sprintf(header.member.m_size, "%lo ", size);
    sprintf(header.member.m_time, "%lo ", sp->st_mtime);
    sprintf(header.member.m_checksum, "%05o", checksum());
}

int         clear_header()
{
    register char *ptr = header.hdr_block;

    while (ptr < &header.hdr_block[BLOCK_SIZE])
	*ptr++ = '\0';
}

int         adjust_boundary()
{
    clear_header();
    mwrite(tar_fd, &header, sizeof(header));

    while (total_blocks < BLOCK_BOUNDARY)
	mwrite(tar_fd, &header, sizeof(header));
}

int         mread(fd, address, bytes)
    int         fd;
    int         bytes;
    char       *address;
{
    if (read(fd, address, bytes) != bytes)
    {
	fprintf(stderr, "%s: read error\n", progname);
	exit(1);
    }
}

int         mwrite(fd, address, bytes)
    int         fd;
    int         bytes;
    char       *address;
{
    int         written;

    if ((written = write(fd, address, bytes)) != bytes)
    {
	fprintf(stderr, "%s: write error\n", progname);
	exit(1);
    }
    total_blocks++;
    if (isfloppy)
	if ((total_blocks % interval) == 0)
	    sync();
}

int         in_list(name)
    char       *name;
{
    char      **np = namelist;
    int         i;
    char       *stristr();

    if (namecount == 0)
	return (1);
    for (i = 0; i < namecount; i++, np++)
    {
	if (stristr(*np, name) != NIL_PTR)
	    return (1);
    }
    return (0);
}

/* malloc space to store a string */

char       *strsave(str)
    char       *str;
{
    char       *cp;

    if ((cp = malloc(strlen(str) + 1)) != NIL_PTR)
	strcpy(cp, str);
    return (cp);
}

/* find a character in forward direction, case insensitive */

char       *strichr(str, ch)
    register char *str;
    register char ch;
{
    register char *cp;

    ch = toupper(ch);

    while ((int) *(cp = str++))
    {
	if (toupper(*cp) == ch)
	    return (cp);
    }
    return (NIL_PTR);
}

/* find a substring str2 in str1, case insensitive */

char       *stristr(str1, str2)
    char       *str1;
    register char *str2;
{
    register char *cp;
    register int l1;
    register int l2;
    char       *strichr();

    l1 = strlen(str1);
    l2 = strlen(str2) - l1 + 1;
    for (cp = str2; l2-- > 0; cp++)
    {
	if ((cp = strichr(cp, *str1)) == NIL_PTR)
	    break;
	if (strnicmp(cp, str1, l1) == 0)
	    return (cp);
    }
    return (NIL_PTR);
}

/* string compare, case insensitive, limited string length */

int         strnicmp(str1, str2, num)
    register char *str1;
    register char *str2;
    register int num;
{
    register int res = 0;

    while (num--)
    {
	register int c = (int) toupper(*str1++);

	res = c - (int) toupper(*str2++);

	if ((res != 0) || (c == 0))
	    break;
    }
    return (res);
}

/* put a string in a list */

int         next_arg(entry)
    char       *entry;
{
    char       *strsave();

    if (namecount < LIST_SIZE)
    {
	if ((namelist[namecount++] = strsave(entry)) != NIL_PTR)
	    return (0);
	fprintf(stderr, "out of memory\n");
    }
    else
	fprintf(stderr, "too many files\n");
    exit(1);
}

/* check whether a path exists */

int         path_exists(path)
    char       *path;
{
    struct stat mystat;
    int result;

    if (stat(path, &mystat) < 0)
	result = NO_PATH;
    else if ((mystat.st_mode & S_IFMT) == S_IFDIR)
	result= IS_DIR;
    else
	result = NO_DIR;
    return(result);
}

/*
 * create all the subdirectories up to the name and the name itself.
 * Because of the selection methode used, the subdirectories may have a
 * different owner than original. There is a solution: while extracting
 * files, create all subdirectories there are in the tarfile even if there
 * will be no files that will get extracted to those directories.
 */

int         path_creat(target, mode)
    char       *target;
    int         mode;
{
    char        path[128];
    int         len;
    int         result;
    char       *cp;
    extern char *strchr();

    cp = target;
    while ((cp = strchr(cp + 1, '/')) != NIL_PTR)
    {
	len = (int) (cp - target);/* the length of the (partial) path */
	strncpy(path, target, len);
	path[len] = '\0';
#ifdef DEBUG
	fprintf(stderr, "exists: %s: ", path);
#endif
	switch (path_exists(path))
	{
	case IS_DIR:
#ifdef DEBUG
	    fprintf(stderr, "ok.\n");
#endif
	    result = 0;
	    break;		  /* exists already */
	case NO_PATH:
#ifdef DEBUG
	    fprintf(stderr, "mkdir\n");
#endif
	    if (verb_fl)
		fprintf(stdout, "%-40s %4d\n", path, 0);
	    else
		fprintf(stdout, "%s\n", path);

	    if ((result = mkdir(path)) == 0)
		break;		  /* done */

	case NO_DIR:
	default:
#ifdef DEBUG
	    fprintf(stderr, "error\n");
#endif
	    return (CR_ERROR);	  /* some error */
	}
    }
#ifdef DEBUG
    fprintf(stderr, "creat: %s\n", target);
#endif
    result = creat(target, mode);
#ifdef DEBUG
    fprintf(stderr, "result %d\n", result);
#endif
    return (result);
}

#ifdef EXTENDED_LIST
/* date - print date in ls format */

#include <time.h>

#define YEAR  (365L * 24L * 3600L)
static long curtime;
char       *moname[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

typedef struct tm type_tm;
extern type_tm *localtime();

date(t)
    long        t;		  /* time in seconds */
{
    /* Print the date. */

    type_tm     tm;

    tm = *localtime(&t);

    /* approximate current time */
    if (curtime == 0)
	curtime = time((long *) 0);

    fprintf(stdout, "%3s %2d", moname[tm.tm_mon], tm.tm_mday);
    if (curtime - t >= YEAR / 2L)
	fprintf(stdout, " %5d", tm.tm_year + 1900);
    else
	fprintf(stdout, " %02d:%02d", tm.tm_hour, tm.tm_min);
}
#endif
