/* backup <device> <files> copies files to backup device. by H. Salomons */

#include "backup.h"
#include "magic.h"

#define NO_DEBUG_ALL

#ifdef DEBUG_ALL
#define DEBUG_MALLOC
#define DEBUG_OPEN
#define DEBUG_REMOVE
#endif

static int  tar_fd = 0;

int         backup_one(myptr, volg)
    fi_ptr      myptr;
    int         volg;
{
    long        blocksize();
    long        perc;
    long        siz;
    char        bname[MAX_NAME];
    char        command[MAX_CMD];
    long        new_size;
    stat_t      arstat;

    /* if not yet compressed and large enough to compress */
    if ((myptr->fi_bsiz == 0L) && (myptr->fi_size > BLOCK_SIZE) &&
	(volg != -1) && do_compress)
    {
	switch (getmagic(myptr->fi_name))
	{
	case MAGIC_ZOO:
	case MAGIC_COMPRESS:
	    /* don't compress further */
	    break;

	case MAGIC_UNKNOWN:
	    sprintf(bname, "%s/bak%04d.Z", dirback, volg);
	    sprintf(command, "compress -cq < %s > %s", myptr->fi_name, bname);

	    fprintf(stdout, "compress %-65s%c", myptr->fi_name, *eoln);
	    fflush(stdout);

	    system(command);
	    if (stat(bname, &arstat) != 0)
		break;
	    if (arstat.st_size == 0)
	    {
		remove(bname);
		break;
	    }
	    if (arstat.st_size < myptr->fi_size)
	    {
		/* sensible to compress */
		myptr->fi_back = strsave(bname);
		myptr->fi_bsiz = arstat.st_size;
	    }
	    else
		/* clean up grown compress file */
		remove(bname);
	    break;
	}
    }

    if (myptr->fi_bsiz > 0L)
	siz = myptr->fi_bsiz;
    else
	siz = myptr->fi_size;

    if (blocksize(siz) > block_max)
    {
	fprintf(stderr, "file %s too large to fit in an archive, skipped\n",
		myptr->fi_name);
	return (0);
    }
    new_size = block_count + blocksize(siz);

    /* if this file still fits: */
    if (new_size < block_max)
    {
	/* save the new block count */
	block_count = new_size;
	if (fnout == NULLF)
	    open_next();

	/* print the result: archive-ing or compress rate */
	if (myptr->fi_back != (char *) 0)
	{
	    if (myptr->fi_size == 0)
		perc = 0L;
	    else
		perc = 100L - myptr->fi_bsiz * 100L / myptr->fi_size;
	    fprintf(stdout, "rate %2d%% %-65s%c",
		(int) perc, myptr->fi_name, *eoln);
	}
	else
	    fprintf(stdout, "archive  %-65s%c", myptr->fi_name,
		(volg == -1) ? '\n' : *eoln);
	fflush(stdout);

	/* add the member to the tar file */
	add_tar(tar_fd, myptr->fi_name, myptr->fi_back);

	/* add the name to the log file */
	fprintf(fnout, "%s\n", myptr->fi_name);

	/* the compressed copy went into the archive, remove that file */
	if (myptr->fi_back != (char *) 0)
	    remove(myptr->fi_back);

	/* return OK */
	return (0);
    }
    /* return FAIL */
    return (1);
}

int         free_info(myptr)
    fi_ptr      myptr;
{
    register char *cp;
    fi_ptr      nxt;

    /* walk the list of pointers */
    for (; myptr != NULL_FI; myptr = nxt)
    {
	/* free the orignal name */
	cp = myptr->fi_name;
	if (cp != (char *) 0)
	    FREE(cp);

	/* free the backup name */
	cp = myptr->fi_back;
	if (cp != (char *) 0)
	    FREE(cp);

	/* save the link to the next */
	nxt = myptr->fi_next;

	/* free the structure */
	FREE(myptr);
    }
}

/* tar */

long        blocksize(fsize)
    long        fsize;
{
    return (1L + (fsize + (long) BLOCK_SIZE - 1L) / BLOCK_SIZE);
}

close_curnt()
{
    if (fnout != NULLF)
    {
	fclose(fnout);
	adjust_boundary();
	close(tar_fd);
	fprintf(stdout, "\n%s: %ld blocks used\n", oname, block_count);
	file_number = 0;
    }
}

#ifdef DEBUG_OPEN
static      first = 0;
#endif
/* open the next list file and tar file */
open_next()
{
    struct stat stbuf;
    char        c;
    int         current;

    current = oseqnr++;
#ifdef DEBUG_OPEN
    fprintf(stdout, "open tar file %d\n", current);
    if (++first > 10)
    {
	fprintf(stderr, "too many archives created, stopping\n");
	exit(1);
    }
#endif
    sprintf(dname, diskname, current);
    sprintf(oname, listname, current);

    if ((fnout = fopen(oname, "w")) == NULLF)
    {
	fprintf(stderr, "can't open %s\n", oname);
	exit(1);
    }

    if (stat(dname, &stbuf) == 0)
    {
	if (((stbuf.st_mode) & S_IFBLK) == S_IFBLK)
	{
	    fprintf(stderr, "please insert %s (# %d) and hit <RETURN> ",
		    dname, current);
	    while (read(0, &c, 1) > 0)
	    {
		if (c == 3)
		    exit(1);
		else if (c == '\n')
		    break;
	    }
	}
    }
    if ((tar_fd = creat(dname, 0644)) < 0)
    {
	fprintf(stderr, "can't open %s\n", dname);
	exit(1);
    }
}

long        getftime(name)
    char       *name;
{
    struct stat statbuf;
    char       *ctime();

    if (stat(name, &statbuf) == 0)
    {
#ifdef DEBUG_TIME
	fprintf(stdout, "since = %s", ctime(&(statbuf.st_ctime)));
#endif
	return (statbuf.st_ctime);
    }
    return (0L);
}

char       *strsave(str)
    char       *str;
{
    char       *cp;

#ifdef DEBUG_STRSAVE
    fprintf(stdout, "try to save '%s' ", str);
#endif
    if ((cp = MALLOC(char, (strlen(str) + 1))) != (char *) 0)
	strcpy(cp, str);
#ifdef DEBUG_STRSAVE_2
    fprintf(stdout, "save %s at %lx\n", str, cp);
#endif
    return (cp);
}
