/*
 * WMAIL -	MicroWalt Extended Mail Agent.
 *		This is the MicroWalt Mail Agent; which is derived
 *		from  the  "Mini-Mail" written by Peter S. Housel.
 *
 * Usage:	wmail [-Qepqrv] [-f file]
 *		wmail [-Qdtv] [-i file] [-s subject] user ...
 *		lmail [-Qlv] [-i file] user ...
 *
 * Author:	Fred van Kempen, MicroWalt Corporation
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <pwd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include "wmail.h"


char *Version = VERSION;
int old_uid, old_gid;			/* "real" ID's of caller */
int remote = FALSE;			/* use RMAIL to deliver (if any) */
int loclink = FALSE;			/* LMAIL: local delivery only! */
int printmode = FALSE;			/* print-and-exit mode */
int immediate = FALSE;			/* send remote immediately! */
int quitmode = FALSE;			/* take interrupts */
int usedrop = TRUE;			/* read the maildrop (no -f given) */
int verbose = FALSE;			/* pass "-v" flag on to mailer */
int needupdate = FALSE;			/* need to update mailbox */
int sayall = TRUE;			/* include line with all recipients */
int checkonly = FALSE;			/* only chack if there is mail */
char home[PATHLEN];			/* user's home directory */
char sender[PATHLEN];			/* who sent the message? */
char forward[PATHLEN];			/* who is the message forwarded to? */
char addressee[PATHLEN];		/* current recipient */
char recipients[PATHLEN];		/* also to... users */
char addr_cc[PATHLEN];			/* Carbon Copy addressee */
char addr_bcc[PATHLEN];			/* Blank CC addressee */
char mailbox[PATHLEN];			/* user's mailbox/maildrop */
char subject[PATHLEN];			/* subject of message */
FILE *infp = (FILE *)NULL;		/* current message input file */
FILE *boxfp = (FILE *)NULL;		/* mailbox file */
char *progname;		 		/* program name */
jmp_buf printjump;	 		/* for quitting out of letters */
LETTER *firstlet, *lastlet;		/* message pointers for readbox() */
int curletno = 0;			/* holds number of current message */
int numlet, seqno;	 		/* number of active letters */
unsigned oldmask;	 		/* saved umask() */ 


extern int getopt(), optind, opterr;	/* from the GETOPT(3) package */
extern char *optarg;
extern int errno;	 		/* missing from "sys/errno.h" */


/* Remove all temporary files and exit. */
void clr_exit(num)
int num;
{
  (void) unlink(tempfn);
  exit(num);
}


/* Catch the SIGINT interrupt for interrupting the printing of letters. */
void onint()
{
  longjmp(printjump, TRUE);
}


/* Chop off the last (file) part of a filename. */
char *basename(name)
register char *name;
{
  register char *p;

  p = strrchr(name, '/');
  if (p == (char *)NULL) return(name);
    else return(p + 1);
}


/* Check if we may perform operation 'mode' on file 'name'. */
int allowed(name, mode)
char *name;			/* name of file to be checked */
unsigned short mode;		/* mode to check (R=4, W=2, X=1) */
{
  char abuf[1024];		/* temp. buf for filename */
  struct stat stb;
  char *p;

  /* Is this 'The Master' calling? */
  if (old_uid == 0) return(TRUE);

  if (stat(name, &stb) < 0) {
	if (errno == ENOENT) {
		strcpy(abuf, name);
		p = strrchr(abuf, '/');	
		if (p == (char *)NULL) getcwd(abuf, 1023);
		  else *p = '\0';
		if (stat(abuf, &stb) < 0) return(FALSE);
	} else return(FALSE);
  }

  /* We now have the status of the file or its parent dir. */
  if (stb.st_uid == old_uid) {
	if ((stb.st_mode >> 6) & mode) return(TRUE);
  	  else return(FALSE);
  } else if (stb.st_gid == old_gid) {
	if ((stb.st_mode >> 3) & mode) return(TRUE);
  	  else return(FALSE);
  } else if (stb.st_mode & mode) return(TRUE);
  return(FALSE);
}


/* Update the mail-box file. */
static void updatebox()
{
  char cpbuff[1024];			/* copy buffer */
  char lockname[PATHLEN];		/* maildrop lock */
  char copytemp[PATHLEN];		/* temporary copy file */
  FILE *copyfp;				/* fp for tempfile */
  LETTER *let;				/* current letter */
  int locktries = 0;			/* tries when box is locked */
  int newlet;				/* true if a new letter is found */
  strcpy(copytemp, COPYTEMP);
  mktemp(copytemp);
  sprintf(lockname, LOCKNAME, sender);
  
  /* Create a new mailbox-file. */
  if ((copyfp = fopen(copytemp, "w")) == (FILE *)NULL) {
	printf("***: Cannot create \"%s\" for update\n", copytemp);
	return;
  }
 
  /* Copy letters from old file to new file. */
  for (let = firstlet; let != NIL_LET; let = let->next) {
	if (let->status != DELETED) printlet(let, copyfp);
  }

  /*
   * If the last letter was deleted, newly received mail will be lost!
   * We try to avoid this by some additional work ...
   */
  if (lastlet->status == DELETED) {
	newlet = FALSE;
	fseek(boxfp, (off_t) (lastlet->location + 1), SEEK_SET);
	while (fgets(cpbuff, sizeof(cpbuff), boxfp) != (char *)NULL) {
		if (!newlet && strncmp(cpbuff,"From ", 5) == 0)
							newlet = TRUE;
		if (newlet == TRUE) {
			fwrite(cpbuff, sizeof(char), strlen(cpbuff), copyfp);
		}
	}
  }

  if ((copyfp = freopen(copytemp, "r", copyfp)) == (FILE *)NULL) {
	sprintf(cpbuff, "%s: temporary file write error", progname);
	perror(cpbuff);
	if (usedrop) unlink(copytemp); 
	return;
  }

  /* Shut off signals during the update. */
  signal(SIGINT, SIG_IGN);
  signal(SIGHUP, SIG_IGN);
  signal(SIGQUIT, SIG_IGN);

  if (usedrop) while(link(mailbox, lockname) != 0) {
	if (++locktries >= LOCKTRIES) {
		fprintf(stderr, "%s: could not lock maildrop for update\n",
								progname);
		return;
	}
	sleep(LOCKWAIT);
  }

  if ((boxfp = freopen(mailbox, "w", boxfp)) == (FILE *)NULL) {
	sprintf(cpbuff, "%s: could not reopen maildrop\n", progname);
       	fprintf(stderr, "%sMail may have been lost; look in %s\n",
							cpbuff, copytemp);
       	unlink(lockname);
       	return;
  }

  /* Copy temp. file to mailbox. */
  while (TRUE) {
	if (fgets(cpbuff, sizeof(cpbuff), copyfp) == (char *)NULL) break;
	fwrite(cpbuff, sizeof(char), strlen(cpbuff), boxfp);
  }
  fflush(copyfp);
  fflush(boxfp);
  fclose(boxfp);
  fclose(copyfp);
  unlink(copytemp); 
  if (usedrop) unlink(lockname);

  if (newlet == TRUE) printf("You have new mail.\n");
}


/*
 * Find the given entry in the mail-header
 * Search for the first occurence of string 'text' in the header.
 * Copy the text following it into the 'let' structure.
 * Return buffer if found, else NULL.
 */
char *find_string(let, text)
LETTER *let;
char *text;
{
  static char inbuff[512];
  off_t curr, limit;
  register char *sp;
  int all;

  fseek(boxfp, let->location, SEEK_SET);
  limit = (off_t) -1L;
  if (let->next != NIL_LET) limit = let->next->location;

  all = FALSE;
  curr = let->location;
  while (curr != limit && all == FALSE) {
	if (fgets(inbuff, sizeof(inbuff), boxfp) == (char *)NULL) all = TRUE;
      	if (inbuff[0] == '\n') all = TRUE; /* end-of-header */

      	if (!strncmp(inbuff, text, strlen(text))) {
		inbuff[strlen(inbuff) - 1] = '\0';
		return(inbuff + strlen(text));	/* return address of buff */
       	}

	curr += (off_t) strlen(inbuff);		/* update message offset */

	if (all == FALSE && limit > 0L)		/* quit if past message */
		if (curr >= limit) all = TRUE;
  }
  return((char *)NULL);
}


/*
 * Check if the first line of the mailbox contains a line like
 *
 *	Forward to XXXX
 *
 * then all mail for the calling user is being forwarded
 * to user XXXX. Return a 1 value if this is the case.
 * Otherwise, return 0 (or -1 for error).
 */
int chk_box(boxname)
char *boxname;
{
  char xbuf[128];
  FILE *fp;
  char *bp;

  if (access(boxname, 04) < 0 || (fp = fopen(boxname, "r")) == (FILE *)NULL) {
	if (usedrop && errno == ENOENT) return(-1);
     	fprintf(stderr, "%s: cannot access mailbox ", progname);
      	perror(boxname);
      	clr_exit(1);
  }

  bp = fgets(xbuf, sizeof(xbuf), fp);
  fclose(fp);

  if (bp != (char *)NULL && !strncmp(xbuf, "Forward to ", 11)) {
	strcpy(forward, strrchr(xbuf, ' ') + 1);
	forward[strlen(forward) - 1] = '\0';
	return(1);
  }
  return(0);
}


/*
 * Clear out any in the given address line.
 * Some mailers add text which shouldn't be in a V6/V7
 * "From user date" address line.  This routine removes
 * all that junk, and returns the address of the new
 * string.  Actually, this means conversion of:
 *
 *	From waltje (Fred van Kempen) Tue, 10 Apr 90 20:30:00
 * to:
 *	From waltje Tue, 10 Apr 90 20:30:00
 */
char *clr_hdr(old)
char *old;
{
  char *buf;
  register char *bp, *sp;
  int parstack, litstack;

  buf = (char *) malloc(strlen(old) + 2);
  if (buf == (char *)NULL) {
	fprintf(stderr, "%s: out of memory: \"%s\"\n", progname, old);
	return((char *)NULL);
  }
  bp = buf; sp = old;
  parstack = 0; litstack = 0;
  while (*sp) switch(*sp) {
	case '(':	/* start comment */
		parstack++;
		sp++;
		break;
	case ')':	/* end comment */
		parstack--;
	 	if (parstack < 0) {
			fprintf(stderr, "%s: bad header: \"%s\"\n",
							progname, old);
			return((char *)NULL);
		} else sp++;
		break;
	case '<':	/* start literal */
		litstack++;
		sp++;
		break;
	case '>':	/* end literal */
		litstack--;
		if (litstack < 0) {
			fprintf(stderr, "%s: bad header: \"%s\"\n",
							progname, old);
			return((char *)NULL);
		} else sp++;
		break;
	default:
		if (parstack == 0 && litstack == 0) *bp++ = *sp;
		sp++;
  }
  *bp = '\0';
  return(buf);
}


/*
 * Decode an old-style (V6/V7) mail header.
 * This is a line like:
 *
 *	From <user> <date> [remote from <host>]
 *
 * We want to find out the <user>, <date> and possible <remote> fields.
 * Warning: some mailers (especially some configurations of the
 *	    SendMail program) allow comments (like (RealName)) to be
 *	    placed in this field.  These comments are removed by the
 *	    clr_hdr() routine.
 */
static void old_hdr(let, text)
LETTER *let;			/* which letter? */
char *text;			/* message header text */
{
  register char *bp, *sp;
  char *buf, *cp;
  int i;

  /* First of all, clear out any junk. */
  buf = clr_hdr(text);
  sp = buf;
  if (sp == (char *)NULL) return;

  /* Then, mark the end of the 'user' field. Skip until <date> field. */
  while (*sp != ' ' && *sp != '\t') sp++;
  *sp++ = '\0';
  while (*sp == ' ' || *sp == '\t') sp++;

  /*
   * SP now contains <date> and (possible) <remote> fields.
   * Parse line to seek out "remote from".
   */
  cp = sp;
  while (TRUE) {
	bp = strchr(sp++, 'r');
	if (bp != (char *)NULL) {
		if (!strncmp(bp, "remote from ", 12)) break;
        } else break;
  }

  if (bp != (char *)NULL) {
	sp = bp + 12;
	*(bp - 1) = '\0';
	bp = strchr(sp, ' ');
	if (bp != (char *)NULL)
		while (*bp == ' ' || *bp == '\t') *bp++ = '\0';
	i = strlen(sp);
  } else {
	i = 0;
  }

  /*
   * Compress the date format in cp[]
   * from   Mon, 12 May 90 12:34:56  or  Mon May 12 12:34:56 1990 
   * to     12 May 12:34             or  May 12 12:34
   */
  if (strchr(cp, ',')) {
	strcpy(let->date, cp + 5);
	strcpy(let->date + 7, cp + 15);
  } else {
	strcpy(let->date, cp + 4);
  }
  if (cp = strrchr(let->date, ':')) *cp = '\0';

  /*
   * Find the return-addresses.
   * First, check for a From: field.  If it fails, use the
   * current "host" and "user" names, and create a bang.
   */
  cp = find_string(let, "From: ");
  if (cp == (char *)NULL) {
	let->sender2 = (char *)malloc(strlen(buf) + i + 4);
	if (let->sender2 == (char *)NULL) {
		fprintf(stderr, "%s: out of memory.\n", progname);
		clr_exit(1);
	}
	/* Create the (dangerous) return-address. */
	if (i > 0) sprintf(let->sender2, "%s!%s", sp, buf);
	  else strcpy(let->sender2, buf);
  } else {
	/* First find a possible <address> field. */
	if (sp = strchr(cp, '<')) {
		cp = ++sp;
		/* Remove the <address> trailing > */
		if (sp = strchr(cp, '>')) 
			*sp = '\0';
	} else {
		/* Remove possible (Real Name) field. */
		if (sp = strchr(cp, '(')) {
			/* Strip also trailing spaces. */
			while ( *(sp - 1) == ' ' || *(sp - 1) == '\t') 
				sp--;
			*sp = '\0';
		}
	}

	let->sender2 = (char *)malloc(strlen(cp) + 2);
	if (let->sender2 == (char *)NULL) {
		fprintf(stderr, "%s: out of memory.\n", progname);
		clr_exit(1);
	}
	strcpy(let->sender2, cp);
  }

  /*
   * The default Reply-address is usually simpler, since most mailers
   * generate a Reply-To: field.  If it is present, use it, otherwise,
   * use the previously decoded address.
   */
  cp = find_string(let, "Reply-To: ");
  if (cp != (char *)NULL) {
	/* First find a possible <address> field. */
	if (sp = strchr(cp, '<')) {
		cp = ++sp;
		/* Remove the <address> trailing > */
		if (sp = strchr(cp, '>')) 
			*sp = '\0';
	} else {
		/* Remove possible (Real Name) field. */
		if (sp = strchr(cp, '(')) {
			/* Strip also trailing spaces. */
			while ( *(sp - 1) == ' ' || *(sp - 1) == '\t') 
				sp--;
			*sp = '\0';
		}
	}

	let->sender1 = (char *)malloc(strlen(cp) + 2);
	if (let->sender1 == (char *)NULL) {
		fprintf(stderr, "%s: out of memory.\n", progname);
		clr_exit(1);
	}
	strcpy(let->sender1, cp);
  } else {	/* not present, use the From: field */
	let->sender1 = let->sender2;
  }

  /* Release the allocated memory. */
  free(buf);
}


/* Read the contents of the Mail-Box into memory. */
static int readbox()
{
  static char lbuff[512];
  register LETTER *let;
  register char *sp, *cp;
  off_t current;
  int no_lines;
 
  firstlet = lastlet = NIL_LET;
  numlet = 0;
  seqno = 1;

  if (chk_box(mailbox) == 1) return(FALSE);

  if ((boxfp = fopen(mailbox, "r")) == (FILE *)NULL) {
	if (usedrop && errno == ENOENT) return(-1);
      	fprintf(stderr, "%s: cannot access mailbox ", progname);
      	perror(mailbox);
      	clr_exit(1);
  }

  /* Determine where all messages start and measure the messages. */
  current = (off_t) 0;
  no_lines = 1;
  while(fgets(lbuff, sizeof(lbuff), boxfp) != (char *)NULL) {
	current = ftell(boxfp);
      	if (!strncmp(lbuff, "From ", 5)) {
		if ((let = (LETTER *)malloc(sizeof(LETTER))) == NIL_LET) {
			fprintf(stderr, "%s: out of memory.\n", progname);
			clr_exit(1);
	 	}
        	if (lastlet == NIL_LET) {
			firstlet = let;
			let->prev = NIL_LET;
	 	} else {
	 		let->prev = lastlet;
	     	 	lastlet->next = let;
			lastlet->no_lines = no_lines;
			no_lines = 1;
			lastlet->no_chars = current - (off_t) strlen(lbuff)
				- lastlet->location;
	    	}
		lastlet = let;
		let->next = NIL_LET;

		let->status = UNREAD;
		let->location = current - (off_t) strlen(lbuff);
		let->seqno = seqno++;
		numlet++;
	}
	else no_lines++;
  }
  if (lastlet != NIL_LET) {
	lastlet->no_lines = no_lines;
	lastlet->no_chars = current - (off_t) strlen(lbuff) - lastlet->location;
  }

  /* We now know where the messages are, read message headers. */
  let = firstlet;
  while (let != NIL_LET) {
	sp = find_string(let, "From ");		/* Find the "From_" field. */
	if (sp == (char *)NULL) {
		fprintf(stderr, "%s: no \"From\"-line.\n", progname);
		clr_exit(-1);
	}

	old_hdr(let, sp);

	/* Find the "Subject" field, if any... */
	sp = find_string(let, "Subject: ");
	if (sp == (char *)NULL) sp = "<none>";
	let->subject = (char *)malloc(strlen(sp) + 1);
	if (let->subject == (char *)NULL) {
		fprintf(stderr, "%s: out of memory.\n", progname);
		clr_exit(1);
	}
	strcpy(let->subject, sp);

	/* Find the real sender name in the From: field, if any... */
	sp = find_string(let, "From: ");
	if (sp != (char *)NULL) {
		if (cp = strchr(sp, '<')) 
			*cp = '\0';
		else if (cp = strchr(sp, '(')) {
			sp = cp + 1;
			if (cp = strchr(sp, ')')) *cp = '\0';
		}
	} else {	/* No From: field, try From_ field */
		sp = find_string(let, "From ");
		if (sp != (char *) NULL) {
			if (cp = strchr(sp, ' ')) 
				*cp = '\0';
			if (cp = strrchr(sp, '!')) 
				sp = cp + 1;
			else 	sp = getpwnam(sp)->pw_gecos;
		} else sp = "Unknown";
	}

	/* Stuff the real name in memory. */
	let->real_name = (char *)malloc(strlen(sp) + 1);
	if (let->real_name == (char *)NULL) {
		fprintf(stderr, "%s: out of memory.\n", progname);
		clr_exit(1);
	}
	strcpy(let->real_name, sp);
	
	let = let->next;
  }
  return(TRUE);
}


/* Check if there is any mail for the calling user. */
static int chk_mail()
{
  FILE *fp;
  char temp[512];

  if ((fp = fopen(mailbox, "r")) != (FILE *)NULL) {
	if (fgets(temp, sizeof(temp), fp) == (char *)NULL) {
		fclose(fp);
		return(1);
	}
      	if (!strncmp(temp, "Forward to ", 11)) {
		fclose(fp);
		return(2);
	}
	fclose(fp);
	return(0);
  }
  return(1);
}


static void usage()
{
  fprintf(stderr, "Usage:\n");
  fprintf(stderr, "\t%s [-Qepqrv] [-f file]\n", progname);
  fprintf(stderr, "\t%s [-Qdtv] [-i file] [-s subject] user ...\n", progname);
  fprintf(stderr, "\t%s [-Qlv] [-i file] user ...\n\n", progname);
}


int main(argc, argv)
int argc;
char *argv[];
{
  struct passwd *pw;
  char buff[PATHLEN];
  int c, st, quick;

  pw = getpwuid(getuid());
  if (pw == (struct passwd *)NULL) {
 	fprintf(stderr, "%s: I do not exist?\n", argv[0]);
	exit(-1);
  }
  strcpy(sender, pw->pw_name);
  strcpy(home, pw->pw_dir);

  old_uid = getuid();
  old_gid = getgid();
  setgid(getegid());
  setuid(geteuid());

  progname = basename(argv[0]);		/* how were we called? */
  if (*progname == 'l') {
	remote = FALSE;			/* 'lmail' link? */
	loclink = TRUE;
  }
  strcpy(tempfn, MAILTEMP);
  mktemp(tempfn);
  oldmask = umask(077);
  infp = stdin;
  
  opterr = 0;
  quick = FALSE;
  while ((c = getopt(argc, argv, "Qdef:i:lpqrs:tv")) != EOF) switch(c) {
	case 'Q':	/* Quick mode. */
		quick = TRUE;
		break;
	case 'd':	/* Deliver immediately. */
		immediate++;
		break;
	case 'e':	/* Only check for mail, do not read it. */
		checkonly++;
		break;
	case 'f':	/* use another mailbox. */
		usedrop = FALSE;
		if (allowed(optarg, 04) == FALSE) {
			fprintf(stderr, "%s: permission denied for %s\n",
				progname, optarg);
			exit(-1);
		}
		strncpy(mailbox, optarg, PATHLEN - 1);
		break;
	case 'i':	/* Use another input-file. */
		if (allowed(optarg, 04) == TRUE) {
			infp = fopen(optarg, "r");
		} else infp = (FILE *)NULL;
		if (infp == (FILE *)NULL) {
			fprintf(stderr, "%s: cannot open %s\n",
						progname, optarg);
			clr_exit(-1);
		}
		break;
	case 'l':	/* Specify 'local' mail; same as 'lmail'. */
		loclink = TRUE;
		break;
	case 'p':	/* Print all messages and exit. */
		printmode++;
		break;
	case 'q':	/* Abort if SIGINT received. */
		quitmode++;
		break;
	case 'r':	/* Show messages in reverse order. */
		break;	/* This is only present for comp. with binmail! */
	case 's':	/* Specify "subject" line. */
		strcpy(subject, optarg);
		break;
	case 't':	/* Show all addressees in "To:" line. */
		sayall = FALSE;	/* it is TRUE by default !! */
		break;
	case 'v':	/* Turn on debugging. */
		verbose++;
		break;
	default:
		usage();
		clr_exit(2);
  }

#ifdef QUICKLOCAL
  /* There is no need to read the RC file for delivery! */
  if (loclink == TRUE) quick = TRUE;
#endif

  if (optind >= argc) {
	if (usedrop) sprintf(mailbox, DROPNAME, sender);

	if (checkonly) {
		st = chk_mail();
		clr_exit(st);
	}

	if (readbox() == FALSE) {
		fprintf(stderr, "Your mail is being forwarded to %s.\n",
								forward);
		clr_exit(1);
	} else {
		st = 0;

		if (quick == FALSE) {
			/* Read the System Mail Config File. */
			sprintf(buff, "%s/%s", SYSDIR, SYSMAILRC);
			rc_read(buff);

			/* Read the User Mail Config File. */
			sprintf(buff, "%s/%s", home, MAILRC);
			rc_read(buff);
		}

	 	if (printmode) printall();
		  else interact();

		if (needupdate) updatebox();

		clr_exit(st);    
	}
  } else {
	if (quick == FALSE) {
		/* Read the System Mail Config File. */
		sprintf(buff, "%s/%s", SYSDIR, SYSMAILRC);
		rc_read(buff);

		/* Read the User Mail Config File. */
		sprintf(buff, "%s/%s", home, MAILRC);
		rc_read(buff);
	}

	st = do_send(argv + optind, (char *)NULL, FALSE);

	if (st != 0) dead_letter();	/* something went wrong... */

	clr_exit(st);    
  }
}
