/*
 * WMAIL -	MicroWalt Extended Mail Agent.
 *		This is the MicroWalt Mail Agent; which is derived
 *		from  the  "Mini-Mail" written by Peter S. Housel.
 *
 *		M A I L    G E N E R A T I O N    M O D U L E
 *
 * Author:	Fred van Kempen, MicroWalt Corporation
 */
#include <sys/types.h>
#include <ctype.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include "wmail.h"


char tempfn[PATHLEN];			/* name of temp. file */
static FILE *tempfp;			/* FILE * of temp. file */


/*
 * Fetch the contents of message or file 'num' and copy them
 * to the 'tempfp' file.  If 'quote' is TRUE, prefix each copied
 * line with "> ".  Also note, that we *must* quote any line
 * beginning with "From_".
 */
static void fetch_msg(num, quote, nchar, nlin)
char *num;
int quote;
off_t *nchar;
int *nlin;
{
  char buff[1024];
  register char *bp;
  off_t nchars, curr, limit;
  int nlines;
  register int c;
  register FILE *ifp;
  LETTER *let;

  nchars = 0L;
  nlines = 0;

  /* Is this a message or a file? */
  if (isdigit(*num)) {
	let = goto_letter(atoi(num));
	if (let == NIL_LET) {
		printf(": no message with number %s\n", num);
		*nchar = nchars;
		*nlin = nlines;
		return;
	}
	fseek(boxfp, (curr = let->location), SEEK_SET);
	limit = (let->next != NIL_LET) ? let->next->location : -1L;

	while(curr != limit) {
		bp = buff;
		do {
			c = fgetc(boxfp);
			if (c != EOF && c != '\n') {
				*bp++ = c;
				nchars++;
			}
			curr++;
		} while (c != EOF && c != '\n' && curr != limit);
		if (c == EOF) curr = limit;
		*bp = '\0';
		nlines++;
		if (quote || !strncmp(buff, "From ", 5))
				fprintf(tempfp, "> %s\n", buff);
		  else fprintf(tempfp, "%s\n", buff);
	}
  } else {
	if (allowed(num, 04) == TRUE) {
		ifp = fopen(num, "r");
		if (ifp != (FILE *)NULL) {
			nchars = 0L;
			nlines = 0;
			while (fgets(buff, 1024, ifp)) {
				if (quote || !strncmp(buff, "From ", 5))
					fprintf(tempfp, "> %s", buff);
				  else fprintf(tempfp, "%s", buff);
				nchars += (off_t) strlen(buff);
				nlines++;
			}
		} else printf(": cannot open file \"%s\"\n", num);
	} else printf(": not allowed to read file \"%s\"\n", num);
  }
  *nchar = nchars;
  *nlin = nlines;
}


/* Add the ".signature" file to the current message. */
static app_signature()
{
  char signame[PATHLEN];
  FILE *sigfp;
  char *sp;
  register int c;

  sp = getenv("SIGNATURE");
  if (sp == (char *)NULL) sprintf(signame, "%s/%s", home, rc_signature);
    else strcpy(signame, sp);

  /* Check if we may read the file. */
  if (allowed(signame, 04) == TRUE) {
	if ((sigfp = fopen(signame, "r")) != (FILE *)NULL) {
		while ((c = fgetc(sigfp)) != EOF) fputc(c, tempfp);
		fclose(sigfp);
	}
  } else {
	fprintf(stderr, "%s: cannot read file \"%s\"\n", progname, signame);
  }
}


/*
 * Run a program in "safe" mode.
 * This means, that we protect the user from being "root"
 * as long as the program is being run.
 */
static void do_run(prg, args)
char *prg;
char *args[];
{
  int status, pid;

  args[0] = prg;
  if ((pid = fork()) == 0) {
	setgid(old_gid);
	setuid(old_uid);
	execv(prg, args);
	exit(0);
  } else if (pid > 0) {
	while (wait(&status) != pid)
		    ;
  }
}


/* Perform a "tilde" action, like in "mailx" style. */
static int do_tilde(text)
char *text;
{
  char *args[4];
  char buff[16];
  register char *sp;
  int nlin;
  off_t nchar;

  /* First off, kill the newline. */
  sp = strchr(text, '\n');
  if (sp != (char *)NULL) *sp = '\0';

  switch(*text) {
	case '?': 	/* Editor Help. */
		printf("\n\t*** W-MAIL Editor Help ***\n\n");
		printf("~?\t\tThis Help Screen\n");
		printf("~!\t\tEscape into Shell\n");
		printf("~A\t\tForce .signature to be appended to message\n");
		printf("~a\t\tAppend .signature to message if not autosig\n");
		printf("~e\t\tInvoke line editor on current message\n");
		printf("~m msgno\tInsert message from the current mailbox\n");
		printf("~q\t\tQuit (with save) sending this message\n");
		printf("~r filename\tInsert file into current message\n");
		printf("~R filename\tInsert quoted file into current message\n");
		printf("~v\t\tInvoke visual editor on current message\n");
		printf("~x\t\tExit (abort) sending this message\n");
		printf("\n");
		break;
	case '!':	/* Shell escape */
		sp = ++text;
		while (*sp == ' ' || *sp == '\t') sp++;
		if (*sp != '\0') {
			args[1] = "-c";
			args[2] = sp;
			args[3] = (char *)NULL;
		} else 	args[1] = (char *)NULL;
		do_run(rc_shell, args);
		printf("\n[continue]\n");
		break;
	case 'a':	/* Append SIGNATURE */
		if (rc_autosig == TRUE) {
			printf("*** Signature will be appended automatically!\n");
			break;
		} else app_signature();
		printf("[continue]\n");
		break;
	case 'A':	/* Force SIGNATURE to be appended */
		app_signature();
		printf("[continue]\n");
		break;
	case 'e':	/* Invoke Line Editor (ED) */
		fclose(tempfp);
		args[1] = tempfn;
		args[2] = (char *)NULL;
		do_run(rc_eedit, args);
		tempfp = fopen(tempfn, "a");
		printf("[continue]\n");
		break;
	case 'm':	/* Input "mailbox" message */
		text++;
		while (*text == ' ' || *text == '\t') text++;
		if (*text == '\0') {	/* no number; use current message */
			if (curletno != 0) {
				printf("\"Message %d\"", curletno);
				sprintf(buff, "%d", curletno);
				text = buff;
			} else {
				printf("*** Need message number!\n");
				break;
			}
		} else printf("\"Message %s\"", text);
		fetch_msg(text, TRUE, &nchar, &nlin);
		if (nchar > 0) printf(" [quoted] %d/%ld",
						nlin, (long) nchar);
		printf("\n[continue]\n");
		break;
	case 'q':	/* Quit (abort) sending message (save message). */
		fclose(tempfp);
		dead_letter();
		/* FALL_THROUGH */
	case 'x':	/* Exit this message (no save). */
		return(FALSE);
	case 'r':	/* Read and insert File */
		text++;
		while (*text == ' ' || *text == '\t') text++;
		printf("\"%s\"", text);
		fetch_msg(text, FALSE, &nchar, &nlin);
		if (nchar > 0) printf(" %d/%ld", nlin, (long) nchar);
		printf("\n[continue]\n");
		break;
	case 'R':	/* Read and insert Quoted File */
		text++;
		while (*text == ' ' || *text == '\t') text++;
		printf("\"%s\"", text);
		fetch_msg(text, TRUE, &nchar, &nlin);
		if (nchar > 0) printf(" [quoted] %d/%ld",
						nlin, (long) nchar);
		printf("\n[continue]\n");
		break;
	case 'v':	/* Invoke Visual Editor (VI) */
		fclose(tempfp);
		args[1] = tempfn;
		args[2] = (char *)NULL;
		do_run(rc_vedit, args);
		tempfp = fopen(tempfn, "a");
		printf("[continue]\n");
		break;
	default:
		printf("*** Unknown escape.  Please use ~? for help.\n");
		break;
  }
  return(TRUE);
}


/*
 * Create a message, including the Mail-header and the final signature.
 * If this is a link to LMAIL and/or if the message is read from a file
 * leave out the signature.
 * This routine copies lines of text from 'stdin' to 'tempfp', until an
 * EOF is typed, or a line containing only a '.'.
 * We complicate things by not setting a line limit.
 * Define V7MAIL to skip the "To:"-field.
 * Return NULL if we quit the message, or if an error occurs.
 */
FILE *edit_mail(msg, quote)
char *msg;			/* ASCII-number of message, NULL for new */
int quote;			/* must the message be quoted? */
{
  char cpbuff[1024];
  char xcc[512];
  char xbcc[512];
  register char *sp;
  off_t chars;
  int lins, c;

  if ((tempfp = fopen(tempfn, "w")) == (FILE *)NULL) {
	fprintf(stderr, "%s: cannot create temp file \"%s\"\n",
						progname, tempfn);
	return((FILE *)NULL);
  }

  /* Make sender owner of temp; he must read it in EDITOR escapes! */
  chown(tempfn, old_uid, old_gid);

  /* 
   * Create the header. This has the form:
   *
   *  From <user> <date>
   *  Subject: <text>
   *  X-Mailer: W-MAIL version ID
   *  To: <userlist>
   *  Cc: <cclist>
   *  Bcc: <bcclist>
   */
  if (loclink == FALSE) {	/* only create header if not LMAIL! */
	fprintf(tempfp, "From %s %s\n", sender, xtime());
	if (isatty(fileno(infp))) {
		if (subject[0] == '\0') {
			if (rc_asksub == TRUE) {
			 	fprintf(stderr, "Subject: ");
				gets(subject);
       				sp = strrchr(subject, '\n');
				if (sp != (char *)NULL) *sp = '\0';
			}
       		}
		if (rc_askcc == TRUE) {
			fprintf(stderr, "Cc: ");
			gets(addr_cc);
			strcpy(xcc, "");
			(void) exp_alias(addr_cc, xcc, 0, (char **)NULL);
			sp = strrchr(xcc, ',');
			if (sp != (char *)NULL) *sp = '\0';
		} else strcpy(addr_cc, "");
		if (rc_askbcc == TRUE) {
			fprintf(stderr, "Bcc: ");
			gets(addr_bcc);
			strcpy(xbcc, "");
			(void) exp_alias(addr_bcc, xbcc, 0, (char **)NULL);
			sp = strrchr(xbcc, ',');
			if (sp != (char *)NULL) *sp = '\0';
		} else strcpy(addr_bcc, "");
	}
#ifndef V7MAIL
	fprintf(tempfp, "X-Mailer: W-MAIL %s\n", Version);
	fprintf(tempfp, "To: %s\n", recipients);
	if (addr_cc[0] != '\0')
		fprintf(tempfp, "Cc: %s\n", xcc);
#endif	/* V7MAIL */
	if (subject[0] != '\0')
		fprintf(tempfp, "Subject: %s\n", subject);
	fprintf(tempfp, "\n");

	/*
	 * We now have a complete header.  Add the message body.
	 * If this stdIN is a terminal, we should process the in-
	 * put in "editor" mode; if the stdIN is a file (redirecttion!)
	 * then we can just copy the data to the temp. file.
	 */
	c = TRUE;
	if (isatty(fileno(infp))) {
		while (c == TRUE) {
			sp = fgets(cpbuff, 1024, infp);
			if (sp == (char *)NULL) break;
			if (cpbuff[0] == '.' && cpbuff[1] == '\n') break;
			  else if (cpbuff[0] == '~' && cpbuff[1] != '~')
						c = do_tilde(&cpbuff[1]);
	        	  else fwrite(cpbuff, sizeof(char), strlen(cpbuff),
								tempfp);
		}
		if (msg != (char *)NULL) {
			/* Add the forwarded message. */
			fprintf(tempfp, "Forwarded message #1:\n");
			fetch_msg(msg, quote, &chars, &lins);
			fprintf(tempfp, "\nEnd of forwarded messages\n\n");
		}
	} else {
		while (TRUE) {
			if (fgets(cpbuff, 1024, infp) == (char *)NULL)
								break;
			fwrite(cpbuff, sizeof(char), strlen(cpbuff), tempfp);
		}
	}

	if (c == FALSE) {	/* Abort mesaage */
		fclose(tempfp);
		return((FILE *)NULL);
	}

	/*
	 * We now have the header plus the body.
	 * Add a .signature file after the message.
	 * Skip this if infp is a file.
	 * Check first for security!
	 */
	if (isatty(fileno(infp))) {
		if (rc_autosig == TRUE) app_signature();
	}
  } else {	/* Fast file copy for local delivery. */
	  /* Copy temp. file to mailbox. */
	  while (TRUE) {
		if (fgets(cpbuff, sizeof(cpbuff), infp) == (char *)NULL)
								      break;
		fwrite(cpbuff, sizeof(char), strlen(cpbuff), tempfp);
	  }
  }

  if (ferror(tempfp) || fclose(tempfp)) {
	fprintf(stderr, "%s: could not copy letter to temporary file\n",
								progname);
      	return((FILE *)NULL);
  }

  /* Flush and close file; reopen for reading. */
  fclose(tempfp);
  tempfp = fopen(tempfn, "r");
  return(tempfp);
}
