/*
 * UMAIL -	MINIX Remote Domain-addressing Mail Router
 *
 *		This version of RMAIL handles message-headers in a much
 *		more "standard" way. It can handle bang-addresses, plus
 *		the new-style Internet addressing (user@host.domain).
 *		It is called by programs as "Mail" and "Uuxqt".
 *
 * 		M A I L   T R A N S P O R T   M O D U L E
 *
 * Author:	F. van Kempen, waltje@minixug.hobby.nl
 *
 * Revisions:
 *		11/07/89 FvK	Edited a little for the new MSS.
 *		11/13/89 FvK	Changed genname() to use the new routines
 *				readseq() and writeseq().
 *				Also, changed some naming conventions
 *				in send_remote(). The 'C'-file is now
 *				equal to the Message-ID.
 *		12/10/89 FvK	Adjust the protection-bits for the DATA file.
 *		12/16/89 FvK	Cleanup.
 *		12/30/89 FvK	Adapted for POSIX (MINIX 1.5)
 *		02/17/90 FvK	Cleaned for release.
 *		05/20/90 FvK	Added existence-check for ERRUSER before
 *				sending (failed!) mail to it...
 *		06/12/90 FvK	Changed "errno.h" include directory.
 *		06/14/90 FvK	Added Ralf Wenk's patches
 *		06/22/90 FvK	Changed sendit() a little to allow for
 *				better delivery programs.
 */
#include <sys/types.h>
#include <sys/errno.h>
#include <dirent.h>
#include <pwd.h>
#include <string.h>
#include <time.h>
#include <stdio.h>
#include "umail.h"
#include "uucp.h"


static char *seqname = SPOOLSEQ;	/* to save some string space */
static char *lseqname = LSPOOLSEQ;	/* to save some string space */


/*
 * Check the host machine name.
 * Return FALSE if not found, else TRUE.
 */
int KnowHost(name)
char *name;
{
  register HOST *hp;

  hp = gethost(name);
  return((hp == NILHOST) ? FALSE : TRUE);
}


/*
 * Check if this is one of our local names.
 * Return TRUE or FALSE.
 */
int islocal(name)
char *name;
{
  register NAME *np;

  np = namelist;
  while (np != NILNAME) {
	if (!strcmp(np->name, name)) return(TRUE);
	np = np->next;
  }
  return(FALSE);
}


/*
 * Read the SPOOLSEQ file and return that ID.
 */
static off_t readseq()
{
  char seqline[10];			/* read buffer */
  off_t seq;				/* the decoded number */
  FILE *spoolseq;			/* the file pointer */
  int i;

  if (access(seqname, 0) != 0) return((off_t) -1);

  while(link(seqname, lseqname) != 0) {
  	sleep(5);
  	if (++i > 5) return((off_t)-1);
  }

  spoolseq = fopen(seqname, "r");
  fgets(seqline, sizeof(seqline), spoolseq);
  fclose(spoolseq);
  unlink(lseqname);

  seq = (off_t) atol(seqline);	/* read the current ID */
  seq++;			/* and use next */

  return(seq);
}


/*
 * Update the SPOOLSEQ file.
 */
static void writeseq(seq)
off_t seq;
{
  FILE *spoolseq;			/* the file pointer */
  int i;

  if (access(seqname, 0) != 0) {	/* create file, owned by UUCP */
	close(creat(seqname, 0600));
	chown(seqname, UUCPUID, UUCPGID);
  }

  while(link(seqname, lseqname) != 0) {	/* lock it */
  	sleep(5);
  	if (++i > 5) return;
  }

  /* Write new contents. */
  if ((spoolseq = fopen(seqname, "w")) != (FILE *)NULL) {
	fprintf(spoolseq, "%ld\n", (long) seq);
	fclose(spoolseq);
  }

  unlink(lseqname);			/* remove lock */
}


/*
 * Create a "short" message-ID.
 * This has the form:
 *
 *	"AAseqno"
 *
 * and return the string.
 */
char *shortid()
{
  static char shidbuf[16];
  off_t seq;

  /* Get the message-ID. */
  seq = readseq();
  if (seq == (off_t)-1) seq == (off_t) 1;

  /* Create the string. */
  sprintf(shidbuf, "AA%04.4ld", (long) seq);

  return(shidbuf);
}


/*
 * Create a "Message-ID" field.
 * This has the form:
 *
 *	"yymmdd.AAseqno@myname"
 *
 * as in "891113.AA12345@minixug.UUCP"
 */
char *mesgid()
{
  static char idbuf[80];
  struct tm *tm;
  char *sp;
  time_t now;

  /* Get the current date and time. */
  time(&now);
  tm = xltime(&now);

  /* Create a short ID. */
  sp = shortid();

  /* Create the string. */
  sprintf(idbuf, "%02.2d%02.2d%02.2d.%s@%s.%s", 
	tm->tm_year - 1900, tm->tm_mon + 1, tm->tm_mday, sp, 
						myname, mydomain);

  return(idbuf);
}



/*
 * Creates a unique UUCP file name, and returns a pointer
 * to it. The filename is a combination of prefix, grade, system name
 * and a sequential number taken from SPOOLSEQ.
 */
char *genname(prefix, grade, sys)
int prefix, grade;
char *sys;
{
  static char _gen_buf[128];
  off_t seq;

  seq = readseq();
  if (seq == (off_t)-1) seq = (off_t) 1;
  writeseq(seq);

  if (grade == 0) sprintf(_gen_buf, "%c.%.7s%lx", prefix, sys, (long) seq);
    else sprintf(_gen_buf, "%c.%.7s%c%lx", prefix, sys, grade, (long) seq);

  return(_gen_buf);
}


/*
 * Deliver this message to a local user.
 * We do this by calling "LMAIL" (which is actually
 * a link to "Mail"; the Local Mail Agent.
 */
int send_local(user, data)
char *user;
char *data;
{
  struct passwd *pw;
  char tmpbuf[128];

  /* See if destination user name exists on this machine. */
  pw = (struct passwd *) getpwnam(user);
  if (pw == (struct passwd *)NULL) {
	sprintf(tmpbuf, "%s ... unknown user at %s", user, myname);
	errmail(tmpbuf, FALSE);
  }

  /* Set the correct protection mask. */
  chown(data, old_uid, old_gid);

#ifdef WMAILER	/* faster than redirecting! */
  sprintf(tmpbuf, "exec %s -i%s %s", LMAIL, data, user);
#else
  sprintf(tmpbuf, "exec %s <%s %s", LMAIL, data, user);
#endif /* WMAILER */

  return(system(tmpbuf));
}


/*
 * Deliver this message to a remote user.
 * We do this by creating the spoolfiles needed by UUCICO.
 * Then we call that program daemon to do the real work.
 * Of course, we can also do this by calling UUX with the
 * correct parameters...
 */
int send_remote(rmtname, rmtuser, data)
char *rmtname;
char *rmtuser;
char *data;
{
  char tmpbuf[128];
  char Bfile[128], Cfile[128], Dfile[128], Xfile[128];
  FILE *fcfile, *fbfile, *fdfile, *fp;

  if (KnowHost(rmtname) == FALSE) {
	sprintf(tmpbuf, "%s ... unknown host machine", rmtname);
	errmail(tmpbuf, FALSE);
  }

  /* Make the spool files for uucico. */
  strcpy(Cfile, genname('C', 'N', rmtname));	/* call-file, also msgID */
  strcpy(Dfile, genname('D', 0, myname));	/* data file */
  strcpy(Bfile, genname('B', 0, rmtname));	/* local Xfilename */
  strcpy(Xfile, Bfile);				/* the remote X-filename */
  Xfile[0] = 'X';

  /* Copy the temp-file to the UUCP data file (D.???). */
  if ((fdfile = fopen(Dfile, "w")) == (FILE *)NULL) {
	perror("rmail 4");
      	exit(1);
  } else {
	  fp = fopen(data, "r");	/* open temp-file */
	  fcopy(fp, fdfile);
	  fclose(fdfile);
	  fclose(fp);
  }
    
  if ((fbfile = fopen(Bfile, "w")) == (FILE *)NULL) {
	perror("rmail 4");
	exit(1);
  } else {
	  fprintf(fbfile, "U %s %s\nF %s\nI %s\nC rmail %s\n",
    				UUCPUSER, myname, Dfile, Dfile, rmtuser);
    	  fclose(fbfile);
  }

  if ((fcfile = fopen(Cfile, "w")) == (FILE *)NULL) {
	perror("rmail 5");
	exit(1);
  } else {
	  fprintf(fcfile,"S %s %s %s - %s 0666\nS %s %s %s - %s 0666\n",
		Dfile, Dfile, UUCPUSER, Dfile, Bfile, Xfile, UUCPUSER, Bfile);
          fclose(fcfile);
  }

  /* UMAIL is setUID root... UUCP cannot read these files! */
  chown(Bfile, UUCPUID, UUCPGID);
  chown(Cfile, UUCPUID, UUCPGID);
  chown(Dfile, UUCPUID, UUCPGID);
  chown(Xfile, UUCPUID, UUCPGID);

  if (immediate == TRUE) {	/* call uucico now! */
	strcpy(tmpbuf, UUCICO);
	sprintf(tmpbuf, "exec %s -s%s -x1 2>/dev/null &", UUCICO, rmtname);
	system(tmpbuf);
  }

  return(FALSE);
}


/*
 * Perform the mail-transport.
 * We do this by calling the appropriate mailer.
 * If the name of the mailer is "$$" then we can use
 * this program to deliver. This saves a lot of memory.
 */
int sendit(who, host, cmd, opts, data)
char *who;			/* who is the addressee? */
char *host;			/* on which machine? */
char *cmd;			/* what command should we use? */
char *opts;			/* which options? */
char *data;			/* name of data (message) file */
{
  char cmdline[512];
  char tmpbuff[512];
  char addressee[1024];
  BOX *box;

  /* Change to UUCP directory. */
  chdir(SPOOLDIR);

  /* Run our own transport routines if possible. */
  if (!strcmp(cmd, "$$")) {
	if (*host=='\0' && aremote == TRUE) {	/* re-route the address */
		strcpy(addressee, who);		/* save the addressee */
		box = convert(addressee);	/* re-route the address */
		if (box == NILBOX) return;
		if (route(box) == FALSE) return;
		if (box->host == '\0') send_local(mailaddr, data);
		  else send_remote(mailhost, mailaddr, data);
	} else {
		if (*host == '\0') send_local(who, data);
		  else send_remote(host, who, data);
	}
  } else {
	  sprintf(cmdline, "exec %s %s ", cmd, opts);
	  sprintf(tmpbuff, cmdline, data);	/* create commandline */
	  sprintf(cmdline, tmpbuff, who);	/* add user address */
	  system(cmdline);			/* execute mailer */
  }
}


/*
 * Send mail to system manager ans sender upon errors.
 * The message is contained in a file called 'dfile'.
 */
void errmail(str, mgronly)
char *str;
int mgronly;
{
  struct passwd *pw;
  FILE *fp, *tp;
  time_t now;
  char fname[32];
  char tmp[128];

  strcpy(fname, "/tmp/umeXXXXXX");
  mktemp(fname);

  tp = fopen(fname, "w");
  fp = fopen(dfile, "r");

  time(&now);

  /* Create header of the report-message. */
  fprintf(tp, "From %s %s\n", ERRUSER, xtime(&now));	
  if (mailsender != (char *)NULL)
	fprintf(tp, "To: %s\n", mailsender);
  fprintf(tp, "Subject: Returned mail\n\n");

  /* Create an error transcript. */
  fprintf(tp, "   ---- Transcript of session follows ----\n\n");
  fprintf(tp, "%s\n", str);
  fprintf(tp, "\n   ---- Unsent message follows ----\n");

  /* Copy the message. */
  while (mfgets(tmp, sizeof(tmp), fp) != (char *)NULL)
				fprintf(tp, "> %s\n", tmp);

  fclose(tp);	/* flush and close message file */
  fclose(fp);	/* flush and close orig. file */

  /* Return mail to system manager (and sender if mgronly == FALSE). */
  if (mgronly == FALSE) sendit(mailsender, "", RMAIL, " <%s", fname);

  /* Send mail to UUCP administrator, if existing. */
  pw = getpwnam(ERRUSER);
  if (pw != (struct passwd *)NULL) {
	sendit(ERRUSER, "", "$$", "", fname);
  }

  unlink(fname);	/* remove data files */
  unlink(dfile);

  exit(1);		/* and exit! */
}
