/*
 * UMAIL -	MINIX Remote Domain-addressing Mail Router
 *
 *		This version of RMAIL handles message-headers in a much
 *		more "standard" way. It can handle bang-addresses, plus
 *		the new-style Internet addressing (user@host.domain).
 *		It is called by programs as "Mail" and "Uuxqt".
 *
 * Usage:	umail [-c <config>] [-d] [-i <infile>] [-n] [-v] <user> ...
 *
 * Author:	F. van Kempen, waltje@minixug.hobby.nl
 *
 * Revisions:
 *		11/07/89 FvK	Edited a little for the new MSS.
 *		12/10/89 FvK	Added 'old_uid' and 'old_gid' for protection-
 *				adjustment in send_local().
 *		12/16/89 FvK	Fixed security-problem with "-i file" option.
 *				Added "allowed()" function.
 *				Added "-v" (verbose) option.
 *				Cleanup.
 *		12/30/89 FvK	Adapted for POSIX (MINIX 1.5)
 *		02/17/90 FvK	Cleaned for release.
 *		04/10/90 FvK	Added some configuration variables, and
 *				fixed some "Header" bugs.
 *		05/20/90 FvK	Added the "mydept" variable.
 *		06/12/90 FvK	Changed "errno.h" include directory.
 *				Added initialization of "mydept".
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "umail.h"


char *Version = VERSION;	/* UMAIL version ID */
int immediate, debug = FALSE;	/* commandline option flags */
int restrict = FALSE;		/* restricted (UUCP) use only */
int old_uid, old_gid;		/* original UID and GID of caller */
int aremote;			/* is the addressee REMOTE or LOCAL ? */
char dfile[128], infile[128];	/* names of message temp-files */
char errmsg[512];		/* global error message */
char mailsender[1024];		/* who sent the message? */
char mailaddr[1024];		/* final routed address to use. */
char mailhost[64];		/* which host to send to */
char mailcmd[512];		/* command to use to send the mail */
char mailopts[64];		/* which options the mailer uses */
NAME *namelist = NILNAME;	/* list of my network names */
VAR *varlist = NILVAR;		/* list of configuration variables */
HOST *hostlist = NILHOST;	/* list of reacheable host names */
ROUTE *routemap = NILROUTE;	/* list of domain routes */

/* Configuration settings. */
char *myname = (char *)NULL;	/* my UUCP site name */
char *mydomain = (char *)NULL;	/* my UUCP domain name */
char *myorg = (char *)NULL;	/* Name of my organization */
char *mydept = (char *)NULL;	/* Name of my department */
int oldmailer = FALSE;		/* does our mailer use old From-lines? */
int usepath = FALSE;		/* add "Path:" field to local mail */
int sendfrom = FALSE;		/* add "From user date [remote from] line */
int escape = FALSE;		/* can we offer a routing-escape? */


extern int getopt(), optind, opterr;	/* from standard library */
extern char *optarg, *fgets();
extern off_t ftell();
extern int errno;


/* 
 * Convert strings S to upper case. 
 */
char *strupr(s)
char *s;
{
  register char *sp;

  sp = s;
  while (*sp) {
	if (islower(*sp)) *sp = *sp - 'a' + 'A';
  	sp++;
  }
  return(s);
}


/* 
 * Convert strings S to lower case. 
 */
char *strlwr(s)
char *s;
{
  register char *sp;

  sp = s;
  while (*sp) {
	if (isupper(*sp)) *sp = *sp - 'A' + 'a';
  	sp++;
  }
  return(s);
}


/*
 * Add 'NAME' to the list of our names.
 */
void add_name(name)
char *name;
{
  register NAME *np, *xp;

  np = (NAME *)malloc(sizeof(NAME));       /* allocate new variable */
  if (namelist == NILNAME) {                /* first variable */
    	namelist = np;
  } else {
          xp = namelist;
          while (xp->next != NILNAME) xp = xp->next;
          xp->next = np;
  }

  np->next = NILNAME;
  np->name = (char *)malloc(strlen(name) + 2);

  strcpy(np->name, name);
}


/*
 * Add host 'NAME' to the list of hosts.
 */
void add_host(name, smart, cmd, opts)
char *name;
int smart;
char *cmd;
char *opts;
{
  register HOST *hp, *xp;

  hp = (HOST *)malloc(sizeof(HOST));       /* allocate new variable */
  if (hostlist == NILHOST) {                /* first variable */
    	hostlist = hp;
  } else {
          xp = hostlist;
          while (xp->next != NILHOST) xp = xp->next;
          xp->next = hp;
  }

  hp->next = NILHOST;
  hp->name = (char *)malloc(strlen(name) + 2);
  hp->command = (char *)malloc(strlen(cmd) + 2);
  hp->opts = (char *)malloc(strlen(opts) + 2);

  strcpy(hp->name, name);
  strcpy(hp->command, cmd);
  strcpy(hp->opts, opts);
  hp->smart = smart;
}


/*
 * Add route 'DOMAIN' to the routing table.
 */
void add_route(domain, host, route)
char *domain;
char *host;
char *route;
{
  register ROUTE *rp, *xp;

  rp = (ROUTE *)malloc(sizeof(ROUTE));	/* allocate new route */
  if (routemap == NILROUTE) {           /* first route */
	routemap = rp;
  } else {
          xp = routemap;
          while (xp->next != NILROUTE) xp = xp->next;
          xp->next = rp;
  }

  rp->next = NILROUTE;
  rp->domain = (char *)malloc(strlen(domain) + 2);
  rp->host = (char *)malloc(strlen(host) + 2);
  rp->route = (char *)malloc(strlen(route) + 2);

  strcpy(rp->domain, domain);
  strcpy(rp->host, host);
  strcpy(rp->route, route);
}


/*
 * Add variable 'NAME' to the variable list.
 */
void add_var(name, val)
char *name;
char *val;
{
  register VAR *vp, *xp;

  strupr(name);

  vp = (VAR *)malloc(sizeof(VAR));	/* allocate new variable */
  if (varlist == NILVAR) {              /* first variable */
    	varlist = vp;
  } else {
          xp = varlist;
          while (xp->next != NILVAR) xp = xp->next;
          xp->next = vp;
  }

  vp->next = NILVAR;
  vp->name = (char *)malloc(strlen(name) + 2);
  vp->value = (char *)malloc(strlen(val) + 2);

  strcpy(vp->name, name);
  strcpy(vp->value, val);
}


/*
 * Get a variable from the variable list.
 * Return NULL if not defined.
 */
char *lookup(what)
char *what;
{
  register VAR *vp;

  vp = varlist;
  while (vp != NILVAR) {
    	if (!strcmp(vp->name, what)) return(vp->value);
    	vp = vp->next;
  }
  return((char *)NULL);
}


/*
 * Return TRUE or FALSE value, depending on
 * the value of the given variable.
 */
int boolean(ascii)
char *ascii;
{
  strupr(ascii);
  if (ascii==(char *)NULL || !strcmp(ascii, "FALSE")) return(FALSE);
    else if (!strcmp(ascii, "TRUE")) return(TRUE);
           else fprintf(stderr, "Bad value of boolean: \"%s\"\n", ascii);
  return(FALSE);
}


/*
 * Lookup a host in our hosts-table.
 */
HOST *gethost(host)
char *host;
{
  register HOST *hp;

  hp = hostlist;
  while (hp != NILHOST) {
	if (!strcmp(hp->name, host)) return(hp);
	hp = hp->next;
  }
  return(NILHOST);
}


/*
 * Lookup a domain in our domain-table.
 */
ROUTE *getdomain(domain)
char *domain;
{
  register ROUTE *rp;

  rp = routemap;
  while (rp != NILROUTE) {
	if (!strcmp(rp->domain, domain)) return(rp);
	rp = rp->next;
  }
  return(NILROUTE);
}


/*
 * mfgets (modified fgets)
 * Same as fgets() only this version deletes '\n'
 */
char *mfgets(s, n, iop)
char *s;
register int n;
register FILE *iop;
{
  register int c;
  register char *cs;

  cs = s;
  while (--n > 0 && (c = getc(iop)) != EOF) {
	if (c == '\n') {
		*cs = '\0';
		break;
        } else *cs++ = c;
  }
  return((c == EOF && cs == s) ? (char *)NULL : s);
}


/*
 * Return the full UUCP ID of the calling user
 */
char *full_id(user)
char *user;
{
  static char fullbuf[48];

  sprintf(fullbuf, "%s@%s.%s", user, myname, mydomain);
  return(fullbuf);
}


/*
 * Return the Real Name of the calling user
 */
char *realname(who)
char *who;
{
  struct passwd *pw;

  if ((pw = getpwnam(who)) != (struct passwd *)NULL) return(pw->pw_gecos);
    else return("unknown flag");
}


/*
 * Make a decent DATE/TIME string.
 * Note, that there are TWO possible date formats:
 *
 *	Sat, 12 Oct 89 20:29:00\0
 * and
 *	Sat 12 Oct 20:29:00 1989\0
 *
 * Most Internet mailers use this first form, so we try
 * to this also. We use the function xtime() for the work...
 */
char *maketime(salt)
time_t *salt;
{
  static char datetime[48];		/* date and time in MET format */
  char *sp;

  sp = lookup("TIMEZONE");		/* get Time Zone from config file */
  if (sp == (char *)NULL) sp = "";	/* must have SOME pointer! */

  sprintf(datetime, "%s %s", xtime(salt), sp);

  return(datetime);
}


/*
 * Check if we may perform operation 'mode' on
 * file 'name'. System Security!
 * If the error is 'ENOENT', then test the parent
 * directory for the desired access.
 */
int allowed(name, mode)
char *name;			/* name of file to be checked */
unsigned short mode;		/* mode to check (R=4, W=2, X=1) */
{
  char abuf[1024];		/* temp. buf for filename */
  struct stat stb;
  char *p;

  /* Is this 'The Master' calling? */
  if (old_uid == 0 && old_gid == 0) return(TRUE);

  if (stat(name, &stb) < 0) {
	if (errno == ENOENT) {			/* file does not exist */
		strcpy(abuf, name);		/* so check its parent dir */
		p = strrchr(abuf, '/');	
		if (p == (char *)NULL)		/* plain filename, */
			getcwd(abuf, 1023);	/* get current dir */
		  else *p = '\0';		/* strip 'file' part */
		if (stat(abuf, &stb) < 0) return(FALSE);	/* error? */
	} else return(FALSE);			/* it exists, other error! */
  }

  /* We now have the status of the file or its parent dir. */
  if (stb.st_uid == old_uid) {			/* we are owner! */
	if ((stb.st_mode >> 6) & mode)
				 return(TRUE);	/* OK, we may do it. */
  	  else return(FALSE);			/* Alas... */
  } else if (stb.st_uid == old_gid) {		/* are we the same group? */
	if ((stb.st_mode >>3) & mode)
				 return(TRUE);	/* OK, we may do it. */
  	  else return(FALSE);			/* Alas... */
  } else if (stb.st_mode & mode)		/* we are 'others' */
				 return(TRUE);	/* OK, we may do it. */
  return(FALSE);				/* Alas... */
}


/*
 * Copy a file from 'inf' to 'outf'.
 */
void fcopy(inf, outf)
register FILE *inf, *outf;
{
  char cpbuff[1024];

  while (TRUE) {
	if (fgets(cpbuff, sizeof(cpbuff), inf) == (char *)NULL) break;
	fwrite(cpbuff, sizeof(char), strlen(cpbuff), outf);
  }
}


/*
 * Load the configuration parameters into their variables.
 */
static void setup(cfg)
char *cfg;
{
  if (scanner(cfg) != 0) {		/* read the configuration file */
          perror(cfg);
          exit(1);
  }

  myname = lookup("SYSTEM");
  if (myname == (char *)NULL) {
	fprintf(stderr, "Missing SYSTEM definition\n");
	exit(-1);
  }
  mydomain = lookup("DOMAIN");
  if (mydomain == (char *)NULL) {
	fprintf(stderr, "Missing DOMAIN definition\n");
	exit(-1);
  }
  myorg = lookup("ORGANIZATION");
  mydept = lookup("DEPARTMENT");
  oldmailer = boolean(lookup("OLDMAILER"));
  escape = boolean(lookup("ESCAPE"));
  sendfrom = boolean(lookup("SENDFROM"));
  usepath = boolean(lookup("USEPATH"));
}


/*
 * Something went wrong.
 * Tell the caller how we should be called!
 */
static void usage()
{
  fprintf(stderr,
	"Usage: umail [-c <config>] [-d] [-i <infile>] [-n] [-v] <users>\n");
}


int main(argc, argv)
int argc;
char *argv[];
{
  FILE *fdfile, *infp;			/* message file pointers */
  BOX *box;				/* conversion/routing addresses */
  char *cfgfile = CONFIG;		/* config file; to save space */
  register int st;			/* error status, to exit() */
  
  if (argv[0][0] == 'r') {		/* 'rmail' link? */
 	restrict = TRUE;		/* yes, restrict usage */
  }

  old_uid = getuid();			/* get current ID's */
  old_gid = getgid();

  opterr = 0;
  while ((st = getopt(argc, argv, "c:di:nv")) != EOF) switch(st) {
	case 'c':	/* use non-standard CONFIG file */
		cfgfile = optarg;
		break;

	case 'd':	/* turn on DEBUG mode */
	case 'v':	/* Verbose mode */
		debug = TRUE;
		break;
	case 'i':	/* use non-stdin input file */
		strncpy(infile, optarg, 128 - 1);
		if (allowed(infile, 04) == FALSE) {
			fprintf(stderr, "%s: cannot read file %s\n", 
							argv[0], infile);
			exit(1);
		}
		break;
	case 'n':	/* call UUCICO after processing */
		immediate = TRUE;
		break;
	default:
		usage();
		exit(1);
  }

  if (optind >= argc) {			/* we need another parameter! */
    	usage();			/* (the addressee ) */
	exit(-1);
  }

  umask(0117);				/* change umask to -rw-rw---- */

  setup(cfgfile);			/* read CONFIG and setup */

  strcpy(dfile, "/tmp/umXXXXXX");	/* create temp. message file */
  mktemp(dfile);
  if ((fdfile = fopen(dfile, "w")) == (FILE *)NULL) {
	perror("rmail 1");
	exit(1);
  }

  box = convert(argv[optind]);		/* convert Internet address to UUCP */
  if (box == NILBOX) st = FALSE;
    else st = route(box);		/* run it through routing tables */

  if (infile[0] != '\0') {		/* open input file if -i option */
	infp = fopen(infile, "r");
	if (infp == (FILE *)NULL) {
		perror(infile);
		exit(1);
	}
  } else infp = stdin;			/* otherwise use stdin! */

  header(infp, fdfile);			/* analyze message header */

  fcopy(infp, fdfile);			/* copy message to the temp. file */

  fclose(fdfile);
  if (infp != stdin) fclose(infp);

  if (st == FALSE) {			/* conversion/routing went wrong? */ 
	errmail(errmsg, FALSE);		/* yes; return the message! */
	st = -1;
  } else st = sendit(mailaddr, mailhost, mailcmd, mailopts, dfile);

  unlink(dfile);			/* remote data file */
  exit(st);				/* and exit! */
}
