/*
 * UUCP	-	Perform  Unix  to Unix  File CoPy Requests.
 *		This  program handles  file  transfers from one system to
 *		another, or local transfers only. It uses UUCP for the
 *		actual file transfer, except when it has to do a local
 *		file copy, in which case it calls "/bin/cp" to do the work.
 *
 *		This program needs work in the handling of some options.
 *
 * Usage:	uucp [-Ccdfjr] [-ggrade] [-muser] [-nuser] 
 *		 			    [-xlevel] <from> <to>
 *
 * Version:	1.6	02/17/90
 *
 * Author:	Fred van Kempen, MINIX User Group Holland
 *			waltje@minixug.hobbynet.nl
 */
#include <sys/types.h>
#include <ctype.h>
#include <pwd.h>
#include <string.h>
#include <stdio.h>
#include "uucp.h"


static char *Version = "@(#) uucp 1.6 (02/17/90)";
static char *seqname = SPOOLSEQ;
static char *lseqname = LSPOOLSEQ;


char cfile[128], dfile[128];	/* filenames for spool-files */
FILE *fcfile, *fdfile;		/* the "C." and "D." spoolfiles */

int opt_c = 1;			/* Link source file, do not copy */
int opt_d = 1;			/* Make all necessary directories */
int opt_f = 0;			/* Do not make intermediate directories */
int opt_j = 0;			/* Output UUCP job number on stderr */
int opt_m = 0;			/* send mail to local user */
int opt_n = 0;			/* send mail to remote user */
int opt_r = 0;			/* Do not call UUCICO, queue the job */

int grade = 'N';		/* Job Grade ID */
int debug = 1;			/* Job Debug-Level */
int fromlocal, tolocal;

char luser[16];			/* local user name */
char mysite[SITENAMELEN];	/* this host's uucpname */
char srcsite[SITENAMELEN];	/* source-file site */
char dstsite[SITENAMELEN];	/* destination-file site */

char srcpath[128];		/* pathname on source-site */
char dstpath[128];		/* pathname on destination-site */

char mailuser[16];
char telluser[16];
char workdir[128];		/* current working directory */


extern int getopt() , optind, opterr;
extern char *optarg;


/*
 * Uucpname() returns a pointer to the local host's UUCP nodename.
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually "/etc/uucpname".
 */
char *uucpname()
{
  FILE *uufile;
  static char uuname[SITENAMELEN];

  if ((uufile = fopen(NODENAME, "r")) == (FILE *)NULL) return("unknown");
  fgets(uuname, sizeof uuname, uufile);
  uuname[strlen(uuname) - 1] = '\0';	/* remove '\n' */
  fclose(uufile);

  return(uuname);
}


/*
 * Knowhost() checks if we know a host with the given name.
 * If so, 1 is returned, else 0.
 */
int knowhost(host)
char *host;
{
  FILE *sysfile;
  char line[256];
  char *p, *q;

  if ((sysfile = fopen(LSYS, "r")) == (FILE *)NULL) return(0);
  while(fgets(line, sizeof line, sysfile) != NULL) {
	for (p = host, q = line; !isspace(*q); p++, q++)
		if (*p != *q) break;

		if ('\0' == *p && isspace(*q)) {
			fclose(sysfile);
			return(1);
		}
	}
  fclose(sysfile);
  return(0);
}


char *whoami()
{
  struct passwd *pw;

  if ((pw = getpwuid(getuid())) != (struct passwd *)NULL) return(pw->pw_name);
    else return("nobody");
}


/*
 * Read the SPOOLSEQ file and return that ID.
 */
long readseq(void)
{
  char seqline[10];			/* read buffer */
  long seq;				/* the decoded number */
  FILE *spoolseq;			/* the file pointer */
  int i;

  if (access(seqname, 0) != 0) return(-1L);

  while(link(seqname, lseqname) != 0) {
  	sleep(5);
  	if (++i > 5) return(-1L);
  }

  spoolseq = fopen(seqname, "r");
  fgets(seqline, sizeof(seqline), spoolseq);
  fclose(spoolseq);
  unlink(lseqname);

  seq = atol(seqline);		/* read the current ID */
  seq++;			/* and use next */

  return(seq);
}


/*
 * Update the SPOOLSEQ file.
 */
void writeseq(seq)
long seq;
{
  FILE *spoolseq;			/* the file pointer */
  int i;

  if (access(seqname, 0) != 0) {	/* create file, owned by UUCP */
	close(creat(seqname, 0600));
	chown(seqname, UUCPUID, UUCPGID);
  }

  while(link(seqname, lseqname) != 0) {	/* lock it */
  	sleep(5);
  	if (++i > 5) return(-1L);
  }

  /* Write new contents. */
  if ((spoolseq = fopen(seqname, "w")) != (FILE *)NULL) {
	fprintf(spoolseq, "%ld\n", seq);
	fclose(spoolseq);
  }

  unlink(lseqname);			/* remove lock */
}


/*
 * Creates a unique UUCP file name, and returns a pointer
 * to it. The filename is a combination of prefix, grade, system name
 * and a sequential number taken from SPOOLSEQ.
 */
char *genname(prefix, grade, sys)
int prefix, grade;
char *sys;
{
  static char _gen_buf[128];
  long seq;

  seq = readseq();
  if (seq == -1L) seq = 1L;
  writeseq(seq);

  if (grade == 0) sprintf(_gen_buf, "%c.%.7s%lx", prefix, sys, seq);
    else sprintf(_gen_buf, "%c.%.7s%c%lx", prefix, sys, grade, seq);

  return(_gen_buf);
}


/*
 * Remove any spool-files and exit.
 */
void bye(status)
int status;
{
  fclose(fcfile);
  fclose(fdfile);
  unlink(cfile);
  unlink(dfile);
  exit(status);
}


char *filesite(name)
char *name;
{
  static char site[SITENAMELEN];
  register char *p, *q;

  if ((p = index(name, '!')) == (char *)NULL) return((char *)NULL);

  q = site;
  while(name < p) *q++ = *name++;
  *q = '\0';

  return(site);
}


char *filepath(name)
char *name;
{
  register char *p;

  if ((p = rindex(name, '!')) == (char *)NULL) return((char *)NULL);

  return(p + 1);
}


/*
 * Create an execute-command file for a remote system.
 */
int do_forward(ssite, spath, dsite, dpath)
char *ssite, *spath;
char *dsite, *dpath;
{
  char tbuff[128];

  sprintf(tbuff, "uux '-r %s!uucp %s %s!%s'",
				ssite, spath, dsite, dpath);

  return(system(tbuff));
}


/*
 * Copy the source-file to the spool-directory.
 * This routine should try to LINK the file first!
 */
int copy_spool(from, to)
char *from, *to;
{
  char namebuf[128];
  FILE *inf, *outf;
  register int ch;

  /* First, make "from" an absolute pathname. */
  if (*from != '/') {
	strcpy(namebuf, workdir);
	if (strlen(namebuf) > 1) strcat(namebuf, "/");
	strcat(namebuf, from);
  } else strcpy(namebuf, from);

  if ((inf = fopen(namebuf, "r")) == (FILE *)NULL) {
	fprintf(stderr, "uucp: cannot open %s\n", namebuf);
	bye(1);
  }

  if ((outf = fopen(to, "w")) == (FILE *)NULL) {
	fprintf(stderr, "uucp: cannot create %s\n", to);
	bye(1);
  }

  while ((ch = fgetc(inf)) != EOF) fputc(ch, outf);

  fclose(outf);
  fclose(inf);
  return(0);
}


/*
 * Create the various spool-files.
 */
void spool_it()
{
  getcwd(workdir, 128);			/* where are we? */
  chdir(SPOOLDIR);			/* change to UUCP spool directory */

  /* generate the right command file name */
  if (fromlocal == 0) strcpy(cfile, genname('C', grade, srcsite));
    else if (tolocal == 0) strcpy(cfile, genname('C', grade, dstsite));

  strcpy(dfile, genname('D', 0, mysite));

  if ((fcfile = fopen(cfile, "w")) == (FILE *)NULL) {
	fprintf(stderr, "uucp: can't open ");
	perror(cfile);
	bye(1);
  }
 
  /*
   * Create the work-file.
   * This is something like:
   * 
   * S <full_src> <full_rmt_dst> <user> - <datafilename>
   *
   * for sending a file, or
   *
   * R <full_rmt_src> <full_dst> <user> - <options -m -d>
   *
   * for receiving a file.
   */

  if (fromlocal==1 && tolocal==0) {	    /* send file to remote */
	fprintf(fcfile, "S %s %s %s - %s 0666\n", 
				srcpath, dstpath, luser, dfile);
	copy_spool(srcpath, dfile);
  } else {  /* receive file from remote */
	  fprintf(fcfile, "R %s %s %s - \n",
				srcpath, dstpath, luser);
    }

  fclose(fcfile);
}


void usage()
{
  fprintf(stderr, "Usage:\nuucp [-Ccdfjr] [-ggrade] [-muser] [-nuser] ");
  fprintf(stderr, "[-xlevel] <from> <to>\n");
}


int main(argc, argv)
int argc; char *argv[];
{
  register int c;
  register char *p;

  umask(0137);				/* set mask to -rw-r----- */
 
  strcpy(mysite, uucpname());
  if (strlen(mysite) == 0) {
	fprintf(stderr, "uucp: can't get my own uucpname\n");
        bye(1);
  }
  strcpy(luser, whoami());

  opterr = 0;
  while((c = getopt(argc, argv, "cdCfg:jm:n:rs:x:")) != EOF) switch(c) {
	case 'c':
		opt_c = 1;
		break;
	case 'd': 
		opt_d = 1;
		break;
	case 'C':
		opt_d = 0;
	        break;
	case 'f':
		opt_f = 1;
		break;
	case 'g':
		if (isalnum(optarg[0]) && '\0' == optarg[1])
						grade = (int) optarg[0];
	    	  else fprintf(stderr, "uucp: %s: illegal grade\n", optarg);
		break;
	case 'j':
		opt_j = 1;
		break;
	case 'm':
		strncpy(mailuser, optarg, 15);
	  	opt_m = 1;
		break;
	case 'n':
		strncpy(telluser, optarg, 15);
	  	opt_n = 1;
		break;
	case 'r':
		opt_r = 1;
		break;
	case 's':
		break;		/* report status to a file */
	case 'x':
		debug = atoi(optarg);
		break;
	default:
		usage();
		bye(1);
  }

  if (optind >= argc) {	/* we must have *something* to do! */
	usage();
	bye(1);
  }

  if ((p = filesite(argv[optind])) == (char *)NULL) {
	strcpy(srcsite, mysite);
	strcpy(srcpath, argv[optind]);
  } else {
	  strcpy(srcsite, p);
     	  if ((p = filepath(argv[optind])) == (char *)NULL) {
		fprintf(stderr, "uucp: illegal source-pathname\n");
               	bye(1);
          }
          strcpy(srcpath, p);
    }
  ++optind;

  if ((p = filesite(argv[optind])) == (char *)NULL) {
	strcpy(dstsite, mysite);
	strcpy(dstpath, argv[optind]);
  } else {
	  strcpy(dstsite, p);
     	  if ((p = filepath(argv[optind])) == (char *)NULL) {
		fprintf(stderr, "uucp: illegal destination-pathname\n");
               	bye(1);
          }
          strcpy(dstpath, p);
    }

  /* Determine if we can suffice with 'cp' */   
  p = "uucp: site %s unknown\n";
  fromlocal = 0; 
  tolocal = 0;
  if (!strcmp(srcsite, mysite)) fromlocal = 1;
    else if (!knowhost(srcsite)) {
 		fprintf(stderr, p, srcsite);
		bye(1);
 	 }

  if (!strcmp(dstsite, mysite)) tolocal = 1;
    else if (!knowhost(dstsite)) {
		fprintf(stderr, p, dstsite);
		bye(1);
	 }
  if (fromlocal==1 && tolocal==1)
    	execl("/bin/cp", "cp", srcpath, dstpath, (char *)NULL);
    else if (fromlocal==1 || tolocal==1) spool_it();	/* do the real work */
	   else do_forward(srcsite, srcpath, dstsite, dstpath);

  if (opt_r==1 || (fromlocal==1 && tolocal==1)) exit(0);
    else execl(UUCICO, UUCICO, "-r1", (char *)NULL);
}
