/*
 * "UUCICO"-	A UUCP File Transport Program.
 * 		This program implements a uucico type file transfer protocol.
 *		It is the revised version  of DCP, which tries to act as the
 *		"real" UUCICO as much as possible on a native UNIX-like system,
 *		as opposed to the original DCP.
 *		The first port to MINIX was done by Peter S. Housel, and the
 *		final edit was done by F. van Kempen. The name of the program
 *		has been changed into UUCICO for practical reasons only.
 *
 * Usage:	uucico [-c] [-xNN] [-rNN] [-sHOST]
 *
 *		-xNN	set debug-level to NN
 *		-rNN	set role number to MASTER (1) or SLAVE (0 == default)
 *		-sHOST 	call system 'HOST' only (for polling etc.)
 *		-c	connect to stdin/stdout, do not dial.
 *
 *
 *		***** Copyright Richard H. Lamb 1985,1986,1987 *****
 *
 * Note:	As a result of a bug in the MINIX C Compiler, some global
 *		variables must be declared as 'static'. Otherwise, some of
 *		them will conflict with variables in the runtime library. -FvK
 *
 * Compile:	cc -i -o uucico uucico.c -lmodemcap -lndir
 *
 * Version:	3.4	05/12/90
 *
 * History:	-Original by R.H. Lamb
 *		-Revised Stuart Lynne May/87
 *		-Modified heavily for MINIX by Peter S. Housel
 *		-Other (also heavy) modifications by Fred van Kempen:
 * 			-Fixed initline() 11/19/89
 *      		-Fixed rfile() 11/19/89
 *			-Added "Sending..." in sfile() 11/19/89
 *			-Edited source for MSS
 *		-Added new 'g'-driver. Fvk 11/30/89
 *		-Added 'setuid()' code in main(). FvK 12/04/89
 *		-Fixed absolute-pathname problem in rfile(). FvK 12/12/89
 *		-Fixed 'hangup' bug. FvK 12/17/89
 *		-Added Housel's fixes d.d. 89/12/20
 *		-Added 'hangup' delay. FvK 12/23/89
 *		-Fixed up for POSIX (MINIX 1.5). FvK 12/30/89
 *		-Fixed unlink() problems. FvK 01/20/90
 *		-Cleaned up for release. Fvk 02/17/90
 *		-More casts and "master bug" fixed. FvK 02/27/90
 *		-Fixed the "garble" bug in rfile(). FvK 05/12/90
 */
#include <sys/types.h>
#include <ctype.h>
#include <dial.h>
#ifdef _BSD
#	include <sys/dir.h>
#	include <strings.h>
#else
#	include <dirent.h>
#	define direct dirent
#	include <string.h>
#endif _BSD
#include <pwd.h>
#include <sgtty.h>
#include <signal.h>
#include <time.h>
#include <stdio.h>
#include "uucp.h"


#define LOGFILE	"/usr/lib/uucp/Log/uucico.log"
#define MSGTIME             20
#define MAXPACK            256
#define DLE		'\020'	/* ASCII 'DLE' character */
#define	SLAVE		    0
#define	MASTER		    1
#define TRUE    	 (-1)
#define FALSE   	    0
#define SAME		    0
#define FAILED		   -1
#define OK		    0

/* log messages */
#define M_NONE		    0	/* no output at all */
#define M_ERROR		    1	/* log regardless of debuglevel */
#define M_CALL		    2	/* the call and pre-protocol negotiation */
#define M_CONVERSE	    3	/* conversation level */
#define M_TRANSFER	    4	/* file transfer */
#define M_SPOOL		    5	/* spool files */
#define M_CALLMSG	    6	/* messages sent out during call/pre-proto */
#define M_HIGHPROTO	    7	/* high-level protocol */
#define M_MEDPROTO	    8	/* med-level protocol */
#define M_LOWPROTO	    9	/* low-level protocol (framing and such) */
#define M_DATA		   10	/* actual packet data */
#define M_INFO		   11	/* reading config files */
#define PROTOS            "g"
#define MAXLOGTRY           3
#define EOTMSG "\004\r\004\r"


typedef short int (*procref) ();

typedef struct
{
  char type;
  procref a;
  procref b;
  procref c;
  procref d;
} Proto;

/* The various protocols available. Add here for others. */
extern procref getpkt, sendpkt, openpk, closepk;
extern short int ggetpkt(), gsendpkt(), gopenpk(), gclosepk();


static char *Version = "@(#) uucico 3.4 (05/12/90)";


char cfile[80];			/* work file name */
char fromfile[132];		/* source of copy */
char tofile[132];		/* destiation of copy */
char Rmtname[20];
char rmtname[20];
char sysline[BUFSIZ];
char s_systems[64];
char s_logfile[64];
char s_syslog[64];
char proto[5];			/* list of protocols */
char *flds[24];
char *cctime;
FILE *logfile;			/* system log file */
FILE *fw;			/* cfile pointer */
FILE *fp;			/* current disk file ptr */
FILE *fsys;
time_t now;			/* current time */
short int pktsize;		/* packet size for this protocol */
short int remote;		/* -1 means we're remote */
short int findwork;
short int msgtime;		/* timeout setting */
short int size;			/* nbytes in buff */
short int swritefd;		/* serial write file descriptor */
short int sreadfd;		/* serial read file descriptor */
short int timedout;
short int kflds;
short int debuglevel;		/* debugging flag */
char state;			/* present state */
char *nodename;			/* UUCP node name */
char sysline[BUFSIZ];
char device[32];
char speed[10];
CALL call;			/* dial(3) structure */
unsigned char rpacket[MAXPACK];
unsigned char spacket[MAXPACK];
procref getpkt, sendpkt, openpk, closepk;


/* The various protocols available. Add here for others. */
extern short int ggetpkt(), gsendpkt(), gopenpk(), gclosepk();

Proto Protolst[] = {
  'g', ggetpkt, gsendpkt, gopenpk, gclosepk,
  '0',
};


extern char *_merr_list[];
extern char *sys_errlist[];
extern short int merrno;


extern int getopt(), optind, opterr;
extern char *optarg;
extern int errno;


/* Forward declarations. */
short int ontime();
short int sigint();
short int swrite();
short int sread();
short int sread2();
short int printmsg();
short int getsystem();
short int checkname();
short int checktime();
short int wmsg();
short int rmsg();
short int prefix();
short int notin();
short int expectstr();
short int writestr();
short int sendexpect();
short int initline();
short int dcpdial();
short int dcpundial();
short int getargs();
short int sdata();
short int sbreak();
short int seof();
short int sfile();
short int sinit();
short int getfile();
short int schkdir();
short int endp();
short int rdata();
short int rfile();
short int rinit();
short int getmsg();
short int sendmsg();
short int callup();
short int scandir();
short int setproto();
short int sysend();
short int startup();
short int recvf();
short int sendf();
short int master();
short int slave();
short int gclosepk();
short int gopenpk();
short int ggetpkt();
short int gsendpkt();
short int gmachine();
short int grpack();
short int gspack();
short int ilog2();
unsigned short int checksum();
char *uucpname();
char *visib();


short int ontime()
{
  signal(SIGALRM, ontime);
  timedout = 1;
  return(0);
}


short int sigint()
{
  fclose(logfile);
  if (sreadfd > 2) dcpundial();
  exit(-1);
}


short int swrite(data, num)
char *data;
short int num;
{
  return(write(swritefd, data, num));
}


short int sread(data, num, timeout)
char *data;
short int num, timeout;
{
  short int ret;

  if (timeout) alarm(timeout);
    else alarm(2);
  signal(SIGALRM, ontime);

  ret = read(sreadfd, data, num);
  alarm(0);
  if (ret < 0) return(0);
  return(ret);
}


short int sread2(data, num)
char *data;
short int num;
{
  return(read(sreadfd, data, num));
}


void cant(file)
char *file;
{
  fprintf(stderr, "uucico: cannot open: \"%s\"\n", file);
  exit(-1);
}


/*
 * printmsg(level, format, ...) prints an error or debugging message
 * into the system error log file.  If not remote, also print to standard
 * error.  All messages at levels less than or equal to the current
 * debuglevel are printed.
 */
/* VARARGS1 */
short int printmsg(level, fmt, a1, a2, a3, a4, a5)
short int level;
char *fmt;
char *a1, *a2, *a3, *a4, *a5;
{
  char msg[512];

  if (level <= debuglevel) {
	sprintf(msg, fmt, a1, a2, a3, a4, a5);
	if (remote == MASTER && debuglevel > 0) fprintf(stderr, "%s\n", msg);
	fprintf(logfile, "%s\n", msg);
  }
}


/*
 * getsystem
 * Process an "L.sys" file entry
 */
short int getsystem()
{
  short int i;

  if (fgets(sysline, BUFSIZ, fsys) == (char *)NULL) return('A');
  printmsg(M_INFO, "%s", sysline);

  kflds = getargs(sysline, flds);
  strcpy(rmtname, flds[FLD_REMOTE]);
  cctime = flds[FLD_CCTIME];
  strcpy(device, flds[FLD_DEVICE]);
  strcpy(speed, flds[FLD_SPEED]);
 
  if (debuglevel >= M_INFO)
	for(i = FLD_EXPECT; i < kflds; i += 2)
     		fprintf(stderr, "expect[%02d]:\t%s\nsend  [%02d]:\t%s\n",
	             i, flds[i], i + 1, flds[i + 1]);

  printmsg(M_INFO, "rmt=%s ctm=%s", rmtname, flds[FLD_CCTIME]);
  printmsg(M_INFO, "dev=%s spd=%s tel=%s ", device, speed, flds[FLD_PHONE]);

  fw = (FILE *)NULL;
  if (checktime(cctime) &&
    ((strcmp(Rmtname, "all") == SAME) ||
     (strcmp(Rmtname, "any") == SAME)) ||
     (strcmp(Rmtname, rmtname) == SAME)) {
       /* Close file in case we matched with scandir(). */
	if (fw != (FILE *)NULL) fclose(fw);
	return('S');		/* startup this system */
  } else return('G');
}


/*
 * checkname
 * Do we know the guy ? 
 */
short int checkname(name)
char *name;
{
  char line[BUFSIZ], tmp[20];	/* can change to 8 if %8s works */
  FILE *ff;
 
  if ((ff = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);
 
  while (fgets(line, BUFSIZ, ff) != (char *)NULL) {
	sscanf(line, "%8s ", tmp);
	printmsg(M_INFO, "rmt= '%s' sys= '%s'", name, tmp);
	if (strncmp(tmp, name, 7) == SAME) {
		fclose(ff);
       		return(OK);
  	}
  }
  fclose(ff);
  return(FAILED);
}


/*
 * checktime
 * check if we may make a call at this time.
 */
short int checktime(xtime)
char *xtime;
{
  return(1);			/* OK go to it */
}


/*
 * wmsg
 * write a ^P type msg to the remote UUCP.
 */
short int wmsg(msg, syn)
short int syn;
char *msg;
{
  short int len;

  printmsg(M_CALL, "wmsg %s", msg);

  len = strlen(msg);
  if (syn == 2) swrite("\0\020", 2);
  swrite(msg, len);
  if (syn == 2) swrite("\0", 1);
}


/*
 * rmsg
 * read a ^P msg from UUCP. 
 */
short int rmsg(msg, syn)
short int syn;
char msg[];
{
  short int ii;
  char c, cc[5];

  c = 'a';
  if (syn == 2) {
	while(c != 0x10) {
		if (sread(cc, 1, msgtime) < 1) return(-1);
		/* Dont ask. MSC needs more than a byte to breathe. */
		c = cc[0] & 0x7f;
	}
  }

  for (ii = 0; ii < 132 && c; ii++) {
	if (sread(cc, 1, msgtime) < 1) return(-1);
	c = cc[0] & 0x7f;
	if (c == '\r' || c == '\n') c = '\0';
	msg[ii] = c;
  }
  return(strlen(msg));
}


short int prefix(sh, lg)
char *sh, *lg;
{
  return(strncmp(sh, lg, strlen(sh)) == SAME);
}


short int notin(sh, lg)
char *sh, *lg;
{
  while(*lg) {
	if (prefix(sh, lg++)) return(FALSE);
  }
  return(TRUE);
}


#define MAXR 300
short int expectstr(str, timeout)
char *str;
{
  static char rdvec[MAXR];
  char *rp = rdvec;
  short int kr;
  char nextch;
 
  printmsg(M_CALL, "wanted %s", str);

  if (strcmp(str, "\"\"") == SAME) return(TRUE);

  *rp = 0;
  while(notin(str, rdvec)) {
	kr = sread(&nextch, 1, timeout);
	if (kr <= 0) return(FALSE);

	if ((*rp = nextch & 0177) != '\0') rp++;

	*rp = '\0';
	if (rp >= rdvec + MAXR) return(FALSE);
  }
  return(TRUE);
}


short int writestr(s)
register char *s;
{
  register char last;
  register char *m;
  short int nocr;
 
  last = '\0';
  nocr = FALSE;
  while(*s) {
	if (last == '\\') {
		switch(*s) {
			case 'd':
			case 'D':	/* delay */
		 		sleep(2);
				break;
			case 'c':
			case 'C':	/* end string don't output CR */
				nocr = TRUE;
				break;
			case 'r':
			case 'R':	/* carriage return */
			case 'm':
			case 'M':
				swrite("\r", 1);
				break;
			case 'n':
			case 'N':
				swrite("\n", 1);
				break;
			case 'b':
			case 'B':
				swrite("\b", 1);
				break;
			case 't':
			case 'T':
				swrite("\t", 1);
				break;
			case 's':
			case 'S':
				swrite(" ", 1);
				break;
			default:
				swrite(s, 1);
		}
		last = '\0';
	} else if (*s != '\\') {
			swrite(s, 1);
		} else {
			last = *s;
		}
	s++;
  }
  return(nocr);
}


/*
 *   void sendthem(str)   send line of login sequence
 *         char *str;
 *
 *   return codes:  none
 */
void sendstr(str)
char *str;
{
  short int nw, ns;
  short int nulls;

  printmsg(M_CALL, "sending %s", str);

#ifdef BREAK
  if (prefix("BREAK", str)) {
	sscanf(&str[5], "%1d", &nulls);
	if (nulls <= 0 || nulls > 10) nulls = 3;
	/* send break */
	ssendbrk(nulls);
	return;
  }
#endif BREAK
   
  if (strcmp(str, "EOT") == SAME) {
	swrite(EOTMSG, strlen(EOTMSG));
	return;
  }
  if (strcmp(str, "\"\"") == SAME) *str = '\0';

  if (strcmp(str, "") != SAME) {
	if (!writestr(str)) {
		swrite("\r", 1);
	}
  } else {
	  swrite("\r", 1);
  }
  return;
}


short int sendexpect(s, e, timeout)
char *s;
char *e;
short int timeout;
{
  sendstr(s);
  return(expectstr(e, timeout));
}


short int initline(infd, outfd)
short int infd, outfd;
{
  struct sgttyb ttyb;

  sreadfd = infd;	/* SLAVE = standard input, MASTER = DIAL() */
  swritefd = outfd;	/* SLAVE = standard output MASTER = DIAL() */

  ioctl(sreadfd, TIOCGETP, &ttyb);	/* set raw mode: 8N1/RAW */
#if _MINIX
  ttyb.sg_flags &= ~(XTABS|EVENP|ODDP|CRMOD|ECHO|CBREAK|BITS5|BITS6|BITS7);
  ttyb.sg_flags |= (RAW | BITS8);
#else
  ttyb.sg_flags &= ~(XTABS|EVENP|ODDP|CRMOD|ECHO|CBREAK|CS5|CS6|CS7);
  ttyb.sg_flags |= (RAW | CS8);
#endif /* _MINIX */
  ioctl(sreadfd, TIOCSETP, &ttyb);

  return(0);
}


short int dcpdial(dev, speed, tel)
char *dev, *speed, *tel;
{
  call.baud = call.speed = atoi(speed);
  call.line = dev;
  call.telno = tel;
  call.modem = 0;

  printmsg(M_CALL, "Dialing at speed %d", call.speed);
  if (tel != (char *)NULL) printmsg(M_CALL, "Calling phone# %s", call.telno);
  if ((sreadfd = swritefd = dial(&call)) < 0) {
	printmsg(M_ERROR, "dial failed, merror=%s (%d)",
				_merr_list[-merrno], merrno);
	return(FALSE);
  }
  return(TRUE);
}


short int dcpundial()
{
  if (swritefd > 2) hangup(swritefd);
}


/*
 * Uucpname() returns a pointer to the local host's UUCP nodename.
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually "/etc/uucpname".
 */
char *uucpname()
{
  static char uuname[SITENAMELEN];
  FILE *uufile;

  if ((uufile = fopen(NODENAME, "r")) == (FILE *)NULL) return("unknown");
  fgets(uuname, SITENAMELEN, uufile);
  uuname[strlen(uuname) - 1] = '\0';
  fclose(uufile);

  return(uuname);
}


/*
 * Visib(data, length) prints the buffer of character data in "visible"
 * format. Printable characters (except '\') are printed as themselves,
 * special control characters (CR, LF, BS, HT) are printed in the usual
 * escape format, and others are printed using 3-digit octal escapes.
 * The usual warning about return value pointing to a static buffer which
 * is overwritten on each call applies here.
 */
char *visib(data, len)
char *data; short int len;
{
  static char buf[256];
  char c, *p;

  p = buf;
  while(len--) {
	c = *data++;
	if (isascii(c) && (isprint(c) || ' ' == c)) *p++ = c;
	  else if ('\n' == c) {
		*p++ = '\\';
		*p++ = 'n';
       	  } else if ('\t' == c) {
		*p++ = '\\';
		*p++ = 't';
          } else if ('\r' == c) {
		*p++ = '\\';
		*p++ = 'r';
         } else if ('\b' == c) {
		*p++ = '\\';
		*p++ = 'b';
         } else {
		sprintf(p, "\\%03o", c);
		p += 4;
        }
  }
  *p = '\0';
  return(buf);
}


/*
 * Getargs(str, argv) breaks up str, which should be a string of
 * whitespace-separated fields, and places a pointer to each field
 * into the array pointed to by argv. The number of fields is
 * returned.
 */
short int getargs(str, argv)
char *str; char *argv[];
{
  short int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while(*str && '\n' != *str) {
	++nflds;				/* field */
	*argv++ = str;

	while(*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while(*str && isspace(*str)) ++str;	/* trailing space */
  }
  return(nflds);
}


/*
 * Expandtilde(filename) expands filename using the usual "~" convention.
 * That is, "~user" expands to the home directory of "user". "~" by itself
 * expands to the home directory of the effective userid, which in this
 * case is usually the /usr/spool/uucppublic directory. Care is taken not
 * to overflow the (static) name buffer.
 */
char *expandtilde(filename)
char *filename;
{
  static char namebuf[PATHLEN];
  struct passwd *pw;
  register char *p;
  extern struct passwd *getpwnam(), *getpwuid();

  if (*filename != '~') return(filename);

  ++filename;
  p = namebuf;
  while (*filename && '/' != *filename && (p - namebuf) < PATHLEN)
							*p++ = *filename++;
  *p = '\0';

  if (strlen(namebuf) == 0) pw = getpwuid(geteuid());
   else pw = getpwnam(namebuf);

  if (pw == (struct passwd *)NULL) return((char *)NULL);

  if (strlen(pw->pw_dir) + strlen(filename) + 1 > PATHLEN)
						return((char *)NULL);
  strcpy(namebuf, pw->pw_dir);
  strcat(namebuf, filename);
  return(namebuf);
}


/*
 * s d a t a 
 *
 * Send File Data 
 */
short int sdata()
{
  while(TRUE) {
	/* Get data; if EOF set state to that */
	if ((size = fread(spacket, 1, pktsize, fp)) <= 0) return('Z');

	if ((*sendpkt) (spacket, size, 0)) return(0);	/* send data */
  }
  return('D');		/* sent data, stay in state D */
}


/*
 * s b r e a k 
 */
short int sbreak()
{
  short int len, i;

  if (sendmsg("H") < 0) return(0);
  if (getmsg(spacket) < 0) return(0);

  printmsg(M_CONVERSE, "Switch modes: %s", spacket);
  if (spacket[1] == 'N') return('G');
  return('Y');
}


/*
 * s e o f 
 *
 * Send End-Of-File. 
 */
short int seof()
{
  short int len, i;

  fclose(fp);
  fp = (FILE *)NULL;

  /* Length zero packet indicates EOF. */
  if ((*sendpkt) (spacket, 0, 0) < 0) return(0);

  if (getmsg(spacket) < 0) return(0);		/* receive CY or CN */
  if (strncmp(spacket, "CY", 2)) return(0);	/* couldn't send file */

  unlink(fromfile);
  printmsg(M_TRANSFER, "Transfer of %s completed.", fromfile);

  return('F');		/* go get the next file to send */
}


/*
 * s f i l e 
 *
 * Send File Header. 
 */
short int sfile()
{
  short int i, len;
  char *cp;

  if (fp == (FILE *)NULL) {	/* If not already open, */
	printmsg(M_SPOOL, "looking for next file...");
	if (getfile()) {	/* get next file from current work */
		fclose(fw);
		unlink(cfile);	/* close and delete completed workfile */
		fw = (FILE *)NULL;
		return('B');	/* end sending session */
	}

	printmsg(M_SPOOL, "Opening %s for sending.", fromfile);
	fp = fopen(fromfile, "r");	/* open the file to be sent */
	if (fp == (FILE *)NULL) {	/* If bad file pointer, give up */
		printmsg(M_ERROR, "Cannot open file %s.", fromfile);
       		return('A');		/* FIXME should be 'F' ! */
	}
  } else return ('A');	/* If something's already open, we're in trouble */

  printmsg(M_ERROR, "Sending \"%s\"", fromfile);

  /* Send the 'S fromfile tofile user - tofile 0666' message. */
  if (sendmsg(tofile) < 0) return(0);
  if (getmsg(spacket) < 0) return(0);
  if (spacket[1] != 'Y') return('A');	/* If other side says no-quit */
  return('D');
}


/*
 * s i n i t 
 *
 * Send Initiate: send this host's parameters and get other side's back. 
 */
short int sinit()
{
  if ((*openpk) ()) return('A');
  return('B');
}


/* 
 * getfile 
 *
 * getfile reads the next line from the presently open workfile (cfile) and
 * determines from this the next file to be sent (file). If there are no more
 * TRUE is returned. A fix for "R from to 0666" should be done here to
 * recieve files in addition to sending them. The appropriate "state
 * letter" i.e. "R" should be returned to the send "master" or "slave"
 * state switching table in "dcp.c" I did not implement this since the
 * majority of uucp transactions appear to be "S from to 0666" type.
 * RHLamb 1/87 
 */
short int getfile()
{
  char line[132], buf[132];
  register char *cp;
  short int i;
 
  if (fgets(line, BUFSIZ, fw) == (char *)NULL) return(TRUE);

  /* FIXME FvK- This should be done differently! */
  sscanf(line, "%*c %*s %*s %*s %*s %s", buf);
  sprintf(fromfile, "%s/%s", SPOOLDIR, buf);

  for(i = 0, cp = line; *cp != '\0'; i++, cp++) {
	if (strncmp(cp, "0666", 4) == 0) break;
  }
  cp += 4;
  *cp = '\0';
  strcpy(tofile, line);
  printmsg(M_SPOOL, "getfile: fromfile=%s, tofile=%s.", fromfile, tofile);
  return(FALSE);
}


/*
 * schkdir
 * scan the dir 
 */
short int schkdir()
{
  char c;

  c = scandir();
  if (c == 'Q') {
	return('Y');
  }
  if (c == 'S') {
	sprintf(rpacket, "HN");
	if ((*sendpkt) (rpacket, 0, 1)) return(0);
  }
  return('B');
}


/* 
 * endp() end protocol 
 *
 */
short int endp()
{
  sendmsg("HY");
  (*closepk) ();
  return('P');
}


/*
 * r d a t a 
 *
 * Receive Data 
 */
short int rdata()
{
  short int len;
 
  if ((*getpkt) (rpacket, &len)) return(0);

  if (len == 0) {
	fclose(fp);

	/* FIXME FvK-We ought to do the copy in a temp file, and move it. */
	if (sendmsg("CY") < 0) return(0);

	printmsg(M_TRANSFER, "transfer complete");
	return('F');
  }

  /* Write the data to the file. */
  if (fwrite(rpacket, len, 1, fp) != 1) return(0);

  return('D');			/* Remain in data state */
}


/*
 * r f i l e 
 *
 * Receive File Header 
 */
short int rfile()
{
  char rfbuf[256], buf[256];	/* receive buffer and copy */
  char tmpfilename[256];	/* Holds the converted file name */
  char *flds[10];
  short int numflds, len, i, fd;
  char *cp, *bp;
 
  printmsg(M_TRANSFER, "rfile entered");
 
  if (getmsg(buf) < 0) return(0);

  if (buf[0] == 'H') return('C');
  strcpy(rfbuf, buf);
  printmsg(M_TRANSFER, "rfile: buf %d \"%s\"", len, buf);

  numflds = getargs(rfbuf, flds);
  cp = flds[2];

  /* Check if absolute pathname given. */
  if (*cp == '/') {
	cp = strrchr(cp, '/');
	cp++;
	sprintf(tmpfilename, "%s/%s", PUBDIR, cp);
  } else {
	/* Check for "~"- destination. */
	bp = expandtilde(cp);
	if (bp == cp) sprintf(tmpfilename, "%s/%s", SPOOLDIR, cp);
	  else strcpy(tmpfilename, bp);
  }
    
  /* Check for dirname only. */
  bp = tmpfilename + strlen(tmpfilename) - 1;
  if (*bp == '\n') *bp-- = '\0';
 
  if (*bp == '/') {
	bp = strrchr(flds[1], '/');
	if (bp == (char *)NULL) bp = flds[1];
	  else bp++;

	strcat(tmpfilename, bp);
  }

  strcpy(tofile, tmpfilename);
  printmsg(M_TRANSFER, "rfile: receive file \"%s\"", tofile);

  /* Try to open a new file. */
  fd = creat(tofile, 0660);
  close(fd);
  if ((fp = fopen(tofile, "w")) == (FILE *)NULL) {
	printmsg(M_ERROR, "cannot create %s err=%s",
				tofile, sys_errlist[errno]);
	return('F');	/* give up if can't */
   }

  printmsg(M_ERROR, "Receiving %s", tofile);
  if (sendmsg("SY") < 0) return(0);

  return('D');		/* Switch to data state */
}


/*
 * r i n i t 
 *
 * Receive Initialization 
 */
short int rinit()
{
  if ((*openpk) ()) return(0);
  return('F');
}


/*
 * Getmsg() receives a null-terminated "conversation-level" message
 * from the communications channel. This may require one or more packets,
 * but all of them will be "long-data" packets containing a full 64 bytes.
 */
short int getmsg(dest)
char *dest;
{
  short int len;

  while(1) {
	if ((*getpkt)(dest, &len) < 0) return(-1);
	*(dest + len) = '\0';		/* make sure it is terminated */
	if (strlen(dest) != len) break;	/* we reached the terminator */
	dest += len;
  }
  return(0);
}


/*
 * Sendmsg(message) sends a null-terminated "conversation-level" message.
 */
short int sendmsg(message)
char *message;
{
  short int len;

  len = strlen(message) + 1;		/* total length including '\0' */

  while(1) {
	/* Send the packet with padding. */
	if ((*sendpkt)(message, 0, 1) < 0) return(-1);
	if ((len -= pktsize) <= 0) break;
	message += pktsize;
  }
  return(0);
}


/*
 * callup
 * script processor - nothing fancy! 
 */
short int callup()
{
  short int flg, kk, jj, ll, firstflg, ok, i;
  char *prsend, *exp, *alternate;
 
  printmsg(M_CALL, "calling host %s", rmtname);
 
  if (strcmp(device, "ACU") == SAME) {
	if (dcpdial((char *)NULL, speed, flds[FLD_PHONE]) == FALSE)
								 return('G');
  } else if (dcpdial(device, speed, (char *)NULL) == FALSE) return('G');
 
  for (i = FLD_EXPECT; i < kflds; i += 2) {
	exp = flds[i];
	printmsg(M_INFO, "callup: expect %d of %d  \"%s\"", i, kflds, exp);
    
	ok = FALSE;
	while (ok != TRUE) {
		alternate = strchr(exp, '-');
		if (alternate != (char *)NULL) *alternate++ = '\0';
	  
		ok = expectstr(exp, 45);
	  
		printmsg(M_INFO, "got %s", ok != TRUE ? "?" : "that");
	  
		if (ok == TRUE) break;

		if (alternate == (char *)NULL) {
			printmsg(M_ERROR, "LOGIN FAILED");
			return('Y');
		}
		exp = strchr(alternate, '-');
		if (exp != (char *)NULL) *exp++ = '\0';
	  
		printmsg(M_INFO, "send alternate");
		sendstr(alternate);
	}
    
	printmsg(M_INFO, "callup: send %d of %d \"%s\"",
					i + 1, kflds, flds[i + 1]);
	sleep(1);
	sendstr(flds[i + 1]);
  }
  return('P');
}


/*
 * scandir
 *
 * scan work dir for "C." files matching current remote host (rmtname)
 * return: A	- abort
 *	   Y	- can't open file
 *	   S	- ok
 *	   Q	- no files 
 */
short int scandir()
{
  char cname[40], tmp[132];
  short int fn, len, i;
  DIR *dirp;
  struct direct *dp;
 
  if ((dirp = opendir(SPOOLDIR)) == (DIR *)NULL) {
	printmsg(M_ERROR, "couldn't open dir %s", SPOOLDIR);
	return('A');
  }
  sprintf(cname, "C.%s", rmtname);
  len = strlen(cname);
  while((dp = readdir(dirp)) != (struct direct *)NULL) {
	printmsg(M_INFO, "scandir: %s", dp->d_name);
	if (strncmp(cname, dp->d_name, len) == SAME) {
		printmsg(M_INFO, "scandir: match!!");
		sprintf(cfile, "%s/%s", SPOOLDIR, dp->d_name);
		closedir(dirp);
		if ((fw = fopen(cfile, "r")) == (FILE *)NULL) return('Y');
		return('S');
	}
  }
  closedir(dirp);
  return('Q');
}


/*
 * set the protocol
 */
short int setproto(pr)
char pr;
{
  short int i;
  Proto *tproto;

  for (tproto = Protolst; tproto->type != '\0' &&
				pr != tproto->type; tproto++) {
	printmsg(M_CALL, "setproto: %c %c", pr, tproto->type);
  }
  if (tproto->type == '\0') {
	printmsg(M_ERROR, "setproto:You said I had it but I cant find it");
	exit(1);
  }
  getpkt = tproto->a;
  sendpkt = tproto->b;
  openpk = tproto->c;
  closepk = tproto->d;
}


/*
 * sysend
 * end UUCP session negotiation 
 */
short int sysend()
{
  char msg[80];

  msg[1] = '\0';
  msgtime = 2 * MSGTIME;
  wmsg("OOOOOO", 2);
  wmsg("OOOOOO", 2);

  /* Clear the RS232-buffer from any HANGUP-garbage. */
#ifdef TIOCFLUSH
  ioctl(sreadfd, TIOCFLUSH, 0);
#else
  while (ioctl(sreadfd, TIOCICNT, 0) > 0) read(sreadfd, msg, 1);
#endif TIOCFLUSH

  /* Hangup the line. */
  dcpundial();

  if (remote == MASTER) return('I');
  return('A');
}


/*
 * startup
 */
short int startup()
{
  char msg[80], tmp1[20], tmp2[20];
  short int i;
  char pro;

  if (remote == MASTER) {
	msgtime = 2 * MSGTIME;
	if (rmsg(msg, 2) == -1) return('Y');
	printmsg(M_INFO, "1st msg = %s", msg);

	/* Only SysV says "Shere", all others say "Shere=myname". */
	if (msg[5] == '=' && strncmp(&msg[6], rmtname, 7)) return('Y');

	sprintf(msg, "S%.7s -Q0 -x%d", nodename, debuglevel);
	printmsg(M_INFO, "Reply to 1st msg = %s", msg);
	wmsg(msg, 2);

	if (rmsg(msg, 2) == -1) return('Y');
	printmsg(M_INFO, "2nd msg = %s", msg);
	if (strncmp(&msg[1], "OK", 2)) return('Y');

	if (rmsg(msg, 2) == -1) return('Y');
	printmsg(M_INFO, "3rd msg = %s", msg);

	for(i = 0; (pro = PROTOS[i]), pro; ++i)
		if (msg[0] == 'P' && strchr(&msg[1], pro) != NULL) {
			sprintf(msg, "U%c", pro);
			printmsg(M_INFO, "Reply to 3rd msg = %s", msg);
			wmsg(msg, 2);
			setproto(pro);
			return('D');
		}
	printmsg(M_ERROR, "Replying 'UN' - no common protocol");
	wmsg("UN", 2);
	return('Y');
  } else {	/* slave */
	  msgtime = 2 * MSGTIME;
	  sprintf(msg, "Shere=%s", nodename);
	  wmsg(msg, 2);

	  if (rmsg(msg, 2) == -1) return('Y');
	  sscanf(&msg[1], "%s %s %s", rmtname, tmp1, tmp2);
	  sscanf(tmp2, "-x%d", &debuglevel);

	  printmsg(M_INFO, "debug level = %d", debuglevel);
	  printmsg(M_CALL, "1st msg from remote = %s", msg);

	  if (checkname(rmtname)) return('Y');
	  wmsg("ROK", 2);

	  sprintf(msg, "P%s", PROTOS);
	  wmsg(msg, 2);
	  if (rmsg(msg, 2) == -1) return('Y');

	  if (msg[0] != 'U' || strchr(PROTOS, msg[1]) == (char *)NULL)
								return('Y');
	  setproto(msg[1]);
	  return('R');
  }
}


/*
 * r e c v f
 *
 * This is the state table switcher for receiving files. 
 */
short int recvf()
{
  state = 'F';			/* Receive-Init is the start state */

  while(TRUE) {
	printmsg(M_TRANSFER, " receive state: %c", state);
	switch(state) {		/* Do until done */
		case 'F':
	      		state = rfile();
			break;			/* Receive-File */
		case 'D':
			state = rdata();
			break;			/* Receive-Data */
		case 'C':
			return('C');		/* Complete state */
		case 'A':
			return('Y');		/* "Abort" state */
		default:
			return('Y');
	}
  }
}


/*
 * s e n d f
 *
 * Sendsw is the state table switcher for sending files. It loops until either
 * it finishes, or an error is encountered.  The routines called by sendsw
 * are responsible for changing the state. 
 *
 */
short int sendf()
{
  fp = (FILE *)NULL;		/* reset file getter/opener */
  state = 'F';			/* send initiate is the start state */

  while(TRUE) {			/* do this as long as necessary */
	printmsg(M_TRANSFER, "send state: %c", state);
	switch(state) {
		case 'F':
	      		state = sfile();
			break;			/* Send-File */
		case 'D':
			state = sdata();
			break;			/* Send-Data */
		case 'Z':
			state = seof();
			break;			/* Send-End-of-File */
		case 'B':
			return('B');		/* Complete */
		case 'A':
			return('Y');		/* "Abort" */
		default:
			return('Y');		/* Unknown, fail */
	}
  }
}


/*
 * master
 */
short int master()
{
  state = 'I';

  while(TRUE) {
	printmsg(M_CONVERSE, "Top level state (master mode) %c", state);
	switch(state) {
		case 'I':
			state = sinit();
			break;
		case 'B':
			state = scandir();
			break;
		case 'S':
			state = sendf();
			break;
		case 'Q':
			state = sbreak();
			break;
		case 'G':
			state = recvf();
			break;
		case 'C':
			state = 'Y';
			break;
		case 'Y':
			state = endp();
			break;
		case 'P':
			return('Y');
		case 'A':
			return('A');
		default:
			return('A');
	}
  }
}


/*
 * slave
 */
short int slave()
{
  state = 'I';

  while(TRUE) {
	printmsg(M_CONVERSE, "Top level state (slave mode) %c", state);
	switch(state) {
		case 'I':
	      		state = rinit();
			break;
		case 'F':
			state = recvf();
			break;
		case 'C':
			state = schkdir();
			break;
		case 'T':
			state = 'B';
			break;
		case 'B':
			state = scandir();
			break;
		case 'S':
			state = sendf();
			break;
		case 'Q':
			state = sbreak();
			break;
		case 'G':
			return('Y');
		case 'Y':
			state = endp();
			break;
		case 'P':
			return('Y');
		case 'A':
			return('A');
		default:
			return('A');
	}
  }
}


main(argc, argv)
int argc;
char *argv[];
{
  FILE *ftmp;
  long now;
  char line[132];
  int c;

  setuid(geteuid());		/* set uid and gid to UUCP.UUCP */
  setgid(getegid());		/* not possible on all systems... */

  umask(117);		/* set protection mask to rw-rw---- */

  if ((logfile = fopen(LOGFILE, "a")) == (FILE *)NULL) cant(LOGFILE);

  signal(SIGINT, sigint);	/* catch INTERRUPT signal */

  time(&now);	/* create a logfile-header */
  fprintf(logfile, "========== %-24.24s ==========\n", ctime(&now));

  debuglevel = 0;
  fp = (FILE *)NULL;
  fw = (FILE *)NULL;
  nodename = uucpname();

  remote = SLAVE;
  strcpy(Rmtname, "any");

  opterr = 0;
  while ((c = getopt(argc, argv, "r:s:x:")) != EOF) switch(c) { 
	case 'x':
		debuglevel = atoi(optarg);
		break;
	case 's':
		remote = MASTER;
		sprintf(Rmtname, "%.7s", optarg);
		break;
	case 'r':
		remote = atoi(optarg);
		break;
	default:
		break;
  }
  printmsg(M_INFO, "nodename = %s", nodename);

  if (remote == MASTER) {
	time(&now);
	printmsg(M_CALL, "Calling %s, debuglevel=%d (%.19s)",
					Rmtname, debuglevel, ctime(&now));

	if ((fsys = fopen(LSYS, "r")) == (FILE *)NULL) cant(LSYS);
	state = 'I';

	while(TRUE) {
		printmsg(M_CALL, "Mstate = %c", state);
		switch(state) {
			case 'I':	/* no system, get one */
		 		state = getsystem();
				break;
			case 'S':	/* got system, call it */
				state = callup();
				/* Initialize the serial line to RAW etc. */
				if (initline(sreadfd, swritefd) < 0)
								state = 'Y';
				break;
			case 'P':	/* called, negotiate protocol */
				state = startup();
				break;
			case 'D':	/* got protocol, start running */
				state = master();
				break;
			case 'Y':	/* abort */
				state = sysend();
				break;
			case 'G':
				state = 'I';
				break;
		}
		if (state == 'A') break;
	}
	fclose(fsys);
  } else {	/* slave */
	  state = '0';
	  while(TRUE) {
		printmsg(M_CALL, "Sstate = %c", state);
		switch(state) {
			case '0':
				/* Initialize stdio to RAW etc. */
				if (initline(0, 1) < 0) state = 'Y';
				  else state = 'I';
				break;
			case 'I':
				  state = startup();
				  break;
			case 'R':
				  state = slave();
				  break;
			case 'Y':
				  state = sysend();
				  break;
		}
		if (state == 'A') break;
	}
	dcpundial();
  }

  sprintf(line, "exec %s", UUXQT);
  if (system(line) != 0) printmsg(M_ERROR, "ERROR in calling UUXQT");

  fclose(logfile);

  return(0);
}


/*$$ file: newgpkt.c
 * Author: Peter S. Housel
 *
 * The |cksum()| routine is taken from UUPC, Copyright 1985, 1986, 1987 by
 * Richard H. Lamb, with (possible) changes Copyright 1987 by Stuart Lynne
 *
 * All other code is Copyright 1989 by Peter S. Housel.
 * Redistribution for any purpose is permitted provided this message
 * is included intact. No warranty of any sort is provided.
 *
 * newgpkt version 1.2 11/30/89
 */

/* This program was written based on the original UUPC 'g' driver,
 * John Gilmore's version of uuslave, and Greg Chesson's protocol
 * description article. The last was especially helpful.
 *
 * This code was written around a severely hacked version of UUPC.
 * The call interface is almost identical to the original, but
 * the internal implementation is quite different. Also, many
 * functions are called that were not available in the original
 * UUPC support functions. It should serve as an adequate framework.
 *
 * The framing strategy requires that a |read()| be interruptable
 * by a |SIGALRM|. No "busy wait" or nonblocking read is required.
 *
 * Note that strncpy() needs to follow the spec in that it should
 * zero-pad the destination string.
 */

/*#include "dcp.h"*/

#define MAXPKT		64	/* incredibly conservative... actually 4096 */
#define SWINDOW		3	/* initial send window size */
#define RWINDOW		3	/* window size we want to recieve */
#define SPKTSIZE	64	/* initial send packet size */
#define RPKTSIZE	64	/* window size we want to recieve */

#define MAXLOST		5	/* max lost packets (closes or such) */
#define TIMEOUT		10	/* max seconds of before timeout */

#define LOSTPKT		-1	/* packet lost, got timeout */
#define BADPKT		-2	/* bad checksum, or data read timed out */

#define ENV_DLE		0	/* framing char at start of envelope */
#define ENV_K		1	/* packet length specifier */
#define ENV_C0		2	/* low-order checksum */
#define ENV_C1		3	/* high-order checksum */
#define ENV_C		4	/* control byte */
#define ENV_X		5	/* xor check byte */
#define ENV_LEN		6	/* overall envelope length */

#define TT_CTRL		0	/* control packet */
#define TT_DATA		2	/* data packet */
#define TT_SDATA	3	/* short data packet */
#define TT_ALTCHAN	1	/* 'alternate channel' - invalid */

#define X_CLOSE		1	/* close down protocol */
#define X_RJ		2	/* reject recieved packet */
#define X_SRJ		3	/* selectively reject packet - invalid */
#define X_RR		4	/* reciever ready */
#define X_INITC		5	/* third init packet */
#define X_INITB		6	/* second init packet */
#define X_INITA		7	/* first init packet */

#define OP_OPEN		1	/* request to open/init protocol */
#define OP_CLOSE	2	/* request to close protocol */
#define OP_WRITE	3	/* request to send packet */
#define OP_READ		4	/* request to read packet */

#define MAGIC (unsigned) 0xAAAA	/* checksum magic value */

/* From Dale Schumacher - determine if a <= b < c, for mod 8 seq numbers. */
#define between(a,b,c)	(((a)<(c)) ? (((a)<=(b))&&((b)<(c))) \
				   : (((a)<=(b))||((b)<(c))))	/* check */

unsigned cksum(/* unsigned char *data, short int len */);
extern char *visib(/* unsigned char *data, short int len */);

struct {
        short sdata;		/* 'is this a short data packet' flag */
	unsigned length;	/* length of this packet */
	unsigned char *bufloc;	/* location of this data pkt's buffer */
} inpbufs[8], outbufs[8]; 	/* input/output queues */

static short int needack;		/* do we need to acknowledge a rcv'd pkt? */
static short int neednack;		/* do we need to reject a recieved pkt? */
static short int recv;		/* seq. number of last correctly rcv'd pkt */
static short int lastread;		/* seq. number of last pkt ret. to caller */
static short int send;		/* first packet in output window */
static short int next;		/* next pkt to send  send <= next < nstuff */
static short int nstuff;		/* next loc. to stuff a pkt in output queue */
				/* (last pkt in output window) + 1 */
static short int initpk;		/* current init sequence send packet */
static short int skipping;		/* skipping out-of-seq packets after RJ */
static short int spktsize;		/* send data size (requested by other end) */
static short int swindow;		/* send output window size (ditto) */
static short int nlost;		/* number of consecutive timeouts */
static short int chanopen = 0;	/* 'channel open' flag */

static unsigned char buf[ENV_LEN + RPKTSIZE];	/* packet framing buffer */
static unsigned char *low, *high;		/* framing buffer limits */

static unsigned char *ibufset, *obufset;	/* i/o packet buffer sets */

/* Gopenpk() opens the 'g' packet protocol on the serial line. It
 * initializes its state variables, allocates buffers, etc., then calls
 * gmachine() to do the actual protocol negotiation/initialization.
 */
short int gopenpk()
{
  short int i;			/* index */
  unsigned char *p, *q;		/* poshort inters short into buffer set */

  high = low = buf;		/* empty input buffer */
  needack = neednack = 0;	/* don't need to accept or reject anything */
  initpk = X_INITA;		/* send INITA during initialization */
  recv = lastread = 0;		/* initial empty read queue, seq=0 */
  send = next = nstuff = 1;	/* first in output queue, seq=1 */
  skipping = nlost = 0;		/* nothing lost yet, not skipping */

  if (gmachine(OP_OPEN) < 0)	/* do the open */
    			return(-1);
  
  /* Allocate send and receive buffers. */
  if ((p = ibufset = (unsigned char *)malloc(8 * RPKTSIZE))
				== (unsigned char *)NULL) return(-1);
  if ((q = obufset = (unsigned char *)malloc(8 * spktsize))
				== (unsigned char *)NULL) return(-1);
 
  for(i = 0; i < 8; ++i) {
	inpbufs[i].bufloc = p;
	p += RPKTSIZE;
	outbufs[i].bufloc = q;
	q += spktsize;
  }
  pktsize = spktsize;	/* for dcp compatibility */

  return(0);
}


/* Gclosepk() closes down the packet protocol using the OP_CLOSE
 * operation of gmachine().
 */
short int gclosepk()
{
  return(gmachine(OP_CLOSE));
}


/* Ggetpkt() reads one packet and returns it. The data is stored in
 * the buffer poshort inted to by |cdata|, and the length is stored in |*lenp|.
 * It calls gmachine() to get the data, and copies it from the proper input
 * buffer to the user's buffer area. "Short data" packets are handled here,
 * as opposed to within gmachine().
 */
short int ggetpkt(cdata, lenp)
unsigned char *cdata; short int *lenp;
{
  short int nextread;
  unsigned char *bufp;

  if (!chanopen) return(-1);

  nextread = (lastread + 1) & 7;
  printmsg(M_HIGHPROTO, "waiting for input pkt seq=%d", nextread);

  if (gmachine(OP_READ) < 0) return(-1);

  *lenp = inpbufs[nextread].length;
  bufp = inpbufs[nextread].bufloc;
  if (inpbufs[nextread].sdata) {
	/* Less than 128 bytes shorter than packet length. */
	if (*bufp < 128) *lenp -= *bufp++;
   	  else {		/* more than 128 bytes shorter */
		*lenp -= (*bufp++ & 127) * 256;
		*lenp -= *bufp++;
    	}
  }

  if (*lenp < 0) return(-1);		/* invalid packet length */

  memcpy(cdata, bufp, *lenp);
  lastread = nextread;
  return(0);
}


/* Gsendpkt() queues the packet poshort inted to by cdata of length len
 * short into the packet driver output buffer, and calls gmachine() to send
 * it. Gmachine() will return when the packet has been transmitted but
 * not necessarily acknowledged, with window size greater than 1.) If
 * |flag| is nonzero, cdata is considered a null-terminated string
 * which will be null-padded to the packet size and transmitted.
 */
short int gsendpkt(cdata, len, flag)
unsigned char *cdata; short int len, flag;
{
  unsigned char *destp;
  unsigned diff;

  if (!chanopen) return(-1);

  destp = outbufs[nstuff].bufloc;
  if (flag && len < spktsize) {
	printmsg(M_HIGHPROTO, "Padded message packet |%s|", cdata);
	strncpy(destp, cdata, spktsize);
	outbufs[nstuff].sdata = 0;
  } else {
	if ((diff = spktsize - len) > 127) {	/* really short packet? */
		*destp++ = (diff >> 8) | 128;
		*destp++ = diff & 255;
		outbufs[nstuff].sdata = 1;
	} else if (diff > 0) {			/* short packet */
		*destp++ = diff;
		outbufs[nstuff].sdata = 1;
	} else outbufs[nstuff].sdata = 0;
	memcpy(destp, cdata, len);		/* copy short into buffer */
  }
  printmsg(M_HIGHPROTO, "queued data packet seq=%d len=%d", nstuff, len);
  outbufs[nstuff].length = spktsize;
  nstuff = (nstuff + 1) & 7;

  return(gmachine(OP_WRITE));
}


/* Gmachine() is the heart of the 'g' packet driver. Its basic strategy
 * is:
 *	- transmit a packet if necessary
 *	- return if possible,
 *	- else read a packet and act on it
 *	- repeat
 *
 * OP_OPEN requests that the channel be opened, and OP_CLOSE requests that
 * it be closed. If why is OP_WRITE, gmachine() will return when the
 * last packet in the output queue has been transmitted (but not necessarily
 * acknowledged). OP_READ requests will return as soon as a new packet errives.
 */
short int gmachine(why)
short int why;
{
  short int xxx, yyy, len;
  unsigned char *bufp;
  short int shortdat;
  short int i;

  while(1) {
	if (OP_CLOSE == why) {
		gspack(TT_CTRL, X_CLOSE, 0, (unsigned char *)NULL, 0);
		chanopen = 0;
		printmsg(M_MEDPROTO, "Sending CLOSE request...");
	} else if (neednack) {
		gspack(TT_CTRL, X_RJ, (recv + 1) & 7,
				(unsigned char *)NULL, 0);
		neednack = 0;
		printmsg(M_MEDPROTO, "Sending RJ... recv=%d", recv);
	} else if (send != nstuff		/* nonzero output queue? */
	       	&& between(send, next, nstuff)	/* 'next' in queue? */
	       	&& between(send, next, (send + swindow) & 7)) {
			printmsg(M_MEDPROTO, "Sending data packet %d", next);
			gspack(outbufs[next].sdata ? TT_SDATA : TT_DATA,
				next, recv, outbufs[next].bufloc,
						outbufs[next].length);
			needack = 0;
			next = (next + 1) & 7;
			/* Go back for more. */
        		if (OP_WRITE == why && next == nstuff)
			return(0);
	} else if (needack) {
		gspack(TT_CTRL, X_RR, recv, (unsigned char *)NULL, 0);
		needack = 0;
		printmsg(M_MEDPROTO, "Sending RR... recv=%d", recv);
	} else if (OP_OPEN == why) {
		if (X_INITB == initpk)
			/* INITB contains packet size, */
	     		i = ilog2(RPKTSIZE) - 5;
		  else i = RWINDOW;	/* INITA, INITC contain window size */
		/* ??? examine for INITC */
		gspack(TT_CTRL, initpk, i, (unsigned char *)NULL, 0);
        }

	if (OP_READ == why && recv != lastread) return(0);

	shortdat = 0;
	bufp = buf + ENV_LEN;
	switch(grpack(&xxx, &yyy, &len)) {
		case LOSTPKT:
			printmsg(M_MEDPROTO, "Lost packet...");
			if (nlost > MAXLOST) return(-1);
			next = send;	/* retransmit last un-ack'ed pkt, */
			if (OP_READ == why)	/* request retransmit */
			   	neednack = 1;
			skipping = 0;
			break;
		case BADPKT:
			printmsg(M_MEDPROTO, "Bad packet...");
			neednack = 1;	/* reject! */
			skipping = 1;	/* ignore the rest of the 'window' */
			break;
		case TT_SDATA:
			shortdat = 1;
			/* fall through */
		case TT_DATA:
			send = (yyy + 1) & 7;	/* recieve acknowledgement */
			if (xxx != ((recv + 1) & 7)) {
				printmsg(M_MEDPROTO, 
				    "Received data out of sequence (%d != %d)",
							xxx, (recv + 1) & 7);
				/* We must have missed one. */
				if (!skipping) {
					neednack = 1;
					skipping = 1;
				} else printmsg(M_MEDPROTO, "(Ignoring)");
			} else {	/* This is most recent correct pkt. */
				recv = xxx;
			 	needack = 1; /* we will ack it */
			 	skipping = 0;
			 	inpbufs[xxx].length = len;
			 	inpbufs[xxx].sdata = shortdat;
			 	memcpy(inpbufs[xxx].bufloc, bufp, len);
			}
			break;
		case TT_CTRL:
			skipping = 0;
			switch(xxx) {
			    case X_CLOSE:
				printmsg(M_MEDPROTO, "* CLOSE *");
				gspack(TT_CTRL, X_CLOSE, 0, (char *)NULL, 0);
				chanopen = 0;
				if (OP_CLOSE == why)
					return(0);	/* expected? */
				  else
					return(-1);	/* nope */
			    case X_RJ:
				printmsg(M_MEDPROTO, "got RJ yyy=%d", yyy);
				if (between(send,yyy,next)) {
				    next = send = yyy;	/* check! */
				} else {
				    printmsg(M_MEDPROTO, "RJ out of window");
				}
				break;
			    case X_RR:
				printmsg(M_MEDPROTO, "got RR yyy=%d", yyy);
				if (between(send,yyy,next)) {
				    send = (yyy + 1) & 7;
				} else {
				    printmsg(M_MEDPROTO, "RR out of window");
				    /* Back up all the way to beginning. */
				    /* of window. */
				    next = send;
				}
				break;
			    case X_INITC:
				printmsg(M_MEDPROTO, "* INITC *");
				swindow = yyy;
				if (X_INITC == initpk) {
				    chanopen = 1;
				    return(0);
				}
				break;
			    case X_INITB:
				printmsg(M_MEDPROTO, "* INITB *");
				spktsize = 32 << yyy;
				if (X_INITB == initpk) initpk = X_INITC;
				break;
			    case X_INITA:
				printmsg(M_MEDPROTO, "* INITA *");
				swindow = yyy;
				initpk = X_INITB;
				break;	
			    default:
				printmsg(M_MEDPROTO,
					"bad control packet: xxx=%d", xxx);
			}
			break;
		default:
			printmsg(M_MEDPROTO, "bad packet type");
			break;
	}
  }
}


/*
 * Grpack() is responsible for reading one 'g'-protocol packet from the
 * input communications channel. This includes framing, detecting timeouts,
 * and checksum validation.
 * The basic strategy is to keep a count of how many bytes have currently
 * been read and how many are needed. When enough bytes for a packet header
 * ("envelope") have been read, it is validated. If it is valid, and it
 * has a nonzero data segment, the data portion is read. When it has
 * everything, it does a checksum test and returns, with the control 
 * information stored in *xxxp, *yyyp, and the data segment length stored
 * in *lenp.
 */
short int grpack(xxxp, yyyp, lenp)
short int *xxxp, *yyyp;
short int *lenp;
{
  short int need;			/* need how many bytes? */
  short int env;			/* 'have pkt envelope' flag */
  short int gotdle;			/* do we have envelope hdr? */
  short int tt;			/* packet type */
  unsigned sum;			/* checksum */
  short int remain;			/* bytes which we stil need */
  short int i;

  env = gotdle = 0;
  need = ENV_LEN;		/* initially, need a header */

  alarm(TIMEOUT);		/* time out if we don't have a packet */
  timedout = 0;

  while(1) {
	if (low == high)		/* prevent framebuffer overruns */
	  	low = high = buf;
	printmsg(M_LOWPROTO, "=> l=%d h=%d g=%d", (short int)(low - buf),
						(short int)(high - buf), gotdle);

	while((remain = need - (high - low)) > 0) {
		if (timedout || (i = sread2(high, remain)) < 0) {
			alarm(0);
			++nlost;
			low = high = buf;/* empty out partial packet, if any */
			return(env ? BADPKT : LOSTPKT);
		}
		high += i;	/* got some data - move upper limit up */
	}
	if (!gotdle) {
		while(low < high) {	/* look for header 'DLE' prefix */
			if (DLE == *low) {
				gotdle = 1;
				break;
			} else ++low;
		}
		continue;
	} else if (!env) {	/* found DLE, but haven't found header yet */
			if (low > buf) { /* move envelope to buf beginning */
				register unsigned char *dst = buf;

				while(low < high) *dst++ = *low++;
				low = buf;
				high = dst;
			}
			if (buf[ENV_X] !=
				(buf[ENV_K]^buf[ENV_C0]^buf[ENV_C1]^buf[ENV_C])
	  		  || buf[ENV_K] < 1 || buf[ENV_K] > 9) {  /* valid? */
				++low;
				gotdle = 0;
				printmsg(M_LOWPROTO,
					"grpack: rejecting an envelope");
				continue;
			}
			env = 1;		/* we have an envelope */
			tt = (buf[ENV_C] >> 6) & 3; 	/* store control info */
			*xxxp = (buf[ENV_C] >> 3) & 7;
			*yyyp = (buf[ENV_C]     ) & 7;
			if (buf[ENV_K] == 9)	/* does it have data? */
				*lenp = 0;
			  else
				/* Size = 32 * 2^(k-1) . */
				*lenp = 16 << buf[ENV_K];
			need += *lenp;		/* now need that many more */
			printmsg(M_LOWPROTO,
				"grpack: tt=%d, xxx=%d, yyy=%d, need=%d",
							tt, *xxxp, *yyyp, need);
			continue;
		} else {			/* have everything we need */
		if (*lenp)
		     sum = MAGIC - (cksum(buf + ENV_LEN, *lenp) ^ buf[ENV_C]);
		  else
		     sum = MAGIC - buf[ENV_C];
		if (((sum >> 8) & 0xFF) != buf[ENV_C1]
		     || (sum & 0xFF) != buf[ENV_C0]) {
			alarm(0);
			nlost = 0;
			printmsg(M_LOWPROTO, "grpack: bad checksum");
			low += ENV_LEN;	/* search bad data seg for a header */
			return(BADPKT);
		} else {
			alarm(0);		/* got it all... return */
			nlost = 0;
			low += need;
			if (*lenp)
				printmsg(M_DATA, "|%s|",
					visib(buf + ENV_LEN, *lenp));
			return(tt);
		}
	}
  }
}


/*
 * gspack simply sends out a packet, by assembling an envelope and writing
 * it, and the data segment, to the communications channel.
 */
short int gspack(tt, xxx, yyy, data, size)
short int tt, xxx, yyy;
unsigned char *data;
short int size;
{
  unsigned char envelope[ENV_LEN];	/* packet envelope */
  unsigned sum;				/* checksum */
  unsigned char ctrl;			/* header control byte */
  short int k;				/* size = 32 * 2^(k-1) */

  ctrl = ((tt & 3) << 6) | ((xxx & 7) << 3) | (yyy & 7);

  if (0 == size) {
	k = 9;				/* no data seg */
	sum = MAGIC - ctrl;
  } else {
	short int tmp = size;
	for(k = -5; tmp != 0; ++k)		/* find k */
				tmp >>= 1;
	sum = MAGIC - (cksum(data, size) ^ ctrl);
  }
  envelope[ENV_DLE] = DLE;
  envelope[ENV_K]   = k;
  envelope[ENV_C1]  = (sum >> 8) & 0xFF;
  envelope[ENV_C0]  = sum & 0xFF;
  envelope[ENV_C]   = ctrl;
  envelope[ENV_X]   = k ^ envelope[ENV_C0] ^ envelope[ENV_C1] ^ ctrl;
  swrite(envelope, ENV_LEN);			/* send envelope */
  if (0 != size) swrite(data, size);		/* send data segment */
  printmsg(M_LOWPROTO, "gspack: tt=%d xxx=%d yyy=%d k=%d sum=%d",
						tt, xxx, yyy, k, sum);
  if (0 != size) printmsg(M_DATA, "|%s|", visib(data, size));
}


/*
 * chksum(data, len) came directly from dcp. It checksums the given data
 * area using the g protocol algorithm.
 */
unsigned cksum(data, len)
short int len;
unsigned char *data;
{
  unsigned short int i, j, tmp, chk1, chk2;
 
  chk1 = 0xffff;
  chk2 = 0;
  j = len;
  for (i = 0; i < len; i++) {
	if (chk1 & 0x8000) {
		chk1 <<= 1;
		chk1++;
	} else {
		chk1 <<= 1;
	}
	tmp = chk1;
	chk1 += (data[i] & 0xff);
	chk2 += chk1 ^ j;
	if ((chk1 & 0xffff) <= (tmp & 0xffff)) chk1 ^= chk2;
	j--;
  }
  return(chk1 & 0xffff);
}


/* |ilog2(value)| returns (short int)floor(log2(value)).
 */
short int ilog2(value)
unsigned value;
{
  short int i;
 
  if (value == 0) return(-1);
  for(i = 0; value > 1; ++i) value >>= 1;

  return(i);
}
