/*
 * UMAIL -	MINIX Remote Domain-addressing Mail Router
 *
 *		This version of RMAIL handles message-headers in a much
 *		more "standard" way. It can handle bang-addresses, plus
 *		the new-style Internet addressing (user@host.domain).
 *		It is called by programs as "Mail" and "Uuxqt".
 *
 *	    	M A I L   H E A D E R   C O N V E R S I O N   M O D U L E
 *
 * Headers:	This module reads the header part of a message into
 *		memory, and then rearranges it into RFC-822 order.
 *		The preferred order of fields is:
 *
 *		1.  From (old-style V6/V7)
 *		2.  From:
 *		3.  Received:	(top to bottom)
 *		4.  Sender:
 *		5.  Reply-To:
 *		6.  Organization:
 *		7.  Department:
 *
 *		N.  Unknown fields (user-defined)
 *
 *		8.  To:
 *		9.  Cc:
 *		10.  Bcc:
 *		11.  Subject:
 *		12. Keywords:
 *		13. Summary:
 *		14. X-Mailer:
 *		15. Message-Id:
 *		16. Date:
 *
 *		This order may be changed and/or expanded in the future,
 *		especially the "Resent-" fields should be added.
 *
 * Author:	F. van Kempen, waltje@minixug.hobbynet.nl
 *
 * Revisions:
 *		11/07/89 FvK	Fixed "From_" problem with local delivery.
 *				This bug caused WMAIL to have a problem with
 *				replying with OTHER smart mailers.
 *				Edited a little for the new MSS.
 *		11/13/89 FvK	Added the 'Message-ID' field to outgoing
 *				remote mail. Also, added subfield 'id' in
 *				the 'Received' field of forwarded mail.
 *				Also, changed the syntax of some fields in
 *				new_hdr() (braces etc.).
 *				Added the "Path:"-field to mail that has to
 *				be delivered locally; which allows the
 *				"From:" field to only contain the last part
 *				of the return path (i.e. the user name).
 *		12/07/89 FvK	Changed "Message-ID" into "Message-Id".
 *		12/16/89 FvK	Cleanup.
 *		12/30/89 FvK	Adapted for POSIX (MINIX 1.5)
 *		02/17/90 FvK	Added the "X-Mailer" field.
 *				Cleaned for release.
 *		04/11/90 FvK	Added some config switches and fixed some
 *				"Header" bugs.
 *		04/28/90 FvK	Added RFC-style comments and literals.
 */
#include <sys/types.h>
#include <string.h>
#include <time.h>
#include <stdio.h>
#include "umail.h"


typedef struct __hdr {
struct __hdr *next;
  int done;		/* 1==READ, 0==UNREAD */
  int count;		/* 1==FIRST, 2.. == NEXT PART */
  int std;		/* 1==RFC, 0==USER_DEFINED */
  char *name;		/* field name */
  char *value;		/* field value */
} HEADER;
#define NILHEAD ((HEADER *)NULL)


#define RF_OFROM	0
#define RF_FROM		1
#define RF_RECVD	2
#define RF_SENDER	3
#define RF_REPLY	4
#define RF_ORG		5
#define RF_DEPT		6
#define RF_UNKNOWN	7
#define RF_TO		8
#define RF_CC		9
#define RF_BCC		10
#define RF_SUBJECT	11
#define RF_KEYWORDS	12
#define RF_SUMMARY	13
#define RF_XMAILER	14
#define RF_MESGID	15
#define RF_DATE		16


static HEADER *hlist = NILHEAD;		/* old message header */
static char *rfc[] = {			/* list of RFC-822 fields */
  "FROM ",	"FROM",		"RECEIVED",	"SENDER",
  "REPLY-TO",	"ORGANIZATION",	"DEPARTMENT",	"",
  "TO",		"CC",		"BCC",		"SUBJECT",
  "KEYWORDS",	"SUMMARY",	"X-MAILER",	"MESSAGE-ID",
  "DATE",	(char *)NULL
};
static char olduser[1024];		/* old user */
static char olddate[48];		/* old date */
static char oldhost[48];		/* old host machine */


/*
 * Look for a field in the header table in memory.
 * If found, return its value, otherwise return NULL.
 */
static char *hfind(name)
char *name;
{
  register HEADER *hp;
  char tmp[48];

  hp = hlist;
  while (hp != NILHEAD) {
	strcpy(tmp, hp->name);
	strupr(tmp);
	if (!strcmp(tmp, name)) return(hp->value);
	hp = hp->next;
  }
  return((char *)NULL);
}


/*
 * Look for a field in the header table in memory.
 * If found, mark the field as READ, and return its address.
 * Otherwise, return NILHEAD.
 */
static HEADER *hsearch(name)
char *name;
{
  register HEADER *hp;
  char tmp[48];

  hp = hlist;
  while (hp != NILHEAD) {
	strcpy(tmp, hp->name);
	strupr(tmp);
	if (!strcmp(tmp, name)) {
		if (hp->done == 0) {
			hp->done = 1;
			return(hp);
		}
	}
	hp = hp->next;
  }
  return(NILHEAD);
}


/*
 * Decode an old-style (V6/V7) mail header.
 * This has a syntax like:
 *
 *	From <user> <date> [remote from <host>]
 *	To: <addressee>
 *	Subject: <text>
 *
 * We want to find out the <user>, <date> and possible <date> fields.
 * Return TRUE for OK, or FALSE if error.
 */
static int get_oldhdr(rmt)
int *rmt;			/* REMOTE or LOCAL mail? */
{
  register char *bp, *sp, *cp;
  int needpar, needlit;

  sp = hfind(rfc[RF_OFROM]);
  if (sp == (char *)NULL) {	/* No From-line??? */
	sprintf(errmsg, "%s: no From line", dfile);
       	return(FALSE);
  }

  /* First of all, clear out any junk. */
  bp = olduser;
  needpar = needlit = FALSE;
  while (*sp) switch(*sp) {
	case '\t':
	case ' ':
		if (*(bp - 1) != ' ') *bp++ = ' ';
		sp++;
		while (*sp == ' ' || *sp == '\t') sp++;
		break;
	case '(':	/* start comment */
		needpar = TRUE;			/* we need the closer! */
		sp++;
		break;
	case ')':	/* end comment */
		if (needpar == TRUE) {
			needpar = FALSE;
			sp++;
		} else {
			fprintf(stderr, "umail: bad header: \"%s\"\n", sp);
			return(FALSE);
		}
		break;
	case '<':	/* start literal */
		needlit = TRUE;
		sp++;
		break;
	case '>':	/* end literal */
		if (needlit == TRUE) {
			needlit = FALSE;
			sp++;
		} else {
			fprintf(stderr, "umail: bad header: \"%s\"\n", sp);
			return(FALSE);
		}
		break;
	default:
		if (needpar == FALSE && needlit == FALSE) *bp++ = *sp;
		sp++;
  }
  *bp = '\0';

  sp = olduser;			/* skip until <date> field */
  while (*sp && *sp!=' ' && *sp!='\t') sp++;
  *sp++ = '\0';			/* mark 'end-of-user' */

  /*
   * SP now contains <date> and (possible) <remote> fields.
   * Parse line to seek out "remote from".
   */
  cp = sp;			/* save the Date-pointer */
  while (TRUE) {
	bp = strchr(sp++, 'r');
	if (bp != (char *)NULL) {	/* we found an 'r' */
		if (!strncmp(bp, "remote from ", 12)) break;
        } else break;
  }

  if (bp != (char *)NULL) {	/* remote host found --> remote mail */
	sp = strrchr(bp, ' ');	/* seek to start of "remote from" text */
	*(bp - 1) = '\0';	/* mark end-of-date */
       	strcpy(olddate, cp);	/* set old date */
	strcpy(oldhost, ++sp);	/* set host name */
	sprintf(mailsender, "%s!%s", oldhost, olduser);
	*rmt = TRUE;
  } else {
	  strcpy(olddate, cp);	/* set old date */
	  strcpy(oldhost, "");	/* no remote host */
	  strcpy(mailsender, olduser);
	  *rmt = FALSE;	
  }
  return(TRUE);
}


/*
 * Analyze the current header.
 *
 * See if this mail was generated locally or came from somewhere else.
 * Note, that old-style mailers use "postmarks" (i.e. header lines
 * looking like "From <user> <date>" with a possible suffix of
 * "remote from <host>". New-style mailers (should) only use the
 * "From: <path>" and "Date: <date>" lines in their headers.
 * UMAIL knows both types. By default it uses new-style headers,
 * but it can use (and generate) old headers by defining OLDMAILER.
 *
 * Return TRUE if we think this mail has been generated remotely,
 * or FALSE if this message was generated by local mail.
 */
static int chk_hdr()
{
  int remmail;				/* remote mail? */
  time_t now;
  register char *sp, *bp;		/* fast scanning pointers */

  bp = hfind(rfc[RF_FROM]);		/* get RFC-From: field */
  sp = hfind(rfc[RF_DATE]);		/* get RFC-Date: field */

  if (sp==(char *)NULL || bp==(char *)NULL) {	/* should have both or none! */
	if (oldmailer == TRUE) {	/* try old-style header */
		if (get_oldhdr(&remmail) == FALSE) {
			strcat(errmsg, "\n\nBad address or header!\n");
			return(FALSE);
		}
	}
  } else {	/* Only use new-style From:=lines. */
	  strcpy(olddate, sp);	/* Save the DATE field */
	  strcpy(oldhost, bp);
	  sp = oldhost;		/* skip comments */
	  while (*sp && *sp!=' ' && *sp!='\t') sp++;
	  *sp = '\0';
	  strcpy(mailsender, oldhost);

	  sp = strchr(oldhost, '!');	/* check for pathname! */
	  if (sp != (char *)NULL) {	/* found one; this was remote! */
		remmail = TRUE;
		*sp++ = '\0';
		strcpy(olduser, sp);
	  } else {
		  remmail = FALSE;
		  strcpy(olduser, bp);
		  strcpy(oldhost, "");
	  }
  }
  return(remmail);
}


/*
 * Create a new RFC-822 message header.
 * This is necessary because we are processing a
 * locally-generated message. The header should become:
 *
 *      From <user> <date> remote from <myname>
 *      From: <myname>!<user> (Real Name)
 *	Received: by <myname> with <proto>;
 *                <receive-date>
 *	Sender: Real Name <user>@<myname>.<mydomain>
 *	To: <user>
 *	Subject: <text>
 *	X-Mailer: <version id>
 *	Message-Id: <id>
 *	Date: <date>
 */
static void new_hdr(outfp)
FILE *outfp;
{
  time_t rcvtm;				/* current time */
  char *date;				/* current date in MET */
  char *fmt1 = "%s:%s\n";		/* to save string space */
  char *fmt2 = "%s\n";			/* to save string space */
  register char *sp;

  /* Get the current date and time. */
  time(&rcvtm); date = maketime(&rcvtm);

  if (oldmailer == TRUE) {
	fprintf(outfp, "From %s %s remote from %s\n",
				olduser, xtime(&rcvtm), myname);
  }

  if (aremote == FALSE) {	/* local mail, add "Path:" field */
	fprintf(outfp, "Path: %s\n", myname);
	fprintf(outfp, "From: %s (%s)\n", olduser, realname(olduser));
  } else {	/* Adressee is remote. add our name! */
	  fprintf(outfp, "From: %s!%s (%s)\n",
		myname, olduser, realname(olduser));
  }
  fprintf(outfp, "Received: by %s.%s (UMAIL %s) with UUCP;\n",
					myname, mydomain, Version);
  fprintf(outfp, "          %s\n", date);
  fprintf(outfp, "Sender: %s <%s>\n", realname(olduser), full_id(olduser));
  if (myorg != (char *)NULL)
	fprintf(outfp, "Organization: %s\n", myorg);
  if ((sp = hfind(rfc[RF_TO])) != (char *)NULL)
			fprintf(outfp, "To: %s\n", sp);
  if ((sp = hfind(rfc[RF_SUBJECT])) != (char *)NULL)
			fprintf(outfp, "Subject: %s\n", sp);
  if ((sp = hfind(rfc[RF_XMAILER])) != (char *)NULL)
			fprintf(outfp, "X-Mailer: %s\n", sp);
  fprintf(outfp, "Message-Id: <%s>\n", mesgid());
  fprintf(outfp, "Date: %s\n", olddate);
  fprintf(outfp, "\n");
}


/*
 * Update the current header.
 * This is necessary because the message may come from a remote system
 * without RFC-conforming mailer. We should include ALL RFC-822 fields
 * in this routine (in a more robust way) !
 */
static void upd_hdr(outfp)
FILE *outfp;
{
  char uhbuf[2048];			/* temp. buffer for "From:" line */
  time_t rcvtm;				/* current time */
  char *date;				/* current date in MET */
  char *fmt1 = "%s: %s\n";
  char *fmt2 = "%s\n";
  register char *sp, *vp, *np;
  register HEADER *hp;

  /* Get the current date and time. */
  time(&rcvtm); date = maketime(&rcvtm);

  /*
   * First of all, get the Old V6/V7 From-line.
   * Note, that we must only add our name if the mail is to be forwarded
   * to another system. If it will be delivered locally, leave it.
   */
  if (oldmailer == TRUE) {
	hp = hsearch(rfc[RF_OFROM]);	/* to make it DONE */
	if (oldhost[0] != '\0') {	/* remote origin */
		if (aremote == FALSE) 	/* local delivery, just leave it! */
			fprintf(outfp, "From %s %s remote from %s\n",
					olduser, xtime(&rcvtm), oldhost);
		  else fprintf(outfp, "From %s!%s %s remote from %s\n",
				oldhost, olduser, xtime(&rcvtm), myname);
	} else fprintf(outfp, "From %s %s remote from %s\n",
				olduser, xtime(&rcvtm), myname);
  }

  /*
   * Write the modified From:-line
   * If the addressee is local, split the contents of the "From:" field
   * into a "Path:"-field and a "From:"-field. The latter contains only
   * the username-part of the return address.
   * If the addressee is remote, do not generate the "Path:"-field, but
   * add our system name to the "From:"-field value.
   */
  hp = hsearch(rfc[RF_FROM]);
  if (hp == NILHEAD) {	/* no From:-line, create one from "From_" */
	sprintf(uhbuf, "%s!%s", oldhost, olduser);
	vp = uhbuf;
	np = "From";
  } else {
	  vp = hp->value;
	  np = hp->name;
  }

  if (aremote == FALSE) {	/* local mail, add "Path:" field */
	sp = strrchr(vp, '!');
	if (sp != (char *)NULL) {
		*sp++ = '\0';	/* terminate the return path */
		fprintf(outfp, "Path: %s\n", vp);
	} else sp = vp;
	fprintf(outfp, "%s: %s\n", np, sp);
  } else {	/* Adressee is remote. add our name! */
	  fprintf(outfp, "%s: %s!%s\n", np, myname, vp);
  }

  /* Our own Received:-line. */
  fprintf(outfp, "Received: by %s.%s (UMAIL %s) with UUCP;\n",
					myname, mydomain, Version);
  if (aremote == TRUE) {	/* remote mail, add 'id' subfield */
	fprintf(outfp, "          id %s; %s\n", shortid(), date);
  } else {
	fprintf(outfp, "          %s\n", date);
  }

  /* Next, all other Received:-lines. */
  while ((hp = hsearch(rfc[RF_RECVD])) != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* The old Sender:-line. */
  hp = hsearch(rfc[RF_SENDER]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* The old Reply-To:-line. */
  hp = hsearch(rfc[RF_REPLY]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* The old Organization:-line. */
  hp = hsearch(rfc[RF_ORG]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* The old Department:-line. */
  hp = hsearch(rfc[RF_DEPT]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* Insert all unknown fields here. */
  hp = hlist;
  while (hp != NILHEAD) {
	if (hp->std == 0) {
		hp->done = 1;
		fprintf(outfp, fmt1, hp->name, hp->value);
	}
	hp = hp->next;
  }

  /* Write the To:-line too. */
  hp = hsearch(rfc[RF_TO]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* The CarbonCopy Cc:-line. */
  hp = hsearch(rfc[RF_CC]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* And the BlindCarbonCopy as well. */
  hp = hsearch(rfc[RF_BCC]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* Finally, the old Subject:-line. */
  hp = hsearch(rfc[RF_SUBJECT]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* plus the old Keywords:-line. */
  hp = hsearch(rfc[RF_KEYWORDS]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* and the old Summary:-line. */
  hp = hsearch(rfc[RF_SUMMARY]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* A possible Mailer version ID. */
  hp = hsearch(rfc[RF_XMAILER]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* A message ID. */
  hp = hsearch(rfc[RF_MESGID]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  }

  /* And the old date of sending. */
  hp = hsearch(rfc[RF_DATE]);
  if (hp != NILHEAD) {
	if (hp->count > 1) fprintf(outfp, fmt2, hp->value);
	  else fprintf(outfp, fmt1, hp->name, hp->value);
  } else fprintf(outfp, fmt1, "Date: ", date);

  /* An empty line marks the end of the header! */
  fprintf(outfp, "\n");
}


/*
 * Read the message-header into memory.
 */
static int read_hdr(infp)
register FILE *infp;
{
  char hdrbuf[1024];
  char lastf[128];
  int i, numfields = 0;
  int lastc = 1;
  register HEADER *hp, *xp;
  register char *bp, *sp;

  while (TRUE) {
	if (mfgets(hdrbuf, 1024, infp) == (char *)NULL) break;
	if (hdrbuf[0] == '\0') break;

	numfields++;
	bp = hdrbuf;

	/* First check if this is the V6/V7 From-line. */
	if (strncmp(hdrbuf, "From ", 5)) {
		/* No From-line. */
		if (*bp==' ' || *bp=='\t') {
			lastc++;	/* next part of previous field */	
			bp = lastf;	/* previous field */
			sp = hdrbuf;	/* value */
		} else {
			sp = strchr(bp, ':');	/* plain field, get sepa */
			if (sp != (char *)NULL) {	/* do we have one? */
				*sp++ = '\0';		/* end it */
				while (*sp && (*sp==' ' || *sp=='\t')) sp++;
				strcpy(lastf, bp);    /* set as prev field */
				lastc = 1;	
			} else sp = bp;	      /* no sepa, use entire field */
	  	}
	} else {
		bp = "From ";
		sp = &hdrbuf[5];
	}

	/* Add a new header field to the message header in memory. */
	hp = (HEADER *) malloc(sizeof(HEADER)); 	/* allocate new variable */
	if (hlist == NILHEAD) {			/* first variable */
    		hlist = hp;
  	} else {
          	xp = hlist;
          	while (xp->next != NILHEAD) xp = xp->next;
          	xp->next = hp;
    	}

  	hp->next = NILHEAD;
  	hp->name = (char *) malloc(strlen(bp) + 2);
  	hp->value = (char *) malloc(strlen(sp) + 2);

  	strcpy(hp->name, bp);
  	strcpy(hp->value, sp);
  	hp->done = 0;		/* not yet read */
  	hp->count = lastc;	/* folding level */
  	hp->std = 0;		/* standard field? */

	/* Now see if this field is an RFC-822 field. */
	i = 0;
  	sp = rfc[i];
	strcpy(hdrbuf, hp->name);	/* convert field name to uppercase */
	strupr(hdrbuf);
  	while (sp != (char *)NULL) {
		if (!strcmp(sp, hdrbuf)) break;
 		sp = rfc[++i];
  	}
  	if (sp != (char *)NULL) hp->std = 1;
  }
}


/*
 * Read the header from the input file 'infd', and adapt some
 * fields to the new values.
 * Then, sort the entries and generate a new header.
 * Put that new header into file 'outfp'.
 * Return TRUE if REMOTE, FALSE if LOCAL mail.
 */
int header(infp, outfp)
register FILE *infp;
register FILE *outfp;
{
  int remote;
  char *sp;

  (void) read_hdr(infp);		/* read in the current header */
	
  remote = chk_hdr(outfp);		/* analyze old header */

  if (remote == FALSE) new_hdr(outfp);	/* locally-generated mail */
    else upd_hdr(outfp);

  return(remote);
}
