/* Lower half of IP, consisting of gateway routines
 * Includes routing and options processing code
 */
#ifdef	TRACE
#include <stdio.h>
#endif

#include "machdep.h"
#include "mbuf.h"
#include "internet.h"
#include "timer.h"
#include "netuser.h"
#include "ip.h"
#include "icmp.h"
#include "iface.h"

struct route *routes[NROUTE];	/* Routing table */
struct route r_default;		/* Default route entry */

int32 ip_addr;
struct ip_stats ip_stats;

/* Route an IP datagram. This is the "hopper" through which all IP datagrams,
 * coming or going, must pass.
 *
 * This router is a temporary hack, since it only does host-specific or
 * default routing (no hierarchical routing yet).
 *
 * If "noforward" is set, the router will discard the packet unless the
 * target interface will accept it (only the local interface should do
 * this). If the destination is unreachable, no ICMP unreachables will
 * be generated. Any interfaces accepting broadcast packets should set
 * this flag.
 */
void
ip_route(bp,noforward)
struct mbuf *bp;
int noforward;
{
	register struct ip_header *ip;	/* IP header being processed */
	int16 ip_len;			/* IP header length */
	int16 buflen;			/* Length of mbuf */
	int16 length;			/* Total datagram length */
	int32 target;			/* Target IP address */
	int32 gateway;			/* Gateway IP address */
	register struct route *rp;	/* Route table entry */
	struct route *rt_lookup();
	int opi;			/* Index into options field */
	int opt_len;			/* Length of current option */
	int strict;			/* Strict source routing flag */
	struct mbuf *sbp;		/* IP header for fragmenting */
	int16 fl_offs;			/* fl_offs field of datagram */
	int16 offset;			/* Offset of fragment */
	char precedence;		/* Extracted from tos field */
	char delay;
	char throughput;
	char reliability;

	ip_stats.total++;
	buflen = len_mbuf(bp);
	if(buflen < sizeof(struct ip_header)){
		/* The packet is shorter than a legal IP header */
		ip_stats.runt++;
		free_p(bp);
		return;
	}
	ip = (struct ip_header *)bp->data;
	length = ntohs(ip->length);
	if(buflen > length){
		/* Packet has excess garbage (e.g., Ethernet padding); trim */
		if(bp->next == NULLBUF){
			/* One mbuf, just adjust count */
			bp->cnt = length;
		} else {
			struct mbuf *nbp;
			/* Copy to a new one */
			nbp = copy_p(bp,length);
			free((char *)bp);
			bp = nbp;
			ip = (struct ip_header *)bp->data;
		}
	}
	ip_len = lonibble(ip->v_ihl) * sizeof(int32);
	if(ip_len < sizeof(struct ip_header)){
		/* The IP header length field is too small */
		ip_stats.length++;
		free_p(bp);
		return;
	}
	if(cksum(NULLHEADER,bp,ip_len) != 0){
		/* Bad IP header checksum; discard */
		ip_stats.checksum++;
		free_p(bp);
		return;
	}
	if(hinibble(ip->v_ihl) != IPVERSION){
		/* We can't handle this version of IP */
		ip_stats.version++;
		free_p(bp);
		return;
	}
	/* Decrement TTL and discard if zero */
	if(--ip->ttl == 0){
		/* Send ICMP "Time Exceeded" message */
		icmp_output(bp,TIME_EXCEED,0,NULLICMP);
		free_p(bp);
		return;
	}
	/* Process options, if any. Also compute length of secondary IP
	 * header in case fragmentation is needed later
	 */
	strict = 0;
	for(opi = sizeof(struct ip_header);opi < ip_len; opi += opt_len){
		char *opt;	/* Points to current option */
		int opt_type;	/* Type of current option */
		int pointer;	/* Pointer field of current option */
		int32 *addr;	/* Pointer to an IP address field in option */

		opt = (char *)ip + opi;
		opt_type = opt[0] & OPT_NUMBER;

		/* Handle special 1-byte do-nothing options */
		if(opt_type == IP_EOL)
			break;		/* End of options list, we're done */
		if(opt_type == IP_NOOP){
			opt_len = 1;	/* No operation, skip to next option */
			continue;
		}
		/* Other options have a length field */
		opt_len = opt[1] & 0xff;

		/* Process options */
		switch(opt_type){
		case IP_SSROUTE:/* Strict source route & record route */
			strict = 1;
		case IP_LSROUTE:/* Loose source route & record route */
			/* Source routes are ignored unless the datagram appears to
			 * be for us
			 */
			if(ntohl(ip->dest) != ip_addr)
				continue;
		case IP_RROUTE:	/* Record route */
			pointer = (opt[2] & 0xff) - 1;
			if(pointer + sizeof(int32) <= opt_len){
				/* Insert our address in the list */
				addr = (int32 *)&opt[pointer];
				if(opt_type != IP_RROUTE)
					/* Old value is next dest only for source routing */
					ip->dest = *addr;
				*addr = htonl(ip_addr);
				opt[2] += 4;
			} else {
				/* Out of space; return a parameter problem and drop */
				union icmp_args icmp_args;

				icmp_args.unused = 0;
				icmp_args.pointer = sizeof(struct ip_header) + opi;
				icmp_output(bp,PARAM_PROB,0,&icmp_args);
				free_p(bp);
				return;
			}
			break;
		}
	}

	/* Note this address may have been modified by source routing */
	target = ntohl(ip->dest);

	/* Look up target address in routing table */
	if((rp = rt_lookup(target)) == NULLROUTE || rp->interface == NULLIF){
		/* Not found, use default route instead */
		rp = &r_default;
	}
	if(rp->interface == NULLIF){
		/* No route exists, return unreachable message */
		if(!noforward)
			icmp_output(bp,DEST_UNREACH,HOST_UNREACH,NULLICMP);
		else
			ip_stats.noforward++;
		free_p(bp);
		return;
	}
	/* Find gateway; zero gateway in routing table means "send direct" */
	if(rp->gateway == (int32)0)
		gateway = target;
	else
		gateway = rp->gateway;

	if(strict && gateway != target){
		/* Strict source routing requires a direct entry */
		if(!noforward)
			icmp_output(bp,DEST_UNREACH,ROUTE_FAIL,NULLICMP);
		else
			ip_stats.noforward++;
		free_p(bp);
		return;
	}
	/* If the input interface has marked this packet "don't forward",
	 * and the output interface won't accept it, then discard
	 */
	if(noforward && !(rp->interface->flags & IF_TAKENF)){
		ip_stats.noforward++;
		free_p(bp);
		return;
	}
	precedence = PREC(ip->tos);
	delay = ip->tos & DELAY;
	throughput = ip->tos & THRUPUT;
	reliability = ip->tos & RELIABILITY;

	if(length <= rp->interface->mtu){
		/* Datagram smaller than interface MTU; send normally */
		/* Recompute header checksum */
		ip->checksum = 0;
		ip->checksum = cksum(NULLHEADER,bp,ip_len);
		(*rp->interface->send)(bp,rp->interface,gateway,
			precedence,delay,throughput,reliability);
		return;
	}
	/* Fragmentation needed */
	fl_offs = ntohs(ip->fl_offs);
	if(fl_offs & DF){
		/* Don't Fragment set; return ICMP message and drop */
		icmp_output(bp,DEST_UNREACH,FRAG_NEEDED,NULLICMP);
		free_p(bp);
		return;
	}
	/* Create copy of IP header for each fragment */
	sbp = copy_p(bp,ip_len);
	pullup(&bp,NULLCHAR,ip_len);
	length -= ip_len;

	/* Create fragments */
	offset = (fl_offs & F_OFFSET) << 3;
	while(length != 0){
		int16 fragsize;		/* Size of this fragment's data */
		struct mbuf *f_header;	/* Header portion of fragment */
		struct ip_header *fip;	/* IP header */
		struct mbuf *f_data;	/* Data portion of fragment */

		f_header = copy_p(sbp,ip_len);
		fip = (struct ip_header *)f_header->data;
		fip->fl_offs = htons(offset >> 3);
		if(length + ip_len <= rp->interface->mtu){
			/* Last fragment; send all that remains */
			fragsize = length;
		} else {
			/* More to come, so send multiple of 8 bytes */
			fragsize = (rp->interface->mtu - ip_len) & 0xfff8;
			fip->fl_offs |= htons(MF);
		}
		fip->length = htons(fragsize + ip_len);
		/* Recompute header checksum */
		fip->checksum = 0;
		fip->checksum = cksum(NULLHEADER,f_header,ip_len);

		/* Extract portion of data and link in */
		f_data = copy_p(bp,fragsize);
		pullup(&bp,NULLCHAR,fragsize);
		f_header->next = f_data;

		(*rp->interface->send)(f_header,rp->interface,gateway,
			precedence,delay,throughput,reliability);
		offset += fragsize;
		length -= fragsize;
	}
	free_p(sbp);
}

/* Add an entry to the IP routing table. Returns 0 on success, -1 on failure */
int
rt_add(target,gateway,metric,interface)
int32 target;
int32 gateway;
int metric;
struct interface *interface;
{
	struct route *rp,**hp,*rt_lookup();
	int16 hash_ip();
	char *malloc();

	/* A target address of 0.0.0.0 refers to the default entry */
	if(target == 0){
		rp = &r_default;
	} else if((rp = rt_lookup(target)) == NULLROUTE){
		/* If the target is not already in the table, create a new
		 * entry and put it in. Otherwise just overwrite the old one.
		 */
		if((rp = (struct route *)malloc(sizeof(struct route))) == NULLROUTE)
			return -1;	/* No space */
		/* Insert at head of table */
		rp->prev = NULLROUTE;
		hp = &routes[hash_ip(target)];
		rp->next = *hp;
		if(rp->next != NULLROUTE)
			rp->next->prev = rp;
		*hp = rp;
	}
	rp->target = target;
	rp->gateway = gateway;
	rp->metric = metric;
	rp->interface = interface;
	return 0;
}

/* Remove an entry from the IP routing table. Returns 0 on success, -1
 * if entry was not in table.
 */
int
rt_drop(target)
int32 target;
{
	register struct route *rp;
	struct route *rt_lookup();

	if(target == 0){
		/* Nail the default entry */
		r_default.interface = NULLIF;
		return 0;
	}
	if((rp = rt_lookup(target)) == NULLROUTE)
		return -1;	/* Not in table */

	if(rp->next != NULLROUTE)
		rp->next->prev = rp->prev;
	if(rp->prev != NULLROUTE)
		rp->prev->next = rp->next;
	else
		routes[hash_ip(target)] = rp->next;

	free((char *)rp);
	return 0;
}

/* Compute hash function on IP address */
static int16
hash_ip(addr)
register int32 addr;
{
	register int16 ret;

	ret = hiword(addr);
	ret ^= loword(addr);
	ret %= NROUTE;
	return ret;
}

/* Look up target in hash table. Return NULL if not found */
static struct route *
rt_lookup(target)
int32 target;
{
	register struct route *rp;
	int16 hash_ip();

	for(rp = routes[hash_ip(target)];rp != NULLROUTE;rp = rp->next){
		if(rp->target == target)
			break;
	}
	return rp;
}
/* Internet checksum routine. The 1's complement algorithm has some nice
 * properties. If you sum a block of data in network order, you end up with
 * the proper network-order checksum regardless of the byte order of the
 * machine doing the summing! Also, summing a byteswapped block is equivalent
 * to byteswapping the sum of an unbyteswapped block. Nice optimizations here.
 */
int16
cksum(ph,bp,length)
struct pseudo_header *ph;	/* Optional IP pseudo-header, MACHINE ORDER */
struct mbuf *bp;
int16 length;
{
	register int16 *wp;
	register int32 sum;
	register int16 wcnt;
	char *buf;
	int16 cnt,tot;
	int16 carries;

	sum = 0;
	if(ph != NULLHEADER){
		sum = hiword(ph->source);
		sum += loword(ph->source);
		sum += hiword(ph->dest);
		sum += loword(ph->dest);
		sum += ph->protocol & 0xff;
		sum += ph->length;
		/* Adjust for 1's complement by doing end-around-carries */
		while((carries = (sum >> 16)) != 0)
			sum = (sum & 0xffff) + carries;
		/* Swap to network order */
		sum = htons((int16)(sum & 0xffff));
	}

	/* Now compute sum of data blocks, already in network order */
	tot = 0;
	while(bp != NULLBUF){
		cnt = min(bp->cnt,length-tot);
		buf = bp->data;
		/* check for leading odd byte */
		if(tot & 1){
			sum += ntohs((int16)(*buf++ & 0xff));
			cnt--;
			tot++;
		}
		/* Do the main bulk of the data */
		wp = (int16 *)buf;
		wcnt = cnt >> 1;
		while(wcnt--)
			sum += *wp++;

		/* Check for trailing odd byte */
		if(cnt & 1){
			buf += cnt & ~1;
			sum += ntohs((int16)((*buf << 8) & 0xff00));
		}
		tot += cnt;
		if(tot == length)
			break;
		bp = bp->next;
	}
	/* Adjust for 1's complement by doing end-around-carries */
	while((carries = (sum >> 16)) != 0)
		sum = (sum & 0xffff) + carries;

	/* All done */
	return(((int16)(~sum & 0xffff)));
}
#ifdef	TRACE
extern int16 trace;

/* Dump an IP datagram header. If it's the first fragment, also dump
 * the next layer header (if known). Dumping is controlled by the low-order
 * 4 bits of the external variable "trace":
 * Level 0: no dump
 * Level 1-2: link level dumps only
 * Level 3: IP and ARP dumps only
 * Level 4: IP header + UDP or TCP header
 * Level 5: All headers + hex dump
 */
void
ip_dump(bp)
struct mbuf *bp;
{
	void dump_tcp(),dump_udp();
	register struct ip_header *ip;
	int32 source,dest;
	int16 ip_len;
	int16 length;
	struct mbuf *tbp;
	int16 offset;
	int i;

	if(bp == NULLBUF)
		return;	
	ip = (struct ip_header *)bp->data;
	ip_len = lonibble(ip->v_ihl) * sizeof(int32);
	length = ntohs(ip->length);
	offset = (ntohs(ip->fl_offs) & F_OFFSET) << 3 ;
	source = ntohl(ip->source);
	dest = ntohl(ip->dest);
	printf("IP: %s",inet_ntoa(source));
	printf("->%s len %u ihl %u ttl %u prot %u",
		inet_ntoa(dest),length,ip_len,ip->ttl & 0xff,
		ip->protocol & 0xff);

	if(ip->tos != 0)
		printf(" tos %u",ip->tos);
	if(offset != 0 || (ntohs(ip->fl_offs) & MF))
		printf(" id %u offs %u",ntohs(ip->id),offset);

	if(ntohs(ip->fl_offs) & DF)
		printf(" DF");
	if(ntohs(ip->fl_offs) & MF)
		printf(" MF");
	if((i = cksum((struct pseudo_header *)NULL,bp,ip_len)) != 0)
		printf(" CHECKSUM ERROR (%u)",i);
	printf("\r\n");

	if((trace & 0xf) > 3){
		if(offset == 0){
			dup_p(&tbp,bp,ip_len,length - ip_len);
			switch(ip->protocol & 0xff){
			case TCP_PTCL:
				dump_tcp(tbp,source,dest);
				break;
			case UDP_PTCL:
				dump_udp(tbp,source,dest);
				break;
			case ICMP_PTCL:
				dump_icmp(tbp,source,dest);
				break;
			}
			free_p(tbp);
		}
	}
	fflush(stdout);
}
/* Dump IP routing table
 * Dest              Interface    Gateway          Metric
 * 192.001.002.003   sl0          192.002.003.004       4
 */
int
dumproute()
{
	register int i;
	extern struct route *routes[];	/* Routing table */
	extern struct route r_default;	/* Default route entry */
	register struct route *rp;

	printf("Dest              Interface    Gateway          Metric\r\n");
	if(r_default.interface != NULL){

		printf("default           ");
		printf("%-13s",r_default.interface->name);
		if(r_default.gateway != 0)
			printf("%-17s",inet_ntoa(r_default.gateway));
		else
			printf("%-17s","");
		printf("%6u\r\n",r_default.metric);
	}
	for(i=0;i<NROUTE;i++){
		for(rp = routes[i];rp != NULL;rp = rp->next){
			printf("%-18s",inet_ntoa(rp->target));
			printf("%-13s",rp->interface->name);
			if(rp->gateway != 0)
				printf("%-17s",inet_ntoa(rp->gateway));
			else
				printf("%-17s","");
			printf("%6u\r\n",rp->metric);
		}
	}
	return 0;
}
#endif
