/* IBM PC console driver (keyboard code).
 *
 * The console driver is split into two parts.  This file contains the keyboard
 * driver code.  The display driver code is contained in the file console.c.
 *
 * The keyboard acumulates characters and passes them on to the TTY task.  To
 * prevent possible race conditions two buffers are maintained.  And while the
 * TTY task processing one it is safe for the keyboard driver to be adding
 * characters to the other.
 *
 * The IBM keyboard is unusual in that keystrokes yield key numbers rather
 * than ASCII codes, and furthermore, an interrupt is generated when a key is
 * depressed (make) and again when it is released (break).  If a key is held
 * down it autorepeats by repeatedly sending the make code.
 */

#include "kernel.h"
#include <sgtty.h>
#include <signal.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "tty.h"

/* Standard and AT keyboard.  (PS/2 MCA implies AT throughout.) */
#define KEYBD		0x60	/* I/O port for keyboard data */

/* AT keyboard.  Most of these values are only used for rebooting. */
#define KB_COMMAND	0x64	/* I/O port for commands on AT */
#define KB_GATE_A20	0x02	/* bit in output port to enable A20 line */
#define KB_PULSE_OUTPUT	0xF0	/* base for commands to pulse output port */
#define KB_RESET	0x01	/* bit in output port to reset CPU */
#define KB_STATUS	0x64	/* I/O port for status on AT */

/* PS/2 model 30 keyboard. */
#define PS_KB_STATUS	0x72	/* I/O port for status on ps/2 (???) */
#define PS_KEYBD	0x68	/* I/O port for data on ps/2 */

/* AT and PS/2 model 30 keyboards. */
#define KB_ACK		0xFA	/* keyboard ack response */
#define KB_BUSY		0x02	/* status bit set when KEYBD port ready */
#define LED_CODE	0xED	/* command to keyboard to set LEDs */
#define MAX_KB_ACK_RETRIES 0x1000	/* max #times to wait for kb ack */
#define MAX_KB_BUSY_RETRIES 0x1000	/* max #times to loop while kb busy */

/* All keyboards. */
#define KBIT		0x80	/* bit used to ack characters to keyboard */

/* Scan codes whose action is not completely captured by maps. */
#define DEL_SCAN	  83	/* DEL for use in CTRL-ALT-DEL reboot */
#define DUTCH_EXT_SCAN	  32	/* 'd' */
#define ESCAPE_CODE	0xE0	/* beginning of escape sequence */
#define F1		  59	/* function key F1, others follow */
#define F2		  60
#define F3		  61
#define F4		  62
#define F5		  63
#define F6		  64
#define F7		  65
#define F8		  66
#define F9		  67
#define F10		  68
#define MINUS_DU	0x35	/* '-' on Dutch extended keybd */
#define NUM_SLASH_DU	0x57	/* numeric keypad slash on Dutch extended kb */
#define OLIVETTI_SCAN	  12	/* '=' key on olivetti */
#define SCODE1		  71	/* Home on numeric pad */
#define SCODE2		  81	/* PgDn on numeric pad */
#define STANDARD_SCAN	  13	/* '=' key on IBM */
#define TOP_ROW		  14	/* codes below this are shifted if CTRL */
#define US_EXT_SCAN	  22	/* 'u' */

#define NR_SCAN_CODES	0x69	/* Number of scan codes */

/* Miscellaneous. */
#define CTRL_S		  31	/* scan code for letter S (for CRTL-S) */
#define MEMCHECK_ADR   0x472	/* address to stop memory check after reboot */
#define MEMCHECK_MAG  0x1234	/* magic number to stop memory check */
#define KB_IBUFSIZE	  32	/* size of keyboard input buffer */

/* Macro to map a console number to a kb_s structure. */
#define kb_addr(num)	(&kb_lines[(num)])

/* Keyboard state. */
PRIVATE int alt;		/* alt key state */
PRIVATE int esc;		/* escape scan code detected? */
PRIVATE int control;		/* control key state */
PRIVATE int caps_off;		/* 1 = normal position, 0 = depressed */
PRIVATE int num_off;		/* 1 = normal position, 0 = depressed */
PRIVATE int shift1;		/* left shift key state */
PRIVATE int shift2;		/* right shift key state */

/* Scan codes to ASCII for unshifted keys for IBM-PC (default). */
PRIVATE char unsh_pc[NR_SCAN_CODES] = {
 0,033,'1','2','3','4','5','6',        '7','8','9','0','-','=','\b','\t',
 'q','w','e','r','t','y','u','i',      'o','p','[',']',015,0202,'a','s',
 'd','f','g','h','j','k','l',';',      047,0140,0200,0134,'z','x','c','v',
 'b','n','m',',','.','/',0201,'*',     0203,' ',0204,0241,0242,0243,0244,0245,
 0246,0247,0250,0251,0252,0205,0210,0267,  0270,0271,0211,0264,0265,0266,0214
,0261,0262,0263,'0',0177,0,0,0,0,	0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0
};

/* Scan codes to ASCII for shifted keys. */
PRIVATE char sh_pc[NR_SCAN_CODES] = {
 0,033,'!','@','#','$','%','^',        '&','*','(',')','_','+','\b','\t',
 'Q','W','E','R','T','Y','U','I',      'O','P','{','}',015,0202,'A','S',
 'D','F','G','H','J','K','L',':',      042,'~',0200,'|','Z','X','C','V',
 'B','N','M','<','>','?',0201,'*',    0203,' ',0204,0221,0222,0223,0224,0225,
 0226,0227,0230,0231,0232,0204,0213,'7',  '8','9',0211,'4','5','6',0214,'1',
 '2','3','0','.',0,0,0,0,		0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0
};

/* Scan codes to ASCII for Olivetti M24 for unshifted keys. */
PRIVATE char unsh_olivetti[NR_SCAN_CODES] = {
 0,033,'1','2','3','4','5','6',        '7','8','9','0','-','^','\b','\t',
 'q','w','e','r','t','y','u','i',      'o','p','@','[','\r',0202,'a','s',
 'd','f','g','h','j','k','l',';',      ':',']',0200,'\\','z','x','c','v',
 'b','n','m',',','.','/',0201,'*',     0203,' ',0204,0241,0242,0243,0244,0245,
0246,0247,0250,0251,0252,023,0210,0267,0270,0271,0211,0264,0265,0266,0214,0261,
0262,0263,'0','.',' ',014,0212,'\r',   0264,0262,0266,0270,032,0213,' ','/',
0253,0254,0255,0256,0257,0215,0216,0217
};

/* Scan codes to ASCII for Olivetti M24 for shifted keys. */
PRIVATE char sh_olivetti[NR_SCAN_CODES] = {
 0,033,'!','"','#','$','%','&',        047,'(',')','_','=','~','\b','\t',
 'Q','W','E','R' ,'T','Y','U','I',     'O','P',0140,'{','\r',0202,'A','S',
 'D','F','G','H','J','K','L','+',      '*','}',0200,'|','Z','X','C','V',
 'B','N','M','<','>','?',0201,'*',     0203,' ',0204,0221,0222,0223,0224,0225,
 0226,0227,0230,0231,0232,0270,023,'7', '8','9',0211,'4','5','6',0214,'1',
 '2','3',0207,0177,0271,014,0272,'\r', '\b','\n','\f',036,032,0273,0274,'/',
 0233,0234,0235,0236,0237,0275,0276,0277
};

PRIVATE char unsh_dutch[NR_SCAN_CODES] = {
 0,033,'1','2','3','4','5','6',		'7','8','9','0','/',0370,'\b','\t',
 'q','w','e','r','t','y','u','i',	'o','p',0,'*','\r',0202,'a','s',
 'd','f','g','h','j','k','l','+',	'\'',0100,0200,'<','z','x','c','v',
 'b','n','m',',','.','-',0201,'*',	0203,' ',0204,0,0,0,0,0,
 0,0,0,0,0,0205,0,183,			184,185,137,180,'5',182,140,177,
 178,179,0,0177,0,0,']','/',0,		0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0
};

PRIVATE char sh_dutch[NR_SCAN_CODES] = {
 0,033,'!','\"','#','$','%','&',	'_','(',')','\'','?',0176,'\b','\t',
 'Q','W','E','R','T','Y','U','I',	'O','P','^','|','\r',0202,'A','S',
 'D','F','G','H','J','K','L',0361,	0,025,0200,'>','Z','X','C','V',
 'B','N','M',';',':','=',0201,'*',	0203,' ',0204,0,0,0,0,0,
 0,0,0,0,0,0205,0,'7',			'8','9',137,'4','5','6',140,'1',
 '2','3','0',',',0,0,'[','/',0,		0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0
};

/* Code table for alt key. */
PRIVATE char altc_dutch[NR_SCAN_CODES] = {
 0,0,0,253,0,172,171,0,		156,'{','}',0,'\\',0,0,0,
 0,0,0,0,0,0,0,0,		0,0,0,0,0,0202,0,0,
 0,0,0,0,0,0,0,0,		0,170,0200,0,174,175,0,0,
 0,0,0,0,0,0,0201,0,		0203,0,0204,0,0,0,0,0,
 0,0,0,0,0,0205,0,0,		0,0,0,0,0,0,0,0,
 0,0,0,0177,0,0,'|',0,0,	0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0
};

/* Unshifted U.S. extended keyboard. */
PRIVATE char unsh_extended[NR_SCAN_CODES] = {
 0,'`','1','2','3','4','5','6',        '7','8','9','0','-','=','\b','\t',
 'q','w','e','r','t','y','u','i',      'o','p','[',']',015,0202,'a','s',
 'd','f','g','h','j','k','l',';',      047,033,0200,0134,'z','x','c','v',
 'b','n','m',',','.','/',0201,'*',     0203,' ',0202,0241,0242,0243,0244,0245,
 0246,0247,0250,0251,0252,0205,0210,0267,  0270,0271,0211,0264,0265,0266,0214
,0261,0262,0263,'0',0177,0,0,0,0,	0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0
};

/* Shifted U.S. extended keyboard. */
PRIVATE char sh_extended[NR_SCAN_CODES] = {
 0,033,'!','@','#','$','%','^',        '&','*','(',')','_','+','\b','\t',
 'Q','W','E','R','T','Y','U','I',      'O','P','{','}',015,0202,'A','S',
 'D','F','G','H','J','K','L',':',      042,'~',0200,'|','Z','X','C','V',
 'B','N','M','<','>','?',0201,'*',    0203,' ',0202,0221,0222,0223,0224,0225,
 0226,0227,0230,0231,0232,0204,0213,'7',  '8','9',0211,'4','5','6',0214,'1',
 '2','3','0','.',0,0,0,0,		0,0,0,0,0,0,0,0,
 0,0,0,0,0,0,0,0
};

PRIVATE char scode_map[] =
		{'H', 'A', 'V', 'S', 'D', 'G', 'C', 'T', 'Y', 'B', 'U'};

/* Keyboard structure, 1 per virtual console. */
struct kb_s {
  int keyb_type;		/* type of keyboard attached */
  char *unsh, *sh, *altc;	/* scan code to ASCII mapping tables, if altc
				 * is NIL then the alt key sets bit 8 */

  int capslock;			/* caps lock key state */
  int numlock;			/* numeric lock key state */

  char *ibuf;			/* start of input buffer */
  char *ibufend;		/* end of input buffer */
  char *iptr;			/* next free spot in input buffer */

  char ibuf1[KB_IBUFSIZE + 1];	/* 1st input buffer, guard at end */
  char ibuf2[KB_IBUFSIZE + 1];	/* 2nd input buffer (for swapping) */
};

/* Keyboard structures. */
PRIVATE struct kb_s kb_lines[NR_CONSOLES];

FORWARD int kb_ack();
FORWARD int kb_wait();
FORWARD int scan_keyboard();
FORWARD void set_leds();

/*===========================================================================*
 *				keyboard				     *
 *===========================================================================*/
PUBLIC void keyboard()
{
/* A keyboard interrupt has occurred.  Process it. */

  int num, minor, code, k;
  struct tty_struct *tp;
  register struct kb_s *kb;

  num = fg_console;
  kb = kb_addr(num);

  /* Fetch the character from the keyboard hardware and acknowledge it. */
  code = scan_keyboard();

  /* The IBM keyboard interrupts twice per key, once when depressed, once when
   * released.  Filter out the latter, ignoring all but the shift-type keys.
   * The shift-type keys 29, 42, 54, 56, 58, and 69 must be processed normally.
   */
  if (kb->keyb_type == DUTCH_EXT)
	if (esc) {
		/* Numeric slash gives scan codes 0xE0 0x35. */
		if (code == MINUS_DU) code = NUM_SLASH_DU;
		esc = FALSE;
	} else
		esc = (code == ESCAPE_CODE);

  k = code - 0200;		/* codes > 0200 mean key release */
  if (k > 0) {
	/* A key has been released. */
	if (k != 29 && k != 42 && k != 54 && k != 56 && k != 58 && k != 69)
		return;		/* don't call tty_task() */
  } else {
	/* Check to see if character is CTRL-S, to stop output.  Setting xoff
	 * to anything other than CTRL-S will not be detected here, but will
	 * be detected later, in the driver.  A general routine to detect any
	 * xoff character here would be complicated since we only have the
	 * scan code here, not the ASCII character.
	 */
	minor = CON_MINOR + num;
	tp = tty_addr(minor);
	if (!(tp->tty_mode & RAW) && control && code == CTRL_S &&
	    tp->tty_xoff == XOFF_CHAR) {
		tp->tty_inhibited = STOPPED;
		return;
	}
  }

  /* Call debugger? Do as early as practical, not in TTY which may be hung. */
  if (code == F10 && db_exists) {
	db_enabled = TRUE;
	db();
  }

  /* Store the character in memory for the TTY task to get later. */
  *kb->iptr = code;
  if (kb->iptr < kb->ibufend) {
	lock();			/* protect shared variable */
	tty_events += EVENT_THRESHOLD;	/* C doesn't guarantee atomicity */
	unlock();
	++kb->iptr;
  }

  /* Else it doesn't fit - discard it. */
}


/*==========================================================================*
 *				kb_read					    *
 *==========================================================================*/
PUBLIC int kb_read(num, bufindirect, odoneindirect)
int num;
char **bufindirect;
unsigned char *odoneindirect;
{
/* Swap the keyboard input buffers, giving the old one to TTY. */

  register char *ibuf;
  register struct kb_s *kb;
  int nread;

  kb = kb_addr(num);
  *odoneindirect = FALSE;
  if (kb->iptr == (ibuf = kb->ibuf)) return 0;
  *bufindirect = ibuf;
  lock();
  nread = kb->iptr - ibuf;
  tty_events -= nread * EVENT_THRESHOLD;
  if (ibuf == kb->ibuf1)
	ibuf = kb->ibuf2;
  else
	ibuf = kb->ibuf1;
  kb->ibufend = ibuf + KB_IBUFSIZE;
  kb->iptr = ibuf;
  unlock();
  kb->ibuf = ibuf;
  return nread;
}


/*===========================================================================*
 *				letter_code				     *
 *===========================================================================*/
PUBLIC int letter_code(num, scode)
int num;
int scode;			/* scan code from key press */
{
/* Convert scan codes from numeric keypad to letters for use in escape seqs. */

  register struct kb_s *kb;

  kb = kb_addr(num);
  if (scode >= SCODE1 && scode <= SCODE2 && (shift1 || shift2 || !kb->numlock))
	return scode_map[scode - SCODE1];
  else
	return 0;
}


/*===========================================================================*
 *				make_break				     *
 *===========================================================================*/
PUBLIC int make_break(num, ch)
int num;
char ch;			/* scan code of key just struck or released */
{
/* Convert scan code to character code.  The scan code might indicate the key
 * was released, or the key might be a shift key.  If this is the case the
 * value -1 will be returned.
 *
 * This routine can handle keyboards that interrupt only on key depression,
 * as well as keyboards that interrupt on key depression and key release.
 * For efficiency, the interrupt routine filters out most key releases.
 */

  int c, make, code;
  register struct kb_s *kb;

  kb = kb_addr(num);

  /* Check for CTRL-ALT-DEL, and if found, reboot the computer. This would
   * be better done in keyboard() in case TTY is hung, except control and
   * alt are set in the high level code.
   */
  if (control && alt && ch == DEL_SCAN) reboot();

  c = ch & 0177;		/* high-order bit set on key release */
  make = (ch & 0200 ? 0 : 1);	/* 1 when key depressed, 0 when key released */

  if (alt && kb->altc != (char *) NIL_PTR)
	code = kb->altc[c];
  else
	code = (shift1 || shift2 ? kb->sh[c] : kb->unsh[c]);

  if (control && c < TOP_ROW) code = kb->sh[c];	/* CTRL-(top row) */

  if (c >= SCODE1 && c <= SCODE2 + 2)	/* numeric pad including DEL, INS */
	code = (shift1 || shift2 || !kb->numlock ? kb->unsh[c] : kb->sh[c]);

  code &= BYTE;
  if (code < 0200 || code >= 0206) {
	/* Ordinary key, i.e. not shift, control, alt, etc. */
	if (kb->capslock)
		if (code >= 'A' && code <= 'Z')
			code += 'a' - 'A';
		else if (code >= 'a' && code <= 'z')
			code -= 'a' - 'A';
	/* Set bit 8 in character code if alt, and no alt map is present. */
	if (alt && kb->altc == (char *) NIL_PTR) code |= 0200;
	if (control) code &= 037;
	if (make == 0) code = -1;	/* key release */
	return(code);
  }

  /* Table entries 0200 - 0206 denote special actions. */
  switch (code - 0200) {
    case 0:	shift1 = make;		break;	/* shift key on left */
    case 1:	shift2 = make;		break;	/* shift key on right */
    case 2:
#if KEYBOARD_84
/* Until IBM invented the 101-key keyboard, the CTRL key was always to the
 * left of the 'A'.  This fix puts it back there on the 101-key keyboard.
 */
		/* Don't toggle caps lock every time a make is received, only
		 * toggle if the previous caps lock code was a break.  This
		 * prevents the caps lock key from autorepeating.
		 */
		if (make && caps_off) {
			kb->capslock = 1 - kb->capslock;
			set_leds(kb);
		}
		caps_off = 1 - make;    break;	/* caps lock */
#else
		control = make;		break;	/* control */
#endif
    case 3:	alt = make;		break;	/* alt key */
    case 4:	
#if KEYBOARD_84
		control = make;		break;	/* control */
#else
		/* Don't toggle caps lock every time a make is received, only
		 * toggle if the previous caps lock code was a break.  This
		 * prevents the caps lock key from autorepeating.
		 */
		if (make && caps_off) {
			kb->capslock = 1 - kb->capslock;
			set_leds(kb);
		}
		caps_off = 1 - make;    break;	/* caps lock */
#endif
    case 5:	if (make && num_off) {
			kb->numlock = 1 - kb->numlock;
			set_leds(kb);
		}
		num_off = 1 - make;	break;	/* num lock */
  }
  return(-1);
}


/*===========================================================================*
 *				change_console				     *
 *===========================================================================*/
PRIVATE void change_console(num)
int num;
{
/* Bring virtual console num to the foreground, or cycle to next console if
 * num is -1.
 */

  int old_console;

  if (num >= NR_CONSOLES) return;

  old_console = fg_console;

  /* Value of fg_console is accessed asynchronously by keyboard() interrupt
   * routine, need to perform lock because update can't be guaranteed to be
   * atomic.
   */
  lock();
  if (num == -1) {
	fg_console++;
	if (fg_console == NR_CONSOLES) fg_console = 0;
  } else
	fg_console = num;
  unlock();

  /* Set leds and display for new console. */
  set_leds(kb_addr(fg_console));
  swap_screen(old_console);
}


/*===========================================================================*
 *				set_leds				     *
 *===========================================================================*/
PRIVATE void set_leds(kb)
register struct kb_s *kb;
{
/* Set the LEDs on the caps lock and num lock keys. */

  int leds, data_port, status_port;

  if (!pc_at && !ps) return;	/* PC/XT doesn't have LEDs */
  leds = (kb->numlock << 1) | (kb->capslock << 2);	/* encode LED bits */


  if (ps) {
	data_port = PS_KEYBD;
	status_port = PS_KB_STATUS;
  } else {
	data_port = KEYBD;
	status_port = KB_STATUS;
  }

  kb_wait(status_port);		/* wait for buffer empty  */
  out_byte(data_port, LED_CODE);/* prepare keyboard to accept LED values */
  kb_ack(data_port);		/* wait for ack response  */

  kb_wait(status_port);		/* wait for buffer empty  */
  out_byte(data_port, leds);	/* give keyboard LED values */
  kb_ack(data_port);		/* wait for ack response  */
}


/*==========================================================================*
 *				kb_wait					    *
 *==========================================================================*/
PRIVATE int kb_wait(status_port)
int status_port;
{
/* Wait until the controller is ready; return zero if wait times out. */

  int retries;

  retries = MAX_KB_BUSY_RETRIES + 1;
  while (--retries != 0 && in_byte(status_port) & KB_BUSY)
	;			/* wait until not busy */
  return(retries);		/* nonzero if ready */
}


/*==========================================================================*
 *				kb_ack					    *
 *==========================================================================*/
PRIVATE int kb_ack(data_port)
int data_port;
{
/* Wait until keyboard acknowledges last command; return zero if wait times
 * out.
 */

  int retries;

  retries = MAX_KB_ACK_RETRIES + 1;
  while (--retries != 0 && in_byte(data_port) != KB_ACK)
	;			/* wait for ack */
  return(retries);		/* nonzero if ack received */
}


/*===========================================================================*
 *				kb_init					     *
 *===========================================================================*/
PUBLIC void kb_init(num)
int num;
{
/* Initialize the keyboard driver for the specified device.  This function must
 * be called, in order, for each of the console devices present.
 */

  register struct kb_s *kb;

  kb = kb_addr(num);

  /* Set up input queue. */
  kb->ibuf = kb->ibuf1;
  kb->ibufend = kb->ibuf1 + KB_IBUFSIZE;
  kb->iptr = kb->ibuf1;

  load_keyboard(num, BOOT_KBD);		/* load key map table */

  /* If the last virtual console is being initialized then initialize shared
   * information and the keyboard itself.
   */
  if (num == NR_CONSOLES - 1) {
	/* Set initial values. */
	fg_console = 0;
	caps_off = 1;
	num_off = 1;

	set_leds(kb_addr(fg_console));	/* turn off leds */

	scan_keyboard();	/* stop lockup from leftover keystroke */
	enable_irq(KEYBOARD_IRQ);	/* safe now everything initialised */
  }
}


/*===========================================================================*
 *				load_keyboard				     *
 *===========================================================================*/
PUBLIC void load_keyboard(num, keyboard_type)
int num;
int keyboard_type;	/* new keyboard type */
{
/* Set keyboard type to keyboard_type. */

  register struct kb_s *kb;

  kb = kb_addr(num);

  if (keyboard_type == BOOT_KBD) {
	/* Determine which keyboard type is attached.  The bootstrap program
	 * asked the user to type an '='.  The scan codes for '=' differ
	 * depending on the keyboard in use.
	*/
	switch (scan_code) {
		case STANDARD_SCAN:	keyboard_type = IBM_PC;		break;
		case OLIVETTI_SCAN:	keyboard_type = OLIVETTI;	break;
		case DUTCH_EXT_SCAN:	keyboard_type = DUTCH_EXT;	break;
		case US_EXT_SCAN:	keyboard_type = US_EXT;		break;
	}
  }

  /* Value of kb->keyb_type is accessed asynchronously by keyboard() interrupt
   * routine, need to perform lock because update can't be guaranteed to be
   * atomic.
   */
  lock();
  kb->keyb_type = keyboard_type;
  unlock();

  switch (keyboard_type) {
	case IBM_PC:
		kb->unsh = unsh_pc;
		kb->sh = sh_pc;
		kb->altc = (char *) NIL_PTR;
		break;

	case OLIVETTI:
		kb->unsh = unsh_olivetti;
		kb->sh = sh_olivetti;
		kb->altc = (char *) NIL_PTR;
		break;

	case DUTCH_EXT:
		kb->unsh = unsh_dutch;
		kb->sh = sh_dutch;
		kb->altc = altc_dutch;
		break;

	case US_EXT:
		kb->unsh = unsh_extended;
		kb->sh = sh_extended;
		kb->altc = (char *) NIL_PTR;
		break;
  }
}


/*===========================================================================*
 *				func_key				     *
 *===========================================================================*/
PUBLIC int func_key(ch)
char ch;			/* scan code for a function key */
{
/* This procedure traps function keys for debugging and control purposes. */

  int minor;

  if (ch < F1 || ch > F10) return(FALSE);	/* not our job */

  if (alt || shift1 || shift2) return(TRUE);	/* ignore these keys */

  if (control) {
	/* Ctrl-FN */
#if AM_KERNEL
#if !NONET
	if (ch == F4) net_init();	/* re-initialise the ethernet card */
#endif
	if (ch == F5) amdump();		/* dump amoeba statistics */
#endif /* AM_KERNEL */

	minor = CON_MINOR + fg_console;
	if (ch == F8) sigchar(tty_addr(minor), SIGINT);
	if (ch == F9) sigchar(tty_addr(minor), SIGKILL);
  } else {
	/* Normal FN */
	if (ch == F1) change_console(0);	/* virtual console 0 */
	if (ch == F2) change_console(1);	/* virtual console 1 */
	if (ch == F3) change_console(2);	/* virtual console 2 */
	if (ch == F4) change_console(3);	/* virtual console 3 */
	if (ch == F5) change_console(-1);	/* cycle virtual console */

	if (ch == F6) toggle_scroll();	/* hardware vs. software scrolling */

	if (ch == F7) p_dmp();			/* print process table */
	if (ch == F8) map_dmp();		/* print memory map */
  }

  return(TRUE);
}


/*==========================================================================*
 *				scan_keyboard				    *
 *==========================================================================*/
PRIVATE int scan_keyboard()
{
/* Fetch the character from the keyboard hardware and acknowledge it. */

  int code;
  int val;

  if (ps) {
	code = in_byte(PS_KEYBD);	/* get the scan code for key struck */
	val = in_byte(0x69);	/* acknowledge it in mysterious ways */
	out_byte(0x69, val ^ 0x10);	/* 0x69 should be equiv to PORT_B */
	out_byte(0x69, val);	/* XOR looks fishy */
	val = in_byte(0x66);	/* what is 0x66? */
	out_byte(0x66, val & ~0x10);	/* 0x72 for PS_KB_STATUS is fishier */
	out_byte(0x66, val | 0x10);
	out_byte(0x66, val & ~0x10);
  } else {
	code = in_byte(KEYBD);	/* get the scan code for the key struck */
	val = in_byte(PORT_B);	/* strobe the keyboard to ack the char */
	out_byte(PORT_B, val | KBIT);	/* strobe the bit high */
	out_byte(PORT_B, val);	/* now strobe it low */
  }
  return code;
}


/*==========================================================================*
 *				reboot					    *
 *==========================================================================*/
PUBLIC void reboot()
{
/* Reboot the machine. */

  static u16_t magic = MEMCHECK_MAG;

  lock();
  eth_stp();			/* stop ethernet (may be unnecessary) */

  /* Stop BIOS memory test. */
  phys_copy(numap(TTY, (vir_bytes) (&magic), sizeof magic),
	  (phys_bytes) MEMCHECK_ADR, (phys_bytes) sizeof magic);
  if (protected_mode) {
	/* Rebooting is nontrivial because the BIOS reboot code is in real
	 * mode and there is no sane way to return to real mode on 286's.
	 */
	if (pc_at) {
		/* Use the AT keyboard controller to reset the processor.
		 * The A20 line is kept enabled in case this code is ever
		 * run from extended memory, and because some machines
		 * appear to drive the fake A20 high instead of low just
		 * after reset, leading to an illegal opode trap.  This bug
		 * is more of a problem if the fake A20 is in use, as it
		 * would be if the keyboard reset were used for real mode.
		 */
		kb_wait();
		out_byte(KB_COMMAND,
			 KB_PULSE_OUTPUT | (0x0F & ~(KB_GATE_A20 | KB_RESET)));
	} else {
		printf("No way to reboot from protected mode on this machine ");
	}
	while (TRUE)
		;		/* no way to recover if the above fails */
  }

  /* In real mode, jumping to the reset address is good enough. */
  reset();
}


/*==========================================================================*
 *				wreboot					    *
 *==========================================================================*/
PUBLIC void wreboot()
{
/* Wait for a keystroke, then reboot the machine.  Don't rely on interrupt
 * to provide the keystroke, since this is usually called after a crash,
 * and possibly before interrupts are initialized.
 */

  int scancode;

  lock();
  milli_delay(1000);		/* pause for a second to ignore key release */
  scan_keyboard();		/* ack any old input */
  printf("Type any key to reboot\r\n");
  scancode = scan_keyboard();	/* quiescent value (0 on PC, last code on AT)*/
  while (scan_keyboard() == scancode)	/* loop until new keypress or */
	;				/* any release */
  reboot();
}
