/* Code and data for the DBCS console driver.		Auther:takamiti@mix */

#include "kernel.h"
#include <sgtty.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/boot.h>
#include <minix/font.h>
#include "protect.h"
#include "tty.h"
#include "vga.h"

/* Definitions used by the console driver. */
#define GO_FORWARD         0	/* scroll forward */
#define GO_BACKWARD        1	/* scroll backward */
#define	ON		   1
#define OFF		   0

/* Beeper. */
#define BEEP_FREQ     0x0533	/* value to put into timer to set beep freq */
#define B_TIME		   3	/* length of CTRL-G beep is ticks */

/* Map from ANSI colors */
PUBLIC int ansi_colors[8] = 
	{BLACK, RED, GREEN, YELLOW, BLUE, MAGENTA, CYAN, WHITE};

PRIVATE int palet[16] =
	{_BLACK, _RED,  _GREEN,  _YELLOW, _BLUE,  _MAGENTA,  _CYAN,  L_GRAY,
	 _BLACK, L_RED, L_GREEN, _BROWN,  L_BLUE, L_MAGENTA, L_CYAN, _WHITE};

/* Private variables used by the console driver. */
PUBLIC unsigned char fontbuf[FONTBUFSIZE];
PUBLIC char ankfont[ANK_FONT_BUFF];
PUBLIC int disp_stat, softscroll;

PRIVATE phys_bytes fontptr;
PRIVATE int cur_stat, ank_tbl;
PRIVATE int one_con_attribute, fg_color, bg_color;
PRIVATE int save_column, save_row, save_fg_attr, save_bg_attr, save_disp;

FORWARD void beep();
FORWARD void do_escape();
FORWARD void move_to();
FORWARD void parse_escape();
FORWARD void stop_beep();
FORWARD void store_queue();
FORWARD void cursor();
FORWARD void swap();
FORWARD int false();
FORWARD long nothing();

extern	int VIDEO_SEG;
extern	void draw_ank();
extern	void draw_dbc();
extern	void scr_up();
extern	void scr_down();
extern	void clr_scr();
extern	void clr_line();
extern	void cur_off();
extern	void cur_on();
extern	void mov_left();
extern	void mov_right();

#if (LANGUAGE == JAPANESE)
FORWARD int mskanji(), euckanji();
FORWARD	long ms2index(), euc2index();	
extern	int ms2jis();
struct font_table *fnt = &sys_font[2];
#endif

struct dbcs_func {
	int (*isdbc)();
	long (*code2idx)();
} dbcs_func[]  = {
	false, nothing,
#if (LANGUAGE == JAPANESE)
	mskanji, ms2index,
	euckanji, euc2index
#endif
};

#ifdef GRAPHICS
#define PLOT_BACKGROUND	0
#define gr_color	one_con_attribute
PRIVATE int gr_x0, gr_y0, gr_x1, gr_y1;
PRIVATE int gr_mode, gr_state, gr_esc, gr_rot = 0;
PRIVATE int gr_ltype = 0xFFFF;	/* line type */
FORWARD void gr_line();
FORWARD void gr_char();
extern	void gr_dot();
#endif

/*===========================================================================*
 *				console					     *
 *===========================================================================*/
PUBLIC void console(tp)
register struct tty_struct *tp;	/* tells which terminal is to be used */
{
/* Copy as much data as possible to the output queue, then start I/O.  On
 * memory-mapped terminals, such as the IBM console, the I/O will also be
 * finished, and the counts updated.  Keep repeating until all I/O done.
 */

  int count;
  int remaining;
  register char *tbuf;

  /* Check quickly for nothing to do, so this can be called often without
   * unmodular tests elsewhere.
   */
  if ((remaining = tp->tty_outleft) == 0 || tp->tty_inhibited == STOPPED)
	return;

  /* Copy the user bytes to tty_buf for decent addressing. Loop over the
   * copies, since the user buffer may be much larger than tty_buf.
   */
  do {
	if (remaining > sizeof(tty_buf)) remaining = sizeof(tty_buf);
	phys_copy(tp->tty_phys, tty_bphys, (phys_bytes)remaining);
	tbuf = tty_buf;

	/* Output each byte of the copy to the screen.  This is the inner loop
	 * of the output routine, so attempt to make it fast. A more serious
	 * attempt should produce
	 *	while (easy_cases-- != 0 && *tbuf >= ' ') {
	 *		*outptr++ = *tbuf++;
	 *		*outptr++ = one_con_attribute;
	 *	}
	 */
	do {
		if (*tbuf < ' ' || tp->tty_esc_state > 0 ||
			tp->tty_column >= LINE_WIDTH - 1 ||
			tp->tty_rwords >= TTY_RAM_WORDS  ||
			tp->tty_1stbyte != 0		 ||
			(*dbcs_func[DispCode].isdbc)((unsigned char)*tbuf) ) {
				out_char(tp, (unsigned char)*tbuf++);
		} else {
			tp->tty_ramqueue[tp->tty_rwords] = (u16_t)*tbuf++;
			tp->tty_atrqueue[tp->tty_rwords] = one_con_attribute;
			tp->tty_rwords++;
			tp->tty_column++;
		}
	} while (--remaining != 0 && tp->tty_inhibited == RUNNING);

	/* Update terminal data structure. */
	count = tbuf - tty_buf;	/* # characters printed */
	tp->tty_phys += count;	/* advance physical data pointer */
	tp->tty_cum += count;	/* number of characters printed */
	tp->tty_outleft -= count;
	if (remaining != 0) break;	/* inhibited while in progress */
  } while ((remaining = tp->tty_outleft) != 0 && tp->tty_inhibited == RUNNING);
  flush(tp);			/* transfer anything buffered to the screen */

  /* If output was not inhibited early, send the appropiate completion reply.
   * Otherwise, let TTY handle suspension.
   */
  if (tp->tty_outleft == 0) finish(tp, tp->tty_cum);
}


/*===========================================================================*
 *				out_char				     *
 *===========================================================================*/
PUBLIC void out_char(tp, c)
register struct tty_struct *tp;	/* pointer to tty struct */
unsigned char c;		/* character to be output */
{
  u16_t wc;

/* Output a character on the console.  Check for escape sequences first. */
  if (tp->tty_esc_state > 0) {
	parse_escape(tp, c);
	return;
  }

  switch(c) {
	case 000:		/* null is typically used for padding */
		tp->tty_1stbyte = tp->tty_2ndbyte = 0;
		return;		/* better not do anything */

	case 007:		/* ring the bell */
		flush(tp);	/* print any chars queued for output */
		beep();
		return;

	case 013:		/* CTRL-K */
		move_to(tp, tp->tty_column, tp->tty_row - 1);
		return;

	case 014:		/* CTRL-L */
		move_to(tp, tp->tty_column + 1, tp->tty_row);
		return;

	case 016:		/* CTRL-N */
		move_to(tp, tp->tty_column, tp->tty_row + 1);
		return;

	case '\b':		/* backspace */
		move_to(tp, tp->tty_column - 1, tp->tty_row);
		return;

	case '\n':		/* line feed */
		if (tp->tty_mode & CRMOD) move_to(tp, 0, tp->tty_row);
		if (tp->tty_row == SCR_LINES-1)
			scroll_screen(tp, GO_FORWARD);
		else
			tp->tty_row++;
		move_to(tp, tp->tty_column, tp->tty_row);
		return;

	case '\r':		/* carriage return */
		move_to(tp, 0, tp->tty_row);
		return;

	case '\t':		/* tab */
		tp->tty_1stbyte = tp->tty_2ndbyte = 0;
		if ( (tp->tty_mode & XTABS) == XTABS) {
		    do {
			if (tp->tty_column >= LINE_WIDTH - 1 ||
			    tp->tty_rwords >= TTY_RAM_WORDS) {
				out_char(tp, ' ');
			} else {
			    tp->tty_ramqueue[tp->tty_rwords]= (u16_t)' ';
			    tp->tty_atrqueue[tp->tty_rwords]=one_con_attribute;
			    tp->tty_rwords++;
			    tp->tty_column++;
			}
		    } while (tp->tty_column & TAB_MASK);
		    return;
		}
		/* Ignore tab if XTABS is off--video RAM has no hardware tab */
		return;

	case 033:		/* ESC - start of an escape sequence */
		flush(tp);	/* print any chars queued for output */
		tp->tty_esc_state = 1;	/* mark ESC as seen */
		return;

#ifdef GRAPHICS
	case 034:	/* FS (point plot mode) */
		gr_mode = 1;
		gr_state = 0;
		tp->tty_esc_state = 10;	/* graphics */
		return;

	case 035:	/* GS (vector mode) */
		gr_mode = 2;
		gr_state = 0;
		tp->tty_esc_state = 10;	/* graphics */
		return;

	case 036:	/* RS (incremental plot mode) */
		gr_mode = 3;	/* 4 for draw */
		gr_state = 0;
		tp->tty_esc_state = 10;	/* graphics */
		return;

	case 037:	/* US (alpha mode) */
		gr_mode = 5;
		gr_state = 0;
		tp->tty_esc_state = 11;	/* alpha-graphics */
		return;
#endif	/* GRAPHICS */

	default:		/* printable chars are stored in ramqueue */
#if !LINEWRAP
		if (tp->tty_column >= LINE_WIDTH) return;	/* long line */
#endif
		if (tp->tty_rwords == TTY_RAM_WORDS) flush(tp);

		if (tp->tty_1stbyte == 0) {
		    if ( (DispCode != ANK_ONLY) &&
			 (*dbcs_func[DispCode].isdbc)(c) ) {
			tp->tty_1stbyte = (u16_t)c;
			return;
		    } else {
			wc = (u16_t)c;
		    }
		} else {
#if (LANGUAGE == JAPANESE)
		    if (DispCode == JIS_MS)
			wc = ((tp->tty_1stbyte << 8) | c);
		    else
#endif
		    switch (tp->tty_1stbyte) {
			    case DBCS_SS_2:
				wc = (u16_t)c;
				break;
			    case DBCS_SS_3:
				if (tp->tty_2ndbyte == 0) {
				    tp->tty_2ndbyte = (u16_t)c;
				    return;
				}
				wc = ((tp->tty_2ndbyte << 8) | c) & 0xff7f;
				tp->tty_2ndbyte = 0;
				break;
			    default:
				wc = ((tp->tty_1stbyte << 8) | c) & 0x7f7f;
		    }
		    tp->tty_1stbyte = 0;

		    if (tp->tty_column >= LINE_WIDTH - 1) {
			if (tp->tty_column == LINE_WIDTH - 1) {
			    tp->tty_ramqueue[tp->tty_rwords]=(u16_t)' ';
			    tp->tty_atrqueue[tp->tty_rwords]=one_con_attribute;
			    tp->tty_rwords++;
			    tp->tty_column++;
		        }
#if LINEWARP
		        flush(tp);
			if (tp->tty_row == SCR_LINES - 1)
			    scroll_screen(tp, GO_FORWARD);
			else
			    tp->tty_row++;
			move_to(tp, 0, tp->tty_row);
#else
			return;
#endif
		    }
		}
		tp->tty_ramqueue[tp->tty_rwords] = wc;
		tp->tty_atrqueue[tp->tty_rwords] = one_con_attribute;
		tp->tty_rwords++;
		tp->tty_column++;
		if (wc > 0xff) tp->tty_column++;
#if LINEWARP
		if (tp->tty_column >= LINE_WIDTH) {
		    flush(tp);
		    if (tp->tty_row == SCR_LINES - 1)
			scroll_screen(tp, GO_FORWARD);
		    else
			tp->tty_row++;
		    move_to(tp, 0, tp->tty_row);
		}
#endif
		return;
  } /* of switch */
}

/*===========================================================================*
 *				scroll_screen				     *
 *===========================================================================*/
PRIVATE scroll_screen(tp, dir)
register struct tty_struct *tp;	/* pointer to tty struct */
int dir;			/* GO_FORWARD or GO_BACKWARD */
{
  int y;

  flush(tp);
  cursor(tp, OFF);
  if (dir == GO_FORWARD) {
	scr_up();	y = SCR_LINES - 1;
  } else {
  	scr_down();	y = 0;
  }
  clr_line(0, y, LINE_WIDTH);
  cursor(tp, ON);
}


/*===========================================================================*
 *				flush					     *
 *===========================================================================*/
PUBLIC void flush(tp)
register struct tty_struct *tp;	/* pointer to tty struct */
{
/* Have the characters in 'ramqueue' transferred to the screen. */
  int i, x, *attr;
  u16_t *wc;

  tp->tty_1stbyte = tp->tty_2ndbyte = 0;
  if (tp->tty_rwords == 0) return;

  cursor(tp, OFF);
  x = (tp->tty_vid >> 8) & BYTE;
  tp->tty_vid &= BYTE;	
  pixel_addr(x, tp->tty_vid);
  wc = tp->tty_ramqueue;
  attr = tp->tty_atrqueue;
  for(i = 0; i < tp->tty_rwords; i++, wc++, attr++) {
	if ((*wc & 0xff00) && (DispCode != ANK_ONLY)) {
#if INTEL_32BITS
	    draw_dbc(*attr, fnt, (*dbcs_func[DispCode].code2idx)(*wc));
#else
	    phys_copy((*dbcs_func[DispCode].code2idx)(*wc), fontptr, (phys_bytes)fnt->fn_size);
	    draw_dbc(*attr, fnt);
#endif
	    x += 2;
	} else {
	    draw_ank(*attr, *wc);
	    x++;
	}
  }

  /* Update the video parameters and cursor. */
  cursor(tp, ON);
  tp->tty_vid |= (x << 8);
  tp->tty_rwords = 0;
}


/*===========================================================================*
 *				move_to					     *
 *===========================================================================*/
PRIVATE void move_to(tp, x, y)
struct tty_struct *tp;		/* pointer to tty struct */
int x;				/* column (0 <= x <= 79) */
int y;				/* row (0 <= y <= 29, 0 at top) */
{
/* Move the cursor to (x, y). */

  flush(tp);			/* flush any pending characters */
  if (x < 0 || x >= LINE_WIDTH || y < 0 || y >= SCR_LINES) return;

  tp->tty_vid = (x << 8) | y;
  tp->tty_column = x;		/* set x co-ordinate */
  tp->tty_row = y;		/* set y co-ordinate */
  cursor(tp, ON);
}


/*===========================================================================*
 *				cursor					     *
 *===========================================================================*/
PRIVATE void cursor(tp, cond)
struct tty_struct *tp;
int cond;
{
  if ((cur_stat & CUR_ACTIVE) == 0) return;
  if (cur_stat & CUR_DISP) cur_off();
  if (cond == OFF) {
	cur_stat &= ~CUR_DISP;
	return;
  }
  cur_on(tp->tty_column, tp->tty_row);
  cur_stat |= CUR_DISP;
}


/*===========================================================================*
 *				parse_escape				     *
 *===========================================================================*/
PRIVATE void parse_escape(tp, c)
register struct tty_struct *tp;	/* pointer to tty struct */
char c;				/* next character in escape sequence */
{
/* The following ANSI escape sequences are currently supported.
 * If n and/or m are omitted, they default to 1.
 *   ESC [nA moves up n lines
 *   ESC [nB moves down n lines
 *   ESC [nC moves right n spaces
 *   ESC [nD moves left n spaces
 *   ESC [m;nH moves cursor to (m,n)
 *   ESC [J clears screen from cursor
 *   ESC [K clears line from cursor
 *   ESC [nL inserts n lines ar cursor
 *   ESC [nM deletes n lines at cursor
 *   ESC [nP deletes n chars at cursor
 *   ESC [n@ inserts n chars at cursor
 *   ESC [nm enables rendition n (0=normal, 1=bold, 5=blinking, 7=reverse)
 *   ESC M scrolls the screen backwards if the cursor is on the top line
 */

  switch (tp->tty_esc_state) {
	case 1: 		/* ESC seen */
		tp->tty_esc_intro = '\0';
		tp->tty_esc_parmp = tp->tty_esc_parmv;
		tp->tty_esc_parmv[0] = tp->tty_esc_parmv[1] = 0;
		/* tp->tty_esc_strp = tp->tty_esc_strv; */
		switch (c) {
		  case '[': 	/* Control Sequence Introducer */
			tp->tty_esc_intro = c;
			tp->tty_esc_state = 2; 
			break;
		  case 'M': 	/* Reverse Index */
			do_escape(tp, c);
			break;
		  default: 
			tp->tty_esc_state = 0; 
			break;
		}
		break;

	case 2: 		/* ESC [ seen */
		if (c == '>') {
		    /* *(tp->tty_esc_strp++) = c; */
		    tp->tty_esc_state = 3;
		    break;
		}
		if (c >= '0' && c <= '9') {
		    if (tp->tty_esc_parmp < tp->tty_esc_parmv + MAX_ESC_PARMS)
			*tp->tty_esc_parmp = *tp->tty_esc_parmp * 10 + (c-'0');
			break;
		}
		else if (c == ';') {
		    if (++tp->tty_esc_parmp < tp->tty_esc_parmv+MAX_ESC_PARMS)
			*tp->tty_esc_parmp = 0;
		    break;
		}
		else {
		    do_escape(tp, c);
		}
		break;

	case 3:
		if (c == '5') {
		    /* *(tp->tty_esc_strp++) = c; */
		    tp->tty_esc_state = 4;
		    break;
		}

	case 4:
		switch (c) {
		    case 'l':		/* cursor on */
				cur_stat |= CUR_ACTIVE;
				cursor(tp, ON);
				break;

		    case 'h':		/* cursor off */
				cursor(tp, OFF);
				cur_stat = 0;
				break;

		    default:  tp->tty_esc_state = 0;
			      break;
		}
		break;

#ifdef GRAPHICS
	/* GRAPHICS mode */
	case 10:	/* point plot, vector, incremental plot mode */
	case 11:	/* alpha mode */
		if (gr_esc) {
			switch (c) {
			case '`':
				gr_ltype = 0xFFFF;
				break;
			case 'a':
				gr_ltype = 0x5555;
				break;
			case 'b':
				gr_ltype = 0x5F5F;
				break;
			case 'c':
				gr_ltype = 0x7777;
				break;
			case 'd':
				gr_ltype = 0x7F7F;
				break;
			}
			gr_esc = 0;
			return;
		}
		else if (c < 0x20) {
			switch (c) {
			case 007:	/* BEL */
				if (tp->tty_esc_state == 10)
					gr_mode++;
				else
					beep();
				break;
			case 030:	/* CAN */
				tp->tty_esc_state = gr_mode = gr_state = 0;
				break;
			case 033:	/* ESC */
				gr_esc = 1;
				break;
			case 034:	/* FS (point plot mode) */
				gr_mode = 1;
				gr_state = 0;
				tp->tty_esc_state = 10;	/* graphics */
				break;
			case 035:	/* GS (vector mode) */
				gr_mode = 2;
				gr_state = 0;
				tp->tty_esc_state = 10;	/* graphics */
				break;
			case 036:	/* RS (incremental plot mode) */
				gr_mode = 3;	/* 4 for draw */
				gr_state = 0;
				tp->tty_esc_state = 10;	/* graphics */
				break;
			case 037:	/* US (alpha mode) */
				gr_mode = 5;
				gr_state = 0;
				tp->tty_esc_state = 11;	/* alpha-graphics */
				break;
			default:
				if (tp->tty_esc_state == 11)
					goto alpha;
			}
			return;
		}
		if (tp->tty_esc_state == 11) {
alpha:
			gr_char(c, DispCode, one_con_attribute);
			return;
		}
		switch (gr_state) {
		    case 0:
			gr_y1 = (c & 0x1f) << 5;
			gr_state++;
			break;

		    case 1:
extra:
			gr_y1 |= c & 0x1f;
			gr_state++;
			break;

		    case 2:
			if (c & 0x60 == 0x06)
				goto extra;
			gr_x1 = (c & 0x1f) << 5;
			gr_state++;
			break;

		    case 3:
			gr_x1 |= c & 0x1f;
			gr_state = 0;
			switch (gr_mode) {
			    case 1:	/* point plot mode */
				gr_dot(gr_x1, gr_y1, gr_color & BYTE);
				break;
			    case 2:	/* vector plot mode (move) */
				gr_mode++;
				break;
			    case 3:	/* vector plot mode (draw) */
				gr_line(gr_x0, gr_y0, gr_x1, gr_y1);
				break;
			    case 4:	/* incremental plot mode */
				break;
			}
			gr_x0 = gr_x1; gr_y0 = gr_y1;
			break;
		}
		break;
#endif	/* GRAPHICS */
	
	default:		/* illegal state */
		tp->tty_esc_state = 0;
		break;
  }
}

/*===========================================================================*
 *				do_escape				     *
 *===========================================================================*/
PRIVATE void do_escape(tp, c)
register struct tty_struct *tp;	/* pointer to tty struct */
char c;				/* next character in escape sequence */
{
  int n, value, src, dst, count;

  /* Some of these things hack on screen RAM, so it had better be up to date */
  flush(tp);

  /* Handle a sequence beginning with just ESC */
  if (tp->tty_esc_intro == '\0') {
	switch (c) {
		case 'M':	/* Reverse Index */
			if (tp->tty_row == 0)
				scroll_screen(tp, GO_BACKWARD);
			else
				tp->tty_row--;
			move_to(tp, tp->tty_column, tp->tty_row);
			break;
		case '7':	/* save cursor position & current attribute */
PushAttr:
			flush(tp);
			save_row = tp->tty_row;
			save_column = tp->tty_column;
			save_fg_attr = fg_color;
			save_bg_attr = bg_color;
			save_disp = disp_stat;
			break;
		case '8':	/* restore cursor position & attribute */
PopAttr:
			move_to(tp, save_column, save_row);
			fg_color = save_fg_attr;
			bg_color = save_bg_attr;
			disp_stat = save_disp;
			one_con_attribute =
				palet[bg_color] << 8 | palet[fg_color];
			break;
		default: break;
	}
  } else {
	/* Handle a sequence beginning with ESC [ and parameters */
	if (tp->tty_esc_intro == '[') {
		value = tp->tty_esc_parmv[0];
		switch (c) {
		    case 'A': 		/* ESC [nA moves up n lines */
			n = (value == 0) ? 1 : value;
			move_to(tp, tp->tty_column, tp->tty_row - n);
			break;

	    	    case 'B':		/* ESC [nB moves down n lines */
			n = (value == 0) ? 1 : value;
			move_to(tp, tp->tty_column, tp->tty_row + n);
			break;

		    case 'C':		/* ESC [nC moves right n spaces */
			n = (value == 0) ? 1 : value;
			move_to(tp, tp->tty_column + n, tp->tty_row);
			break;

		    case 'D':		/* ESC [nD moves left n spaces */
			n = (value == 0) ? 1 : value;
			move_to(tp, tp->tty_column - n, tp->tty_row);
			break;

		    case 'H':		/* ESC [m;nH" moves cursor to (m,n) */
			move_to(tp, MAX(1, MIN(LINE_WIDTH, tp->tty_esc_parmv[1])) - 1,
			    MAX(1, MIN(SCR_LINES, tp->tty_esc_parmv[0])) - 1 );
			break;

		    case 'J':		/* ESC [J clears screen from cursor */
			cursor(tp, OFF);
		    	switch(value) {
		    	case 0:	 /* clear from cursor to end of screen */
				if (tp->tty_row | tp->tty_column) {
				    n = LINE_WIDTH - tp->tty_column;
				    clr_line(tp->tty_column, tp->tty_row, n);
				    for(n = tp->tty_row+1; n < SCR_LINES; n++)
					clr_line(0, n, LINE_WIDTH);
				    break;
				}
			case 2:	  /* clear screen and goto home */
				clr_scr();
				move_to(tp, 0, 0);
				break;

			case 1:	 /* clear from beginning of screen to cursor */
				for(n = 0; n < tp->tty_row; n++)
					clr_line(0, n, LINE_WIDTH);
				clr_line(0, tp->tty_row, tp->tty_column + 1);
				break;
			}
			cursor(tp, ON);
			break;				

		    case 'K':		/* ESC [K clears line from cursor */
			cursor(tp, OFF);
		    	switch(value) {
		    	case 0:		/* clear from cursor to end of line */
				n = LINE_WIDTH - tp->tty_column;
				clr_line(tp->tty_column, tp->tty_row, n);
				break;
			case 1:     /* clear from begining of line to cursor */
				clr_line(0, tp->tty_row, tp->tty_column + 1);
				break;
			case 2:	    /* clear entire line containing cursor */
				clr_line(0, tp->tty_row, LINE_WIDTH);
				break;
			}
			cursor(tp, ON);
			break;

		    case 'L':		/* ESC [nL inserts n lines at cursor */
			n = (value < 1) ? 1: value;
			if (n > (SCR_LINES - tp->tty_row)) 
				n = SCR_LINES - tp->tty_row;
			src = SCR_LINES - n - 1;
			dst = SCR_LINES - 1;
			count = SCR_LINES - n - tp->tty_row;
			cursor(tp, OFF);
			for ( ; count > 0; src--, dst--, count--)
				mov_left(0, src, 0, dst, LINE_WIDTH);
			for (dst = tp->tty_row; n > 0; dst++, n--)
				clr_line(0, dst, LINE_WIDTH);
			cursor(tp, ON);
			break;

		    case 'M':		/* ESC [nM deletes n lines at cursor */
			n = (value < 1) ? 1: value;
			if (n > (SCR_LINES - tp->tty_row)) 
				n = SCR_LINES - tp->tty_row;
			src = tp->tty_row + n;
			dst = tp->tty_row;
			count = SCR_LINES - n - tp->tty_row;
			cursor(tp, OFF);
			for ( ; count > 0; src++, dst++, count--)
				mov_left(0, src, 0, dst, LINE_WIDTH); 
			for (dst = SCR_LINES - n; dst < SCR_LINES; dst++)
				clr_line(0, dst, LINE_WIDTH);
			cursor(tp, ON);
			break;

		    case 'P':		/* ESC [nP deletes n chars at cursor */
			n = (value < 1) ? 1: value;
			if (n > (LINE_WIDTH - tp->tty_column))
				n = LINE_WIDTH - tp->tty_column;
			src = tp->tty_column + n;
			dst = tp->tty_column;
			count = LINE_WIDTH - tp->tty_column - n;
			cursor(tp, OFF);
			mov_left(src, tp->tty_row, dst, tp->tty_row, count);
			clr_line(dst + count, tp->tty_row, n);
			cursor(tp, ON);
			break;

		    case '@':  		/* ESC [n@ inserts n chars at cursor */
			n = (value < 1) ? 1: value;
			if (n > (LINE_WIDTH - tp->tty_column))
				n = LINE_WIDTH - tp->tty_column;
			src = LINE_WIDTH - n - 1;
			dst = LINE_WIDTH - 1;
			count = LINE_WIDTH - tp->tty_column - n;
			cursor(tp, OFF);
			mov_right(src, tp->tty_row, dst, tp->tty_row, count);
			clr_line(tp->tty_column, tp->tty_row, n);
			cursor(tp, ON);
			break;

	   	    case 'm':		/* ESC [nm enables rendition n */
	 		switch (value) {
	 		    case 0: /* change DEFAULT */
				fg_color = DEF_FG_COLOR;
				bg_color = DEF_BG_COLOR;
				disp_stat = NORMAL_MODE;
	 		    	break;

 			    case 1: /*  BOLD  */
				if ( !(disp_stat & BOLD_BIT)) {
				    disp_stat |= BOLD_BIT;
				    fg_color += 8;
				}
 				break;

 			    case 4: /*  UNDERLINE */
				disp_stat |= UNDER_LINE_BIT;
 				break;

 			    case 5: /*  BLINKING */
				disp_stat |= BLINK_BIT;
 				break;

 			    case 7: /*  REVERSE  */
				swap(&fg_color, &bg_color);
  				break;

			    case 8: /* see through :-) */
				disp_stat |= NO_BACK_GR_BIT;
				break;

 			    default:
 			    	/* foreground color */
 			    	if (value >= 30 && value <= 37) {
				    fg_color = ansi_colors[(value - 30)];
				} else if (value >= 40 && value <= 47) {
				    bg_color = ansi_colors[(value - 40)];
				}
  				break;
	 		}
			one_con_attribute
				= palet[bg_color] << 8 | palet[fg_color];
			break;

		    case 's':	goto PushAttr;
		    case 'u':	goto PopAttr;
	   	    default:
			break;
		}	/* closes switch(c) */
	}	/* closes if (tp->tty_esc_intro == '[') */
  }
  tp->tty_esc_state = 0;
}


/*===========================================================================*
 *				beep					     *
 *===========================================================================*/
PRIVATE int beeping = FALSE;

PRIVATE void beep()
{
/* Making a beeping sound on the speaker (output for CRTL-G).
 * This routine works by turning on the bits 0 and 1 in port B of the 8255
 * chip that drive the speaker.
 */

  message mess;

  if (beeping) return;
  out_byte(TIMER_MODE, 0xB6);	/* set up timer channel 2 (square wave) */
  out_byte(TIMER2, BEEP_FREQ & BYTE);	/* load low-order bits of frequency */
  out_byte(TIMER2, (BEEP_FREQ >> 8) & BYTE);	/* now high-order bits */
  lock();			/* guard PORT_B from keyboard intr handler */
  out_byte(PORT_B, in_byte(PORT_B) | 3);	/* turn on beep bits */
  unlock();
  beeping = TRUE;

  mess.m_type = SET_ALARM;
  mess.CLOCK_PROC_NR = TTY;
  mess.DELTA_TICKS = B_TIME;
  mess.FUNC_TO_CALL = (void (*)()) stop_beep;
  sendrec(CLOCK, &mess);
}


/*===========================================================================*
 *				stop_beep				     *
 *===========================================================================*/
PRIVATE void stop_beep()
{
/* Turn off the beeper by turning off bits 0 and 1 in PORT_B. */

  lock();			/* guard PORT_B from keyboard intr handler */
  out_byte(PORT_B, in_byte(PORT_B) & ~3);
  beeping = FALSE;
  unlock();
}


/*===========================================================================*
 *				scr_init				     *
 *===========================================================================*/
PUBLIC void scr_init(minor)
int minor;
{
/* Initialize the screen driver. */
  struct tty_struct *tp;

  ank_tbl = ((sys_font[1].flag & USABLE_FONT) == USABLE_FONT) ? 1 : 0;
  fontptr = umap(proc_ptr, D, (vir_bytes)ankfont, 4096);
  phys_copy(sys_font[ank_tbl].ft_base, fontptr, (phys_bytes)4096);
  VIDEO_SEG = protected_mode ? COLOR_SELECTOR: (COLOR_BASE >> 4);

  tp = &tty_struct[0];
  if ((sys_font[2].flag & USABLE_FONT) == USABLE_FONT) {
	DispCode = _DEF_DBCS_CODE;
	tp->tty_mode |= DEF_CON_MODE;
	fontptr = umap(proc_ptr, D, (vir_bytes)fontbuf, FONTBUFSIZE);
	sys_font[2].top_ofs = ((CHAR_HIGH - sys_font[2].fn_high) >> 1)
					* BYTE_PER_LINE;
  } else
	DispCode = ANK_ONLY;

#if (LANGUAGE == JAPANESE)
  if (DispCode != ANK_ONLY && (sys_font[3].flag & USABLE_FONT) == USABLE_FONT)
	sys_font[3].top_ofs = ((CHAR_HIGH - sys_font[3].fn_high) >> 1)
					* BYTE_PER_LINE;
#endif

  scr_reset(tp); /* initialize screen & SCR buffer */
  move_to(tp, 0, SCR_LINES - 1);  /* move cursor to lower left */
}

/*===========================================================================*
 *				putc					     *
 *===========================================================================*/
PUBLIC void putc(c)
char c;				/* character to print */
{
/* This procedure is used by the version of printf() that is linked with
 * the kernel itself.  The one in the library sends a message to FS, which is
 * not what is needed for printing within the kernel.  This version just queues
 * the character and starts the output.
 */

  out_char(&tty_struct[0], c);
}

/*===========================================================================*
 *				toggle_scroll				     *
 *===========================================================================*/
PUBLIC void toggle_scroll(c)
{
/* Toggle between hardware and software scroll. */

  softscroll = 1 - softscroll;
  cursor(&tty_struct[0], OFF);
  clr_scr();
  move_to(&tty_struct[0], 0, 0);
  printf("%sware scrolling enabled.\n", softscroll ? "Soft" : "Hard");
}

PUBLIC void xchg_ankfont()
{
    if ((sys_font[1].flag & FONT_ACTIVE) == 0) return;
    ank_tbl = 1 - ank_tbl;
    phys_copy(sys_font[ank_tbl].ft_base,
	umap(proc_ptr, D, (vir_bytes)ankfont, 4096), (phys_bytes)4096);
}

PRIVATE void swap(a1, a2) int *a1, *a2; { int n; n = *a1; *a1 = *a2; *a2 = n; }
PRIVATE int false()	{ return(0); }		/* dummy routine */   
PRIVATE long nothing()	{ return(0L); }		/* dummy */

#if (LANGUAGE == JAPANESE)
PRIVATE int mskanji(c)
unsigned char c;
{
    return(((0x81 <= c) && (c < 0xa0)) || ((0xe0 <= c) && (c <= 0xfc)));
}

PRIVATE int euckanji(c)
unsigned char c;
{
    return(c & 0x80);
}

PRIVATE long ms2index(wc)
u16_t wc;
{
    wc = ms2jis(wc) - 0x2121;
    return( fnt->ft_base + ((wc >> 8) * 94L + (wc & BYTE)) * fnt->fn_size );
}

PRIVATE long euc2index(wc)
u16_t wc;
{
    fnt = ( ((sys_font[3].flag != FONT_DISABLE)
		&& (wc & 0x8000)) ? &sys_font[3]: &sys_font[2] );
    wc = (wc & 0x7f7f) - 0x2121;
    return( fnt->ft_base + ((wc >> 8) * 94L + (wc & BYTE)) * fnt->fn_size );
}

PUBLIC void sw_knj_code(tp)
struct tty_struct *tp;
{
    if (DispCode == ANK_ONLY) return;
    switch (tp->tty_mode & (JIS_MS_BIT | JIS_EUC_BIT)) {
	case 0:					/* default mode */
		DispCode = _DEF_DBCS_CODE;
		tp->tty_mode |= DEF_CON_MODE;
		break;
	case JIS_MS_BIT:			/* use MS-JIS */
		DispCode = JIS_MS;
		break;
	case JIS_EUC_BIT:			/* use EUC-JIS */
		DispCode = JIS_EUC;
		break;
	case (JIS_MS_BIT | JIS_EUC_BIT):	/* switch Kanji code */
		if (DispCode == JIS_MS) {
			DispCode = JIS_EUC;
			tp->tty_mode &= ~JIS_MS_BIT;
		} else {
			DispCode = JIS_MS;
			tp->tty_mode &= ~JIS_EUC_BIT;
		}
    }
    tp->tty_1stbyte = tp->tty_2ndbyte = 0;
    fnt = &sys_font[2];
}
#endif /* LANGUAGE == JAPANESE */

PUBLIC void scr_reset(tp)
struct tty_struct *tp;
{
    fg_color = save_fg_attr = DEF_FG_COLOR;
    bg_color = save_bg_attr = DEF_BG_COLOR;
    disp_stat = save_disp= 0;
    one_con_attribute = palet[bg_color] << 8 | palet[fg_color];
    cur_stat = CUR_ACTIVE;
    save_row = SCR_LINES - 1;
    save_column = 0;
    tp->tty_1stbyte = tp->tty_2ndbyte = 0;
    tp->tty_esc_state = 0;
    cursor(tp, OFF);
    clr_scr();
    cursor(tp, ON);
#ifdef GRAPHICS
    gr_mode = gr_state = gr_esc = 0;
#endif
}

#ifdef GRAPHICS
PRIVATE void gr_Hline(y, x1, x2, ltype, fg, bg)
int y, x1, x2, ltype, fg, bg;
{
    int step, len, mask = 0x80;

    step = ((x1 > x2) ? -1: 1);
    for(len = abs(x2 - x1); len >= 0; len--, x1 += step) {
#if PLOT_BACKGROUND
	gr_dot(x1, y, ((ltype & mask) ? fg: bg));
#else
	if (ltype & mask) gr_dot(x1, y, fg);
#endif			
	if ( !(mask >>= 1)) mask = 0x80;
    }
}

PRIVATE void gr_Vline(x, y1, y2, ltype, fg, bg)
int x, y1, y2, ltype, fg, bg;
{
    int step, len, mask = 0x80;
    step = ((y1 > y2) ? -1: 1);
    for(len = abs(y2 - y1); len >= 0; len--, y1 += step) {
#if PLOT_BACKGROUND
	gr_dot(x, y1, ((ltype & mask) ? fg: bg));
#else
	if (ltype & mask) gr_dot(x, y1, fg);
#endif
	if ( !(mask >>= 1)) mask = 0x80;
    }
}


PRIVATE void gr_line(x1, y1, x2, y2)
int x1, y1, x2, y2;
{
    int fg, bg;
    int mask = 0x8000;
    int x, y, dx, dy, inc, incr1, incr2, d, d2;

    fg = gr_color & BYTE;
    bg = (gr_color >> 8) & BYTE;

    dy = abs(y2 - y1);
    if (dy == 0) {
	gr_Hline(y1, x1, x2, gr_ltype, fg, bg);
	return;
    }

    dx = abs(x2 - x1);
    if (dx == 0) {
	gr_Vline(x1, y1, y2, gr_ltype, fg, bg);
	return;
    }

    if (dx > dy) {
	/* slope < 1 */
	if (x1 > x2) {
	    x = x2; y = y2;
	    inc = ((y2 > y1) ? -1: 1);
	} else {
	    x = x1; y = y1;
	    inc = ((y2 > y1) ? 1: -1);
	}
   	d2 = dy << 1;
	d = d2 - dx;
	incr1 = d2;
	incr2 = (dy - dx) << 1;
	do {
#if PLOT_BACKGROUND
	    gr_dot(x, y, ((gr_ltype & mask) ? fg: bg));
#else
	    if (gr_ltype & mask) gr_dot(x, y, fg);
#endif
	    x++;
	    if ( !(mask >>= 1)) mask = 0x8000;
	    if (d < 0) d += incr1;
	    else { d += incr2; y += inc; }
	} while (dx-- > 0);
    } else {
	/* slope > 1 */
	if (y1 > y2) {
	    y = y2; x = x2;
	    inc = ((x2 > x1) ? -1: 1);
	} else {
	    y = y1; x = x1;
	    inc = ((x2 > x1) ? 1: -1);
	}
	d2 = dx << 1;
	d = d2 - dy;
	incr1 = d2;
	incr2 = (dx - dy) << 1;
	do {
#if PLOT_BACKGROUND
	    gr_dot(x, y, ((gr_ltype & mask) ? fg: bg));
#else
	    if (gr_ltype & mask) gr_dot(x, y, fg);
#endif
	    y++;
	    if ( !(mask >>= 1)) mask = 0x8000;
	    if (d < 0) d += incr1;
	    else { d += incr2; x += inc; }
	} while (dy-- > 0); 
    }
}


PRIVATE void gr_outch(fbit, high, a)
char *fbit;
int high, a;
{
    int fg, bg, i, x, y;

    fg = gr_color & BYTE;
    bg = (gr_color >> 8) & BYTE;
/*    switch(gr_rot) {
	case 0: */
	    for (i = 0, y = gr_y0 + high; i < high; i++, y--, fbit++) {
		gr_Hline(y, gr_x0, gr_x0 + CHAR_WIDTH-1, (u16_t)*fbit, fg, bg);
		if (a == 2) {
		    fbit++;
		    gr_Hline(y, gr_x0 + CHAR_WIDTH,
				gr_x0 +(CHAR_WIDTH*2)-1, (u16_t)*fbit, fg, bg);
		}
	    }
	    gr_x0 += (CHAR_WIDTH * a);
	    if (gr_x0 > X_AXIS_MAX) gr_x0 = 0;
/*	    break;
	case 1:
	    for(i = 0, x = gr_x0 + high; i < high; i++, x--, fbit++) {
		gr_Vline(x, gr_y0, gr_y0 - CHAR_WIDTH+1, (u16_t)*fbit, fg, bg);
		if (a == 2) {
		    fbit++;
		    gr_Vline(x, gr_y0 - CHAR_WIDTH,
				gr_y0 -(CHAR_WIDTH*2)+1, (u16_t)*fbit, fg, bg);
		}
	    }
	    gr_y0 -= (CHAR_WIDTH * a);
	    if (gr_y0 < 0) gr_y0 = Y_AXIS_MAX;
	    break;
	case 2:
	    for(i = 0, y = gr_y0 - high; i < high; i++, y++, fbit++) {
		gr_Hline(y, gr_x0, gr_x0 - CHAR_WIDTH+1, (u16_t)*fbit, fg, bg);
		if (a == 2) {
		    fbit++;
		    gr_Hline(y, gr_x0 - CHAR_WIDTH,
				gr_x0 -(CHAR_WIDTH*2)+1, (u16_t)*fbit, fg, bg);
		}
	    }
	    gr_x0 -= (CHAR_WIDTH * a);
	    if (gr_x0 < 0) gr_x0 = X_AXIS_MAX;
	    break;
	case 3:
	    for(i = 0, x = gr_x0 - high; i < high; i++, x++, fbit++) {
		gr_Vline(x, gr_y0, gr_y0 + CHAR_WIDTH-1, (u16_t)*fbit, fg, bg);
		if (a == 2) {
		    fbit++;
		    gr_Vline(x, gr_y0 + CHAR_WIDTH,
				gr_y0 +(CHAR_WIDTH*2)-1, (u16_t)*fbit, fg, bg);
		}
	    }
	    gr_y0 += (CHAR_WIDTH * a);
	    if (gr_y0 > Y_AXIS_MAX) gr_y0 = 0;
	    break;
    } */
}

PRIVATE void gr_char(ch, dispcode, color)
unsigned char ch;
int dispcode, color;
{
    static u16_t ch_1st = 0, ch_2nd = 0;
    u16_t wc;

    switch (ch) {
	case 000:
		return;
	case 007:
		beep();
		return;
	case '\b':
		gr_x0 -= CHAR_WIDTH;
		if (gr_x0 < 0) gr_x0 += X_AXIS_MAX;
		return;
	case '\r':
		gr_x0 = 0;
		return;
	case '\n':
		gr_y0 -= CHAR_HIGH;
		if (gr_y0 < 0) gr_y0 += Y_AXIS_MAX;
		return;
	case '\t':
		do {
		    gr_x0 += CHAR_WIDTH;
		    if (gr_x0 > (X_AXIS_MAX - 1)) gr_x0 -= X_AXIS_MAX;
		} while ((gr_x0 << 3) & TAB_MASK);
		return;
	default:
		if (ch < 0x20) return;	
		if (ch_1st == 0) {
		    if ( (dispcode != ANK_ONLY) &&
			 (*dbcs_func[dispcode].isdbc)(ch) ) {
			ch_1st = (u16_t)ch;
			return;
		    } else {
			wc = (u16_t)ch;
		    }
		} else {
#if (LANGUAGE == JAPANESE)
		    if (dispcode == JIS_MS)
			wc = ((ch_1st << 8) | ch);
		    else
#endif
			switch (ch_1st) {
			    case DBCS_SS_2:
				wc = (u16_t)ch;
				break;
			    case DBCS_SS_3:
				if (ch_2nd == 0) {
				    ch_2nd = (u16_t)ch;
				    return;
				}
				wc = ((ch_2nd << 8) | ch) & 0xff7f;
				ch_2nd = 0;
				break;
			    default:
				wc = ((ch_1st << 8) | ch) & 0x7f7f;
			}
		    ch_1st = 0;
		}

		if ((wc & 0xff00) && (dispcode != ANK_ONLY)) {
		    phys_copy( (*dbcs_func[dispcode].code2idx)(wc),
				fontptr, (phys_bytes)fnt->fn_size);
		    gr_outch(fontbuf, fnt->fn_high, 2);
		} else {
		    gr_outch(ankfont + ((wc & BYTE) << 4), CHAR_HIGH, 1);
		}
    } /* of switch(ch) */
}
#endif	/* GRAPHICS */
