/*****************************************************************************
**                                                                          **
**          The Clam Shell is Copyright (C) 1988 by Callum Gibson.          **
**       This file is part of Clam Shell. You may freely use, copy and      **
**     distribute it, but if you alter any source code do not distribute    **
**   the altered copy. i.e. you may alter this file for your own use only.  **
**                                                                          **
*****************************************************************************/
/******************************************************************************
**                                                                           **
**                                 meta.c                                    **
**     This file contains functions relevant to Metacharacter expansion,     **
**   including `` ! * ? [] {} ~ ^ $. This module lies between the command    **
**     line editor and the main parsing routines.   8/8/88. CG               **
**                                                                           **
******************************************************************************/

#include "header.h"
/* errors for Loadpattern */
#define EOLN 0
#define FINE 1
#define BADPAT -2

bool Curly,Pattern,Shriek,Bquote,Tilde,Caret,Dollar;
char *matchers[MAXARG];
int mindex;
/* forward decl */
#ifndef __STDC__
static int matchdir();
#else
static int matchdir(char *directory, char *pattern);
#endif

static
void prefilter(line)
  char *line;
{
  int i,quote=0;	/* quote=0 no quote, =1 in "", =2 in '', =3 \ */

  if (line[0]=='^') Caret=TRUE;
  for (i=0;line[i];i++)
    switch(quote)
    {
      case 0 : switch(line[i])
	       {
		 case '\\': quote=1;
		 case '\'': quote++;
		 case '"' : quote++;
			    break;
		 case '!' : Shriek=TRUE;
			    break;
	       }
	       break;
      case 1 : switch(line[i])
	       {
		 case '"': quote=0;
			   break;
		 case '!': Shriek=TRUE;
			   break;
	       }
	       break;
      case 2 : if (line[i]=='\'') quote=0;
	       else if (line[i]=='!') Shriek=TRUE;
	       break;
      case 3 : quote=0;	/* don't need to do anything, just ensure \* isn't interpreted */
	        break;
      default: write(2,"Help! Looney with a stick in prefilter!!\n",41);
	       exit(1);
    }
}

static
void postfilter(line)
  char *line;
{
  int i,star=0,quote=0;	/* quote=0 no quote, =1 in "", =2 in '', =3 \, =4 $ */

/* remove redundancies **... from the input line */
  for (i=0;line[i];i++)
    switch(quote)
    {
      case 0 : switch(line[i])
	       {
		 case '*' : if (star)
			    {
			      strcpy(&line[i],&line[i+1]);
			      i--;		/* to compensate for the fact that we just moved the line back */
			    }
			    else star=1;
			    Pattern=TRUE;
			    break;
		 case '\\': quote=1;
		 case '\'': quote++;
		 case '"' : quote++;
			    break;
		 case '$' : Dollar=TRUE;
			    quote=4;
			    star=0;
			    break;
		 case '~' : Tilde=TRUE;
			    star=0;
			    break;
		 case '`' : Bquote=TRUE;
			    star=0;
			    break;
		 case '{' : Curly=TRUE;
			    star=0;
			    break;
		 case '?' : case '[' :
			    Pattern=TRUE;
		 default  : star=0;
			    break;
	       }
	       break;
      case 1 : switch(line[i])
	       {
		 case '"': quote=0;
			   break;
		 case '`': Bquote=TRUE;
			   break;
		 case '$': Dollar=TRUE;
	       }
	       break;
      case 2 : if (line[i]=='\'') quote=0;
	       break;
      case 3 : quote=0;	/* don't need to do anything, just ensure \* isn't interpreted */
	        break;
      case 4 : switch(line[i])
	       {
		 case ' ':case ';':case '|':case '<':case '>':
		 case '&':case '\t':case '=':
		   quote=0;
		   break;
		 default :break;
	       }
	       break;
      default: write(2,"Help! Looney with a stick in postfilter!!\n",42);
	       exit(1);
    }
}

static int pipepid;

static
FILE *mypopen(command)
  char *command;
{
  int pfd[2];
  extern FILE *fdopen();

  if (pipe(pfd)<0)
  {
    perror("pipe");
    return(NULL);
  }
  switch (pipepid = fork())
  {
    case -1:
	perror("fork");
	return(NULL);
    case 0:
	close(pfd[0]);		/* close reading end of pipe */
	dup2(pfd[1],1);		/* duplicate the writing end to be stdout */
	close(pfd[1]);		/* and now we're finished with the pipe fd */
	execl(CLAMSH,"clam","-c",command,(char *) 0);
	exit(127);	/* a nice odd value */
    default:
	close(pfd[1]);		/* close the writing end of pipe */
	return(fdopen(pfd[0],"r"));	/* return a readable stream */
  }
}

static
int mypclose(fp)
  FILE *fp;
{
#ifdef ATT
  int status;
#else
# ifdef MINIX
  int status;
# else
  union wait status;
# endif
#endif
  int wpid;

#ifdef DEBUG
fprintf(stderr,"closing pipe\n");
#endif
  fclose(fp);
  while ((wpid=wait(&status)) != -1)
  {
    if (wpid == pipepid) break;
  }
  if (wpid == -1)
#ifdef ATT
    status= -1;
  return(status);
#else
# ifdef MINIX
    status= -1;
    return(status);
# else
    status.w_retcode= -1;
  return((int)status.w_retcode);
# endif
#endif
}

static
bool getcom(line,output)
  char *line,*output;
{
  extern SYM_T intercom();
  extern void resetsh();
  extern int fromfile;
#ifdef JOB
  extern void checkjobs();
#endif
  char command[MAXWL];
  int i,c;
  FILE *progout,*popen();

  for (i=0;line[i];i++)
  {
    if (line[i]=='`') 
    {
      command[i]=EOS;
      break;
    }
    command[i]=line[i];
  }
  if (line[i]) 
  {
#ifdef UCB
    signal(SIGCHLD,SIG_IGN);
#endif
#ifdef DEBUG
fprintf(stderr,"command is %s;\n",command);
#endif
/* Probably better make environment before doing this unless always updated */
/*****************/
    if ((progout=mypopen(command))==NULL)
    {
      fprintf(stderr,"%s: Cannot execute.\n",command);
#ifdef JOB
      signal(SIGCHLD,checkjobs);
#endif
      return(FALSE);
    }
#ifdef DEBUG
fprintf(stderr,"Finished popen\n");
#endif
    i=0;
    while ( (c = getc(progout)) != EOF)
    {
      switch(c)
      {
	case '\r':case '\n':case '\t':
	  output[i]=' ';
	  break;
	default :
	  output[i]=c;
	  break;
      }
      i++;
      if (i==MAXLL) break;
    }
    if (i>0) output[--i]=EOS;
    else output[i]=EOS;
    if (mypclose(progout)==127)
    {
      fprintf(stderr,"%s: Cannot execute.\n",command);
#ifdef JOB
      signal(SIGCHLD,checkjobs);
#endif
      return(FALSE);
    }
/* put all the clam stuff back in case it got mangled */
    resetsh();
#ifdef JOB
    signal(SIGCHLD,checkjobs);
#endif
    return(TRUE);
  }
  else
  {
    write(2,"Unmatched backquote `.\n",23);
    return(FALSE);
  }
}

static
int backquote(line)
  char *line;
{
  char newline[MAXLL],word[MAXLL];
  int i,j,quote=0;	/* Note: quote=1 "" and 3 \ not used in this routine */

  for (i=0,j=0;line[i] && j<MAXLL;i++,j++)
    if (!quote)
      switch(line[i])
      {
	case '\\': newline[j++]=line[i++];
		   newline[j]=line[i];
		   break;
	case '`' : i++;
		   if (getcom(&line[i],word)==TRUE)
		   {
		     strncpy(&newline[j],word,MAXLL-j);
		     newline[MAXLL-1]=EOS;
		     for (;line[i]!='`';i++);
		     for (;newline[j];j++);
		     if (j) j--;	/* gets incremented at end of loop */
#ifdef DEBUG
fprintf(stderr,"j in ` %d\n",j);
#endif
		     break;
		   }
		   else return(BQ_ERR);
	case '\'': quote=2;
	default  : newline[j]=line[i];
		   break;
      }
    else
    {
      if (line[i]=='\'')
	quote=0;
      newline[j]=line[i];
    }
  newline[j]=EOS;
  strcpy(line,newline);
  return(OK);
}

static
void prevbit(line,pos,bit)
  char *line,*bit;
  int pos;
{
  int stpos=pos,l=1;

  while (l && stpos>0)
    switch(line[stpos-1])
    {
      case ' ':case '\t':case '>':case '<':case '|':case '"':
      case ';':case '\'':case '=':case '+':case '&':
	l=0;break;
      default:
	stpos--;break;
    }
  for (l=0;stpos<pos;stpos++,l++)
    bit[l]=line[stpos];
  bit[l]=EOS;
}

static
int nextbit(line,pos,bit)
  char *line,*bit;
  int *pos;
{
  int l=1,i=0,moretodo=0;

  while (l)
    switch(line[*pos])
    {
      case ' ':case '\t':case '>':case '<':case '|':case '"':
      case ';':case '\'':case '=':case '+':case '&':case EOS:
	l=0;
	bit[i]=EOS;
	break;
      case '{': moretodo=1;
      default:
	bit[i++]=line[(*pos)++];
	break;
    }
  return(moretodo);
}

static
int curly(line)
  char *line;
{
  char newline[MAXLL],listelm[MAXWL],pbit[MAXWL],nbit[MAXWL];
  int i,j,k,l,quote=0,listindex,moretodo=0;

  for (i=0,j=0;line[i];i++,j++)
  {
    switch(quote)
    {
      case 0 : switch(line[i])
	       {
		 case '{' : i++;		/* don't load the { */
			    listindex=i;	/* start of list */
/* get the previous bit and following bit */
			    prevbit(line,listindex-1,pbit);
			    for (;line[i]!=EOS && line[i]!='}';i++);
			    if (line[i]==EOS)
			    {
			      write(2,"Unmatched brace '{'.\n",21);
			      return(CY_ERR);
			    }
			    i++;
			    moretodo=nextbit(line,&i,nbit);
			    i--;
			    j-=strlen(pbit);
/* now step thru list adding words to newline until you find } */
			    l=1;k=0;
			    while (l)
			    {
			      switch(line[listindex])
			      {
				case '}':
				  l=0;		/* we've reached the last elmnt */
				case ',':
				  listelm[k]=EOS;
				  strcpy(&newline[j],pbit);
				  strcat(newline,listelm);
				  strcat(newline,nbit);
				  for (;newline[j];j++);
				  if (l) newline[j++]=' ';
				  k=0;
				  break;
				case EOS: write(2,"Huh?? Already got EOS\n",22);
				case '=':case ';':case '$':case '<':case '\'':
				case '|':case '/':case '&':case '>':case '"':
				  write(2,"Unmatched brace '{'.\n",21);
				  return(CY_ERR);
				case '\\':
				  listelm[k++]=line[listindex++];
				default:
				  listelm[k]=line[listindex];
				  k++;
				  break;
			      }
			      listindex++;
			    }
			    j--;
			    break;
		 case '\\': quote=1;
		 case '\'': quote++;
		 case '"' : quote++;
		 default  : newline[j]=line[i];
	       }
	       break;
      case 1 : if (line[i]=='"') quote=0;
	       newline[j]=line[i];
	       break;
      case 2 : if (line[i]=='\'') quote=0;
	       newline[j]=line[i];
	       break;
      case 3 : quote=0;
	       newline[j]=line[i];
	       break;
      default: write(2,"Help! Looney with a stick in curly!!\n",37);
	       exit(1);
    }
  }
  newline[j]=EOS;
  strcpy(line,newline);
  if (moretodo) return(curly(line));
  return(OK);
}

static
void finddir(word,dir)
  char *word,*dir;
{
  char c;
  int i=0,j,l=1;

/* This function finds the directory to start the matching from. */
  while(l)
    switch(c=word[i])
    {
      case '*':case '?':case '[':
	for (;dir[i]!='/' && i;i--);		/* go back to end of previous component */
	if (i) dir[i++]=EOS;
	else
	  if (dir[0]=='/') dir[++i]=EOS;
	  else 
	    dir[0]=EOS;
	l=0;
        break;
      default :
	dir[i++]=c;
    }
  for (l=i,j=0;word[l];l++,j++)
    word[j]=word[l];
  word[j]=EOS;
}

static
bool loadword(line,pos,word)
  char *line,*word;
  int *pos;
{
  int l=1,insym=0,inword=0,i=0;

  word[0]=EOS;
  while(l)
    switch(line[*pos])
    {
      case EOS:
	l=0;
	word[i++]=line[(*pos)++];
	break;
      case '\t':case '<':case '>':case '|':case '`':
      case ';':case '=':case '&':
	if (inword)
	{
	  l=0;
	  word[i]=EOS;
	}
	else
	{
	  insym=1;
	  word[i++]=line[(*pos)++];
	}
	break;
      case ' ':
#ifdef DEBUG
fprintf(stderr,"inword %d insym %d\n",inword,insym);
#endif
	if (inword || insym)
	{
	  l=0;
	  word[i]=EOS;
	}
	else (*pos)++;
	break;
      case '"':
#ifdef DEBUG
fprintf(stderr,"copying to next \"\n");
#endif
	inword=1;
	while(line[*pos]!=EOS)
	{
	  word[i++]=line[*pos];
	  if (line[*pos]=='"')
	  {
	    (*pos)++;
	    break;
	  }
	  (*pos)++;
	}
	break;
      case '\'':
#ifdef DEBUG
fprintf(stderr,"copying to next '\n");
#endif
	inword=1;
	while(line[*pos]!=EOS)
	{
	  word[i++]=line[*pos];
	  if (line[*pos]=='\'')
	  {
	    (*pos)++;
	    break;
	  }
	  (*pos)++;
	}
	break;
      case '\\':
	inword=1;
	word[i++]=line[(*pos)++];
	if (line[*pos]!=EOS) word[i++]=line[(*pos)++];
	break;	/* copy over both \ and next char, almost no questions asked */
      default:
	if (insym)
	{
	  l=0;
	  word[i]=EOS;
	  break;
	}
	inword=1;
	word[i++]=line[(*pos)++];
    }
  if (word[0]==EOS) return(FALSE);
  return(TRUE);
}

static
int loadpattern(line,pos,word,pattern)
  char *line,*word;
  int *pos;
  bool *pattern;
{
  char c;
  int i=0,quote=0,foundquote=0;		/* none=0, "=1, '=2, \=3, $=4 */

/* This function loads words from the input line and indicates whether or not
  they are to be pattern matched. This indication is given in the pattern
  bool variable. The return variable is EOLN when the end of line is reached */

  *pattern=FALSE;
  if (loadword(line,pos,word)==FALSE)
    return(EOLN);			/* end of line encountered */
#ifdef DEBUG
fprintf(stderr,"word from loadword is %s<\n",word);
#endif
/* this loop will detect if there's any * ? [] to match */
  while(c=word[i++])
    switch(quote)
    {
      case 0 : switch(c)
	       {
		 case '*':case '?':case '[':
		   if (foundquote)
		   {
		     write(2,"Can't have *,?,[] and quotes in same word.\n",43);
		     return(BADPAT);
		   }
		   *pattern=TRUE;
		   return(FINE);
		   break;
		 case '$':
		   quote=4;
		   break;
		 case '\\':
		   i++;
		   break;
		 case '\'':
		   quote=2;
		   break;
		 case '"':
		   quote=1;
		   break;
	       }
	       break;
      case 1 : if (c=='"')
	       {
	         quote=0;
		 foundquote=1;
	       }
	       break;
      case 2 : if (c=='\'')
	       {
		 quote=0;
		 foundquote=1;
	       }
	       break;
      case 4 : break;	     /* once we have a $, only delimited by another $ */
      default: write(2,"Help! Looney with a stick in loadpattern!!\n",43);
	       exit(1);
    }
  return(FINE);
}

static
int match(string,pattern,accumdir)
  char *string,*pattern,*accumdir;
{
  char c,*findex,*pindex,*fmatch,*pmatch,rempat[MAXWL],where[MAXPL];
  int i,mismatch,found,star=0;

  findex=string;
  pindex=pattern;
  if ((*findex=='.') && (*pindex!='.'))
    return(NX_ERR);	/* don't match dots files unless explicitly asked for */
  mismatch=0;
  while (*findex && !mismatch)
  {
    switch(*pindex)
    {
      case '*' : pindex++;
		 pmatch=pindex;
		 fmatch=findex;
		 star=1;
		 break;
      case '?' : pindex++;   /* just increment, letter automatically matches. */
		 findex++;
		 break;
      case '[' : found=0;
		 for (;(*pindex) && (*pindex!=']');pindex++)
		 {
		   switch (*pindex)
		   {
		     case '\\':
		       if (*(pindex+1)==*findex)
		       found=1;
		       break;
		     case '-' :
		       if ((*(pindex+1)==']') || (*(pindex-1)=='['))
			 if (*pindex==*findex)
			 {
			   found=1;
			   break;
			 }	
		       for (c= *(pindex-1)+1;c <= *(pindex+1) && !found && c!=']';c++)
			 if (c==*findex)
			   found=1;
		       break;
		     default  :
		       if (*pindex==*findex)
			 found=1;
		       break;
		   }
		 }
/* We could exit the loop as soon as found is set to 1 but we have to keep
   going to the end of the list so we can recommence matching after the ']'.
   This also allows us to test for the unmatched bracket. */
		 if (*pindex==EOS)
		 {
		   write(2,"Unmatched bracket '['\n",22);
		   return(BR_ERR);
		 }
		 if (found)
		 {
		   pindex++;		/* move past the ']' */
		   findex++;		/* and on to the next letter */
		   break;
		 }
		 if (!star)
		   mismatch=1;		/* failure for this letter of file */
		 else
		 {
		   pindex=pmatch;
		   findex= ++fmatch;
		 }
		 break;
      case '\\': pindex++;	/* go on to next symbol */
      default  : if (*findex!=*pindex)
		   if (!star)
		     mismatch=1;
		   else
		   {
		     pindex=pmatch;
		     findex= ++fmatch;
		   }
		 else
		 {
		   pindex++;
		   findex++;
		 }
		 break;
    }
  }
  if (mismatch) return(NX_ERR);		/* it didn't match, go on to next one */
  for (;*pindex=='*';pindex++);      /* get rid of trailing stars in pattern */
  if (*findex==*pindex)
  {
/* add it to the candidate list */
    where[0]=EOS;
    if (*accumdir)
    {
      strcpy(where,accumdir);
      strcat(where,"/");
    }
    strcat(where,string);
    matchers[mindex]=(char *) malloc ((unsigned)(strlen(where)+1));
    strcpy(matchers[mindex++],where);
    return(OK);
  }
  else
    if (*pindex=='/')			/* test if file is a directory */
    {
      for (i=0;*pindex;i++,pindex++)
        rempat[i]= *(pindex+1);		/* even copies null across */
      strcpy(where,accumdir);
      if (where[0]!=EOS)
      {
	strcat(where,"/");
	strcat(where,string);
      }
      else strcpy(where,string);
      return(matchdir(where,rempat));	/* recursively call this pair again */
    }
    else				/* it just doesn't match */
      return(NX_ERR);
}

static
int matchdir(directory,pattern)
  char *directory,*pattern;
{
  DIR *dirp,*opendir();
#ifdef ATT
  struct dirent *entry,*readdir();
#else
# ifdef MINIX
  struct dirent *entry;
# else
  struct direct *entry,*readdir();
# endif
#endif
  int foundany=0;

  if (*directory!=EOS)
  {
    if ((dirp=opendir(directory))==NULL)
      return(NX_ERR);
  }
  else
    if ((dirp=opendir("."))==NULL)
      return(NX_ERR);
  while((entry=readdir(dirp))!=NULL)
    if (!match(entry->d_name,pattern,directory))
      foundany++;
  closedir(dirp);
  if (foundany) return(OK);
  else return(NX_ERR);
}

static
int cmp(a,b)
  char **a,**b;
{
  return(strcmp(*a,*b));
}

static
int pattern_match(line)
  char *line;
{
  char newline[MAXLL],word[MAXWL],dir[MAXPL];
  int i=0,j,k,index,m_err=(-2),err,lasterr,nopat=1;
  bool pattern;

  for (j=0;j<MAXLL;j++) newline[j]=EOS;
  j=0;
/* loadpattern puts words from line into word and indicates in pattern if
   they need pattern matching. If not, they're just added to newline. */
  while ((err=loadpattern(line,&i,word,&pattern))==FINE)
  {
#ifdef DEBUG
fprintf(stderr,"word was %s<\n",word);
#endif
    if (pattern==FALSE)			/* if no * ? [] then add to line */
    {
      if (j!=0) newline[j++]=' ';
      for (index=0;word[index];index++,j++)
	newline[j]=word[index];
      newline[j]=EOS;
      continue;
    }
#ifdef DEBUG
fprintf(stderr,"newline in pattern_match\n%s\n",newline);
#endif
/* if an absolute path name is given, then chop it off the front of word and
   put it in dir. Now word contains the patterned stuff onwards. */
  
    dir[0] = EOS;	/* finddir seems to assume this */
    finddir(word,dir);

/* Ok, here goes with the first call to the recursive functions. The parameters
   are: dir - directory to take files from; pattern - the remaining regular
   expression pattern to apply to the files in this directory. Now below we
   set m_err to OK as soon as one matches. Otherwise, lasterr contains the
   last error that occurred. */
    mindex=0;nopat=0;
    if (!(err=matchdir(dir,word))) m_err=OK;
    else lasterr=err;		/* keep record of last error */
#ifdef __GNUC__
	qsort((char *)matchers,(unsigned long)mindex,
	  (unsigned long)sizeof(char *),cmp); 
#else
        qsort((char *)matchers,mindex,(int)sizeof(char *),cmp);
#endif
    for (k=0;k<mindex;k++)
    {
      strcat(newline," ");
      strcat(newline,matchers[k]);
      free(matchers[k]);
    }
    for (;newline[j];j++);
    pattern=FALSE;
  }
  if (err==BADPAT) return(BADPAT);
  if ((!m_err) || (nopat==1))
  {
    strcpy(line,newline);
    return(OK);		/* if m_err isn't -2 anymore then at least 1 matched
			   err will still be -99 if no patterns */
  }
  else
  {
    write(2,"No match.\n",10);
    return(lasterr);	/* otherwise just report the most recent error */
  }
}

void getdir(fp,line)
  FILE *fp;
  char *line;
{
  int i,c;

  for (i=0,c=getc(fp);c!='\n' && c!=EOF;c=getc(fp),i++)
    line[i]=c;
}

static
bool getname(line,pos,name)
  char *line,*name;
  int *pos;
{
  int l=1,i=0;

  while(l)
    switch(line[*pos])
    {
      case EOS:
/* All these chars are illegal in usernames and so act as terminators */
      case ' ':case '/':case ';':case '\t':case '<':case '>':
      case '$':case '\'':case '=':case '&':case '|':case '"':
	name[i]=EOS;
	l=0;
	break;
      default:
	name[i++]=line[(*pos)++];
    }
  if (name[0]==EOS) return(FALSE);
  return(TRUE);
}

static
int getarrsub(line,pos)
  char *line;
  int *pos;
{
  int val=0,innum=0;
  char c;
  
  while(1)
    switch(c=line[(*pos)++])
    {
      case '0':case '1':case '2':case '3':case '4':
      case '5':case '6':case '7':case '8':case '9':
		if (innum)
		{
		  val*=10;
		  val=val+c-48;
		  break;
		}
		else return(-3);		/* this should not get used */
      case ']': if (innum) return(val);
		return(-2);			/* missing [ */
      case '[': if (!innum)
		{
		  innum=1;
		  break;
		} else return(-2);		/* can't have a [ here */
      default : if (innum)
		  return(-2);			/* this isn't a num or ] */
		(*pos)--;		/* put back the char, it's valid */
		return(-1);			/* Not an array. Should've
						been delimited by a $;" etc */
    }
}

int isanum(string)
  char *string;
{
  int i;

  for (i=0;string[i];i++)
    if (!isdigit(string[i])) return(0);
  if (i==0) return(0);
  return(1);
}

static
int dollar(line)
  char *line;
{
  extern bool getword();
  extern char **arguments,*getenv(),*vget(),*vwget();
  extern int numargs;
  char newline[MAXLL],vname[VARNL];
  int i,j,quote=0,num;		/* don't need quote=1 for $variables */

  for (i=0,j=0;line[i];i++,j++)
  {
    switch(quote)
    {
      case 0  : switch(line[i])
		{
		  case '\'':quote=2;
			    newline[j]=line[i];
			    break;
		  case '\\':quote=3;
			    newline[j]=line[i]; /* need to copy \ for aliases */
			    break;
		  case '$' :if (getword(line,&i,vname,FALSE)==TRUE)
			    {
			      if (isanum(vname))
			      {
				num=atoi(vname);
				if (num<=numargs-1)
				{
#ifdef DEBUG
fprintf(stderr,"Getting an argument $%d %s\n",num,arguments[num]);
#endif
				  strcpy(&newline[j],arguments[num]);
				  for (;newline[j];j++);
				  if (j) j--;
				}
				else j--;	/* don't want the increment */
				i--;
				break;
			      }
			      if (!strcmp(vname,"*"))
			      {
				newline[j]=EOS;		/* make sure */
				if (numargs>1)
				{
				  for (num=1;num<numargs;num++)
				  {
				    strcat(newline,arguments[num]);
				    strcat(newline," ");
				  }
				  for (;newline[j];j++);	/* lose last ' ' */
				  if (j>1) j-=2;
				}
				else j--;	/* don't want the increment */
				i--;
				break;
			      }
/* Check for array subscripts by looking for '[' in the next position of the
   line. Then if we find one, get the number and call vwget (matching other
   bracket) otherwise just call vget for entire variable. Note that '[' cannot
   be used for pattern matching after a $ so there is no ambiguity. */
			      if (vget(vname)!=(char *)UNDEF)
				switch(num=getarrsub(line,&i))
				{
				  case -3: write(2,"Looney!!\n",9);
					   exit(1);
				  case -2: write(2,"Missing '[' or ']'\n",19);
					   return(DL_ERR);
				  case -1: strcpy(&newline[j],vget(vname));
					   break;
				  default:
				     { char *tmp, *vwget();
				       strcpy(&newline[j],
					      (tmp = vwget(vname,num))); 
				       free(tmp);
				     }
				}
			      else
				if (getenv(vname))
				  strcpy(&newline[j],getenv(vname));
				else
				{
				  fprintf(stderr,"No such variable : %s\n",vname);
				  fflush(stderr);
				  return(DL_ERR);
				}
			      for (;newline[j];j++);
			      if (j) j--;
			      i--;		/* move line pointer back */
			    }
			    else
			    {
			      write(2,"Illegal variable name.\n",23);
			      return(DL_ERR);
			    }
			    break;
		  default  :newline[j]=line[i];
		}
		break;
      case 2  : if (line[i]=='\'') quote=0;
		newline[j]=line[i];
		break;
      case 3  : quote=0;
		newline[j]=line[i];
		break;
      default : write(2,"Looney with a stick in dollar!!\n",32);
		exit(1);
    }
  }
  newline[j]=EOS;
  strcpy(line,newline);
  return(OK);
}

static
int tilde(line)
  char *line;
{
  extern char *getenv();
  char newline[MAXLL],username[USERNL],pwent[256];
  int i,j,k,quote=0,colon,len;
  FILE *passwdp,*fopen();

  for (i=0,j=0;line[i];i++,j++)
  {
    colon=0;
    switch(quote)
    {
      case 0 : switch(line[i])
	       {
		 case '~' : i++;	/* don't load the ~ */
			    if (line[i]=='#')
			    {
			      newline[j]=line[--i];	/* leave ~# alone */
			      break;
			    }
			    if (getname(line,&i,username)==FALSE)
			    {
			      extern char *getlogin(), *getenv();
			      struct passwd *pw;
			      register char *p;

			/* $USER not there on ST Minix so need to be pedantic */
			      if((p=getenv("USER"))==(char *)NULL)
				if ((p=getlogin())==(char *)NULL)
				{
				  if ((pw=getpwuid(getuid())) == NULL)
				  {
				    fprintf(stderr,"Cannot determine USER\n");
				    return(TL_ERR);
				  }
				  else p=pw->pw_name;
				}
			      strncpy(username,p,USERNL);
			    }
				
			    i--;	/* put back the pointer */
			    if ((passwdp=fopen(PASSWD,"r"))==NULL)
			    {
			      fprintf(stderr,"Can't open file: %s\n",PASSWD);
			      return(TL_ERR);
			    }
			    len=strlen(username);
			    while (!feof(passwdp))
			    {
			      getdir(passwdp,pwent);
			      if ((!strncmp(username,pwent,len)) && (pwent[len]==':'))
			      {
				for (k=0;pwent[k] && colon<5;k++)
				  if (pwent[k]==':') colon++;
				if (colon==5)
				{
				  for (;pwent[k] && pwent[k]!=':';k++,j++)
				    newline[j]=pwent[k];
				  j--;		/* put newline pointer back */
				}
				else
				{
				  fprintf(stderr,"Bad passwd entry for %s.\n",username);
/*				  close(passwdp); */
				  fclose(passwdp);
				  return(TL_ERR);
				}
				break;
			      }
			    }
/*			    close(passwdp); */
			    fclose(passwdp);
/* This test is a repeat of the one above but will give a different error
   message if we can't find a matching username */
			    if (colon!=5)
			    {
			      fprintf(stderr,"No such user: %s\n",username);
			      return(TL_ERR);
			    }
			    break;
		 case '\\': quote=1;
		 case '\'': quote++;
		 case '"' : quote++;
		 default  : newline[j]=line[i];
	       }
	       break;
      case 1 : if (line[i]=='"') quote=0;
	       newline[j]=line[i];
	       break;
      case 2 : if (line[i]=='\'') quote=0;
	       newline[j]=line[i];
	       break;
      case 3 : quote=0;
	       newline[j]=line[i];
	       break;
      default: write(2,"Help! Looney with a stick in tilde!!\n",37);
	       exit(1);
    }
  }
  newline[j]=EOS;
  strcpy(line,newline);
  return(OK);
}

static
int caret(line)
  char *line;
{
  extern char *getnumhist();
  extern int curr_hist;
  char *ptr,templine[MAXLL],search[MAXWL],subst[MAXWL];
  int i,j,len;

  if ((ptr=getnumhist(curr_hist-1))==(char *)UNDEF)
  {
    write(2,"Can't get previous line from history.\n",38);
    return(CT_ERR);
  }
  strcpy(templine,ptr);
#ifdef DEBUG
fprintf(stderr,"templine %s<\n",templine);
#endif
  for (i=1;line[i]!='^' && line[i]!=EOS;i++)
    search[i-1]=line[i];		/* create search string */
  search[i-1]=EOS;
#ifdef DEBUG
fprintf(stderr,"search %s<\n",search);
#endif
  len=i-1;
  if (line[i]==EOS)
    subst[0]=EOS;
  else
  {
    for (i++,j=0;line[i]!=EOS;i++,j++)
      subst[j]=line[i];			/* create substitution string */
    subst[j]=EOS;
  }
#ifdef DEBUG
fprintf(stderr,"subst %s<\n",subst);
#endif
  for (i=0,j=0;templine[j+len-1]!=EOS;)
    if (!strncmp(search,&templine[j],len))
    {
      j+=len;				/* move ptr to templine */
      strcpy(&line[i],subst);		/* copy in the substitution string */
      for (;line[i];i++);		/* get to end of line */
    }
    else
      line[i++]=templine[j++];
  for (;templine[j];j++,i++)
    line[i]=templine[j];
  line[i]=EOS;
  return(OK);
}

static
int shriek(line,printline)
  char *line;
  int *printline;
{
  extern SYM_T retsym();
  extern bool gethist();
  char newline[MAXLL],word[MAXLL];
  int i,j=0;

  for (i=0,j=0;line[i] && j<1024;i++,j++)
    switch(line[i])
    {
      case '!' : i++;			/* don't let retsym load the ! */
		 word[0]=EOS;
		 if (line[i]!='!')	/* check if !! */
		 {
		   /* call retsym with allow_colon (or ") FALSE */
		   if (retsym(line,word,&i,FALSE)!=WORD)
		   {
		     write(2,"Missing event description.\n",27);
		     return(SH_ERR);
		   }
#ifdef DEBUG
fprintf(stderr,"i from retsym %d\n",i);
#endif
		   i--;			/* now put back the pointer */
/* note: all this i++,i-- stuff is silly, but it saves having to look one
   ahead in retsym, and writing a new function to do it. */
		 }
		 else
		   strcpy(word,"!");
#ifdef DEBUG
fprintf(stderr,"word %s< j %d\n",word,j);
#endif
		 if (gethist(word)==TRUE)
		 {
		   strncpy(&newline[j],word,MAXLL-j);
		   for (;newline[j];j++);		/* go to end */
		   if (j) j--;		/* j incremented at end of loop */
		   *printline=1;
		 }
		 else
		 {
		   fprintf(stderr,"Cannot find event !%s\n",word);
		   return(SH_ERR);
		 }
		 break;
      case '\\': newline[j++]=line[i++]; /* need to copy \ for aliases */
      default  : newline[j]=line[i];
		 break;
    }
  newline[j]=EOS;
  strcpy(line,newline);
  return(OK);
}

/*
   Meta has now been split into two parts. This, the first part, does any
   substitution relating to the history (while the line is still intact,
   and echoes the history-substituted line if necessary, after saving it.
*/
int meta_1(line,nosave)
  char *line;
  int nosave;
{
  extern bool getword();
  extern int curr_hist,maxhist,savehist();
  char word[MAXWL];
  int m_err=0,m_err2=0,pos=0,printline=0;

  Dollar=Curly=Pattern=Shriek=Bquote=Tilde=Caret=FALSE;
  prefilter(line);			/* detect any ! or ^ */
#ifdef DEBUG
fprintf(stderr,"line before shriek:\n%s<\n",line);
#endif
  if (Shriek==TRUE)
    m_err=shriek(line,&printline);
#ifdef DEBUG
fprintf(stderr,"line before caret:\n%s<\n",line);
#endif
  if (Caret==TRUE)
  {
    m_err2=caret(line);
    printline=1;
  }

/* History saved now that ! is gone otherwise we get nested !'s (YUK!). This
position may become optional with another up further after expansion is done */
  if (!nosave && !(getword(line,&pos,word,TRUE)==TRUE && (!strcmp(word,"history"))) && (line[1]!=EOS))
    if (savehist(line,curr_hist,maxhist)) curr_hist++;

/* Having saved the history, we return if there was an error from either of
   the history substitutions. */
  if (m_err) return(m_err);
  if (m_err2) return(m_err2);

/* if we used the ! command then we need to print out the interpreted line */
  if (printline)
  {
    write(1,line,strlen(line));
    write(1,"\n",1);
  }
  return(OK);
}

/*
   Second part of meta, called from within the parsing loop. This does the
   pattern-matching substitution, variables, command output, and the like.
*/
int meta_2(line)
  char *line;
{
  int m_err=0;

  postfilter(line);			/* detect other metachars */
  if (Dollar==TRUE)
    if (m_err=dollar(line)) return(m_err);
#ifdef DEBUG
fprintf(stderr,"line after var sub\n%s\n",line);
#endif
  if (Curly==TRUE)
    if (m_err=curly(line)) return(m_err);
  if (Tilde==TRUE)
    if (m_err=tilde(line)) return(m_err);
#ifdef DEBUG
fprintf(stderr,"line before pattern_match\n%s\n",line);
#endif
  if (Pattern==TRUE)
    if (m_err=pattern_match(line)) return(m_err);
#ifdef DEBUG
fprintf(stderr,"line after pattern_match\n%s\n",line);
#endif
  if (Bquote==TRUE)
    if (m_err=backquote(line)) return(m_err);

/* other optional history save in here */
#ifdef DEBUG
fprintf(stderr,"Returning OK from meta_2 with line %s<\n",line);
#endif

  return(OK);
}
