/*****************************************************************************
**                                                                          **
**          The Clam Shell is Copyright (C) 1988 by Callum Gibson.          **
**       This file is part of Clam Shell. You may freely use, copy and      **
**     distribute it, but if you alter any source code do not distribute    **
**   the altered copy. i.e. you may alter this file for your own use only.  **
**                                                                          **
*****************************************************************************/
/******************************************************************************
**                                                                           **
**                                  job.c                                    **
**  Job Control functions. The actual builtins are in builtin.c. This file   **
**  has functions and structures that form the backbone of the job control.  **
**  Also special waitfor function. Att job control function goes here.       **
**                                                                           **
******************************************************************************/


	/* Basically, since Minix doesn't have any job control, we use */
	/* the ATT code; hence all the horrible ifdefs following!      */

#include "header.h"

static char *siglist[] = { "Done","Hangup","Interrupt","Quit","Illegal Instruction",
"Trace/BPT Trap","IOT Trap","EMT Trap","Floating Point Exception","Killed",
"Bus Error","Segmentation Violation","Bad System Call","Broken Pipe","Alarm",
"Terminated"
# ifdef JOB
,"Urgent Socket Condition","Stopped (signal)","Stopped","Continue",
"Child Status Change","Stopped (tty input)","Stopped (tty output)","I/O",
"Cpu Time Limit","File Size Limit","Virtual Time Alarm","Profile Alarm"
# endif
 };

#ifdef JOB
struct job *jtop=NULL,*exitop,*currentjob;

void joblist()
{
  struct job *ptr;

  for (ptr=jtop;ptr;ptr=ptr->next)
  {
    if (currentjob->pid==ptr->pid)
      printf("    + [%d] %d",ptr->jobnumber,ptr->pid);
    else
      printf("      [%d] %d",ptr->jobnumber,ptr->pid);
    if (ptr->status.w_stopval==WSTOPPED) printf(" %s   ",siglist[ptr->status.w_stopsig]);
    printf(" %s\n",ptr->name);
  }
  fflush(stdout);
}


bool stoppedjobs()
{
  struct job *ptr;

  for (ptr=jtop;ptr;ptr=ptr->next)
    if (ptr->status.w_stopval==WSTOPPED) return(TRUE);

  return(FALSE);
}


struct job *findjob(pid)
  int pid;
{
  struct job *ptr;

  for (ptr=jtop;ptr;ptr=ptr->next)
    if (ptr->pid==pid) return(ptr);
  return(0);
}


int pidfromjob(jobno)
  int jobno;
{
  struct job *ptr;

  for (ptr=jtop;ptr;ptr=ptr->next)
    if (ptr->jobnumber==jobno) return(ptr->pid);
  return(-1);
}


void addreport(newone)
  struct job *newone;
{
  extern int jobdone;
  struct job *ptr,*old;
  int i;

#ifdef DEBUG
fprintf(stderr,"adding report for %d %s\n",newone->pid,newone->name);
#endif
  jobdone=1;
  for (i=0,old=ptr=exitop;ptr;i++,old=ptr,ptr=ptr->next)
    ;
#ifdef DEBUG
fprintf(stderr,"i is %d\n",i);
#endif
  if (i)
  {
    old->next=ptr=(struct job *) malloc ((unsigned)(sizeof(struct job)));
    if (ptr==0)
    {
      write(2,"Malloc failed. Cannot add report.\n",34);
      return;
    }
    ptr->next=0;
  }
  else
  {
    exitop=(struct job *) malloc ((unsigned)(sizeof(struct job)));
    if (exitop==0)
    {
      write(2,"Malloc failed. Cannot add report.\n",34);
      return;
    }
    exitop->next=0;
    ptr=exitop;
  }
  ptr->jobnumber=newone->jobnumber;
  ptr->pid=newone->pid;
  ptr->status.w_status=newone->status.w_status;
  ptr->name=(char *) malloc ((unsigned)(strlen(newone->name)+1));
  if (ptr->name)
    (void) strcpy(ptr->name,newone->name);
  else
    write(2,"Malloc failed in addreport.\n",28);
  ptr->dir=(char *) malloc ((unsigned)(strlen(newone->dir)+1));
  if (ptr->dir)
    (void) strcpy(ptr->dir,newone->dir);
  else
    write(2,"Malloc failed in addreport.\n",28);
  ptr->rdrout=newone->rdrout;
  ptr->lastmod=newone->lastmod;
}


int addjob(pid,name,outfd,bckgnd)
  int pid,outfd;
  char *name;
  bool bckgnd;/* this isn't used yet */
{
  int jobno,diff=(-1);
  struct job *ptr,*old,*new;
  char execdir[MAXPL];

  for (old=ptr=jtop,jobno=1;ptr;jobno++,old=ptr,ptr=ptr->next)
    if (jtop->jobnumber>1 || (diff=old->jobnumber-ptr->jobnumber+1)<0) break;
  if (diff<0)
    if (jobno!=1)			/* insertion between old and ptr */
    {
      new=(struct job *) malloc ((unsigned)(sizeof(struct job)));
      if (new==0)
      {
	write(2,"Malloc failed. Cannot add job.\n",31);
	return(jobno);
      }
      old->next=new;
      new->next=ptr;
      ptr=new;
    }
    else			/* insertion before jtop */
    {
      old=(struct job *) malloc ((unsigned)(sizeof(struct job)));
      if (old==0)
      {
	write(2,"Malloc failed. Cannot add job.\n",31);
	return(jobno);
      }
      old->next=jtop;
      ptr=jtop=old;
    }
  else					/* append at end */
  {
    ptr=(struct job *) malloc ((unsigned)(sizeof(struct job)));
    if (ptr==0)
    {
      write(2,"Malloc failed. Cannot add job.\n",31);
      return(jobno);
    }
    old->next=ptr;
    ptr->next=0;
  }
  ptr->jobnumber=jobno;
  ptr->pid=pid;
  ptr->name=(char *) malloc ((unsigned)(strlen(name)+1));
  if (ptr->name)
    (void) strcpy(ptr->name,name);
  else
    write(2,"Malloc failed in addjob.\n",25);
  ptr->status.w_status=0;
  if (getwd(execdir))
  {
    ptr->dir=(char *) malloc ((unsigned)(strlen(execdir)+1));
    if (ptr->dir)
      (void) strcpy(ptr->dir,execdir);
    else
      write(2,"Malloc failed in addjob.\n",25);
  }
  else ptr->dir=" ??? ";
#ifdef DEBUG
fprintf(stderr,"execdir from getwd is %s ptr->dir is %s\n",execdir,ptr->dir);
#endif
  ptr->lastmod=time((long *)0);
  currentjob=ptr;			/* a new current job */
#ifdef DEBUG
fprintf(stderr,"added jobno %d pid %d\n",jobno,pid);
#endif
  return(jobno);
}

void newcurr()
{
  struct job *ptr;

/* set new current job */
  currentjob=0;
  for (ptr=jtop;ptr;ptr=ptr->next)
    if (currentjob==0 || ptr->lastmod>currentjob->lastmod)
      currentjob=ptr;
#ifdef DEBUG
if (currentjob)
  fprintf(stderr,"New current job is [%d] %d\n",currentjob->jobnumber,currentjob->pid);
#endif
}


void rmjob(pid)
  int pid;
{
  struct job *ptr,*old;
  int i;

#ifdef DEBUG
fprintf(stderr,"removing pid %d from list\n",pid);
fflush(stderr);
#endif
  for (i=0,old=ptr=jtop;ptr;i++,old=ptr,ptr=ptr->next)
    if (ptr->pid==pid)
      if (i)
      {
	old->next=ptr->next;
	free(ptr->name);
	free(ptr);
	break;
      }
      else
      {
	jtop=jtop->next;
	free(ptr->name);
	free(ptr);
	break;
      }
  newcurr();
}


int update(pid,status)
  int pid;
  union wait status;
{
  struct job *ptr;

#ifdef DEBUG
fprintf(stderr,"updating %d\n",pid);
fflush(stderr);
#endif
  for (ptr=jtop;ptr!=NULL && ptr->pid!=pid;ptr=ptr->next)
    ;
  if (ptr!=NULL && ptr->pid==pid)
  {
    ptr->status.w_status=status.w_status;
    ptr->lastmod=time((long *)0);
    return(1);
  }
/* otherwise ignore it */
#ifdef DEBUG
fprintf(stderr,"Couldn't update %d\n",pid);
#endif
  return(0);
}


void checkjobs()
{
  int wpid;


  struct rusage rusage;
  union wait status;

#ifdef DEBUG
write(2,"In checkjobs\n",13);
#endif
  wpid=wait3(&status,WNOHANG|WUNTRACED,&rusage);
  if (wpid)
  {
    if (update(wpid,status))
      addreport(findjob(wpid));
    if (status.w_stopval!=WSTOPPED)
      rmjob(wpid);	/* if not stopped, delete. */
  }
#ifdef DEBUG
write(2,"Out checkjobs\n",14);
#endif

}
#endif

int waitfor(pid)	/* Clam versions without job control still need this */
  int pid;
{
  int wpid;

#ifndef JOB
# ifdef UCB
  union wait status;

/* very temporary. Must use att job control algorilla */
  while ((wpid=wait(&status))!=pid && wpid!=-1)
    /* update list */;
  return(status.w_status);

# else
  int status;
  char *message;

/* use exactly the same code as for ATT */
  do	{
	wpid = wait(&status);
	if (status & 0xff) 
		{message=siglist[status &0x7f];
		if (wpid != pid)
			printf("%d ",wpid);
		if (status &0200)
			printf("%s (core dumped)\n",message);
		else
			printf("%s\n",message);
		}
# ifdef chatter
	else
	      {
		if (wpid != pid)
			printf("%d: Exit %d\n", wpid, status >>8);
	      }
# endif

	} while (wpid != pid && wpid!=-1);

  return(status);
# endif
#else


  extern char *vget();
  struct rusage rusage;
  union wait status;
  struct job *ptr;
  char *c,currdir[MAXPL];

  while (1)
  {
#ifdef DEBUG
fprintf(stderr,"wait for %d\n",pid);
#endif
    if ((wpid=wait3(&status,WUNTRACED,&rusage))==-1)
    {
      perror("wait");
      write(2,"You fg-ed a non-child. Watch out for looneys!\n",46);
      if (setpgrp(pid,pid)==-1)
      {
	perror("setpgrp");
	fprintf(stderr,"Oh no! Can't stop it. Runaway process number %d\n",pid);
	fflush(stderr);
      }
      return(1);
    }
#ifdef DEBUG
fprintf(stderr,"wpid returned from wait3 %d\n",wpid);
#endif
    if (wpid>0)
    {
      if (update(wpid,status))
        if (wpid!=pid) addreport(findjob(wpid));
      ptr=findjob(pid);
      strncpy(currdir,ptr->dir,MAXPL);
      if (status.w_stopval!=WSTOPPED)
	rmjob(wpid);	/* if not stopped, delete. */
      if (wpid==pid)
      {
	if (status.w_termsig)
	{
	  if (status.w_stopval==WSTOPPED)
	  {
	    printf("\n%s\n",siglist[status.w_stopsig]);
	  }
	  else
	    if (status.w_coredump)
	      printf("\n%s (core dumped)\n",siglist[status.w_termsig]);
	    else
	      printf("\n%s\n",siglist[status.w_termsig]);
	  fflush(stdout);
	}
	c=vget("cwd");
#ifdef DEBUG
fprintf(stderr,"currdir (ptr->dir) is %s. cwd is %s.\n",currdir,c);
#endif
	if (strcmp(currdir,c))			/* if directory has changed */
	{
	  write(2,"(wd now: ",8);
	  write(2,c,strlen(c));
	  write(2,")\n",2);
	}
	break;
      }
    }
  }
  return(status.w_retcode);
#endif
}

#ifdef JOB
void reportjobs()
{
  extern int jobdone;
  struct job *ptr;
  char *message,mess[30];

  for (ptr=exitop;ptr;exitop=ptr)
  {
    ptr=ptr->next;
    if (exitop->status.w_stopval==WSTOPPED)
    {
      message=siglist[exitop->status.w_stopsig];
      printf("[%d] %d %s   %s\n",exitop->jobnumber,exitop->pid,message,exitop->name);
    }
    else
    {
      if (exitop->status.w_retcode) 
      {
	sprintf(mess,"Exit %d",exitop->status.w_retcode);
	message=mess;
      }
      else
        message=siglist[exitop->status.w_termsig];
      if (exitop->status.w_coredump)
	printf("[%d] %d %s (core dumped)   %s\n",exitop->jobnumber,exitop->pid,message,exitop->name);
      else
	printf("[%d] %d %s   %s\n",exitop->jobnumber,exitop->pid,message,exitop->name);
    }
    free(exitop->name);free(exitop->dir);free(exitop);
  }
  fflush(stdout);
  jobdone=0;
}
#endif
