/*****************************************************************************
**                                                                          **
**          The Clam Shell is Copyright (C) 1988 by Callum Gibson.          **
**       This file is part of Clam Shell. You may freely use, copy and      **
**     distribute it, but if you alter any source code do not distribute    **
**   the altered copy. i.e. you may alter this file for your own use only.  **
**                                                                          **
*****************************************************************************/
/******************************************************************************
**                                                                           **
**                                   hash.c                                  **
**      This function sets up the hash table for the path. executable is     **
**       not used at this point in time (system calls too slow)              **
**                                                                           **
******************************************************************************/

#include "header.h"
#include <sys/stat.h>

#ifdef HASH
struct har hasharray[HSHSIZ];
char priority;
#endif

/* Builtin list for hashing. If it's not here, it ain't hashed. */

char *builtin_name[40]=
{ "alias","cd","chdir","edit","exec","exit","export",
  "history","list","load","lock",
  "logout","source","unalias",
  "unset","unsetenv","which",
#ifdef JOB
  "bg","fg","jobs","kill","stop","suspend","wait",
#endif
#ifdef HASH
  "hashstat","rehash",
#endif
#ifdef SCRIPT
  "case","for","if","repeat","read","shift","while",
#endif
#ifndef MINIX
  "help","limit","unlimit",
#endif
  NULL };


/* Not used here. Called from getpath in exec.c */
bool executable(dir,entname,nojoin)
  char *dir,*entname;
  int nojoin;
{
  struct stat buf;
  char fpn[MAXWL];

  if (nojoin)
  {
#ifdef DEBUG
fprintf(stderr,"ex f1\n");
#endif
    if (stat(dir,&buf)) return(FALSE);
  }
  else
  {
#ifdef DEBUG
fprintf(stderr,"ex f2\n");
#endif
    sprintf(fpn,"%s/%s",dir,entname);
    if (stat(fpn,&buf)) return(FALSE);
  }
  if (((buf.st_mode & S_IFMT)==S_IFREG) && (buf.st_mode & 0111))
  {
#ifdef DEBUG
fprintf(stderr,"ex s\n");
#endif
    return(TRUE);
  }
#ifdef DEBUG
fprintf(stderr,"ex f3\n");
#endif
  return(FALSE);
}

#ifdef HASH
int hash(name)
  char *name;
{
  int base=4999,i,hashval=0;

  for (i=0;name[i];i++)
  {
    hashval=hashval+name[i]*base;
    base=base*2017+1;
  }
  hashval%=HSHSIZ;
  if (hashval<0) hashval+=HSHSIZ;
  return(hashval);
}

int hashaliases()
{
  extern struct alias *atop;
  extern struct adefn *checkalias();
  extern int h_hits,h_misses;
  struct alias *aptr;
  int hashval,times;

  for (aptr=atop;aptr;aptr=aptr->next)
  {
    hashval=hash(aptr->a_name);
    for (times=0;hasharray[hashval].name!=NULL && times<HSHSIZ;times++)
    {
      hashval++;
      hashval%=HSHSIZ;
    }
    if (times==0) h_hits++;
    else
    {
      h_misses++;
      if (times==HSHSIZ)
      {
	write(2,"Hash table is full. Cannot complete hash.\n",42);
			/* allow the partial hash and just return */
	return(1);
      }
    }
/* Having found a blank position we now malloc some space and copy the values */
    hasharray[hashval].name=(char *) malloc ((unsigned)(strlen(aptr->a_name)+1));
    strcpy(hasharray[hashval].name,aptr->a_name);
    hasharray[hashval].exec_ptr.alias_defn=checkalias(aptr->a_name);
    hasharray[hashval].flag=ALIASEX|priority;
  }
  return(0);
}


int hashbuiltins()
{
  extern void (*checkbuiltins())();
  extern int h_hits,h_misses;
  int i,hashval,times;

  for (i=0;builtin_name[i];i++)
  {
    hashval=hash(builtin_name[i]);
    for (times=0;hasharray[hashval].name!=NULL && times<HSHSIZ;times++)
    {
      hashval++;
      hashval%=HSHSIZ;
    }
    if (times==0) h_hits++;
    else
    {
      h_misses++;
      if (times==HSHSIZ)
      {
	write(2,"Hash table is full. Cannot complete hash.\n",42);
			/* allow the partial hash and just return */
	return(1);
      }
    }
/* Having found a blank position we now malloc some space and copy the values */
    hasharray[hashval].name=(char *) malloc ((unsigned)strlen(builtin_name[i])+1);
    strcpy(hasharray[hashval].name,builtin_name[i]);
    hasharray[hashval].exec_ptr.builtin_fn=checkbuiltins(builtin_name[i]);
    hasharray[hashval].flag=BUILTEX|priority;
  }
  return(0);
}


int tildehashisin(path)
  char *path;
{
  int pos;

  for (pos=0;path[pos];pos++)
    if (path[pos]=='~')
      if (path[pos+1]=='#' && (path[pos+2]==' ' || path[pos+2]==EOS))
	if (pos==0 || path[pos-1]==' ')
	{
#ifdef DEBUG
fprintf(stderr,"~# is in PATH\n");
#endif
	  return(1);
	}
#ifdef DEBUG
fprintf(stderr,"No ~# in PATH\n");
#endif
  return(0);
}


void inihash()
{
  int i;

  for (i=0;i<HSHSIZ;i++)
  {
    hasharray[i].name=NULL;
    hasharray[i].flag=0;
  }
}


void hashpath()
{
  extern void vdel();
  extern bool hashed,dotinpath;
  extern char *vget();
  extern int h_hits,h_misses;
  SYM_T t,retsym();
  int times,pos=0,pcnum=0,hashval;
  char path[VARVL],dir[MAXWL];
  DIR *dd,*opendir();
#ifdef ATT
  struct dirent *entry,*readdir();
#else
  struct direct *entry,*readdir();
#endif

  /* This function takes each component of the path and calculates the hash
     val for each file in each directory. This values gives the position in
     the hash-array of structures which contains the name (to check for double
     hash values) and the directory that contains it. Any collisions cause the
     search to continue to the next element in the hash-array */

#ifdef DEBUG
fprintf(stderr,"In hashpath\n");
#endif
  priority=BEFOREDOT;
  dotinpath=FALSE;
  if (vget("PATH")==(char *)UNDEF)
  {
    write(2,"No PATH variable!!\n",19);
    vdel("HASH");
    hashed=FALSE;
    return;
  }
  strcpy(path,vget("PATH"));
/* free up previous hash table */
  for (times=0;times<HSHSIZ;times++)
  {
    if (hasharray[times].name!=NULL)
    {
      free(hasharray[times].name);
      hasharray[times].name=NULL;
    }
    if (!(hasharray[times].flag&EXTYPE))		/* it's a string */
      free(hasharray[times].exec_ptr.dir);
    hasharray[times].exec_ptr.dir=NULL;		/* nullify pointer */
  }
  h_hits=h_misses=0;			/* reset hashing stats */
/* hash builtins here if not in PATH variable. */
  if (!tildehashisin(path))
  {
    if (hashaliases()) return;			/* should never happen */
    if (hashbuiltins()) return;			/* ditto */
  }
  while((t=retsym(path,dir,&pos,FALSE))!=ENDLN)
  {
    pcnum++;
    if (t!=WORD)
    {
      fprintf(stderr,"Component %d of $PATH is invalid. Ignoring error.\n",pcnum);
      continue;
    }
    if (dir[0]=='~' && dir[1]=='#')
    {
		/* hash aliases and builtins */
      if (hashaliases()) return;		/* return if full */
      if (hashbuiltins()) return;			/* ditto */
      continue;
    }
/*
   If we find . in the path, set the priority flag to AFTERDOT and then continue
   hashing the rest of the path. Dot is no longer hashed in the table.
*/
    if (dir[0]=='.' && dir[1]==EOS)
    {
      priority=AFTERDOT;
      dotinpath=TRUE;
      continue;
    }

/* shouldn't have to malloc for entry since opendir does it, hopefully */
    if ((dd=opendir(dir))!=NULL)
    {
#ifdef DEBUG
fprintf(stderr,"Hashing '%s'\n",dir);
#endif
      while ((entry=readdir(dd))!=NULL)
      {
	if (entry->d_name[0]=='.' && (entry->d_name[1]==EOS || entry->d_name[1]=='.')) continue;
	hashval=hash(entry->d_name);
	for (times=0;hasharray[hashval].name!=NULL && times<HSHSIZ;times++)
	{
	  hashval++;
	  hashval%=HSHSIZ;
	}
	if (times==0) h_hits++;
	else
	{
#ifdef DEBUG
	  fprintf(stderr,"Hash missed on %d\n",hash(entry->d_name));
#endif
	  h_misses++;
	  if (times==HSHSIZ)
	  {
	    write(2,"Hash table is full. Cannot complete hash.\n",42);
			/* allow the partial hash and just return */
	    return;
	  }
	}

/* Having found a blank position we now malloc some space and copy the values */
	hasharray[hashval].name=(char *) malloc ((unsigned)(strlen(entry->d_name)+1));
	strcpy(hasharray[hashval].name,entry->d_name);
	hasharray[hashval].exec_ptr.dir=(char *) malloc ((unsigned)(strlen(dir)+1));
	strcpy(hasharray[hashval].exec_ptr.dir,dir);
	hasharray[hashval].flag=priority;	/* actually 0|priority */
      }
      closedir(dd);
    }
    else
      fprintf(stderr,"Can't open directory: %s\n",dir);
  }
}
#endif
