/*****************************************************************************
**                                                                          **
**          The Clam Shell is Copyright (C) 1988 by Callum Gibson.          **
**       This file is part of Clam Shell. You may freely use, copy and      **
**     distribute it, but if you alter any source code do not distribute    **
**   the altered copy. i.e. you may alter this file for your own use only.  **
**                                                                          **
*****************************************************************************/
/******************************************************************************
**                                                                           **
**                               builtin1.c                                  **
**     This file contains the functions for the builtin shell commands.      **
**                                                                           **
******************************************************************************/

#include "header.h"

	/* To make Clam fit under Minix, and also to keep the code */
	/* down to a reasonable size, I have removed all of the    */
	/* job control code (which Minix can't do), and also the   */
	/* hashing & script functions, most of the latter are not  */
	/* implemented yet */

#ifdef ATARI_ST
#define lock _LOCK	/* conflicts with libc lock prototype */
#endif

#define FOUND 0
#define NONEXIST 2
#define NOTEXEC 13

extern char **environ;

static
void Cd(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void vset();
  extern char *vget();
  extern int fromfile,errno;
  SYM_T t,retsym();
  char cdpath[VARVL],dir[MAXWL],home[VARVL];
  char fulldirname[MAXFNL], *cdpathptr, *s;
  int pos=0, cdpcnum=0,olderr=0;
#ifdef MINIX
  char dirname[128];
  extern char *getcwd();
#endif

  if (argc==1) argv[1]=vget("HOME");
  if (argc>2)
  {
    write(2,"cd: Too many arguments.\n",24);
    return;
  }
  if (chdir(argv[1])!=0)
  {
    /* absolute, so don't use cdpath, it failed */
    if (*argv[1]=='/' || !strncmp(argv[1],"./",2) || !strncmp(argv[1],"../",3))
      perror("cd");
    else
    {
      /* use cdpath */
      if ((cdpathptr=vget("cdpath"))==(char *)UNDEF)
	perror("cd");
      else
      {
	olderr=errno;
	strcpy(cdpath,cdpathptr);
	/* search through cdpath for a suitable directory */
	while((t=retsym(cdpath,dir,&pos,FALSE))!=ENDLN)
	{
	  cdpcnum++;
	  if (t!=WORD)
	  {
	    fprintf(stderr,"Component %d of cdpath is illegal.\n",cdpcnum);
	    continue;
	  }
	  strcpy(fulldirname,dir);
	  strcat(fulldirname,"/");
	  strcat(fulldirname,argv[1]);
	  if (chdir(fulldirname)==0)
	  {
	    /* chdir was successful */
	    if (!fromfile && strcmp(dir,"."))
	    {
	      /* print the new directory if an interactive shell
		 and the new directory was not relative to the old */
	      strcpy(home,vget("HOME"));
	      if (strncmp(home,fulldirname,strlen(home)))
		write(2,fulldirname,strlen(fulldirname));
	      else
	      {
		/* substitute ~ for the home directory */
		write(2,"~/",2);
		s=fulldirname+(strlen(home)+1);
		write(2,s,strlen(s));
	      }
	      write(2,"\n",1);
	    }              
	    break;
	  }
	}
	if (t==ENDLN)
	{
	  errno=olderr;		/* we want the original error if cdpath fails */
	  perror("cd");
	}
      }
    }
  }
  /* else if (hashed==TRUE) hashpath(); don't need this now */
#ifndef MINIX
  if (argc==2) argv[1]=(char *) realloc (argv[1],(unsigned)(MAXPL));
  else argv[1]=(char *) malloc ((unsigned)(MAXPL));
#endif
#ifdef ATT
  if (getcwd(argv[1]),MAXPL) vset("cwd",argv[1]);
#else
# ifdef MINIX
  if (getcwd(dirname, 128)) vset("cwd",dirname);
# else
  if (getwd(argv[1])) vset("cwd",argv[1]);
# endif
#endif
  else write(2,"Can't get cwd properly\n",23);
#ifndef MINIX
  free(argv[1]);	/* just in case? */
#endif
}

static
void Exit(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void setdown();
  extern bool loginsh;
#ifdef JOB
  extern bool stoppedjobs();
  extern int stopped;
#endif
  extern int fromfile,isanum();
  void logout();	/* just down from here... */
#ifdef DEBUG
 printf("Entering builtin Exit\n");
#endif
#ifdef JOB
  if (!stopped && stoppedjobs()==TRUE)
  {
    write(1,"There are stopped jobs.\n",24);
    stopped=2;
    return;
  }
#endif
#ifdef DEBUG
printf("Exit: testing if login shell\n");
#endif
  if (loginsh==TRUE)
    logout(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped);
  if (argc>2 || (argc!=1 && !isanum(argv[1])))
  {
    write(2,"Bad argument to exit.\n",22);
    return;
  }
#ifdef DEBUG
printf("Exit: Not login shell, calling setdown\n");
#endif
  if (!fromfile) setdown();
  if (argc==1) exit(0);
  if (argc==2) exit(atoi(argv[1]));
  else
    fprintf(stderr,"usage: exit [ status ]\n");
}

static
void unset(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void vdel();
#ifdef HASH
  extern bool hashed;
#endif
  extern int nohistdup;
  int i;

  for (i=1;argv[i];i++)
  {
    vdel(argv[i]);
#ifdef HASH
    if (!strcmp(argv[i],"HASH")) hashed=FALSE;
#endif
    if (!strcmp(argv[i],"umask")) umask(022);
    if (!strcmp(argv[i],"nohistdup")) nohistdup=0;
  }
}

static
void unsetenv(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void venvdel();
  int i;

  for (i=1;argv[i];i++) venvdel(argv[i]);
}

static
void list(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void venvprint(),vprint();

  if (argc>2)
  {
    fprintf(stderr,"usage: list [ env ]\n");
    return;
  }
  if (argc==1)
  {
    vprint();
    return;
  }
  if (!strcmp(argv[1],"env"))
    venvprint();
  else fprintf(stderr,"usage: list [ env ]\n");
}

static
void Exec(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void (*builtin_func)(),setdown(),redirect(),runalias();
  extern struct adefn *alias_defn;
  extern int errno,fromfile,getpath();
  char path[MAXPL],**startarg;

  if (argc==1)
  {
    fprintf(stderr,"usage: exec <program> [ <args> ]\n");
    return;
  }
  if ((errno=getpath(path,argv[1],0))==FOUND)
  {
#ifdef DEBUG
fprintf(stderr,"path is %s\n",path);
#endif
    startarg= &argv[1];
    redirect(infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd);
    if (alias_defn!=0)
    {
      runalias(alias_defn,argc-1,startarg,infd,outfd,errfd,ifil,ofil,efil,
	       appnd,bckgnd,piped);
      if (!fromfile) setdown();
      exit(0);
    }
    if (builtin_func!=0)
    {
      (*builtin_func)(argc-1,startarg,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped);
      if (!fromfile) setdown();
      exit(0);
    }
    if (!fromfile) setdown();
    execve(path,startarg,environ);
  }
  perror(argv[1]);
}

static
void Export(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern bool export();
  char *name;

  switch (argc)
  {
    case 3 :
	name=argv[2];
	argv[2]=argv[1];
	break;
    case 2 :
	argv[2]="";
    case 1 :
	name=argv[1];
	break;
    default:
	fprintf(stderr,"usage: export [ [ - ] <variable> ]\n");
	return;
  }
  if (export(name,argv[2])==FALSE)
    fprintf(stderr,"No such variable: %s\n",name);
}

static
void edit(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void show(),vset();
  extern bool getline();
  extern char *vget();
  extern int lenprompt;
  char line[MAXLL];
  int i,curs[2];

  if (argc!=2)
  {
    fprintf(stderr,"usage: edit <variable>\n");
    return;
  }
  if (vget(argv[1])==(char *)UNDEF)
  {
    fprintf(stderr,"No such variable : %s\n",argv[1]);
    return;
  }
  for (i=0;i<MAXLL;i++) line[i]=EOS;
  strcpy(line,vget(argv[1]));
  lenprompt=0;		/* There is no prompt for this editor */
  curs[0]=0;curs[1]=0;	/* we start at the beginning of the line */
#ifdef DEBUG
fprintf(stderr,"editing %s.\n",argv[1]);
#endif
  show(line,curs,TRUE);	/* clearing is true since start on new line */
  if (getline(line,&i,1)==TRUE)	/* Use the CLE to edit the variable */
    vset(argv[1],line);		/* now restore the variable */
}

#ifdef HASH
static
void hashstat(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern h_hits,h_misses,hits,misses;
  float ratio;

  if (argc==1)
  {
    if (hits>0)
      ratio=(float) hits/(hits+misses)*100;
    else
      ratio=0;
    printf("Actual:\nhits %d     misses %d     hit ratio %.1f%%\n",hits,misses,ratio);
    fflush(stdout);
  }
  else
    if (!strcmp(argv[1],"x"))
    {
      if (h_hits>0)
#ifdef MINIX
        ratio=100*h_hits/(h_hits+h_misses);
#else
	ratio=(float) h_hits/(h_hits+h_misses)*100;
#endif
      else
	ratio=0;
#ifdef MINIX
      printf("Expected:\nhits %d     misses %d     hit ratio %d%%\n",h_hits,h_misses,ratio);
      printf("Table %d%% full.\n",100*(h_hits+h_misses)/HSHSIZ);
#else
      printf("Expected:\nhits %d     misses %d     hit ratio %.1f%%\n",h_hits,h_misses,ratio);
      printf("Table %.0f%% full.\n",(float)(h_hits+h_misses)/HSHSIZ*100);
#endif
      fflush(stdout);
    }
    else
      fprintf(stderr,"usage: hashstat [ x ]\n");
}
#endif

static
void History(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void history();

  history(argv);
}

static
void lock(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  char key1[64],key2[64];

  write(1,"Enter key:",10);
  scanf("%64s",key1);
  write(1,"\nAgain:",7);
  scanf("%64s",key2);
  if (!strcmp(key1,key2))
  {
    key1[0]=EOS;
    while (strcmp(key1,key2))
    {
      write(1,"\nKey:",5);
      scanf("%64s",key1);
    }
  }
  else write(1,"\nKeys are different!\n",21);
}

#ifdef __STDC__
void alias(int argc, char **argv, int infd, int outfd, int errfd,
	   char *ifil, char *ofil, char *efil, bool appnd, bool bckgnd)
#else
void alias(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
#endif
{
  extern void aliaslist(),makealias(),savealias(),editalias(),makelinealias();
#ifdef HASH
  extern bool hashed;
  extern void hashpath();
#endif
  FILE *fp;
  char line[MAXLL];
  int i;

#ifdef DEBUG
  fprintf(stderr,"In alias with argc %d argv[1] %s\n",argc,argv[1]);
#endif

  if (argc==1)
  {
    aliaslist(NULL);
    return;
  }
  if (argc==2)
  {
    aliaslist(argv[1]);
    return;
  }
  if (!strcmp(argv[1],"-e"))
    for (i=2;i<argc;i++)
      editalias(argv[2]);
  else if (!strcmp(argv[1],"-l"))
    for (i=2;i<argc;i++)
    {
      if ((fp=fopen(argv[i],"r"))==NULL)
      {
	fprintf(stderr,"Can't open file: %s\n",argv[i]);
	continue;
      }
      makealias(argv[i],fp);
      fclose(fp);
#ifdef HASH
      if (hashed==TRUE) hashpath();
#endif
    }
  else if (!strcmp(argv[1],"-s"))
    for (i=2;i<argc;i++)
    {
      if ((fp=fopen(argv[i],"w"))==NULL)
      {
	fprintf(stderr,"Can't open file: %s\n",argv[i]);
	continue;
      }
      savealias(argv[i],fp);
      fclose(fp);
    }
  else
  {
    line[0]=EOS;
    for (i=2;i<argc;i++)
    {
      if (i!=2) strcat(line," ");
      strcat(line,argv[i]);
    }
    makelinealias(argv[1],line);
#ifdef HASH
    if (hashed==TRUE) hashpath();
#endif
  }
}

static
void unalias(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void delalias();
#ifdef HASH
  extern bool hashed;
  extern void hashpath();
#endif
  int i;

  for (i=1;argv[i];i++)
    delalias(argv[i]);
#ifdef HASH
  if (hashed==TRUE) hashpath();
#endif
}

#ifdef HASH
static
void rehash(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void hashpath(),vset();

  vset("HASH","");
  hashpath();
}
#endif

static
void source(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void File(),resetsh();
#ifdef HASH
  bool oldhashed;
  extern bool hashed;
#endif
  extern char **arguments;
  extern int numargs,fromfile;
  FILE *fp;
  char **oldargstemp;
  int oldnumargtemp;

  if (argc==1)
  {
    write(2,"source: No source file given.\n",30);
    return;
  }
  if ((fp=fopen(argv[1],"r"))==NULL)
  {
    write(2,"Cannot open file: ",18);
    write(2,argv[1],strlen(argv[1]));
    write(2,"\n",1);
    return;
  }
#ifdef HASH
  oldhashed=hashed;
  hashed=FALSE;
#endif
  oldargstemp=arguments;
  arguments= &argv[1];
  oldnumargtemp=numargs;
  numargs=argc-1;
#ifdef DEBUG
fprintf(stderr,"Calling file to source %s\n",argv[1]);
#endif
  File(fp,1);
  fclose(fp);
  arguments=oldargstemp;
  numargs=oldnumargtemp;
#ifdef HASH
  hashed=oldhashed;
#endif
  if (!fromfile) resetsh();
}


#ifdef __STDC__
void logout(int argc, char **argv, int infd, int outfd, int errfd,
	    char *ifil, char *ofil, char *efil, bool appnd, bool bckgnd)
#else
void logout(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
#endif
{
  extern void resetsh(),setdown(),File();
  extern bool loginsh;
  extern int fromfile;
  FILE *fp;
  char dir[MAXPL];

  if (loginsh==FALSE)
  {
    write(2,"Not login shell\n",16);
    if (!fromfile) resetsh();
    return;
  }
  strcpy(dir,getenv("HOME"));
  strcat(dir,"/.logout");
  if ((fp=fopen(dir,"r"))!=NULL)
  {
    File(fp,1);
    fclose(fp);
  }
  setdown();
  exit(0);
}


static
void which(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void (*builtin_func)();
  extern struct adefn *alias_defn,*checkalias();
  extern int alias_depth,getpath();
  extern char *current_alias[],*vget();
  struct adefn *defn_top,*dptr;
  char path[MAXPL];
  int i,nometa;

  if (argc==1)
  {
    write(2,"usage: which [ -o | -O ] <program_name> ...\n",32);
    return;
  }
  i=1;nometa=0;
  if (!strcmp(argv[1],"-o"))
  {
    nometa=1;
    i++;
  }
  if (!strcmp(argv[1],"-O"))
  {
    nometa=2;
    i++;
  }
  for (;i<argc;i++)
  {
    alias_defn=NULL;
    builtin_func=NULL;
    if (getpath(path,argv[i],nometa)==FOUND)
    {
      if (alias_defn!=NULL)
      {
	printf("~#/%s aliased to: ",argv[i]);
	defn_top=0;
	defn_top=checkalias(argv[i]);
	if (defn_top)
	  for (dptr=defn_top;dptr;dptr=dptr->nextln)
	    printf("%s\n",dptr->a_line);
	else write(2,"No such alias. Looney!\n",23);
	continue;
      }
      if (builtin_func!=NULL)
      {
	printf("~#/%s is a Clam builtin\n",argv[i]);
	continue;
      }
      printf("%s\n",path);
    }
    else
      fprintf(stderr,"Can't find %s in %s\n",argv[i],vget("PATH"));
  }
}

static
void load(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern int fromfile,loadline;
  extern char saveline[];
  int i;

  if (fromfile)
  {
    write(2,"You can't use load in a script\n",31);
    return;
  }
  if (argc==1)
  {
    write(2,"usage: load line_text\n",22);
    return;
  }
  for (i=0;i<MAXLL;i++) saveline[i]=EOS;
  for (i=1;i<argc;i++)
  {
    if (i!=1) strcat(saveline," ");
    strcat(saveline,argv[i]);
  }
  loadline=1;
}

	/* Several of the following functions implement job control under
	   BSD 4.x . However, they don't work under non-BSD Unixes,so they
	   have been excluded by Ifdefs	*/

static int wait_int()
{
	printf ("wait: interrupted\n");
}

static
void Wait(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
#ifndef UCB
  int status;
  int (*oldsig)(), wait_int ();
  oldsig = signal (SIGINT, wait_int);/* Necess. for making wait inter. RR*/
  setdown();
  while ( wait(&status) != -1 ) ;	/* wait for all childs */
  resetsh();
  signal (SIGINT, oldsig);
#else
  /* I'll think about it */
#endif
}

#ifdef JOB
static
void suspend(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void resetsh(),setdown();
  extern bool loginsh;
  extern int fromfile;

  if (loginsh==TRUE)
  {
    write(2,"Can't suspend a login shell.\n",29);
    return;
  }
  setdown();
  signal(SIGTSTP,SIG_DFL);
  kill(0,SIGTSTP);
  if (!fromfile) resetsh();
  signal(SIGTSTP,SIG_IGN);
}


static
void jobs(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void joblist();

/* Have -i or -l flag which gives more info. (stuff in job struct) */
  joblist();
}

void Kill(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void newcurr();
  extern int isanum(),pidfromjob(),convert();
  char *c;
  int pid,sig,startjob;

  if (argc>1)
  {
    c=argv[1];
    if (*c=='-')
    {
      startjob=2;
      c++;
      if (isanum(c))
	sig=atoi(c);
      else
	sig=convert(c);
    }
    else
    {
      startjob=1;
      sig=SIGTERM;
    }
  }
  else
  {
    write(2,"usage: kill [ <signal> ] <pid>\n",31);
    return;
  }
  for (;argv[startjob];startjob++)
  {
    c=argv[startjob];
    if (*c=='%')
    {
      c++;
      if (!isanum(c) || ((pid=pidfromjob(atoi(c)))==-1))
      {
	fprintf(stderr,"No such job number: %s\n",c);
	continue;
      }
    }
    else
    {
      if (!isanum(c)) continue;		/* ignore bad pids */
      pid=atoi(c);
    }
#ifdef DEBUG
fprintf(stderr,"Sending %d to %d\n",sig,pid);
#endif
    if (kill(pid,sig))
      perror("kill");
    else if (sig==SIGTERM || sig==SIGHUP)
	   kill(pid,SIGCONT);
  }
  fflush(stderr);
  newcurr();
}


static
void fg(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern struct job *currentjob,*findjob();
  extern void setexec(),newcurr();
  extern int fromfile,waitfor(),update();
  struct job *ptr;
  char *c;
  int pid,i;

  if (fromfile)
  {
    write(2,"No job control in shell scripts.\n",33);
    return;
  }
  signal(SIGCHLD,SIG_IGN);		/* so that checkjobs doesn't update */
  if (argc==1)
  {
#ifdef DEBUG
fprintf(stderr,"Trying to use current job pointer\n");
#endif
    if (currentjob)
    {
      pid=currentjob->pid;				/* current job */
      if (pid!=getpgrp(pid))
      {
	fprintf(stderr,"Can't bring pid:%d to the foreground. Impossible?\n",pid);
	fflush(stderr);
	return;
      }
#ifdef DEBUG
      fprintf(stderr,"setting terminal pg to %d\n",pid);
#endif
      if (ioctl(0,TIOCSPGRP,&pid)) perror("fg spg");
      setdown();
      /*if (setpgrp(pid,getpgrp(0))==-1)  /* set process group to the shell's 
      {
        write(2,"Can't fg this pid\n",18);
        return;
      } *** This is wrong. All are in their own process group now */
      c=vget("cwd");
      if (strcmp(currentjob->dir,c))	/* if directory has changed */
	fprintf(stderr,"[%d] %d %s (wd now: %s)\n",currentjob->jobnumber,currentjob->pid,currentjob->name,currentjob->dir);
      else
	fprintf(stderr,"[%d] %d %s\n",currentjob->jobnumber,currentjob->pid,currentjob->name);
      fflush(stderr);
      update(pid,0);		/* clear status */
      newcurr();		/* change current job to this one (with luck) */
#ifdef DEBUG
      fprintf(stderr,"sending CONT to %d grp\n",pid);
#endif
      if (killpg(pid,SIGCONT)) perror("killpg");
      else (void) waitfor(pid);			/* do the waiting here. */
      return;
    }
    else
    {
      write(2,"No current job.\n",16);
      return;
    }
  }
  for (i=1;argv[i];i++)
  {
    c=argv[i];
    if (*c=='%')
    {
      c++;
      if (!isanum(c) || ((pid=pidfromjob(atoi(c)))==-1))
      {
	fprintf(stderr,"No such job number: %s\n",c);
	fflush(stderr);
	continue;
      }
    }
    else
    {
      if (!isanum(c)) continue;
      pid=atoi(c);	/* this has strange effects if pid not a child */
    }
#ifdef DEBUG
fprintf(stderr,"fg-ing %d\n",pid);
#endif
    if (pid!=getpgrp(pid))
    {
      fprintf(stderr,"Can't bring pid:%d to the foreground.\n",pid);
      fflush(stderr);
      continue;
    }
    setexec(pid);	/* 
    /*if (setpgrp(pid,getpgrp(0))==-1)	/* set process group to the shell's 
    {
      write(2,"Can't fg this pid\n",18);
      return;
    } *** This is wrong. All things are in their own group now */
    ptr=findjob(pid);
    c=vget("cwd");
    if (strcmp(ptr->dir,c))	/* if directory has changed */
      fprintf(stderr,"[%d] %d %s (wd now: %s)\n",ptr->jobnumber,ptr->pid,ptr->name,ptr->dir);
    else
      fprintf(stderr,"[%d] %d %s\n",ptr->jobnumber,ptr->pid,ptr->name);
    fflush(stderr);
    update(pid,0);		/* clear status */
    newcurr();			/* change current job to this one (with luck) */
#ifdef DEBUG
    fprintf(stderr,"sending CONT to %d pg\n",pid);
#endif
    if (killpg(pid,SIGCONT)) perror("killpg");
    else (void) waitfor(pid);			/* do the waiting here. */
  }
}

static
void bg(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern struct job *currentjob,*findjob();
  extern void newcurr();
  extern int fromfile,update();
  struct job *ptr;
  char *c;
  int pid,i;

  if (fromfile)
  {
    write(2,"No job control in shell scripts.\n",33);
    return;
  }
  if (argc==1)
  {
    if (currentjob)
    {
      pid=currentjob->pid;				/* current job */
      /*if (setpgrp(pid,pid)==-1)
      {
        write(2,"Can't bg this pid\n",18);
        return;
      }   *** Don't need this, already done now */
      fprintf(stderr,"[%d] %d %s\n",currentjob->jobnumber,currentjob->pid,currentjob->name);
      fflush(stderr);
      update(pid,0);		/* clear status */
      newcurr();		/* change current job to this one (with luck) */
#ifdef DEBUG
      fprintf(stderr,"killpg on pid %d\n",pid);
#endif
      if (killpg(pid,SIGCONT)) perror("killpg");
    }
    else
    {
      write(2,"No current job.\n",16);
      return;
    }
  }
  for (i=1;argv[i];i++)
  {
    c=argv[i];
    if (*c=='%')
    {
      c++;
      if (!isanum(c) || ((pid=pidfromjob(atoi(c)))==-1))
      {
	fprintf(stderr,"No such job number: %s\n",c);
	fflush(stderr);
	continue;
      }
    }
    else
    {
      if (!isanum(c)) continue;
      pid=atoi(c);
    }
    /*if (setpgrp(pid,pid)==-1)
    {
      write(2,"Can't bg this pid\n",18);
      return;
    }  *** Don't need this, now, already done */
    ptr=findjob(pid);
    fprintf(stderr,"[%d] %d %s\n",ptr->jobnumber,ptr->pid,ptr->name);
    fflush(stderr);
    update(pid,0);		/* clear status */
    newcurr();			/* change current job to this one (with luck) */
    if (killpg(pid,SIGCONT)) perror("killpg");
  }
}

static
void stop(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void newcurr();
  extern struct job *currentjob;
  char *c;
  int pid,i;

  if (argc==1)
  {
    if (currentjob)
    {
      pid=currentjob->pid;				/* current job */
      if (kill(pid,SIGSTOP)== -1)
	perror("kill");
      newcurr();
      return;
    }
    else
    {
      write(2,"No current job.\n",16);
      return;
    }
  }
  for (i=1;argv[i];i++)
  {
    c=argv[i];
    if (*c=='%')
    {
      c++;
      if (!isanum(c) || ((pid=pidfromjob(atoi(c)))==-1))
      {
	fprintf(stderr,"No such job number: %s\n",c);
	fflush(stderr);
	continue;
      }
    }
    else
    {
      if (!isanum(c)) continue;
      pid=atoi(c);
    }
    if (kill(pid,SIGSTOP)== -1)
      perror("kill");
  }
  newcurr();
}
#endif


#ifdef SCRIPT
static
void shift(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern char **arguments;
  extern int numargs;

  numargs--;
  arguments= &arguments[1];
}


static
void Case(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
}

static
void If(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
}

static
void While(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
}

static
void For(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
}

static
void repeat(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
}

static
void Read(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern FILE *zin;
  extern void vadd();
  extern int fromfile,getnextword();
  extern bool getfileline();
#ifdef ATT
  struct termio tbuf;
#else
  struct sgttyb termod;
#endif
  char line[MAXLL],word[MAXWL];
  int pos,i;

  for (i=0;i<MAXLL;i++) line[i]=EOS;
  if (!fromfile)
  {
#ifdef ATT
    if (ioctl(0,TCGETA,&tbuf)) perror("ioctl in Read");
    tbuf.c_lflag |= ICANON | ECHO;
    if (ioctl(0,TCSETA,&tbuf)) perror("ioctl1 R");
#else
    if (ioctl(0,TIOCGETP,&termod)) perror("ioctl in Read");
    termod.sg_flags &= (~CBREAK);
    termod.sg_flags |= ECHO;
    if (ioctl(0,TIOCSETN,&termod)) perror("ioctl2 R");
#endif
  }
  (void) getfileline(line,zin,1);
#ifdef DEBUG
fprintf(stderr,"calling getnextword\n");
#endif
  pos=0;
  for (i=1;i<argc;i++)
    if (getnextword(line,&pos,word))
      vset(argv[i],word);
    else
      vset(argv[i],"");
#ifdef DEBUG
fprintf(stderr,"while\n");
#endif
  if (argc>1)
    while (getnextword(line,&pos,word))
    {
      vadd(argv[argc-1]," ");vadd(argv[argc-1],word);
    }
  if (!fromfile)
  {
#ifdef ATT
    if (ioctl(0,TCGETA,&tbuf)) perror("ioctl in Read");
    tbuf.c_lflag &= (~ICANON) & (~ECHO);
    if (ioctl(0,TCSETA,&tbuf)) perror("ioctl3 R");
#else
    if (ioctl(0,TIOCGETP,&termod)) perror("ioctl in Read");
    termod.sg_flags |= CBREAK;
    termod.sg_flags &= (~ECHO);
    if (ioctl(0,TIOCSETN,&termod)) perror("ioctl4 R");
#endif
  }
#ifdef DEBUG
fprintf(stderr,"finished\n");
#endif
}
#endif	/* of script functions */


#ifndef MINIX
static
void Help(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
/* may not be implemented */
}

void limit(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  extern void printlimit(),setlimit();

  if (argc>3)
  {
    write(2,"usage: limit [ <resource>  [ <limit> ] ]\n",41);
    return;
  }
  if (argc<3)
    printlimit(argv[1]);
  else
    setlimit(argv[1],argv[2]);
}

void unlimit(argc,argv,infd,outfd,errfd,ifil,ofil,efil,appnd,bckgnd,piped)
  int argc,infd,outfd,errfd;
  char *argv[],*ifil,*ofil,*efil;
  bool appnd,bckgnd,piped;
{
  if (argc>2)
  {
    write(2,"usage: unlimit [ <resource> ]\n",30);
    return;
  }
}
#endif

void (*checkbuiltins(name))()
  char *name;
{
/* this will be changed to use hashing */

  if (!strcmp(name,"cd") || !strcmp(name,"chdir"))
    return(Cd);
  if (!strcmp(name,"logout"))
    return(logout);
  if (!strcmp(name,"exit"))
    return(Exit);
  if (!strcmp(name,"unset"))
    return(unset);
  if (!strcmp(name,"unsetenv"))
    return(unsetenv);
  if (!strcmp(name,"list"))
    return(list);
  if (!strcmp(name,"history"))
    return(History);
  if (!strcmp(name,"exec"))
    return(Exec);
  if (!strcmp(name,"export"))
    return(Export);
  if (!strcmp(name,"edit"))
    return(edit);
  if (!strcmp(name,"lock"))
    return(lock);
  if (!strcmp(name,"alias"))
    return(alias);
  if (!strcmp(name,"unalias"))
    return(unalias);
  if (!strcmp(name,"source"))
    return(source);
  if (!strcmp(name,"which"))
    return(which);
  if (!strcmp(name,"load"))
    return(load);
#ifdef HASH
  if (!strcmp(name,"hashstat"))
    return(hashstat);
  if (!strcmp(name,"rehash"))
    return(rehash);
#endif
#ifdef JOB
  if (!strcmp(name,"suspend"))
    return(suspend);
  if (!strcmp(name,"jobs"))
    return(jobs);
  if (!strcmp(name,"kill"))
    return(Kill);
  if (!strcmp(name,"fg"))
    return(fg);
  if (!strcmp(name,"bg"))
    return(bg);
  if (!strcmp(name,"stop"))
    return(stop);
#endif
  if (!strcmp(name,"wait"))
    return(Wait);
#ifdef SCRIPT
  if (!strcmp(name,"case"))
    return(Case);
  if (!strcmp(name,"for"))
    return(For);
  if (!strcmp(name,"if"))
    return(If);
  if (!strcmp(name,"while"))
    return(While);
  if (!strcmp(name,"repeat"))
    return(repeat);
  if (!strcmp(name,"shift"))
    return(shift);
  if (!strcmp(name,"help"))
    return(Help);
  if (!strcmp(name,"read"))
    return(Read);
#endif
#ifndef MINIX
  if (!strcmp(name,"limit"))
    return(limit);
  if (!strcmp(name,"unlimit"))
    return(unlimit);
#endif
  return(NULL);
}
