/* This file contains the multiprocessor specification for intel architecture:
   - Local APIC
   - I/O APIC (still not used)
   - MP table reading
   - Start and Stop MP system
   - CPU hadling

  Other files related: mp.h, xmp.s, xmp.h, mpinfo.c, mptable.h

  xmp.s		is an extension of mp.c, with assembler routines
  xmp.h		is the header file for xmp.s
  mp.h		is the header file for mp.c
  mpinfo.c	contains routines for printing MP information during MP start
  mptable.h	contains definitions of MP table structures
 */

#include <minix/config.h>    /* for mp configuration */

#include "kernel.h"
#include "mp.h"
#include "xmp.h"
#include "mptable.h"
#include "proc.h"
#include <minix/com.h>		/* for CLOCK */

struct floating_pointer fps;	/* Structures for loading configuration */
struct mp_config_header mph;
struct io_apic_entry ioapic;

extern int appm_flag;	/* flag to detect AP protected mode activity */


int mp_command[MP_NR_CPUS];	/* command for cpu to execute in IPI */
int mp_param[MP_NR_CPUS];	/* paremeter of IPI command */
int cpu_available[MP_NR_CPUS];	/* is this cpu running now? */

int ap_running_flag;		/* is new cpu alive? */

/**************************************************************************/
/** LOCAL APIC BASIC HANDLING DEFINITIONS				 **/
/**************************************************************************/

u32_t local_apic_base=0x0FEE00000;

#define LOCAL_APIC_ICR_LOW	0x0300	/* offset for IRC low regiter */
#define LOCAL_APIC_ICR_HIGH	0x0310	/* offset for IRC high register */
#define LOCAL_APIC_ESR		0x0280	/* offset for ESR register */
#define LOCAL_APIC_SPIV		0x00F0	/* offset for SPIV register */
#define LOCAL_APIC_TASKP	0x0080	/* offset for TASK PRIORITY register */
#define LOCAL_APIC_LDR		0x00D0	/* offset for LD register */
#define LOCAL_APIC_DFR		0x00E0	/* offset for DF register */
#define LOCAL_APIC_IDR		0x0020	/* offset for ID register */
#define LOCAL_APIC_LVTL0	0x0350	/* offset for LVT LINTIN0 */
#define LOCAL_APIC_LVTL1	0x0360	/* offset for LVT LINTIN1 */
#define LOCAL_APIC_EOI		0x00B0	/* offset for EOI register */

#define LVT_DM_FIXED		0	/* delivery mode in L.APIC LVT */
#define LVT_DM_NMI		4	/* delivery mode in L.APIC LVT */
#define LVT_DM_EXTINT		7	/* delivery mode in L.APIC LVT */

#define LVT_MASKED_SHIFT	16	/* shift for masked LINTINx bit */
#define LVT_DM_SHIFT		8	/* shift for delivery mode field */

#define DELIVERY_FIXED	0	/* 000 = FIXED */
#define DELIVERY_INIT	5	/* 101 = INIT IPI */
#define DELIVERY_STARTUP 6	/* 110 = STARTUP IPI */
#define DELIVERY_SHIFT	8	/* delivery bit position in dword */
#define LEVEL_SHIFT 	14	/* level bit position in dword */
#define TRIGGER_SHIFT	15	/* trigger mode bit position in dword */

#define PHYSICAL_DEST	0	/* physical destination addr mode */
#define LOGICAL_DEST	1	/* logical destination addr mode */
#define DEST_MODE_SHIFT	11	/* destination mode bit position in dword */

#define DELIVERY_STATUS_SHIFT	12	/* status of ipi */
#define DELIVERY_PENDING	1	/* ipi is being sent */
#define DELIVERY_IDLE	 	0	/* sent: no work to be done */
#define DELIVERY_STATUS  	(LOCAL_APIC_READ(LOCAL_APIC_ICR_LOW) \
                                              & (1<<DELIVERY_STATUS_SHIFT))

#define DEST_FIELD	0	/* destination by field */
#define DEST_SELF	1	/* destination is self cpu */
#define DEST_ALL	2	/* destination is all cpus */
#define DEST_OTHERS	3	/* destination is all cpus but self */
#define DEST_SHORT_SHIFT 18	/* destination shorthand bit position */

#define DEST_FIELD_SHIFT (56-32)	/* destination field bits position */

#define ENABLE_APIC_SHIFT	8	/* enable APIC bit position in SPIV reg */




void LOCAL_APIC_WRITE(u32_t reg, u32_t val) {
   phys_copy_dword( vir2phys(&val), local_apic_base+reg );
}

u32_t LOCAL_APIC_READ(u32_t reg) {
   u32_t val;
   phys_copy_dword( local_apic_base+reg, vir2phys(&val) );
   return val;
}


/**************************************************************************/
/** IO APIC BASIC HANDLING DEFINITIONCMOS				 **/
/**************************************************************************/

u32_t io_apic_base=0xFEC00000;

#define IO_APIC_ACC_REG_SEL	0x00
#define IO_APIC_ACC_WIN_REG	0x10

#define IO_APIC_ID_REG			0x00
#define IO_APIC_VERSION_REG		0x01
#define IO_APIC_ARBITR_REG		0x02
#define IO_APIC_REDTBL_REG_LOW(n)	(0x10 + (2 * n))
#define IO_APIC_REDTBL_REG_HIGH(n)	(0x11 + (2 * n))

#define IOA_ID_SHIFT	24	/* id field in io-apic id reg. */
#define IOA_VER_SHIFT	0	/* version field in io-apic version reg. */
#define IOA_MXENT_SHIFT	16	/* max entries field in io-apic version reg. */
#define IOA_ARB_SHIFT	24	/* arb. field in io-apic arbitration reg. */
#define IOA_ID_MASK	(0x0F<<IOA_ID_SHIFT)
#define IOA_VER_MASK	(0xFF<<IOA_VER_SHIFT)
#define IOA_MXENT_MASK	(0xFF<<IOA_MXENT_SHIFT)
#define IOA_ARB_MASK	(0x0F<<IOA_ARB_SHIFT)

void IO_APIC_WRITE(u32_t reg, u32_t val) {
   reg &= 0xFF;
   phys_copy_dword( vir2phys(&reg), io_apic_base+IO_APIC_ACC_REG_SEL);
   phys_copy_dword( vir2phys(&val), io_apic_base+IO_APIC_ACC_WIN_REG);
}

u32_t IO_APIC_READ(u32_t reg) {
   u32_t val;
   phys_copy_dword( vir2phys(&reg), io_apic_base+IO_APIC_ACC_REG_SEL);
   phys_copy_dword( io_apic_base+IO_APIC_ACC_WIN_REG, vir2phys(&val));
   return val;
}



/**************************************************************************/
/** PRINTING MP INTIALIZACION STATUS ON CONSOLE				 **/
/**************************************************************************/

char MP_STATUS[1000];		/* MP starting messages */
char *actpos=MP_STATUS;		/* printing position */

#define XADDS_MP_STATUS(x)
#define XADDN_MP_STATUS(a,b,c)	

void ADDS_MP_STATUS(char *mensaje)
{
   while(mensaje[0]) {
      actpos[0]=mensaje[0];
      actpos++;
      mensaje++;
   } 
   actpos[0]=0;
}


void ADDN_MP_STATUS(u32_t numero, u32_t base, u32_t tam)
{
   char buffer[33]; 
   long int i;
   u32_t n;
   int d;

   if ((base<1) || (base>16)) base=10;
   i=0;
   n=numero;
   if (tam<1) tam=1;

   while(n>0) {n/=base;i++;}
   if (i<tam) i=tam;
   buffer[i--]=0;
   while (i>=0) {
      d=numero%base;
      switch (d) {
        case 0: case 1: case 2: case 3: case 4: 
        case 5: case 6: case 7: case 8: case 9:
           buffer[i--]='0'+d;
           break;
        default :
           buffer[i--]='A'-10+d;
           break;
      }
      numero /= base;
   }
   ADDS_MP_STATUS(buffer);
}



#include "mpinfo.c"


/**************************************************************************/
/** CMOS BASIC HANDLING DEFINITIONS                                      **/
/**************************************************************************/

void cmos_write(u8_t addr, u8_t value) {
   out_byte(0x70, addr);
   out_byte(0x71, value);
}

u8_t cmos_read(u8_t addr) {
   out_byte(0x70, addr);
   return in_byte(0x71);
}


/**************************************************************************/
/**  FLOATING POINTER STRUCTURE AND MP TABLE                             **/
/**************************************************************************/

/*===========================================================================*
 *				test_fps_checksum			     *
 *===========================================================================*/
int test_fps_checksum(struct floating_pointer *fps) {
   unsigned char *data, checksum;
   int i;
 
/* Check in a FPS structure for a valid signature and checksum 
   Return noncero on success */

   /* fisrt look for signature */
   for (i=0; i<4; i++)
      if (fps->fp_signature[i] != SIGN_FPS[i]) return 0;

   /* then calculate checksum*/
   data=(unsigned char *)fps;
   checksum=0;
   for (i=0; i<sizeof (struct floating_pointer); i++)
      checksum+= data[i];
   return (!checksum);  /* checksum ok if sums 0 */
}



/*===========================================================================*
 *				load_fps				     * 
 *===========================================================================*/
int load_fps(void) {
/* Find and load a valid FPS structure. Retun nonzero on success */

   unsigned addr;

   /* for the moment, scan the whole base memory, in
      16-byte steps */
   for (addr=0xF5B30; addr<0xFFFF0; addr+=0x10) {
      phys_copy ( addr , vir2phys(&fps) , sizeof(fps) );
      if (test_fps_checksum(&fps)) {
         /*ADDS_MP_STATUS("MP floating pointer struct found at 0x");
         ADDN_MP_STATUS(addr,16,0);
         ADDS_MP_STATUS("\n");*/
         return 1;
      }
   }
   ADDS_MP_STATUS("MP scan FAILLED! MP config table not found in base memory\n");
   return 0;
}


/*===========================================================================*
 *			test_mptable_checksum				     *
 *===========================================================================*/
int test_mptable_checksum(void) {
/* Cheks a MP table header structure for a valid signature and checksum. 
   Return noncero on success */

   unsigned char data, checksum;
   int i;

   /* first match the signature */
   for (i=0; i<4; i++) 
      if (mph.mpch_signature[i] != SIGN_MP_C_HEADER[i] )
         return 0;

   /* then calculate checksum */
   checksum=0;
   for (i=0; i<mph.mpch_length; i++) {
      /* the complete table is in another segment, so we need copy 
         each byte into kernel address space (mph is only the header) */
      phys_copy(fps.fp_mp_table+i, vir2phys(&data), 1);
      checksum +=data;
   }
   return (!checksum);  /* checksum ok if sums 0 */
}



/*===========================================================================*
 *				load_mph				     * 
 *===========================================================================*/
int load_mph(void) {
/* Find and load a valid MP config header struct. Return noncero on success */

   unsigned addr;

   phys_copy ( fps.fp_mp_table , vir2phys(&mph) , sizeof(mph) );
   if (test_mptable_checksum()) return 1;
   ADDS_MP_STATUS("MP config table checksum FAILLED!\n");
   return 0;
}



/*===========================================================================*
 *			process_mp_configuration			     * 
 *===========================================================================*/
void process_mp_configuration(void) {
/* Scan MP table and find some information about MP configuration */

   u32_t next;
   int cpu_count, bus_count, apic_count, ioint_count, lint_count;
   int i=mph.mpch_entry_count;
   u8_t this_entry;
   struct io_apic_entry	ae;
   struct cpu_entry	ce;
   struct io_int_entry	ioie;
   struct l_int_entry	lie;

   cpu_count= bus_count= apic_count= ioint_count= lint_count= 0;

   next=fps.fp_mp_table+sizeof (struct mp_config_header);
   while (i-- > 0) {
      phys_copy(next, vir2phys(&this_entry), 1);
      switch (this_entry) {
         case CPU_ENTRY_ID : 
            if (++cpu_count > 2)
               ADDS_MP_STATUS("MP PANIC: only 2 cpus currently supported!\n");
            phys_copy(next,vir2phys(&ce),sizeof(struct cpu_entry));
            next+=sizeof(struct cpu_entry);
            PRINT_CPU_ENTRY(cpu);
            if ((ce.ce_flags & CE_EN_FLAG_MASK)==0) 
               ADDS_MP_STATUS("MP PANIC: disabled apics not currently supported\n");
            break;
         case BUS_ENTRY_ID :
            bus_count++;
            next+=sizeof(struct bus_entry);
            break;
         case IO_APIC_ENTRY_ID :
            if (++apic_count > 1)
               ADDS_MP_STATUS("MP PANIC: only 1 I/O APIC currently supported!\n");
            phys_copy(next,vir2phys(&ae),sizeof(struct io_apic_entry));
            next+=sizeof(struct io_apic_entry);
            /*PRINT_IO_APIC_ENTRY(ae);*/
            if ((ae.ae_flags & AE_EN_FLAG_MASK)==0) 	
               ADDS_MP_STATUS("MP PANIC: disabled apic not currently supported!");
            io_apic_base=ae.ae_address;
            break;
         case IO_INT_ENTRY_ID :
            ioint_count++;
            phys_copy(next,vir2phys(&ioie),sizeof(struct io_int_entry));
            next+=sizeof(struct io_int_entry);
            /*PRINT_IO_INT_ENTRY(ioie);*/
            break;
         case LOCAL_INT_ENTRY_ID :
            lint_count++;
            phys_copy(next,vir2phys(&lie),sizeof(struct l_int_entry));
            next+=sizeof(struct l_int_entry);
            /*PRINT_LOCAL_ENTRY(lie);*/
            break;
         default :
            ADDS_MP_STATUS("MP config table corrupt!\n");
            i=0; /* break loop */
            break;
      }
   } 
   /*PRINT_MP_SUMMARY(cpu_count,bus_count,apic_count,
		      ioint_count,lint_count);*/
   if ((apic_count!=1))
      ADDS_MP_STATUS("MP PANIC: only 1 I/O APIC currently supported!\n");
}



/*===========================================================================*
 *				disable_operand_size				     * 
 *===========================================================================*/
void disable_operand_size(u32_t trampoline_addr, u32_t trampoline_sz) {
/* Change operand-size modifier codes (66h & 67h) on trampoline's machine 
   code introduced by assembler and replace they with <nop> code (90h) */

   u16_t code;
   u32_t code_addr=vir2phys(&code);

   while (trampoline_sz>1) {	/* last byte not necessary to scan */
      phys_copy(trampoline_addr, code_addr, 2);
      if ((code==0x6766)) {	/* o16 a16 */
         code=0x9090;		/* nop nop */
         phys_copy(code_addr, trampoline_addr, 2);
         trampoline_addr+=2;
         trampoline_sz-=2;
      }
      else {
         trampoline_addr++;
         trampoline_sz--;
      }
   }   
} 


/*===========================================================================*
 *				find_trampoline				     * 
 *===========================================================================*/
void init_ap(void);
void end_init_ap(void);
/* Declaration of this points in xmp.s */

u32_t find_trampoline(void) {
/* Find a memory zone suitable for allocating trampoline code.
   It will be a 0's zone, suposing this is free. */

   u32_t addr8;
   u32_t addr;
   u32_t i;
   u8_t c;

   /* size of tampoline code */
   u32_t tramp_len=(u32_t)end_init_ap-(u32_t)init_ap; 

   /* Look for a hole of free memory in each valid position of base memory */
   /* For some reason 0x0000F000 is not valid!! so start form 0x10 */
   for (addr8=0x11; addr8<0x100; addr8++) {
      if (addr8==0xA0) addr8=0xC0;  /* vectors A0..BF are reserved */  
      addr=addr8<<12;               /* aligned in 4kb boundaries */
      for (i=0; i<tramp_len; i++) {
         phys_copy(addr+i, vir2phys(&c), 1);
         if (c) break;
      }
      if (i==tramp_len) {
         /* Prepare and copy the tampoline */
         copy_registers_to_trampoline();
         phys_copy(vir2phys(init_ap), addr, tramp_len);
         disable_operand_size(addr,tramp_len);
         /*dump_trampoline(addr,tramp_len);*/
         return addr;                                   /* return it */
      }
   }
   return 0;
}

/*===========================================================================*
 *				free_trampoline				     * 
 *===========================================================================*/
void free_trampoline(u32_t addr) {
/* Restore with 0's the memory zone used for trampoline */

   char dummy=0;
   u32_t tramp_len=(u32_t)end_init_ap-(u32_t)init_ap; 
   while (tramp_len--) phys_copy((u32_t)&dummy,addr++,1);
}


/**************************************************************************/
/** HANDLING AND COMUNICATING PROCESSORS				 **/
/**************************************************************************/

/*===========================================================================*
 *				apic_error_status			     * 
 *===========================================================================*/
u8_t apic_error_status(void) {
/* Return Local APIC error status */

   LOCAL_APIC_WRITE(LOCAL_APIC_ESR,0);  /* clear previous error */
   return (LOCAL_APIC_READ(LOCAL_APIC_ESR) & 0xEF);
}



/*===========================================================================*
 *			wait_for_ipi_completion				     * 
 *===========================================================================*/
void wait_for_ipi_completion(void) {
/* Wait until current IPI is delivered */

   u32_t status = (DELIVERY_STATUS>>DELIVERY_STATUS_SHIFT);
   int count=0,err;

   if (status!=DELIVERY_PENDING) return;
   err=0;
   while ( (DELIVERY_STATUS == (DELIVERY_PENDING<<DELIVERY_STATUS_SHIFT)) 
        && (err==0) ){
      err=apic_error_status();
      count++;
      if (count>2000) {
         ADDS_MP_STATUS("ERROR: IPI timed out after 2000 loops\n");
         break;
      }
   }
   if (err) {
      ADDS_MP_STATUS("APIC IPI ERROR: STATUS ");
      ADDN_MP_STATUS(err, 2, 8);
      ADDS_MP_STATUS("\n"); 
   }
}


 
/*===========================================================================*
 *				send_init_ipi				     * 
 *===========================================================================*/
void send_init_ipi(u32_t trampoline_addr, int which_cpu) {
/* Send INIT IPI to a processor in order to start execution in it.
   First, send an INIT assert, and then, an INIT deassert */

   u32_t icr_h,icr_l;
   u16_t value;
   u8_t old_cmos;
   u32_t old_vector;

   /* Reprogram warm reset: write code 0A at CMOS addr 0F */
   old_cmos=cmos_read(0x0F);
   cmos_write(0x0F, 0x0A);

   /* save old reset vector at 40:67 (dw) */
   phys_copy(0x467,vir2phys(&old_vector), sizeof(u32_t));

   /* program reset vector to point at trampoline */
   value=(trampoline_addr >> 4);          /* trampoline segment */
   phys_copy(vir2phys(&value), 0x469, sizeof(u16_t));
   value=(trampoline_addr & 0xF);         /* trampoline offset */
   phys_copy(vir2phys(&value), 0x467, sizeof(u16_t));

   /* prepare to send INIT IPI */
   icr_h = LOCAL_APIC_READ(LOCAL_APIC_ICR_HIGH);
   icr_l = LOCAL_APIC_READ(LOCAL_APIC_ICR_LOW);

   icr_l &= 0xFFF00000;  /* clear */
   icr_h &= 0xF0FFFFFF;  /* clear */

   icr_l |= (DELIVERY_INIT<<DELIVERY_SHIFT);	/* stablish INIT */
   icr_l |= (1 << LEVEL_SHIFT);			/* level = assert */
   icr_l |= (1 << TRIGGER_SHIFT);		/* trigger = level */
   icr_l |= (PHYSICAL_DEST << DEST_MODE_SHIFT);	/* destination = physical */
   icr_l |= (DEST_FIELD << DEST_SHORT_SHIFT);	/* destination by field */

   icr_h |= (which_cpu << DEST_FIELD_SHIFT);	/* cpu to interrupt */

   /* send the IPI (assert) */
   apic_error_status();		/* first, clear previous errors */
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_HIGH, icr_h);
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_LOW,  icr_l);
   wait_for_ipi_completion();

   /* send the IPI (deassert) */
   apic_error_status();		/* first, clear previous errors */
   icr_l ^= (1<<LEVEL_SHIFT);	/* switch to deassert */
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_HIGH, icr_h);
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_LOW,  icr_l);
   wait_for_ipi_completion();

   /* restore old reset vector and cmos shutdown code */
   phys_copy(vir2phys(&old_vector), 0x467, sizeof(u32_t));
   cmos_write(0x0F, old_cmos);	
}



/*===========================================================================*
 *				send_startup_ipi			     * 
 *===========================================================================*/
void send_startup_ipi(u32_t trampoline_addr, int which_cpu) {
/* Send STARTUP IPI to a processor in order to start execution in it */

   u32_t icr_h,icr_l;

   if (DELIVERY_STATUS) {
      ADDS_MP_STATUS("PANIC: APIC is busy, no startup IPI has been sent!!!\n");
      return;
   } 
   /* put addr in 8 bit lower */
   trampoline_addr = (trampoline_addr >> 12) & 0xFF;

   /* prepare to send STARTUP IPI */
   icr_h = LOCAL_APIC_READ(LOCAL_APIC_ICR_HIGH);
   icr_l = LOCAL_APIC_READ(LOCAL_APIC_ICR_LOW);
 
   icr_l &=~0xCDFFF;  /* clear */
   icr_h &= 0xF0FFFFFF;  /* clear */

   icr_l |= (DELIVERY_STARTUP<<DELIVERY_SHIFT);	/* stablish STARTUP */
   icr_l |= (1 << LEVEL_SHIFT);			/* level = assert */
   icr_l &= ~(1 << TRIGGER_SHIFT);		/* trigger = edge */
   icr_l |= (PHYSICAL_DEST << DEST_MODE_SHIFT);	/* destination = physical */
   icr_l |= (DEST_FIELD << DEST_SHORT_SHIFT);	/* destination by field */

   icr_l |= trampoline_addr;			/* vector field */
   icr_h |= (which_cpu << DEST_FIELD_SHIFT);	/* cpu to interrupt */

   /* send the IPI */
   apic_error_status();
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_HIGH, icr_h);
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_LOW,  icr_l);
   wait_for_ipi_completion();
}



/*===========================================================================*
 *				AP_running				     * 
 *===========================================================================*/
int AP_running(void) {
/* Wait for an AP to write an value in the flag that means it is running */
   int max_loops=100;

   while ((ap_running_flag!=AP_LIFE_FLAG_MARK) && (--max_loops)) 
	milli_delay(1);
   return ap_running_flag==AP_LIFE_FLAG_MARK;
}


/*===========================================================================*
 *			forward_vector_to_cpu				     * 
 *===========================================================================*/
void forward_vector_to_cpu(u8_t vector, int cpu) {
/* Send a interrupt vector to a CPU */
   u32_t icr_h,icr_l;

   while (DELIVERY_STATUS); 
   /* Wait for local APIC to complete previous IPI */
   
   /* prepare to send IPI */
   icr_h = LOCAL_APIC_READ(LOCAL_APIC_ICR_HIGH);
   icr_l = LOCAL_APIC_READ(LOCAL_APIC_ICR_LOW);
 
   icr_l &=~0x000CDFFF;  /* clear non-reserved fields */
   icr_h &= 0x00FFFFFF;  /* clear non-reserved fields */

   /* Next inetructions does nothig because the values are all 0's,
      so, skip it for speed */
 /*icr_l |= (DELIVERY_FIXED<<DELIVERY_SHIFT);*/	/* stablish FIXED (000) */
 /*icr_l &= ~(1 << TRIGGER_SHIFT);*/		/* trigger = edge */
 /*icr_l |= (PHYSICAL_DEST << DEST_MODE_SHIFT);*/	/* destination = physical */
 /*icr_l |= (DEST_FIELD << DEST_SHORT_SHIFT);*/	/* destination by field */

   icr_l |= vector;				/* vector field */
   icr_h |= (cpu << DEST_FIELD_SHIFT);		/* cpu to interrupt */

   /* Clear previous error???? We dont hand it, so skip for speed */
   /*apic_error_status();*/

   /* send the IPI */
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_HIGH, icr_h);
   LOCAL_APIC_WRITE(LOCAL_APIC_ICR_LOW,  icr_l);
}



/*===========================================================================*
 *				interrupt_cpu				     * 
 *===========================================================================*/
void interrupt_cpu(int cpu, int command, int param) {
/* Send a message to a CPU consisting in a command and an optional parameter */

   mp_command[cpu]=command;
   mp_param[cpu]=param;
   forward_vector_to_cpu(VECTOR(16),cpu);
}


/*===========================================================================*
 *				mp_IPI_c				     * 
 *===========================================================================*/
void mp_IPI_c(void) {
/* This is the C section of mp_IPI, the interrupt handler of IPIs used for
   inter-CPU messaging */

   int cpu;

   LOCAL_APIC_WRITE(LOCAL_APIC_EOI,0); /* AKC interrupt to local APIC */

   cpu=this_cpu;
   switch(mp_command[cpu]) {
      case MP_CM_NONE:
	break;
      case MP_CM_DISABLE:
	disable_cpu(cpu,mp_param[cpu]);
	break;
      case MP_CM_ENABLE:
	enable_cpu(cpu,mp_param[cpu]);
	break;
      case MP_CM_PICKPROC:
        if (proc_ptr[cpu]==proc_addr_IDLE[cpu])
		lock_pick_proc();
	break;
      case MP_CM_SCHED:
	if (isuserp(proc_ptr[cpu])) lock_pick_proc();
	break;
      case MP_CM_REBOOT:
	wreboot(mp_param[cpu]);
	break;
      case MP_CM_HALT:
	halt_cpu(cpu);
	break;
   }
   mp_command[cpu]=MP_CM_NONE;	/* done */
}



/*===========================================================================*
 *				enable_apic_ints			     * 
 *===========================================================================*/
void enable_apic_ints(void) {
/* Enable current APIC and open to interrputs from PIC */
   u32_t reg;

   reg = LOCAL_APIC_READ(LOCAL_APIC_SPIV);
   reg |= (1<<ENABLE_APIC_SHIFT);		/* Enable APIC */
   LOCAL_APIC_WRITE(LOCAL_APIC_SPIV, reg); 

   reg = LOCAL_APIC_READ(LOCAL_APIC_LVTL0);
   reg &= ~(7<<LVT_DM_SHIFT);			/* clear delivery mode */
   reg &= ~(1<<LVT_MASKED_SHIFT);		/* unmask LINTIN0 */
   reg |= (LVT_DM_EXTINT<<LVT_DM_SHIFT);	/* ExtINT at LINTINT0 */
   LOCAL_APIC_WRITE(LOCAL_APIC_LVTL0, reg);

   reg = LOCAL_APIC_READ(LOCAL_APIC_LVTL1);
   reg &= ~(7<<LVT_DM_SHIFT);			/* clear delivery mode */
   reg &= ~(1<<LVT_MASKED_SHIFT);		/* ummask LINTIN1 */
   reg |= (LVT_DM_NMI<<LVT_DM_SHIFT);		/* NMI at LINTINT1 */
   LOCAL_APIC_WRITE(LOCAL_APIC_LVTL1, reg);

}



/*===========================================================================*
 *				disable_apic_ints			     * 
 *===========================================================================*/
void disable_apic_ints(void) {
/* Disable current APIC and close interrupts from PIC */
   u32_t reg;

   reg = LOCAL_APIC_READ(LOCAL_APIC_SPIV);
   reg &= ~(1<<ENABLE_APIC_SHIFT);		/* Disable apic */
   /*LOCAL_APIC_WRITE(LOCAL_APIC_SPIV, reg);*/
   /* Let APIC enabled in order to listen IPI and make posible 
      re-enabling */

   reg = LOCAL_APIC_READ(LOCAL_APIC_LVTL0);
   reg |= (1<<LVT_MASKED_SHIFT);		/* mask LINTIN0 */
   LOCAL_APIC_WRITE(LOCAL_APIC_LVTL0, reg);

   reg = LOCAL_APIC_READ(LOCAL_APIC_LVTL1);
   reg &= ~(1<<LVT_MASKED_SHIFT);		/* ummask LINTIN1 */
   LOCAL_APIC_WRITE(LOCAL_APIC_LVTL1, reg);
}






/*===========================================================================*
 *				enable_cpu				     * 
 *===========================================================================*/
PUBLIC void enable_cpu(int cpu, int echo) {
/* Mark a CPU as usable, enables it, and open interrupts.
   If echo is nonzero, print a message of successfully enabling */

   if (cpu_available[cpu]==CPU_ENABLED) return;

   /* A CPU only can only enable its own APIC */
   if (cpu==this_cpu) {
      enable_apic_ints();
      cpu_available[cpu]=CPU_ENABLED;
      lock_pick_proc();
      if (echo) printk("CPU%d enabled\n",cpu);
   }
   else {
      interrupt_cpu(cpu,MP_CM_ENABLE,echo);
   }
}



/*===========================================================================*
 *				disable_cpu				     * 
 *===========================================================================*/
PUBLIC void disable_cpu(int cpu, int echo) {
/* Mark a CPU as non usable, diables it, and close interrupts.
   If echo is nonzero, print a message of successfully disabling */

   if (cpu_available[cpu]==CPU_DISABLED) return;

   /* A CPU only can disable itself. */
   if (cpu==this_cpu) {
      disable_apic_ints();
      cpu_available[cpu]=CPU_DISABLED;
      if (!istaskp(proc_ptr[cpu])) {
         proc_ptr[cpu]->p_currentcpu=NONE_CPU;
         bill_ptr[cpu]=proc_ptr[cpu]=proc_addr_IDLE[cpu];
      }
      if (echo) printk("CPU%d disabled\n",cpu);
   }
   else {
      cpu_available[cpu]=CPU_DISABLING;
      interrupt_cpu(cpu,MP_CM_DISABLE,echo);
   }
}





/*===========================================================================*
 *				halt_cpu				     * 
 *===========================================================================*/
PUBLIC void halt_cpu(int cpu) {
/* Stop a CPU */

   /* A CPU only can stop itself. */
   if (cpu==this_cpu) {
      disable_apic_ints();
      cpu_available[cpu]=CPU_HALTED;
      if (! istaskp(proc_ptr[cpu])) {
         proc_ptr[cpu]->p_currentcpu=NONE_CPU;
         bill_ptr[cpu]=proc_ptr[cpu]=proc_addr_IDLE[cpu];
         printk("CPU%d halted\n",cpu);
         unlock_mp_kernel();
         disable_cache();
         halt();	/* Never return */
      }
   }
   else {
      cpu_available[cpu]=CPU_DISABLING;
      interrupt_cpu(cpu,MP_CM_HALT,MP_CM_ARG_NONE);
   }
}




/*===========================================================================*
 *				ap_main				     * 
 *===========================================================================*/
void ap_main(void) {
/* This is the main routine for AP's. This is the entry point before
   the CPU has been started and jumped to protected mode */

   /* Tell BSP we are running */
   ap_running_flag=AP_LIFE_FLAG_MARK;

   /* Now we're ready to take some work. We find any task and call 
      restart() to execute it (or to idle), but we must synchonize 
      other cpus before enter kernel code */
   lock_mp_kernel();	/* restart() will unlock later */

   /* Enable APIC interrupt aceptance */
   enable_cpu(this_cpu,WITHOUT_ECHO);

   /* Now, kernel is our! */
   lock_pick_proc();	/* Is there any work? */   
   restart();		/* Let's go... */
}






/**************************************************************************/
/** START AND STOP MP SYSTEM						 **/
/**************************************************************************/

 
/*===========================================================================*
 *				mp_start				     * 
 *===========================================================================*/
void mp_start() {
/* Load MP configuraton and wake up each CPU in the system */

   u32_t trampoline_addr;
   int cpu;

   /* Start messaging to console */  
   MP_STATUS[0]=0;
   ADDS_MP_STATUS("IntelMP multiprocessor kernel support v" \
                  MP_RELEASE "." MP_VERSION "\n");

   /* Try to load MP structures from memory */
   if (load_fps()) {
      /*print_fps_info();*/
      if (load_mph()) {
         /*print_mp_configuration();*/
         local_apic_base=mph.mpch_apic_addr;
         process_mp_configuration();
         /* Try to allocate the trampoline for APs to start */
         if ((trampoline_addr=find_trampoline())) {
            /* Start each processor */
            FOR_EACH_AP(cpu) {
               if (cpu!=1) milli_delay(100);
               ap_running_flag=0;
               send_init_ipi(trampoline_addr, cpu);
               send_startup_ipi(trampoline_addr, cpu);
               if (! AP_running()) {
                  ADDS_MP_STATUS("\n\n*** WARNING! AP#");
                  ADDN_MP_STATUS(cpu,10,1);
                  ADDS_MP_STATUS(" is not running ***\n\n");
               }
            }
            free_trampoline(trampoline_addr);   
         }
         else ADDS_MP_STATUS("MP ERROR!: Unable to allocate trampoline\n");
      }
   }
}



/*===========================================================================*
 *				mp_stop					     * 
 *===========================================================================*/
void mp_stop(void) {
/* Stop all processor excepting BSP */
   int dummy_trampoline_addr=0;
   int cpu,n;

   if (this_cpu!=0) {
      printk("Only BSP can stop MP!!!!\n");
      return;
   }
   printk("Disabling CPUs...\n");

   /* Disabling CPUs need to interrupt them, so unlock the kernel 
      and let them CPUs to enter kernel for disabling */
   unlock_mp_kernel();	

   FOR_EACH_AP(cpu)
      halt_cpu(cpu);
   FOR_EACH_AP(cpu)
      for (n=0; n<500; n++)
         if (cpu_available[cpu]==CPU_DISABLED) break;
         else milli_delay(1);

   /* Stop all APs */
   /* Sending INIT to a processor makes it to wait in a halt state */
   FOR_EACH_AP(cpu)
      send_init_ipi(dummy_trampoline_addr, cpu);

   /* Ensure BSP has APIC interrupts enabled */
   if (cpu_available[0]!=CPU_ENABLED) enable_cpu(0,WITH_ECHO);
}





