/*
 * local.c
 *
 * Copyright (C) 1993, 1994, John Kilburg.
 *
 * See copyright.h for details.
 */
#include "copyright.h"

#include "options.h"

#include <stdio.h>
#include <errno.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>

#if defined(SYSV) || defined(SVR4) || defined(__arm) || __minix
#include <dirent.h>
#define DIRSTUFF struct dirent
#else
#include <sys/dir.h>
#define DIRSTUFF struct direct
#endif

/* Jim Rees fix */
#ifndef S_ISDIR
#define S_ISDIR(mode) (((mode) & S_IFMT) == S_IFDIR)
#endif

#define DEFAULT_TELNET_PORT 23

#include "url.h"
#include "mime.h"
#include "document.h"
#include "local.h"
#include "ftp.h"
#include "input.h"
#include "util.h"
#include "net.h"
#include "stringdb.h"

/*
 * LoadFile
 *
 * Load the contents of a file into memory.
 */
static Document *
LoadFile(filename, size, mtlist)
char *filename;
int size;
MIMEType *mtlist;
{
  FILE *fp;
  char *b;
  char *content;
  Document *d;

  fp = fopen(filename, "r");
  if (fp == NULL) return(NULL);

  b = (char *)alloc_mem(size + 1);
  if (fread(b, 1, size, fp) < size)
  {
    free(b);
    fclose(fp);
    return(NULL);
  }
  b[size] = '\0';

  fclose(fp);

  content = Ext2Content(mtlist, filename);
  if (content == NULL) content = "text/plain";
  d = BuildDocument(b, size, content, 0, 0);

  return(d);
}

typedef struct namelist {	/* Obviously a list of names. */
  struct namelist *next;
  char		  *name;
} namelist_t;

/*
 * Sort
 *
 * A stable mergesort disguised as line noise.  Must be called like this:
 *	if (L!=NULL && L->next!=NULL) Sort(&L);
 */
static void
Sort(anl)
namelist_t **anl;
{
  /* static */ namelist_t *nl1, **mid;	/* Need not be local */
  namelist_t *nl2;

  nl1= *(mid= &(*anl)->next);
  do {
    if ((nl1= nl1->next) == NULL) break;
    mid= &(*mid)->next;
  } while ((nl1= nl1->next) != NULL);

  nl2= *mid;
  *mid= NULL;

  if ((*anl)->next != NULL) Sort(anl);
  if (nl2->next != NULL) Sort(&nl2);

  nl1= *anl;
  for (;;) {
    if (strcmp(nl1->name, nl2->name)<=0) {
      if ((nl1= *(anl= &nl1->next)) == NULL) {
        *anl= nl2;
        break;
      }
    } else {
      *anl= nl2;
      nl2= *(anl= &nl2->next);
      *anl= nl1;
      if (nl2 == NULL) break;
    }
  }
}

/*
 * Collect
 *
 * Return a sorted list of directory entries.  Returns null with errno != 0
 * on error.  (Author: Kees J. Bot <kjb@cs.vu.nl>)
 */
static namelist_t *
Collect(dir)
char *dir;
{
  namelist_t *names, **pn= &names;
  DIR *dp;
  struct dirent *entry;

  if ((dp= opendir(dir)) == NULL) return NULL;

  while ((entry= readdir(dp)) != NULL) {
    if (entry->d_name[0] == '.'
      && (entry->d_name[1] == 0
        || (entry->d_name[1] == '.'
          && entry->d_name[2] == 0))) {
      continue;
    }
    *pn= (namelist_t *) alloc_mem(sizeof(**pn));
    (*pn)->name= (char *) alloc_mem(strlen(entry->d_name) + 1);
    strcpy((*pn)->name, entry->d_name);
    pn= &(*pn)->next;
  }
  closedir(dp);
  *pn= NULL;
  errno= 0;
  if (names != NULL && names->next != NULL) Sort(&names);
  return names;
}

/*
 * LoadDir
 *
 * Reads a local directory and converts it into HTML so the user can
 * navigate the local filesystem.
 */
static Document *
LoadDir(filename)
char *filename;
{
  int flen;
  int filelen;
  int formlen;
  namelist_t *nl, *junk;
  char *f;
  static char *format = "<li> <a href=file:%s/%s> %s </a>\n";
  static char *rformat = "<li> <a href=file:%s%s> %s </a>\n";
  static char *header = "<title>Local Directory %s</title>\n<h1>Local Directory %s</h1>\n<ul>\n";

  filelen = strlen(filename);
  formlen = strlen(format);

  flen = strlen(header) + filelen * 2 + 1;
  f = (char *)alloc_mem(flen);
  sprintf (f, header, filename, filename);

  nl = Collect(filename);
  if (nl == NULL && errno != 0) return(NULL);

  while (nl != NULL)
  {
    if (nl->name[0] != '.')
    {
      flen += formlen + strlen(nl->name) * 2 + filelen + 1;
      
      f = (char *)realloc_mem(f, flen);

      if (filename[strlen(filename) - 1] == '/')
      {
	sprintf(f + strlen(f), rformat, filename, nl->name, nl->name);
      }
      else
      {
	sprintf(f + strlen(f), format, filename, nl->name, nl->name);
      }
    }
    junk = nl;
    nl = nl->next;
    free(junk->name);
    free(junk);
  }

  return(BuildDocument(f, strlen(f), "text/html", 0, 0));
}

/*
 * file
 *
 * Grabs a local file or to grab the file using FTP (if the hostname
 * is not localhost).
 */
Document *
file(up, mtlist)
URLParts *up;
MIMEType *mtlist;
{
  char *domain;
  struct stat s;
  char *filename;
  Document *d;
  char *hostname;

  hostname = up->hostname != NULL ? up->hostname:"localhost";
  if (mystrcmp(hostname, "localhost") != 0)
  {
    domain = net_gethostname(); /* gets the full domain name */
    if (domain == NULL) return(NULL);
    if (mystrcmp(domain, hostname) != 0)
    {
      char myhostname[256];

      gethostname(myhostname, sizeof(myhostname)); /* gets only hostname ? */
      if (mystrcmp(hostname, myhostname) != 0)
      {
	return(ftp(up, mtlist));
      }
    }
  }

  filename = FixFilename(up->filename);
  if (filename == NULL || stat(filename, &s) < 0)
  {
    char *msg = GetFromStringDB("localaccess");

    return(BuildDocument(msg, strlen(msg), "text/html", 1, 0));
  }

  if (S_ISDIR(s.st_mode)) d = LoadDir(filename);
  else d = LoadFile(filename, (int) s.st_size, mtlist);

  return(d);
}

/*
 * telnet_main
 *
 * handles telnet and tn3270 URLs
 */
static Document *
telnet_main(up, content)
URLParts *up;
char *content;
{
  char *username;
  char *password;
  char *t;
  int portno;
  static char *format = "%s %d %s %s\n";

  if (up->hostname == NULL) return(NULL);

  if (up->username == NULL) username = alloc_string("");
  else username = up->username;

  if (up->password == NULL) password = alloc_string("");
  else password = up->password;

  if (up->port == 0) portno = DEFAULT_TELNET_PORT;
  else portno = up->port;

  t = alloc_mem(strlen(format) +
		strlen(up->hostname) +
		12 +
		strlen(username) +
		strlen(password));
  sprintf (t, format, up->hostname, portno, username, password);

  return(BuildDocument(t, strlen(t), content, 0, 0));
}

/*
 * telnet
 *
 * handles telnet URLs
 */
Document * 
telnet(up, mtlist)
URLParts *up;
MIMEType *mtlist;
{
  return(telnet_main(up, "application/x-telnet"));
}

/*
 * tn3270
 *
 * handles tn3270 URLs
 */
Document *
tn3270(up, mtlist)
URLParts *up;
MIMEType *mtlist;
{
  return(telnet_main(up, "application/x-tn3270"));
}
