/*	startXserver 1.1 - start X11 server		Author: Kees J. Bot
 *								6 Mar 1994
 * The problem with xinit is that it wants to start both the
 * server and the client processes.  We want to start the server first and
 * then worry about the kind of client to start.
 *
 * This program starts the server like xinit does, waits for it to get ready
 * and then exit to let the clients be started.  Also it doesn't fiddle with
 * environment variables or other things you don't expect.
 */
#define nil 0
#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <errno.h>

void fatal(char *label)
{
	fprintf(stderr, "startXserver: %s: %s\n", label, strerror(errno));
	exit(1);
}

void catch(int sig)
{
#if __minix
	signal(sig, catch);
#endif
	/* SIGUSR1 may arrive before the wait(), so we keep a signal going. */
	alarm(1);
}

int main(int argc, char **argv)
{
	int errp[2];
	int pid, status;

	if (argc <= 1) {
		fprintf(stderr, "Usage: startXserver Xserver [arg ...]\n");
		exit(1);
	}

	signal(SIGALRM, catch);
	signal(SIGUSR1, catch);

	if (pipe(errp) < 0) fatal("pipe()");

	switch ((pid= fork())) {
	case -1:
		fatal("fork()");
	case 0:
		close(errp[0]);
		fcntl(errp[1], F_SETFD, fcntl(errp[1], F_GETFD) | FD_CLOEXEC);

		/* Don't hang on read/write to control tty. */
#ifdef SIGTTIN
		signal(SIGTTIN, SIG_IGN);
#endif
#ifdef SIGTTOU
		signal(SIGTTOU, SIG_IGN);
#endif

		/* Ignore SIGUSR1 to tell the server we want to be signalled. */
		signal(SIGUSR1, SIG_IGN);

		/* Rewire stdin to null, stdout to stderr. */
		close(0);
		open("/dev/null", O_RDONLY);
		dup2(2, 1);

		/* Start server. */
		execvp(argv[1], argv + 1);

		/* Exec failed, return the errno value. */
		write(errp[1], &errno, sizeof(errno));
		_exit(-1);
	}

	/* Wait until we are signalled by SIGUSR1 or until 15 seconds have
	 * passed.
	 */
	alarm(15);
	if (waitpid(pid, &status, 0) == pid) {
		/* Server shouldn't exit!  Exec failed? */
		close(errp[1]);
		if (read(errp[0], &errno, sizeof(errno)) != 0) fatal(argv[1]);

		fprintf(stderr, "startXserver: %s died", argv[1]);
		if (WIFEXITED(status)) {
			fprintf(stderr, ", exit status %d\n",
				WEXITSTATUS(status));
		} else {
			fprintf(stderr, " by signal %d%s\n",
				WTERMSIG(status),
				status & 0x80 ? " - core dumped" : "");
		}
		exit(1);
	}
	alarm(0);

	/* Return server process id. */
	printf("%d\n", pid);
	exit(0);
}
