/*	uname 1.1 - Display system info.		Author: Kees J. Bot
 *								5 Dec 1992
 */
#define nil 0
#include <sys/types.h>
#include <sys/utsname.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#define arraysize(a)	(sizeof(a) / sizeof((a)[0]))
#define arraylimit(a)	((a) + arraysize(a))

typedef struct action {
	char	*name;		/* Program name. */
	int	def;		/* Default option. */
	char	*opts;		/* Allowed options. */
} action_t;

action_t actions[] = {
	{ "sysname",	's',	"s"		},
	{ "nodename",	'n',	"n"		},
	{ "release",	'r',	"r"		},
	{ "version",	'v',	"v"		},
	{ "machine",	'm',	"m"		},
	{ "arch",	'p',	"pkmhnrvsba"	},
	{ "kernel",	'k',	"k"		},
	{ "hostname",	'h',	"hn"		},
	{ "bus",	'b',	"b"		},
	{ "uname",	's',	"snrvmpkhba"	},
};
action_t *action;

char active[_UTS_MAX + 1];	/* The options to act upon. */
char *value[_UTS_MAX];		/* The values to set for the options. */

int translate(int option)
/* Translate an option letter to an utsname field identifier. */
{
	switch (option) {
	case 's':	return _UTS_SYSNAME;
	case 'n':	return _UTS_NODENAME;
	case 'r':	return _UTS_RELEASE;
	case 'v':	return _UTS_VERSION;
	case 'm':	return _UTS_MACHINE;
	case 'p':	return _UTS_ARCH;
	case 'k':	return _UTS_KERNEL;
	case 'h':	return _UTS_HOSTNAME;
	case 'b':	return _UTS_BUS;
	default:	return -1;
	}
}

void usage(void)
/* Usage: uname -snrvma [string...] */
{
	fprintf(stderr,
		action->opts[1] != 0
			? "Usage: %s -[%s] [string...]\n"
			: "Usage: %s [string]\n",
		action->name,
		action->opts);
	exit(1);
}

char *string(size_t n)
/* A string of length n please. */
{
	static char *s= nil;

	s= s == nil ? malloc(n * sizeof(*s))
		: realloc(s, n * sizeof(*s));

	if (s == nil) {
		fprintf(stderr, "%s: %s\n", action->name, strerror(errno));
		exit(1);
	}

	return s;
}

int main(int argc, char **argv)
{
	char *arg0;
	int av;
	int i;
	int ex= 0;
	int all= 0;	/* uname -a? */

	/* Find out under what name this program is called. */

	if ((arg0= strrchr(argv[0], '/')) == nil) arg0= argv[0]; else arg0++;

	for (action= actions; action < arraylimit(actions); action++) {
		if (strcmp(action->name, arg0) == 0) break;
	}

	if (action == arraylimit(actions)) {
		fprintf(stderr, "uname: don't know what to do as \"%s\"\n",
			arg0);
		exit(1);
	}

	/* Use the default option? */
	if (argc <= 1 || argv[1][0] != '-')
		active[0]= action->def;

	/* 'av' matches a string argument to an option. */
	av= 0;

	/* Process options and arguments. */
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			/* Options, check them one by one for the command
			 * at hand.
			 */
			char *p;

			for (p= argv[i]+1; *p != 0; p++) {
				/* Action has this option? */
				if (strchr(action->opts, *p) == nil) usage();

				/* Duplicate option? */
				if (strchr(active, *p) != nil) usage();

				/* 'uname -a' or 'arch -a'? */
				if (*p == 'a') {
					if (active[0] != 0) usage();

					if (strcmp(action->name, "uname") == 0)
						strcpy(active, "snrvm");
					else
					if (strcmp(action->name, "arch") == 0)
						strcpy(active, "pkmhnrvsb");
					else
						usage();
					all= 1;
					continue;
				}

				/* Add option. */
				active[strlen(active)]= *p;
			}
		} else {
			/* An argument, match it to an option. */

			/* There must be an option to match to. */
			if (all || active[av] == 0) usage();

			/* Add it. */
			value[av++]= argv[i];
		}
	}

	if (value[0] != nil) {
		/* There are string arguments, so try to set them. */

		/* There must be enough. */
		if (active[av] != 0) usage();

		/* Upload each of the values. */
		for (i= 0; active[i] != 0; i++) {
			if (sysuname(_UTS_SET,
				translate(active[i]),
				value[i],
				strlen(value[i]) + 1) < 0
			) {
				fprintf(stderr, "%s -%c %s: %s\n",
					action->name,
					active[i],
					value[i],
					strerror(errno));
				ex= 1;
			}
		}
	} else {
		/* No string arguments, display values in action->opts order. */
		size_t len= 256;
		char *opt;
		int space= 0;

		for (opt= action->opts; *opt != 0; opt++) {
			if (strchr(active, *opt) == nil) continue;

			for (;;) {
				string(len)[len - 1] = 0;

				if (sysuname(_UTS_GET,
					translate(*opt),
					string(len),
					len) < 0
				) {
					fprintf(stderr, "%s -%c: %s\n",
						action->name,
						*opt,
						strerror(errno));
					exit(1);
				}
				/* If the string is null-terminated, then
				 * it is long enough to contain the value.
				 */
				if (string(len)[len - 1] == 0) break;

				len*= 2;
			}
			if (space) putchar(' '); else space= 1;
			printf("%s", string(len));
		}
		putchar('\n');
	}
	/* Strange, not an uname() call or struct utsname in sight. */
	exit(ex);
}
