/*	mount/umount 1.14 - mount and unmount file systems
 *							Author: Kees J. Bot
 *								15 Nov 1992
 */
#define nil 0
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <dirent.h>
#include <time.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/mnttab.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mount.h>
#include <sys/diskio.h>
#include <sys/svrctl.h>
#include <minix/cfg_public.h>
#include <minix/swap.h>
#include <minix/queryparam.h>

#if SH_VERSION != 1
#error Swap header version not as expected
#endif

/* Parameters determining the size of the executable cache.  The cache size
 * goes up with the square root of the memory size.  The parameters below
 * modify the outcome.
 */
#define XCACHE_FACTOR	1012		/* Cache = factor * sqrt(mem). */
#define XCACHE_VMIN	(14*1024*1024L)	/* Minimum leftover virtual memory. */

#define VERBOSE		1	/* Default verbosity level. */

typedef struct mnttab mnttab_t;

#define arraysize(a)	(sizeof(a) / sizeof((a)[0]))
#define arraylimit(a)	((a) + arraysize(a))

const char FSTAB[]=	"/etc/fstab";	/* File system mount table. */
const char MTAB[]=	"/etc/mtab";	/* Currently mounted file systems. */

const char MKFS[]=	"/usr/bin/mkfs";
const char FSCK[]=	"/usr/bin/fsck";

char DEFOPT[]=		"";	/* Default mount options (don't change). */

#define MOUNT		0	/* Mount it. */
#define UMOUNT		1	/* No, unmount it. */
int action;			/* One of the above. */

int verbose= VERBOSE;	/* -vq: Idle talk about mounts and unmounts */
int all= 0;		/* -a:  All fstab/mtab entries. */
int fake= 0;		/* -f:  Fake a mount/umount in mtab. */
int notab= 0;		/* -n:  Do not change mtab. */
int pure= 0;		/* -p:  Talk in fstab style, otherwise readable. */
int rdonly= 0;		/* -r:	Read-only. */
char *options= DEFOPT;	/* -o:	Mount options. */
char *type= nil;	/* -t:  Device type. */

char now[sizeof(long) * 3];	/* Current time in decimal. */

int exstat= 0;			/* Exit status. */

uid_t uid;			/* Uid of caller. */
int super= 0;			/* Privileged? */

const char *arg0;

void usage(void)
{
	int mount= arg0[0] != 'u';

	static char use_mount[]=
		"-[apfnrvq] [-t type] [-o options] [device] [directory]";
	static char use_umount[]=
		"-[afnvq] [-t type] device|directory";

	fprintf(stderr, "Usage: %s [-%c] %s\n",
		arg0, mount ? 'm' : 'u', mount ? use_mount : use_umount);

	fprintf(stderr, "       %s -%c %s\n",
		arg0, mount ? 'u' : 'm', mount ? use_umount : use_mount);
	exit(1);
}

void report(const char *label)
{
	const char *err= strerror(errno);

	if (*label == 0)
		fprintf(stderr, "%s: %s\n", arg0, err);
	else
		fprintf(stderr, "%s: %s: %s\n", arg0, label, err);
	exstat= 1;
}

void fatal(const char *label)
{
	report(label);
	exit(1);
}

void *alloc(void *mem, size_t len)
{
	mem= mem == nil ? malloc(len) : realloc(mem, len);
	if (mem == nil) fatal("");
	return mem;
}

char *copystr(const char *s)
{
	char *c= alloc(nil, (strlen(s) + 1) * sizeof(*s));
	strcpy(c, s);
	return c;
}

u32_t isqrt(u32_t i)
/* There are many fast integer square root implementations, this is not one
 * of them.
 */
{
	int n;
	u32_t s, j;

	for (s= 1, j= i; j != 0; s *= 2, j /= 4) {}
	for (n= 0; n < 10; n++) s= (s + i / s) / 2;
	return s;
}

void set_execcache(int setit)
/* Determine the size of the executable cache from the amount of RAM and the
 * current amount of swap space.  There is a lot of guessing here!
 */
{
	struct mmexeccache xcache;
	u32_t total_dmacore_mem, total_coremem, total_swapmem;
	u32_t ram, swap, virtual, limit;
	struct svrqueryparam qpar;
	static char param[]= "total_dmacore_mem,total_coremem,total_swapmem";
	char value[3 * (2 * sizeof(u32_t) + 1)];
	char *pv;
	char bxcache[256], *end;
	unsigned long bxc;

	if (svrctl(MMGEXECCACHE, &xcache) < 0) {
		static int told= 0;
		if (!told) {
			fprintf(stderr,
				"%s: can't query executable cache size: %s\n",
				arg0, strerror(errno));
			told= 1;
		}
		return;
	}

	if (!setit) {
		/* Only want to kill the cache to be able to remove swap. */
		limit= 0;
	} else {
		/* Get some kernel parameters. */
		qpar.param= param;
		qpar.psize= sizeof(param);
		qpar.value= value;
		qpar.vsize= sizeof(value);
		if (svrctl(SYSQUERYPARAM, &qpar) < 0) {
			static int told= 0;
			if (!told) {
				fprintf(stderr,
			"%s: can't query virtual memory parameters: %s\n",
					arg0, strerror(errno));
				told= 1;
			}
			return;
		}

		pv= value;
		paramvalue(&pv, &total_dmacore_mem, sizeof(total_dmacore_mem));
		paramvalue(&pv, &total_coremem, sizeof(total_coremem));
		paramvalue(&pv, &total_swapmem, sizeof(total_swapmem));

		ram= (total_dmacore_mem + total_coremem) * CLICK_SIZE;
		swap= total_swapmem * CLICK_SIZE;
		virtual= ram + swap;

		/* Computed cache size. */
		limit= XCACHE_FACTOR * isqrt(virtual);

		/* Cache size set by XCACHE boot variable? */
		if (sysenv("XCACHE", bxcache, sizeof(bxcache)) != -1) {
			limit= strtoul(bxcache, &end, 10);
			if (end == bxcache || *end != 0) {
				fprintf(stderr,
			"%s: bad executable cache size setting: XCACHE=%s\n",
					arg0, bxcache);
				limit= 0;
			}
		}

		/* Don't waste good memory if there is not much swap. */
		if (limit > swap) limit= swap;

		/* Make sure there is some memory left. */
		if (virtual - limit < XCACHE_VMIN) {
			limit= virtual < XCACHE_VMIN ? 0 : virtual-XCACHE_VMIN;
		}
	}

	xcache.mec_limit= limit;
	if (svrctl(MMSEXECCACHE, &xcache) < 0) {
		static int told= 0;
		if (!told) {
			fprintf(stderr,
				"%s: can't set executable cache size: %s\n",
				arg0, strerror(errno));
			told= 1;
		}
		return;
	}
}

int type2flags(const char *type)
{
	if (strcmp(type, "dev") == 0) return M_DEV;
	if (strcmp(type, "1") == 0) return M_V1;
	if (strcmp(type, "2") == 0) return M_V2;
	if (strcmp(type, "1f") == 0) return M_V1F;
	if (strcmp(type, "2f") == 0) return M_V2F;
	if (strcmp(type, "lo") == 0) return M_LO;
	return 0;
}

char *flags2type(int flags)
{
	switch (flags & M_TYPE) {
	case M_DEV:	return "dev";
	case M_V1:	return "1";
	case M_V2:	return "2";
	case M_V1F:	return "1f";
	case M_V2F:	return "2f";
	case M_LO:	return "lo";
	default:	return "?";
	}
}

/* Internal mount options: */
#define IMO_NOAUTO	0x01	/* Not mounted on 'mount -a'. */
#define IMO_MKFS	0x02	/* Make a filesystem on automatic mount. */
#define IMO_SWAP	0x04	/* Shared swapspace. */
#define IMO_USER	0x08	/* Allow a user to mount if checked. */

void opts2flags(const char *opts, int *flags, int *iflags)
{
	mnttab_t dummy;

	dummy.mnt_mntopts= (char *) opts;
	if (hasmntopt(&dummy, "rw")) *flags&= ~M_RDONLY;
	if (hasmntopt(&dummy, "ro")) *flags|= M_RDONLY;
	if (hasmntopt(&dummy, "suid")) *flags&= ~M_NOSUID;
	if (hasmntopt(&dummy, "nosuid")) *flags|= M_NOSUID;
	if (hasmntopt(&dummy, "grpid")) *flags|= M_GRPID;
	if (hasmntopt(&dummy, "nogrpid")) *flags&= ~M_GRPID;
	if (hasmntopt(&dummy, "nf")) *flags|= M_NF;
	if (hasmntopt(&dummy, "nonf")) *flags&= ~M_NF;
	if (hasmntopt(&dummy, "remount")) *flags|= M_REMOUNT;

	if (hasmntopt(&dummy, "noauto")) *iflags|= IMO_NOAUTO;
	if (hasmntopt(&dummy, "mkfs")) *iflags|= IMO_MKFS;
	if (hasmntopt(&dummy, "swap")) *iflags|= IMO_SWAP;
	if (hasmntopt(&dummy, "noswap")) *iflags&= ~IMO_SWAP;
	if (hasmntopt(&dummy, "user")) *iflags|= IMO_USER;
}

char *flags2opts(int flags, int iflags)
{
	char opts[100];

	if (flags & M_RDONLY) strcpy(opts, "ro"); else strcpy(opts, "rw");
	if (flags & M_NOSUID) strcat(opts, ",nosuid");
	if (flags & M_GRPID) strcat(opts, ",grpid");
	if (flags & M_NF) strcat(opts, ",nf");
	if (iflags & IMO_SWAP) strcat(opts, ",swap");
	if (iflags & IMO_USER) strcat(opts, ",user");
	return copystr(opts);
}

char *finddev(const char *dir)
/* Find the device that is mounted on dir. */
{
	DIR *dp;
	struct dirent *entry;
	dev_t rdev;
	struct stat st;
	char *name= nil;

	if (stat(dir, &st) < 0) fatal(dir);
	rdev= st.st_dev;

	if ((dp= opendir("/dev")) == nil) fatal("/dev");

	while ((entry= readdir(dp)) != nil) {
		name= alloc(name, (6 + strlen(entry->d_name)) * sizeof(*name));
		strcpy(name, "/dev/");
		strcpy(name+5, entry->d_name);

		if (stat(name, &st) >= 0 && S_ISBLK(st.st_mode)
					&& st.st_rdev == rdev) break;
	}
	(void) closedir(dp);

	return entry != nil ? name : "/dev/unknown";
}

typedef struct mntlist {	/* An /etc/mtab pulled in core. */
	struct mntlist	*next;
	int		stale;
	mnttab_t	ent;
} mntlist_t;

mntlist_t *fstab, *mtab;	/* The two tables in core. */
int dirty= 0;			/* Must mtab be rewritten? */

mntlist_t **searchtab(mntlist_t **ptab, mnttab_t *look)
/* Look for *look in *ptab, if one of mnt_special, mnt_mountp, or mnt_fstype
 * is nonnil, then it is checked to be equal.  "dev" matches any device.
 * Returns a hook to the entry found, otherwise to the end.
 */
{
	mntlist_t *tab;

	while ((tab= *ptab) != nil) {
		int found= !tab->stale;

		if (look->mnt_special != nil) {
			struct stat tabst, lookst;

			if (stat(look->mnt_special, &lookst) < 0
				|| stat(tab->ent.mnt_special, &tabst) < 0
				|| !S_ISBLK(lookst.st_mode)
				|| !S_ISBLK(tabst.st_mode)
				|| lookst.st_rdev != tabst.st_rdev
			) found= 0;
		}
		if (look->mnt_mountp != nil) {
			struct stat tabst, lookst;

			if (stat(look->mnt_mountp, &lookst) < 0
				|| stat(tab->ent.mnt_mountp, &tabst) < 0
				|| lookst.st_dev != tabst.st_dev
				|| lookst.st_ino != tabst.st_ino
			) found= 0;
		}
		if (look->mnt_fstype != nil
			&& strcmp(look->mnt_fstype, tab->ent.mnt_fstype) != 0)
		{
			int looktype= type2flags(look->mnt_fstype);
			int tabtype= type2flags(tab->ent.mnt_fstype);

			if (looktype == 0 || tabtype == 0)
				found= 0;

			if (looktype != M_DEV && tabtype != M_DEV)
				found= 0;
		}
		if (found) break;

		ptab= &tab->next;
	}
	return ptab;
}

mntlist_t **searchlast(mntlist_t **ptab, mnttab_t *mtp)
/* Like strrchr is to strchr. */
{
	mntlist_t **last= nil;

	while (*(ptab= searchtab(ptab, mtp)) != nil) {
		last= ptab;
		ptab= &(*ptab)->next;
	}
	return last == nil ? ptab : last;
}

void copyent(mnttab_t *mt1, const mnttab_t *mt2)
{
	mt1->mnt_special= copystr(mt2->mnt_special);
	mt1->mnt_mountp= copystr(mt2->mnt_mountp);
	mt1->mnt_fstype= copystr(mt2->mnt_fstype);
	mt1->mnt_mntopts= copystr(mt2->mnt_mntopts);
	mt1->mnt_freq= copystr(mt2->mnt_freq);
	mt1->mnt_fsckpass= copystr(mt2->mnt_fsckpass);
	mt1->mnt_time= copystr(mt2->mnt_time);
}

mntlist_t *getmtab(const char *file)
/* Read an mtab file into core. */
{
	FILE *mfp;
	mnttab_t mte;
	int fdrw;
	int r;
	char *err;
	int stale= 0;
	mntlist_t *tab, **last= &tab, *new;
	struct flock how;

	/* Try to open the /etc/mtab file read-write.  This will fail at boot
	 * time with EROFS, because / is initially read-only.  The information
	 * in it is stale anyway, so you don't want it.  We lock the file if
	 * possible, to keep two mounts out of each others hair.
	 */
	if (file == MTAB) {
		setuid(0);
		if ((fdrw= open(file, O_RDWR)) < 0) {
			if (errno == EROFS) stale= 1;
		} else {
			how.l_type= F_WRLCK;
			how.l_whence= SEEK_SET;
			how.l_start= 0;
			how.l_len= 0;
			(void) fcntl(fdrw, F_SETLKW, &how);
		}
		setuid(uid);
	}

	if (!stale && (mfp= fopen(file, "r")) != nil) {
		while ((r= getmntent(mfp, &mte)) >= 0) {
			switch (r) {
			case MNT_TOOLONG:
				err= "line too long";			break;
			case MNT_TOOMANY:
				err= "entry has too many fields";	break;
			case MNT_TOOFEW:
				err= "entry has too few fields";	break;
			}
			if (r != 0) {
				fprintf(stderr, "%s: %s in %s\n",
					arg0, err, file);
				if (file == FSTAB) continue;
			}

			new= alloc(nil, sizeof(*new));
			new->stale= 0;

			copyent(&new->ent, &mte);

			*last= new;
			last= &new->next;
		}
		(void) fclose(mfp);
	}
	*last= nil;

	if (stale) {
		/* Construct an entry for / at bootup. */
		new= alloc(nil, sizeof(*new));
		new->stale= 0;
		new->ent.mnt_special= finddev("/");
		new->ent.mnt_mountp= "/";
		new->ent.mnt_fstype= "dev";
		new->ent.mnt_mntopts= "ro";
		new->ent.mnt_freq= "0";
		new->ent.mnt_fsckpass= "0";
		new->ent.mnt_time= now;

		new->next= nil;
		tab= new;
		dirty= 1;

		/* Exit status is 2 if the device doesn't match with fstab */
		if (fstab != nil && *searchtab(&fstab, &tab->ent) == nil)
			exstat= 2;
	}
	return tab;
}

void putmtab(const char *file, mntlist_t *tab)
/* Write the in core mtab to a file. */
{
	FILE *mfp;

	if (notab) dirty= 0;
	if (file == MTAB && !dirty) return;

	setuid(0);
	if ((mfp= fopen(file, "w")) == nil) fatal(file);

	while (tab != nil) {
		if (putmntent(mfp, &tab->ent) == EOF) fatal(file);
		tab= tab->next;
	}
	if (fclose(mfp) == EOF) fatal(file);
	setuid(uid);
}

void tell(int new, int flags, int iflags, const mnttab_t *mtp)
{
	if (new && !verbose) return;

	if (pure) {
		(void) putmntent(stdout, mtp);
	} else
	if (strcmp(mtp->mnt_fstype, "swap") == 0) {
		printf("%s %s swap\n",
			mtp->mnt_special,
			new ? "added to" : "is");
	} else
	if (strcmp(mtp->mnt_fstype, "lo") == 0) {
		printf("%s is loopback mounted on %s\n",
			mtp->mnt_special,
			mtp->mnt_mountp);
	} else {
		printf("%s is %s %smounted on %s\n",
			mtp->mnt_special,
			flags & M_RDONLY ? "read-only" : "read-write",
			flags & M_REMOUNT ? "re" : "",
			mtp->mnt_mountp);
	}
}

int run(const char *device, const char *cmd, char **argv)
{
	int pid, status, fd;

	switch (pid= fork()) {
	case -1:
		report("fork()");
		return 0;
	case 0:
		if ((fd= open("/dev/null", O_WRONLY)) != -1) {
			if (fd != 1) {
				dup2(fd, 1);
				close(fd);
			}
		}
		execv(cmd, argv);
		fatal(cmd);
	}

	if (waitpid(pid, &status, 0) < 0) {
		report("wait()");
		return 0;
	}
	if (status != 0) {
		fprintf(stderr, "%s: %s: %s failed\n", arg0, device, argv[0]);
		exstat= 1;
	}
	return status == 0;
}

int mkfs(mnttab_t *mtp)
{
	char *argv[5];

	argv[0]= "mkfs";
	argv[1]= "-t";
	argv[2]= mtp->mnt_fstype;
	argv[3]= mtp->mnt_special;
	argv[4]= nil;

	return run(mtp->mnt_special, MKFS, argv);
}

int fsck(mnttab_t *mtp)
{
	char *argv[5];

	argv[0]= "fsck";
	argv[1]= "-n";
	argv[2]= mtp->mnt_special;
	argv[3]= nil;

	return run(mtp->mnt_special, FSCK, argv);
}

int do_swapon(const char *device, int opt)
{
	int fd= -1, r;
	swap_hdr_t hdr;
	dio_swapon_t dio_swapon;
	struct stat st;
	int ok= 1;
	off_t hdr_offset;

	/* Offset depends on it being a swap partition or optional swap. */
	hdr_offset= opt ? OPTSWAP_BOOTOFF : SWAP_BOOTOFF;

	/* Open the device and fetch the header. */
	if ((fd= open(device, O_RDONLY)) < 0
		|| fstat(fd, &st) < 0
		|| (errno= ENOTBLK, !S_ISBLK(st.st_mode))
		|| lseek(fd, hdr_offset, SEEK_SET) == -1
		|| (r= read(fd, (char *) &hdr, sizeof(hdr))) < 0
	) {
		report(device);
		ok= 0;
	}

	if (ok && r != sizeof(hdr)) {
		fprintf(stderr, "%s: %s: unexpected EOF\n", arg0, device);
		exstat= 1;
		ok= 0;
	}

	if (ok && (hdr.sh_magic[0] != SWAP_MAGIC0
		|| hdr.sh_magic[1] != SWAP_MAGIC1
		|| hdr.sh_magic[2] != SWAP_MAGIC2
		|| hdr.sh_magic[3] != SWAP_MAGIC3
		|| hdr.sh_version > SH_VERSION
	)) {
		errno= EINVAL;
		report(device);
		ok= 0;
	}

	if (hdr.sh_version == 0 && !opt) {
		/* Version 0 swap header misses the sh_offset field. */
		memmove(&hdr.sh_swapsize, &hdr.sh_offset,
			sizeof(hdr.sh_swapsize) + sizeof(hdr.sh_priority));
		hdr.sh_offset= SWAP_OFFSET;
	}

	/* Tell the kernel to use this device as a swap device. */
	dio_swapon.ds_offset= hdr.sh_offset;
	dio_swapon.ds_size= hdr.sh_swapsize;
	dio_swapon.ds_priority= hdr.sh_priority;

	if (ok && ioctl(fd, DIOCSWAPON, &dio_swapon) < 0) {
		report(device);
		ok= 0;
	}
	if (fd >= 0) close(fd);
	set_execcache(1);
	return ok;
}

int do_swapoff(const char *device)
/* Try to remove a block special from the list of swap partitions. This may
 * fail if the core memory and other swap devices do not offer enough space
 * to release the partition.
 */
{
	int fd;
	dio_swapinfo_t dio_swapinfo;
	int firsttime, ok= 1;

	/* Open the device. */
	if ((fd= open(device, O_RDONLY)) < 0) {
		report(device);
		return 0;
	}

	/* Kill the executable cache, it may use a lot of virtual memory. */
	set_execcache(0);

	/* Call swapoff until the device is no longer used for swap. */
	for (firsttime= 1; ok; firsttime= 0) {
		if (ioctl(fd, DIOCSWAPINFO, &dio_swapinfo) < 0) {
			if (errno == ESRCH) break;

			report(device);
			ok= 0;
		}

		if (ok && swapoff(dio_swapinfo.dsi_index) < 0) {
			report(device);
			ok= 0;
		}
	}
	if (ok && firsttime) {
		fprintf(stderr, "%s: %s: not a swap device\n", arg0, device);
		ok= 0;
	}
	close(fd);
	set_execcache(1);
	return ok;
}

void mount_action(char *dev, char *dir)
{
	mntlist_t *tab, **ptab, **pmtab;
	mnttab_t look, found;
	int flags, optflags, iflags, optiflags, tabiflags, swap, r;
	struct stat st;

	/* Mount options. */
	optflags= optiflags= 0;
	opts2flags(options, &optflags, &optiflags);
	if (rdonly) optflags|= M_RDONLY;

	/* Use fstab for a normal mount, mtab for a remount. */
	ptab= (optflags & M_REMOUNT) ? &mtab : &fstab;

	/* If adding swap then dir must be nil, and we can't remount. */
	if (type != nil && strcmp(type, "swap") == 0) {
		if (dir != nil || (optflags & M_REMOUNT)) usage();
		dir= "swap";
	}

	/* Fill in a mntent to search for. */
	look.mnt_special= dev;
	look.mnt_mountp= dir;
	look.mnt_fstype= type;
	look.mnt_mntopts= options;
	look.mnt_freq= "0";
	look.mnt_fsckpass= "0";
	look.mnt_time= now;

	if (!all) {
		/* Look for the entry to see if it exists.  Look for the
		 * last entry in mtab, because they are the most current
		 * for layered mounts.
		 */
		if (optflags & M_REMOUNT)
			ptab= searchlast(ptab, &look);
		else
			ptab= searchtab(ptab, &look);

		if (*ptab == nil) {
			/* There is no fstab or mtab entry, fatal? */

			if (dev != nil && dir != nil) {
				/* 'mount dev dir' can be done, add entry. */
				*ptab= alloc(nil, sizeof(**ptab));
				(*ptab)->next= nil;
				(*ptab)->ent= look;
				if (type == nil)
					(*ptab)->ent.mnt_fstype= "dev";
			} else {
				fprintf(stderr, "%s: %s: not in %s\n",
					arg0, dev == nil ? dir : dev,
					optflags & M_REMOUNT ? MTAB : FSTAB);
				exstat= 1;
				return;
			}
		}
	}

	/* Scan the table. */
	for (; (tab= *ptab) != nil; ptab= searchtab(&tab->next, &look)) {
		fflush(nil);

		found= tab->ent;

		/* Take current options and update them to "-o". */
		flags= iflags= 0;
		opts2flags(found.mnt_mntopts, &flags, &iflags);
		tabiflags= iflags;
		if ((optflags & M_REMOUNT) && hasmntopt(&look, "fstab")) {
			/* Remount using the options in /etc/fstab. */
			mntlist_t *opttab;

			if ((opttab= *searchtab(&fstab, &found)) != nil) {
				/* Entry present in fstab, use options. */
				opts2flags(opttab->ent.mnt_mntopts,
						&flags, &iflags);
			} else
			if (dir != nil && strcmp(dir, "/") == 0) {
				/* Remounting a strange root, use "rw". */
				flags&= ~M_RDONLY;
			} else {
				fprintf(stderr, "%s: %s: not in %s\n",
					arg0,
					dir != nil ? dir : found.mnt_special,
					FSTAB);
				exstat= 1;
				continue;
			}
		}
		opts2flags(options, &flags, &iflags);
		if (rdonly) flags|= M_RDONLY;

		if (!(tabiflags & IMO_USER)) iflags&= ~IMO_USER;
		if (!(optflags & M_REMOUNT)) tabiflags= 0;
		flags|= type2flags(found.mnt_fstype);

		if (!super) {
			/* User mounts are always "nosuid", never "swap". */
			flags|= M_NOSUID;
			iflags&= ~IMO_SWAP;
		} else {
			/* Super-user mounts are never "user". */
			iflags&= ~IMO_USER;
		}

		/* Possibly skip "ignore", "noauto", or "/". */
		if (strcmp(found.mnt_fstype, "ignore") == 0) continue;
		if (all && (iflags & IMO_NOAUTO)) continue;
		if (all && strcmp(found.mnt_mountp, "/") == 0) continue;

		/* Adding swap? */
		swap= strcmp(found.mnt_fstype, "swap") == 0;

		/* Do I know about this type? */
		if (!swap && (flags & M_TYPE) == 0) {
			if (verbose) {
				printf("%s not mounted, type \"%s\" unknown\n",
					found.mnt_special, found.mnt_fstype);
			}
			continue;
		}

		if (optflags & M_REMOUNT) {
			pmtab= ptab;
			if ((flags & M_TYPE) == 0) {
				/* Skip non-device entries? */
				if (all && type == nil) continue;
				usage();
			}
		} else {
			/* Take care not to layer mounts using -a. */
			mnttab_t old= found;

			if (swap) old.mnt_mountp= nil;
			else
			if (all) old.mnt_special= nil;
			pmtab= searchtab(&mtab, &old);

			if (*pmtab != nil) {
				if (!all || verbose) {
					printf("%s is already %s%s\n",
						(*pmtab)->ent.mnt_special,
						swap ? "" : "mounted on ",
						(*pmtab)->ent.mnt_mountp);
				}
				if (!all) exstat= 1;
				continue;
			}
			if (all && (iflags & IMO_MKFS) && !mkfs(&found))
				continue;

			if (!super && (iflags & IMO_USER) && !fsck(&found))
				continue;

			/* New entry, new timestamp. */
			found.mnt_time= now;
		}

		/* Pre-check for better error reports. */
		if (!fake && !swap && stat(found.mnt_mountp, &st) < 0) {
			report(found.mnt_mountp);
			continue;
		}

		/* Do the mount call. */
		if (!fake && !swap) {
			if (iflags & IMO_USER) setuid(0);
			flags= mount(found.mnt_special, found.mnt_mountp,
									flags);
			if (iflags & IMO_USER) setuid(uid);

			if (flags == -1) {
				/* Mount failed. */
				report(found.mnt_special);
				continue;
			}
		}

		/* Do the swapon or add shared swap. */
		if (!fake && (swap ||
			(!(tabiflags & IMO_SWAP) && (iflags & IMO_SWAP))
		)) {
			if (iflags & IMO_USER) setuid(0);
			r= do_swapon(found.mnt_special, iflags & IMO_SWAP);
			if (iflags & IMO_USER) setuid(uid);

			if (!r) {
				/* No swap added. */
				if (swap) continue;
				iflags &= ~IMO_SWAP;
			}
		}

		/* Or remove shared swap. */
		if (!fake && !swap &&
			((tabiflags & IMO_SWAP) && !(iflags & IMO_SWAP))
		) {
			if (iflags & IMO_USER) setuid(0);
			r= do_swapoff(found.mnt_special);
			if (iflags & IMO_USER) setuid(uid);

			if (!r) {
				/* Not gone yet. */
				iflags |= IMO_SWAP;
			}
		}

		/* Mount succeeded, update /etc/mtab. */
		if (*pmtab == nil) {
			*pmtab= alloc(nil, sizeof(**pmtab));
			(*pmtab)->next= nil;
		}
		found.mnt_mntopts= swap ? "rw" : flags2opts(flags, iflags);
		found.mnt_fstype= swap ? "swap" : flags2type(flags);
		(*pmtab)->ent= found;
		dirty= 1;
		tell(1, flags, iflags, &found);
	}
}

void umount_action(char *dev, char *dir)
{
	mntlist_t *tab, **ptab, dummy;
	mnttab_t look, found;
	int flags, iflags, swap, r, layered;

	/* Fill in a mntent to search for. */
	look.mnt_special= dev;
	look.mnt_mountp= dir;
	look.mnt_fstype= type;
	look.mnt_mntopts= options;
	look.mnt_freq= "0";
	look.mnt_fsckpass= "0";
	look.mnt_time= now;

	/* Unmount everything that matches "look".  Do it in reverse,
	 * because file systems may be mounted on each other.
	 */

	do {
		fflush(nil);

		tab= *(ptab= searchlast(&mtab, &look));
		layered= 0;
		if (tab != nil) {
			look.mnt_special= nil;
			if (searchlast(&mtab, &look) != ptab) layered= 1;
			look.mnt_special= dev;
		}

		/* Use a dummy entry if not listed in mtab. */
		if (tab == nil) {
			if (all) break;

			dummy.ent= look;
			dummy.ent.mnt_special= dev;
			dummy.ent.mnt_mountp= dir;
			dummy.next= nil;
			tab= &dummy;
		}

		tab->stale= 1;	/* Don't find it again. */

		swap= tab->ent.mnt_fstype != nil
			&& strcmp(tab->ent.mnt_fstype, "swap") == 0;

		/* Do I know about this type? */
		if (!swap && tab->ent.mnt_fstype != nil
			&& type2flags(tab->ent.mnt_fstype) == 0
		) {
			if (verbose) {
			    printf("%s not unmounted, type \"%s\" unknown\n",
					found.mnt_special, found.mnt_fstype);
			}
			continue;
		}

		flags= iflags= 0;
		opts2flags(tab->ent.mnt_mntopts, &flags, &iflags);

		/* Swapoff? */
		if (!fake && (swap || (iflags & IMO_SWAP))) {
			if (iflags & IMO_USER) setuid(0);
			r= do_swapoff(tab->ent.mnt_special);
			if (iflags & IMO_USER) setuid(uid);

			if (!r && swap) {
				/* Swapoff failed. */
				continue;
			}
			if (r && (iflags && IMO_SWAP)) {
				/* Remove "swap" option. */
				flags&= ~IMO_SWAP;
				tab->ent.mnt_mntopts= flags2opts(flags, iflags);
				dirty= 1;
			}
		}

		/* Unmount? */
		if (!fake && !swap) {
			char *handle;

			/* Prefere unmounting the directory, use the device
			 * if the mount is layered or no directory is known.
			 */
			if (!layered && tab->ent.mnt_mountp != nil) {
				handle= tab->ent.mnt_mountp;
			} else {
				handle= tab->ent.mnt_special;
			}

			if (iflags & IMO_USER) setuid(0);
			r= umount(handle);
			if (iflags & IMO_USER) setuid(uid);

			if (r < 0) {
				/* It failed, complain (probably busy). */
				report(handle);
				continue;
			}
		}

		if (tab == &dummy) {
			/* Managed to unmount an unlisted device. */
			if (verbose) {
				printf("%s %s\n",
					tab->ent.mnt_special,
					swap ? "removed as swap" : "unmounted");
			}
		} else {
			/* Cut this entry out of mtab. */
			if (verbose) {
				printf("%s %s %s\n",
					tab->ent.mnt_special,
					swap ? "removed as" : "unmounted from",
					tab->ent.mnt_mountp);
			}
			*ptab= tab->next;
			dirty= 1;
		}
	} while (all);
}

char *canonical(char *path)
/* Remove extra pathname separators. */
{
	char *p, *c;

	c= path;

	for (p= path; *p != 0; p++) {
		if (*p == '/' && c > path && c[-1] == '/') continue;
		*c++ = *p;
	}
	if (c > path+1 && c[-1] == '/') c--;
	*c= 0;
	return path;
}

int main(int argc, char **argv)
{
	char **argp;
	struct stat st;
	gid_t groups[NGROUPS_MAX];
	int ngroups;
	int g, i;

	if ((arg0= strrchr(argv[0], '/')) == nil) arg0= argv[0]; else arg0++;

	action= arg0[0] != 'u' ? MOUNT : UMOUNT;

	/* Determine privileges.  Are we allowed to mount anything? */
	if (getgid() == 0) super= 1;
	ngroups= getgroups(NGROUPS_MAX, groups);
	for (g= 0; g < ngroups; g++) if (groups[g] == 0) super= 1;
	if (super) {
		uid= 0;
	} else {
		uid= getuid();
		setuid(uid);
	}

	/* Scan the argument line for options, don't stop at the first
	 * argument, "-r" may be at the end for hysterical reasons.
	 */
	argp= argv+1;
	for (i= 1; i < argc; i++) {
		if (argv[i][0] == '-') {	/* Parse options. */
			char *opt= argv[i] + 1;

			while (*opt != 0) switch (*opt++) {
			case 'm':	action= MOUNT;	break;
			case 'u':	action= UMOUNT;	break;
			case 'a':	all= 1;		break;
			case 'p':	pure= 1;	break;
			case 'f':	fake= 1;	break;
			case 'n':	notab= 1;	break;
			case 'v':	verbose= 1;	break;
			case 'q':	verbose= 0;	break;
			case 'r':	rdonly= 1;	break;
			case 't':
				if (*opt != 0) {
					type= opt;
					opt= "";
				} else {
					if (i == argc) usage();
					type= argv[++i];
				}
				break;
			case 'o':
				if (*opt != 0) {
					options= opt;
					opt= "";
				} else {
					if (i == argc) usage();
					options= argv[++i];
				}
				break;
			default:
				usage();
			}
		} else {	/* Not an option, so it's an argument */
			*argp++= canonical(argv[i]);
		}
	}
	*argp= nil;
	argc= argp - argv;

	if (all && argc > 1) usage();

	if (!super && (fake || notab)) {
		fprintf(stderr,
			"%s: flags -f and -n are for the superuser only\n",
			arg0);
		exit(1);
	}

	/* Timestamp for new mounts. */
	sprintf(now, "%lu", (unsigned long) time((time_t *) nil));

	fstab= getmtab(FSTAB);
	mtab= getmtab(MTAB);

	if (action == MOUNT && argc <= 1 && !all) {
		/* List mtab. */
		mntlist_t *tab;

		for (tab= mtab; tab != nil; tab= tab->next) {
			int flags= 0, iflags= 0;
			opts2flags(tab->ent.mnt_mntopts, &flags, &iflags);
			tell(0, flags, iflags, &tab->ent);
		}
		exit(exstat);
	}

	if (action == MOUNT) {
		switch (argc) {
		case 1:		/* mount */
			mount_action(nil, nil);
			break;
		case 2:
			if (stat(argv[1], &st) >= 0 && S_ISBLK(st.st_mode)) {
				/* mount device */
				mount_action(argv[1], nil);
			} else {
				/* mount directory */
				mount_action(nil, argv[1]);
			}
			break;
		case 3:		/* mount device directory */
			mount_action(argv[1], argv[2]);
			break;
		default:
			usage();
		}
	} else {
		if (options != DEFOPT || rdonly || pure) usage();

		switch (argc) {
		case 1:		/* umount */
			if (!all) usage();
			umount_action(nil, nil);
			break;
		case 2:	
			if (stat(argv[1], &st) >= 0 && S_ISBLK(st.st_mode)) {
				/* umount device */
				umount_action(argv[1], nil);
			} else {
				/* umount directory */
				umount_action(nil, argv[1]);
			}
			break;
		default:
			usage();
		}
	}

	putmtab(MTAB, mtab);
	exit(exstat);
}
